<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Block ghs is defined here.
 *
 * @package     block_ghs
 * @copyright   2019 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

/**
 * ghs block.
 *
 * @package    block_ghs
 * @copyright  2019 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class block_ghs extends block_list {

    /**
     * @var array
     */
    private $courses;

    /**
     * block_ghs constructor.
     * @throws coding_exception
     */
    function __construct() {
        parent::__construct();
        $this->courses = enrol_get_my_courses();
    }

    /**
     * Initializes class member variables.
     */
    public function init() {
        // Needed by Moodle to differentiate between blocks.
        $this->title = get_string('pluginname', 'block_ghs');
    }

    /**
     * Returns the block contents.
     *
     * @return stdClass The block contents.
     */
    public function get_content() {
        global $OUTPUT;

        if ($this->content !== null) {
            return $this->content;
        }

        $this->content = new stdClass();
        $this->content->items = array();
        $this->content->icons = array();
        $this->content->footer = '';

        $contextsystem = context_system::instance();

        $viewsla = has_capability('report/ghs:viewsla', $contextsystem) || $this->has_capability_in_any_course('report/ghs:viewsla');
        $viewgrades = has_capability('report/ghs:viewgrades', $contextsystem) || $this->has_capability_in_any_course('report/ghs:viewgrades');
        $viewenrollment = has_capability('report/ghs:viewenrollment', $contextsystem) || $this->has_capability_in_any_course('report/ghs:viewenrollment');
        $viewgroupenrollment = has_capability('report/ghs:viewgroupenrollment', $contextsystem) || $this->has_capability_in_any_course('report/ghs:viewgroupenrollment');
        $viewviewenddates = has_capability('report/ghs:viewenddates', $contextsystem) || $this->has_capability_in_any_course('report/ghs:viewenddates');
        $viewactivityextensions = has_capability('report/ghs:viewactivityextensions', $contextsystem) || $this->has_capability_in_any_course('report/ghs:viewactivityextensions');
        $viewhsdmoverview = has_capability('report/ghs:viewdmoverview', $contextsystem) || $this->has_capability_in_any_course('report/ghs:viewdmoverview');


        if (!$viewsla && !$viewgrades && !$viewenrollment && !$viewgroupenrollment) {
            return $this->content;
        }

        $reporticon = html_writer::img($OUTPUT->image_url('i/report'), '', array('class' => 'icon icon-inner'));

//        if ($viewsla) {
//            $url = new moodle_url('/report/ghs/ghs_sla.php');
//            $this->content->icons[] = $reporticon;
//            $this->content->items[] = html_writer::link($url, get_string('reportsla', 'block_ghs'));
//        }

        if ($viewgrades) {
            $url = new moodle_url('/report/ghs/ghs_grades.php');
            $this->content->icons[] = $reporticon;
            $this->content->items[] = html_writer::link($url, get_string('reportgrades', 'block_ghs'));
        }

        if ($viewenrollment) {
            $url = new moodle_url('/report/ghs/ghs_enrollment.php');
            $this->content->icons[] = $reporticon;
            $this->content->items[] = html_writer::link($url, get_string('reportenrollment', 'block_ghs'));
        }

        if ($viewgroupenrollment) {
            $url = new moodle_url('/report/ghs/ghs_group_enrollment.php');
            $this->content->icons[] = $reporticon;
            $this->content->items[] = html_writer::link($url, get_string('reportgroupenrollment', 'block_ghs'));
        }

        if ($viewviewenddates) {
            $url = new moodle_url('/report/ghs/ghs_enddate_extension.php');
            $this->content->icons[] = $reporticon;
            $this->content->items[] = html_writer::link($url, get_string('ghsenddateextension', 'block_ghs'));
        }

        if ($viewactivityextensions) {
            $url = new moodle_url('/report/ghs/ghs_activity_extension.php');
            $this->content->icons[] = $reporticon;
            $this->content->items[] = html_writer::link($url, get_string('ghsactivityextension', 'block_ghs'));
        }

        if ($viewhsdmoverview) {
            $url = new moodle_url('/report/ghs/ghs_dm_overview.php');
            $this->content->icons[] = $reporticon;
            $this->content->items[] = html_writer::link($url, get_string('ghsdmoverview', 'block_ghs'));
        }

        return $this->content;
    }

    /**
     * Defines configuration data.
     *
     * The function is called immediatly after init().
     */
    public function specialization() {
        if ($title = get_config('block_ghs', 'title')) {
            $this->title = get_string('pluginname', 'block_ghs');
        } else {
            $this->title = $title;
        }
    }

    /**
     * Allow multiple instances in a single course?
     *
     * @return bool True if multiple instances are allowed, false otherwise.
     */
    public function instance_allow_multiple() {
        return true;
    }

    /**
     * @return bool
     */
    function has_config() {
        return true;
    }

    /**
     * Sets the applicable formats for the block.
     *
     * @return string[] Array of pages and permissions.
     */
    public function applicable_formats() {
        return array(
            'all' => false,
            'site' => true,
            'course-*' => true,
            'my' => true
        );
    }

    /**
     * @param $capability
     * @return bool
     * @throws coding_exception
     */
    public function has_capability_in_any_course($capability) {
        foreach ($this->courses as $course) {
            if (has_capability($capability, context_course::instance($course->id))) {
                return true;
            }
        }
        return false;
    }
}
