<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * @package    block_ned_teacher_tools
 * @copyright  Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

use block_ned_teacher_tools as NED;
use block_ned_teacher_tools\course_users_status;
use block_ned_teacher_tools\deadline_manager;
use block_ned_teacher_tools\output\menu_bar as MB;
use block_ned_teacher_tools\shared_lib as SH;
use block_ned_teacher_tools\cta_tracker;
use local_ned_controller\marking_manager\marking_manager as MM;

require_once($CFG->dirroot . '/user/lib.php');
require_once(__DIR__ . '/lib.php');

/**
 * Simple  class for block ned_teacher_tools
 *
 * @copyright 2011 Moodlefn
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class block_ned_teacher_tools extends block_list {

    /**
     * An object to contain the information to be displayed in the block.
     * @var block_ned_teacher_tools_block_content $content
     */
    var $content = NULL;
    protected $_config;
    /** @var MB */
    protected $_MB;
    /** @var MM */
    protected $_MM;

    /**
     * Sets the block title
     *
     * @return void
     */
    public function init(){
        $this->title = SH::str('blocktitle');
    }

    /**
     * @return MB
     */
    public function get_MB(){
        if (is_null($this->_MB)){
            $this->_MB = new MB($this->page->url->out(), [], $this->page->course->id);
        }
        return $this->_MB;
    }

    /**
     * @return MM
     */
    public function get_MM(){
        if (is_null($this->_MM)){
            $this->_MM = MM::get_MM_by_params($this->get_MB()->parameters_for_MM(['cmstudents' => false]));
        }
        return $this->_MM;
    }

    /**
     * Controls the block title based on instance configuration
     *
     * @return void
     */
    public function specialization(){
        $this->_config = SH::get_site_and_course_block_config($this->page->course->id, SH::TT_NAME);
        if (!isset($this->config)){
            $this->config = new stdClass;
        }

        if ($this->page->context->contextlevel == CONTEXT_COURSE){
            $this->_check_redirects();
        }

        if (!isset($this->config->customlinks)){
            $this->config->customlinks = '';
        }

        $this->title = $this->config->block_title ?? ($this->_config->blocktitlecourselevel ?: SH::str('blocktitle'));
        if (($this->page->course->id == SITEID) || ($this->instance->pagetypepattern == 'my-index')){
            $this->title = SH::span($this->title, 'title');
        }
        // Default setting values.
        if ($this->page->course->id == SITEID || !$this->_config->numberoflinks) return;

        $numberoflinks = $this->_config->numberoflinks;
        for ($i = 0; $i <= $numberoflinks; $i++){
            $var = 'sitelink_'.$i;
            if (!isset($this->config->$var)){
                $this->config->$var = 1;
            }
        }
    }

    /**
     * Subclasses should override this and return true if the
     * subclass block has a settings.php file.
     *
     * @return boolean
     */
    public function has_config(){
        return true;
    }

    /**
     * Controls the block title based on instance configuration
     *
     * @return bool
     */
    public function instance_allow_config(){
        return true;
    }

    /**
     * Return the block content
     *
     * @return object - block content
     */
    public function get_content(){
        if (empty($this->content)){
            $this->content = new stdClass;
            $this->content->items = array();
            $this->content->icons = array();
            $this->content->text = '';
            $this->content->footer = '';

            try {
                $this->_get_content();
            } catch (\Throwable $ex){
                $this->content->items[] = '';
                $this->content->footer = NED\print_hidden_error($ex, [], SH::div(SH::str('blockcontenterror'), 'alert-error'));
            }
        }

        return $this->content;
    }

    /**
     * If we need - redirect user somewhere else
     *
     * @return void
     */
    protected function _check_redirects(){
        global $SCRIPT;
        if ($SCRIPT == SH::PAGE_COURSE && $this->get_MB()->has_group_selected() && cta_tracker::is_responsible_grader($this->page->course->id)){
            $cache = SH::get_user_cache();
            $cache_key = SH::CACHE_USERS_KEY_CTA_COURSE_CHECK.$this->page->course->id.'_'.$this->get_MB()->groupid;
            $last_cta_check = $cache->get($cache_key) ?: 0;
            $now = time();
            if (($last_cta_check + HOURSECS) < $now){
                $cache->set($cache_key, $now);
                $MM = $this->get_MM();
                $filter = [
                    MM::TAG_NAME_LIST_HAVE_ANY => SH::TAG_CTA,
                    MM::SUM => true,
                    MM::ST_UNMARKED => true,
                ];
                $MM->filter_hide_grades_before_midn_if_need($filter);

                /** @var \local_ned_controller\marking_manager\mm_data_by_activity[] $mm_data */
                $mm_data = $MM->get_data($filter, false, cta_tracker::CM_TYPES);
                foreach (cta_tracker::get_render()::MODS_ALL as $mod => $cm_key){
                    if (!empty($mm_data[$mod]->unmarked)){
                        SH::redirect($this->get_MB()->get_url(MB::PAGE_CTA, [SH::PAR_CM => $cm_key]));
                        die;
                    }
                }
            }
        }
    }

    /**
     * Creates the real blocks main content
     *
     * @return object
     */
    protected function _get_content(){
        // Need the bigger course stdClass.
        if (($this->page->course->id != SITEID) && ($this->instance->pagetypepattern != 'my-index') ){
            $this->_get_standard_content();
        }

        return $this->content;
    }

    /**
     * Defines where the block can be added
     *
     * @return array
     */
    public function applicable_formats(){
        // Default case: the block can be used in all course types.
        return array(
            'all'      => false,
            'site'     => true,
            'course-*' => true,
            'my'       => true,
        );
    }

    /**
     * Return any HTML attributes that you want added to the outer <div> that
     * of the block when it is output.
     *
     * Because of the way certain JS events are wired it is a good idea to ensure
     * that the default values here still get set.
     * I found the easiest way to do this and still set anything you want is to
     * override it within your block in the following way
     *
     * <code php>
     * function html_attributes(){
     *    $attributes = parent::html_attributes();
     *    $attributes['class'] .= ' mynewclass';
     *    return $attributes;
     * }
     * </code>
     *
     * @return array attribute name => value.
     */
    public function html_attributes(){
        $attributes = parent::html_attributes();
        $attributes['class'] .= ' path-blocks-NED block-NED';
        return $attributes;
    }

    /**
     * Function to return the standard content, used in all versions.
     */
    protected function _get_standard_content(){
        global $CFG, $DB;
        require_once($CFG->dirroot.'/mod/forum/lib.php');

        $MB = $this->get_MB();
        if (!$MB->isteacher){
            return '';
        }
        $MM = $this->get_MM();
        $showtitles = $this->_config->showtitles ?? false;
        $context = $MB->context;

        $showcourselink = $this->_config->showcourselink ?? false;

        if ($MB->courseid == SITEID){
            return $this->content;
        }

        $sections = $DB->get_records('course_sections', ['course' => $MB->courseid], 'section ASC', 'section, sequence');

        // Settings.
        $div_line = SH::div('<hr />', '', ['style' => 'width:156px;']);
        $this->render_group_selector_and_show_inactive_checkbox($MB->groupid, $MB->allgroups, $MB->show_inactive);

        if ($showcourselink){
            $this->content->items[] = SH::link(['/course/view.php', ['id' => $MB->courseid]], $MB->course->shortname, 'course-link');
            $this->content->icons[] = SH::img('i/course');
        }

        // MARKING MANAGER
        if ($showtitles){
            $this->content->items[] = SH::div(SH::$C::str('markingmanager'), 'block-subtitle first');
            $this->content->icons[] = '';
        } else {
            if ($showcourselink){
                $this->content->items[] = $div_line;
                $this->content->icons[] = '';
            }
        }

        $activity_options = MM::SELECT_STATUS;

        // check, what conditions we really check
        foreach ($activity_options as $key => $status){
            $conf_opt = 'show'.$status;
            if (!isset($this->_config->$conf_opt)){
                unset($activity_options[$key]);
            }
        }

        // calculate
        $mm_data = null;
        if ($MB->act && !empty($activity_options)){
            $filter = $activity_options;
            $MM->filter_hide_grades_before_midn_if_need($filter);
            $filter[MM::SUM] = true;
            if (!empty($this->_config->excludewithoutsla)){
                $filter[MM::TAG_NAME_LIST_HAVE_ANY] = SH::SLA_TAGS;
            }
            $mm_data = $MM->get_data_all($filter);
        }

        // show
        $fn_params = ['courseid' => $MB->courseid, 'navlevel' => 'top'];
        foreach ($activity_options as $status){
            $num = $mm_data ? $mm_data->$status : '';
            $this->content->items[] = SH::link(['~/marking_manager.php', array_merge($fn_params, ['show' => $status])],
                $num.' '.SH::$C::str($status));
            $this->content->icons[] = SH::div(SH::icon_get_nge_by_status($status), 'fa');
        }

        // QUICK LINKS
        if ($showtitles){
            $this->content->items[] = SH::div(SH::str('quicklinks'), 'block-subtitle');
        } else {
            $this->content->items[] = $div_line;
        }
        $this->content->icons[] = '';

        if ($MB->is_show_page($MB::PAGE_CP)){
            $this->content->items[] = $MB->get_menu_link(MB::PAGE_CP);
            $this->content->icons[] = SH::fa('fa-check-square-o');
        }

        if ($MB->is_show_page($MB::PAGE_SP)){
            $this->content->items[] = $MB->get_menu_link(MB::PAGE_SP);
            $this->content->icons[] = SH::fa('fa-columns');
        }

        $kicaerrors = '';
        if ($MB->kicainstalled && $MB->act && has_capability('moodle/grade:edit', $context)){
            $kica_stats = block_ned_teacher_tools_get_kica_activity_stats($MB->course, $MB->groupid);

            $numberofactivities = $kica_stats['incomplete'] + $kica_stats['a_mismatch'] + $kica_stats['a_flagged'];
            if ($numberofactivities > 0){
                $kicaerrors .= SH::fa('fa-exclamation-triangle text-warning',
                    SH::link(['/local/kica/notifications.php', ['courseid' => $this->page->course->id]],
                        $numberofactivities, ['class' => 'color-red'])
                );
            }

            if ($kica_stats['a_ungraded'] > 0){
                $kicaerrors .= SH::fa('fa-clock-o',
                    SH::link(['/local/kica/notifications.php', ['courseid' => $this->page->course->id]],
                        $kica_stats['u_ungraded'], ['class' => 'color-red']
                    )
                );
            }
        }

        if ($MB->is_show_page($MB::PAGE_GB)){
            $this->content->items[] = $MB->get_menu_link(MB::PAGE_GB);
            $this->content->icons[] = SH::fa('fa-table');
        }

        if ($MB->is_show_page($MB::PAGE_KICA)){
            $this->content->items[] = $MB->get_menu_link(MB::PAGE_KICA).
                ' '.SH::span($kicaerrors, 'color-red');
            $this->content->icons[] = SH::fa('fa-table');
        }

        // Deadline manager.
        if ($MB->is_show_page($MB::PAGE_DM)){
            $deadlinemanagerwarnings = '';
            if ($MB->act && SH::has_capability('view_own_groups', $context)){
                $deadlinemanager = new deadline_manager($MB->courseid);
                if ($deadlinemanager->has_missed_schedule($MB->groupid)){
                    $deadlinemanagerwarnings .= SH::fa('fa-exclamation-triangle text-warning');
                    $reviewurl = $MB->get_url(MB::PAGE_DM);
                    $this->content->footer = SH::str('missedschedulefooter', $reviewurl->out(false));
                }
            }

            $this->content->items[] = $MB->get_menu_link(MB::PAGE_DM).
                ' '.SH::span($deadlinemanagerwarnings, 'color-red');
            $this->content->icons[] = SH::fa('fa-calendar');
        }

        // CTA Tracker
        if ($MB->is_show_page($MB::PAGE_CTA)){
            $this->content->items[] = $MB->get_menu_link(MB::PAGE_CTA);
            $this->content->icons[] = SH::fa('fa-life-ring');
        }

        // Proxy overview.
        if (SH::is_proxy_local_exists()){
            if (has_capability('assignsubmission/proxy:manage', $context)){
                $this->content->items[] = SH::link(['/local/proxy/overview.php', ['id' => $MB->courseid]],
                    get_string('proxysubmissions', SH::PROXY_LOCAL));
                $this->content->icons[] = SH::fa('fa-folder');
            }
        }

        // CT submissions.
        if (SH::is_ctsubmission_exists()){
            if (has_any_capability(['assignsubmission/ctsubmission:submit', 'assignsubmission/ctsubmission:approve'], $context)){
                if ($activities = \assignsubmission_ctsubmission\helper::get_ctsubmission_assignments($MB->courseid)){
                    foreach ($activities as $activity){
                        $cm = SH::get_cm_by_params($activity->course, SH::ASSIGN, $activity->assignment);
                        $this->content->items[] = SH::link(['/mod/assign/submission/ctsubmission/overview.php',
                            ['courseid' => $MB->courseid, 'cmid' => $cm->id]], $activity->name);
                        $this->content->icons[] = SH::fa('fa-cloud-upload');
                    }
                }
            }
        }

        // Custom links.
        if ($custommenuitems = $this->render_custom_links($this->config->customlinks)){
            $numberofcourselinks = count($custommenuitems);
        } else {
            $numberofcourselinks = 0;
        }

        $numberofsitelinks = $this->_config->numberoflinks;
        $enabledsitelinks = 0;

        if ($numberofsitelinks){
            for ($i = 1; $i <= $numberofsitelinks; $i++){
                $var = 'sitelink_'.$i;
                $enabledsitelinks += (int)$this->config->$var;
            }
        }

        $links_items = $links_icons = [];
        if ($numberofcourselinks){
            foreach ($custommenuitems as $custommenuitem){
                $cls = '';
                if ($custommenuitem->itembehavior == '_popup'){
                    $cls = 'popup';
                }
                if (!empty($custommenuitem->itemtitle)){
                    $link = SH::link([$custommenuitem->itemurl], $custommenuitem->itemtitle,
                        $cls, ['target' => $custommenuitem->itembehavior]);
                    $links_items[] = format_text($link, FORMAT_HTML);
                    $links_icons[] = SH::fa($custommenuitem->itemicon);
                }
            }
        }

        if ($enabledsitelinks){
            for ($i = 1; $i <= $numberofsitelinks; $i++){
                $var = 'sitelink_'.$i;
                if (!isset($this->_config->{'iconcode_'.$i})){
                    continue;
                }
                $icon = $this->_config->{'iconcode_'.$i};
                $target = $this->_config->{'linkbehaviour_'.$i};
                $url = $this->_config->{'customlinkurl_'.$i};
                $title = $this->_config->{'customlinkstitle_'.$i};

                $cls = '';
                if ($target == '_popup'){
                    $cls = 'popup';
                }
                if (!empty($title) && (int)$this->config->$var){
                    $links_items[] = SH::link([$url], $title, $cls, ['target' => $target]);
                    $links_icons[] = SH::fa($icon);
                }
            }
        }

        if (count($links_items) > 0){
            $this->content->items[] = SH::div($this->_config->customlinkstitle, 'block-subtitle');
            $this->content->icons[] = '';
            $this->content->items = array_merge($this->content->items, $links_items);
            $this->content->icons = array_merge($this->content->icons, $links_icons);
        } else {
            $this->content->items[] = $div_line;
            $this->content->icons[] = '';
        }

        // NOTICES
        if (!$MB->act) return $this->content;

        $daysnotlogged = $this->_config->daysnotlogged;
        $daysnotsubmited = $this->_config->daysnotsubmited;
        $percent = $this->_config->setpercentmarks;

        $showstudentnotsubmittedassignment = $this->_config->showstudentnotsubmittedassignment;
        $showstudentmarkslessthanfiftypercent = $this->_config->showstudentmarkslessthanfiftypercent;

        $now = time();
        $lastweek = $now - (60 * 60 * 24 * $daysnotsubmited);
        $numnotsubmittedany = $numfailing = 0;
        if ($showstudentnotsubmittedassignment){
            $numnotsubmittedany =
                block_ned_teacher_tools_get_notsubmittedany($MB->course, $lastweek, true, $sections, null, $MB->groupid);
            $showstudentnotsubmittedassignment = (bool)$numnotsubmittedany;
        }
        if ($showstudentmarkslessthanfiftypercent){
            $numfailing = block_ned_teacher_tools_count_failing($MB->course, $percent, $MB->groupid);
            $showstudentmarkslessthanfiftypercent = (bool)$numfailing;
        }

        if (!$showstudentnotsubmittedassignment && !$showstudentmarkslessthanfiftypercent) return $this->content;

        if ($showtitles){
            $this->content->items[] = SH::div(SH::str('notices'), 'block-subtitle');
        } else {
            $this->content->items[] = $div_line;
        }
        $this->content->icons[] = '';

        $strstudents = get_string('students');
        $exclamation_icon = SH::img('exclamation.png');
        $params = ['id' => $MB->courseid, 'navlevel' => 'top', 'days' => $daysnotlogged, 'daysnotsubmited' => $daysnotsubmited];

        if ($showstudentnotsubmittedassignment){
            $this->content->items[] = SH::span(
                SH::link(['~/fn_summaries.php', array_merge($params, ['show' => 'notsubmittedany'])],
                    "$numnotsubmittedany $strstudents").
                SH::str('notsubmittedany', $daysnotsubmited),
                'fn_summaries');
            $this->content->icons[] = $exclamation_icon;
        }

        if ($showstudentmarkslessthanfiftypercent){
            $this->content->items[] = SH::span(
                SH::link(['~/fn_summaries.php', array_merge($params, ['show' => 'failing', 'percent' => $percent])],
                    "$numfailing $strstudents").
                SH::str('overallfailinggrade', $percent),
                'fn_summaries');
            $this->content->icons[] = $exclamation_icon;
        }

        return $this->content;
    }

    /**
     * @param $customlinkstext
     *
     * @return array
     */
    public function render_custom_links($customlinkstext){
        $customlinks = array();

        $lines = explode("\n", $customlinkstext);
        foreach ($lines as $linenumber => $line){
            $line = trim($line);
            if (empty($line)){
                continue;
            }
            // Parse item settings.
            $linkitem = new stdClass();
            $linkitem->itemicon = null;
            $linkitem->itemtext = null;
            $linkitem->itemurl = null;
            $linkitem->itembehavior = null;

            $linkbehaviouroptions = array(
                'new-window' => '_blank',
                'same-window' => '_self',
                'pop-up' => '_popup'
            );

            $settings = explode(',', $line);
            foreach ($settings as $i => $setting){
                $setting = trim($setting);
                if (!empty($setting)){
                    switch ($i){
                        case 0:
                            $linkitem->itemicon = $setting;
                            break;
                        case 1:
                            $linkitem->itemtext = $setting;
                            $linkitem->itemtitle = $linkitem->itemtext;
                            break;
                        case 2:
                            if ($setting == 'auto-link'){
                                // Find course module.
                                if ($activityurl = $this->activity_link($linkitem->itemtext)){
                                    $linkitem->itemurl = $activityurl;
                                } else {
                                    $linkitem->itemurl = '';
                                }
                            } else if ($setting == 'section-link'){
                                // Find section.
                                if ($sectionurl = $this->section_link($linkitem->itemtext)){
                                    $linkitem->itemurl = $sectionurl;
                                } else {
                                    $linkitem->itemurl = '';
                                }
                            } else {
                                $linkitem->itemurl = $setting;
                            }
                            break;
                        case 3:
                            if (isset($linkbehaviouroptions[$setting])){
                                $linkitem->itembehavior = $linkbehaviouroptions[$setting];
                            } else {
                                $linkitem->itembehavior = '_blank';
                            }
                            break;
                    }
                }
            }
            $customlinks[] = $linkitem;
        }
        return $customlinks;
    }

    /**
     * @param $activityname
     *
     * @return false|string
     */
    public function activity_link($activityname){
        $courseid = $this->page->course->id;
        if (!SH::ctx($courseid)){
            return $activityname;
        }

        $modinfo = get_fast_modinfo($courseid);
        if (!empty($modinfo->cms)){
            foreach ($modinfo->cms as $cm){
                // Use normal access control and visibility, but exclude labels and hidden activities.
                if ($cm->visible and $cm->has_view() and $cm->uservisible){
                    if (trim($cm->name) == $activityname){
                        $url = new moodle_url('/mod/'.$cm->modname.'/view.php', array('id'=>$cm->id));
                        return $url->out();
                    }
                }
            }
        }
        return false;
    }

    /**
     * @param $sectionname
     *
     * @return false|string
     */
    function section_link($sectionname){
        $courseid = $this->page->course->id;
        // Activity selection.
        $modinfo = get_fast_modinfo($courseid);
        $sections = $modinfo->get_section_info_all();
        $counter = 0;

        foreach ($sections as $section){
            if (trim($section->__get('name'))){
                $sectiontitle = trim($section->__get('name'));
            } else {
                $sectiontitle = get_string('defaultsectionname', 'nedactivitycluster', $section->section);
            }
            if (trim($sectiontitle) == $sectionname){
                $url = new moodle_url('/course/view.php', array('id'=>$courseid, 'section' => $counter));
                return $url->out();
            }
            $counter++;
        }
        return false;
    }

    /**
     * @param $groupid
     * @param $allowedgroups
     * @param $show_inactive
     */
    function render_group_selector_and_show_inactive_checkbox($groupid, $allowedgroups, $show_inactive){
        $course = $this->page->course;
        $course_context = context_course::instance($course->id);

        $select = '';
        if (!empty($allowedgroups)){
            $select .= MB::group_selector($this->page->url, $allowedgroups, $groupid, MB::PAR_GROUP_BLOCK, null, $show_inactive);
        }

        $output = SH::div_start('tt-box-group');
        $output .= $select;
        $output .= SH::div(SH::fa('fa-user-o'), 'icon column c0');

        [$count_active,$count_inactive] = $this->get_count_active_and_inactive_by_course_and_group($course,$course_context,$groupid);

        $role_names = NED\get_role_names_by_course_and_rolename($course->id, 'student');
        $text_link = 'students';
        if (!empty($role_names))
            $text_link = $role_names;

        $text_status = '';
        $icon_off = $icon_on = SH::div(SH::fa('fa-user') . SH::fa('fa-folder'), 'tt-icon-user-folder');
        $icon_off .= SH::span($count_inactive, 'tt-icon-user-folder-count');
        if (!$show_inactive){
            $text_status .= SH::str('activestudents', ['active' => $count_active,'role' => $text_link]);
        } else {
            $text_status .= SH::str('inactivestudents', ['active' => $count_active, 'inactive' => $count_inactive,'role' => $text_link]);
        }
        $user_params = ['contextid' => $course_context->id];
        if (!empty($groupid)){
            $user_params["group"] = $groupid;
        }
        $us_user_params = $user_params;
        $ct_user_params = $user_params;
        $ot_user_params = $user_params;
        $roles = get_all_roles();
        foreach ($roles as $role){
            if ($role->shortname == 'student' && $role->archetype == 'student'){
                $us_user_params["roleid"] = $role->id;
            } elseif ($role->shortname == course_users_status::ROLE_CT_SHORTNAME){
                $ct_user_params["roleid"] = $role->id;
            } elseif ($role->shortname == course_users_status::ROLE_OT_SHORTNAME){
                $ot_user_params["roleid"] = $role->id;
            }
        }

        $checkbox_status = SH::single_checkbox2($this->page->url, $show_inactive, MB::PAR_SHOWINACTIVE_BLOCK, $icon_on, $icon_off, 'show-inactive-checkbox',
            SH::str('hideinactiveusers'), SH::str('showinactiveusers'));
        $text_status = SH::link(['/user/index.php', $us_user_params], $text_status) . $checkbox_status;
        $output .= SH::div($text_status, 'column c1');

        if ($text_ct = $this->render_show_count_teachers_by_course_and_group($course->id, $course_context->id, $groupid, course_users_status::ROLE_CT_SHORTNAME)){
            $output .= SH::div(SH::div(SH::str('ct'), 'tt-role-icon icon'), 'column c0 icon');
            $output .= SH::div(SH::link(['/user/index.php', $ct_user_params], $text_ct), 'column c1');
        }

        if ($text_ot = $this->render_show_count_teachers_by_course_and_group($course->id, $course_context->id, $groupid)){
            $output .= SH::div(SH::div(SH::str('ot'), 'tt-role-icon icon'), 'column c0 icon');
            $output .= SH::div(SH::link(['/user/index.php', $ot_user_params], $text_ot), 'column c1');
        }

        $output .= SH::div(SH::fa('fa-user-o'), 'icon column c0');
        $output .= SH::div(SH::link(['/user/index.php', $user_params], 'listallusers'), 'column c1');

        $output .= SH::div_end(); // tt-box-group
        $this->content->items[] = SH::div($output, 'block-group');
        $this->content->icons[] = '';
    }

    /**
     * @param $courseid
     * @param $context_course_id
     * @param $group_id
     * @param string $rolename
     *
     * @return lang_string|string
     */
    function render_show_count_teachers_by_course_and_group($courseid, $context_course_id, $group_id, $rolename = course_users_status::ROLE_OT_SHORTNAME){
        $teachers = NED\get_group_teachers_by_course_context($courseid, $context_course_id, $group_id, $rolename);
        if (count($teachers) == 1){
            $teachers = reset($teachers);
            $text = fullname($teachers, true);
        } elseif (($count_teachers = count($teachers)) > 1){
            if ($rolename == course_users_status::ROLE_OT_SHORTNAME){
                $text = SH::str('countonlineteacher', $count_teachers);
            } elseif ($rolename == course_users_status::ROLE_CT_SHORTNAME){
                $text = SH::str('countclassroomteacher', $count_teachers);
            } else {
                $text = SH::str('multiple');
            }
        } else {
            return false;
        }
        return $text;
    }

    /**
     * @param $course
     * @param $course_context
     * @param $groupid
     *
     * @return int[]
     */
    function get_count_active_and_inactive_by_course_and_group($course,$course_context,$groupid){
        $tt_users_and_groups = block_ned_teacher_tools_get_users_and_groups($course, false, false);
        $suspended_users = get_suspended_userids($course_context);
        $id_users = 0;
        $id_groups = 1;
        $count_inactive = 0;
        $count_active = 0;
        $tt_users = $tt_users_and_groups[$id_users];
        if (!empty($groupid)){
            if (isset($tt_users_and_groups[$id_groups][$groupid]))
                $tt_users = $tt_users_and_groups[$id_groups][$groupid]->users;
        }
        foreach ($tt_users as $us){
            if ($us->suspended || isset($suspended_users[$us->id])){
                $count_inactive++;
            } else {
                $count_active++;
            }
        }
        return [$count_active,$count_inactive];
    }

    /**
     * Serialize and store config data
     *
     * @param      $data
     * @param bool $nolongerused
     */
    public function instance_config_save($data, $nolongerused=false){
        parent::instance_config_save($data, $nolongerused);
    }
}
