<?php
/**
 * Activity status class
 * Part of common lib, be careful when add something, that other plugin may have not
 *
 * @package    block_ned_teacher_tools
 * @subpackage NED
 * @copyright  2020 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace block_ned_teacher_tools;
use local_ned_controller\marking_manager\marking_manager as MM;
use block_ned_teacher_tools\shared_lib as SH;

include_once(__DIR__ . '/../common_lib.php');

const USUAL_ACTIVITY = 0;
const FORMATIVE_ACTIVITY = 1;
const SUMMATIVE_ACTIVITY = 2;

/**
 * Class activity_status
 *
 * @package block_ned_teacher_tools
 */
class activity_status
{
    const STATUS_ALL = SH::STATUS_ALL;
    const STATUS_EXCLUDED = SH::STATUS_EXCLUDED;
    const STATUS_COMPLETED = SH::STATUS_COMPLETED;
    const STATUS_PASSED = SH::STATUS_PASSED;
    const STATUS_INCOMPLETED = SH::STATUS_INCOMPLETED;
    const STATUS_DRAFT = SH::STATUS_DRAFT;
    const STATUS_NOTATTEMPTED = SH::STATUS_NOTATTEMPTED;
    const STATUS_WAITINGFORGRADE = SH::STATUS_WAITINGFORGRADE;
    const STATUS_GRADED_KICA = SH::STATUS_GRADED_KICA;
    const STATUS_GRADED_KICA_SUMMATIVE = SH::STATUS_GRADED_KICA_SUMMATIVE;
    const STATUS_GRADED_KICA_FORMATIVE = SH::STATUS_GRADED_KICA_FORMATIVE;
    const STATUS_UNGRADED_KICA = SH::STATUS_UNGRADED_KICA;
    const STATUS_UNGRADED_KICA_SUMMATIVE = SH::STATUS_UNGRADED_KICA_SUMMATIVE;
    const STATUS_UNGRADED_KICA_FORMATIVE = SH::STATUS_UNGRADED_KICA_FORMATIVE;
    const STATUS_INCOMPLETED_SUMMATIVE = SH::STATUS_INCOMPLETED_SUMMATIVE;
    const STATUS_INCOMPLETED_FORMATIVE = SH::STATUS_INCOMPLETED_FORMATIVE;
    const STATUS_KICA_ZEROGRADE = SH::STATUS_KICA_ZEROGRADE;
    const STATUS_AWARD_ZERO = SH::STATUS_NGC_ZERO;

    const PP_CATEGORIES = [self::STATUS_COMPLETED, self::STATUS_WAITINGFORGRADE, self::STATUS_INCOMPLETED];

    private $fs_activity_status = [FORMATIVE_ACTIVITY => null, SUMMATIVE_ACTIVITY => null];
    private $empty_list = [];

    protected $status_list = [];    // [activity_id => $show]
    protected $fs_status_list = []; // [activity_id => *_ACTIVITY]
    protected $offlineoverridestatus = [];  // // [activity_id => ['gradeoverride' => 1, 'offlinegraded' => 1]]
    protected $fs_activities = [FORMATIVE_ACTIVITY => 0, SUMMATIVE_ACTIVITY => 0]; // count of fs activities
    protected $tags_list = [];  // [activity_id => [tag_id => tag_name]]
    protected $all_tags = [];   // [tag_id => tag_name]


    public $all = 0;
    public $excluded = 0;
    public $completed = 0;
    public $passed = 0;
    public $incompleted = 0;
    public $draft = 0;
    public $notattempted = 0;
    public $waitingforgrade = 0;
    public $graded_kica = 0;
    public $graded_kica_summative = 0;
    public $graded_kica_formative = 0;
    public $ungraded_kica = 0;
    public $ungraded_kica_summative = 0;
    public $ungraded_kica_formative = 0;
    public $incompleted_summative = 0;
    public $incompleted_formative = 0;
    public $kica_zerograde = 0;
    public $ngc_zero = 0;

    public function __construct($use_participation_power=true){
        if ($use_participation_power){
            $this->fs_activity_status = [
                FORMATIVE_ACTIVITY => new self(false),
                SUMMATIVE_ACTIVITY => new self(false)
            ];
        }
        $this->empty_list = $this->get_list();
    }

    /**
     * @param                $key
     * @param \stdClass|null $mm_data
     *
     * @return array
     */
    public static function precheck($key, $mm_data=null){
        $score_type = USUAL_ACTIVITY;
        if ($mm_data){
            $tags = explode(',', $mm_data->tags);
            if ($mm_data->is_summative){
                $score_type = SUMMATIVE_ACTIVITY;
            } elseif ($mm_data->is_formative){
                $score_type = FORMATIVE_ACTIVITY;
            }
        } else {
            $tags = get_tags($key);
            if (in_array(TAG_SUMMATIVE, $tags) !== false){
                $score_type = SUMMATIVE_ACTIVITY;
            } elseif(in_array(TAG_FORMATIVE, $tags) !== false){
                $score_type = FORMATIVE_ACTIVITY;
            }
        }

        return [$score_type, $tags];
    }

    /**
     * @param \cm_info|\stdClass $mod
     * @param int                $userid
     * @param bool               $count_only_pp
     * @param \stdClass|null     $use_kica
     * @param \stdClass|null     $mm_data
     * @param bool               $activity_completion - if null, we calculate it here
     *
     * @return string|null
     */
    public function check_activity($mod, $userid=0, $count_only_pp=false, $use_kica=null, $mm_data=null, $activity_completion=null){
        global $USER;
        $courseid = $mod->course;
        $key = $mod->id;
        if (!$userid) {
            $userid = $USER->id;
        }

        list($score_type, $tags) = self::precheck($key, $mm_data);
        $this->fs_status_list[$key] = $score_type;

        if ($count_only_pp && !$score_type){
            return null;
        }

        do {
            // if we have mm data, check by it
            if ($mm_data){
                $this->status_list[$key] = SH::get_status_by_mm_data($mm_data);
                break;
            }

            $NGC = SH::$ned_grade_controller;
            $ngc_records = $NGC::get_records_by_params($mod->id, $userid);
            if (!empty($ngc_records)){
                foreach ($ngc_records as $ngc_record){
                    if ($ngc_record->status == $NGC::ST_DONE && $ngc_record->reason == $NGC::GT_AWARD_ZERO){
                        $this->status_list[$key] = static::STATUS_AWARD_ZERO;
                        break 2;
                    }
                }
            }

            $use_activity_completion = $mod->completion != COMPLETION_TRACKING_NONE;

            if ($use_activity_completion && is_null($activity_completion)){
                $options = ['courseid' => $courseid, 'cmid' => $key, 'userid' => $userid];
                $activity_completion = get_course_modules_completion($options, true, true);
            }
            // check $activity_completion
            if ($use_activity_completion && $activity_completion){
                $this->status_list[$key] = static::STATUS_COMPLETED;
                break;
            }

            // default status: Ungraded (Unmarked)
            $this->status_list[$key] = static::STATUS_NOTATTEMPTED;
            $instance = get_instance($mod->modname,  $mod->instance);
            if (!(($mod->modname != 'forum') || ($instance->assessed > 0))){
                if ($use_activity_completion){
                    break;
                }
                return null;
            }

            if (!$item = get_grade_item($courseid, $mod->modname, $mod->instance)) {
                $item = new \stdClass();
                $item->id = $mod->id;
                $item->gradepass = 0;
                $item->grademax = 0;
            }

            $gradefunction = get_grade_function($mod);
            if (!$gradefunction){
                if ($use_activity_completion){
                    break;
                }
                return null;
            }

            $rawgrade = false;

            if ($mod->modname == SH::QUIZ || $mod->modname == SH::FORUM || $mod->modname == SH::JOURNAL) {
                $grade = $gradefunction($instance, $userid);

                if ($mod->modname == SH::JOURNAL){
                    if (isset($grade[$userid]) && !is_numeric($grade[$userid]->rawgrade)){
                        $this->status_list[$key] = static::STATUS_WAITINGFORGRADE;
                        break;
                    }
                }

                $rawgrade = $grade[$userid]->rawgrade ?? false;
            } elseif ($modstatus = assignment_status($mod, $userid)){
                switch ($modstatus){
                    case 'offlinegraded':
                        $this->offlineoverridestatus[$key]['offlinegraded'] = 1;
                        break;
                    case 'gradeoverride':
                        $this->offlineoverridestatus[$key]['gradeoverride'] = 1;
                        break;
                }

                switch ($modstatus) {
                    case SH::STATUS_SUBMITTED:
                    case 'offlinegraded':
                    case 'gradeoverride':
                    case 'graded':
                        $rawgrade = gradebook_grade($item->id, $userid);
                        break;
                    case SH::STATUS_DRAFT:
                    case 'saved':
                        // Saved
                        $this->status_list[$key] = static::STATUS_DRAFT;
                        break;
                    case SH::STATUS_WAITINGFORGRADE:
                    case 'waitinggrade':
                        // Waitinggrade
                        $this->status_list[$key] = static::STATUS_WAITINGFORGRADE;
                        break;
                    default:
                        trigger_error ("Unknown assignment status \"$modstatus\".", E_USER_NOTICE);
                        $this->status_list[$key] = static::STATUS_NOTATTEMPTED;
                }
            }

            if ($rawgrade !== false){
                // check kica
                if ($use_kica && $kica_item = SH::ki_get_by_cm_and_course($mod, $courseid)) {
                    if (!$use_activity_completion && $grade = get_kica_is_graded($userid, $kica_item->id)) {
                        $this->status_list[$key] = static::STATUS_GRADED_KICA;
                    } else {
                        $this->status_list[$key] = static::STATUS_INCOMPLETED;
                    }
                    break;
                } elseif ($item->gradepass > 0){
                    if ($use_activity_completion){
                        $this->status_list[$key] = static::STATUS_INCOMPLETED;
                    } else {
                        if ($rawgrade >= $item->gradepass){
                            // Passed.
                            $this->status_list[$key] = static::STATUS_COMPLETED;
                        } else {
                            // Fail.
                            $this->status_list[$key] = static::STATUS_INCOMPLETED;
                        }
                    }
                } else {
                    if ($use_activity_completion){
                        $this->status_list[$key] = static::STATUS_NOTATTEMPTED;
                    } else {
                        // Graded (grade-to-pass is not set).
                        $this->status_list[$key] = static::STATUS_COMPLETED;
                    }
                }
            }

        } while (false);

        if (!$mm_data){
            if ($score_type == SUMMATIVE_ACTIVITY || $score_type == FORMATIVE_ACTIVITY){
                $is_summative = $score_type == SUMMATIVE_ACTIVITY;
                switch ($this->status_list[$key]){
                    case static::STATUS_COMPLETED:
                    case static::STATUS_PASSED:
                    case static::STATUS_GRADED_KICA:
                        $this->status_list[$key] = $is_summative ? static::STATUS_GRADED_KICA_SUMMATIVE : static::STATUS_GRADED_KICA_FORMATIVE;
                        break;
                    case static::STATUS_WAITINGFORGRADE:
                    case static::STATUS_UNGRADED_KICA:
                        $this->status_list[$key] = $is_summative ? static::STATUS_UNGRADED_KICA_SUMMATIVE : static::STATUS_UNGRADED_KICA_FORMATIVE;
                        break;
                    case static::STATUS_INCOMPLETED:
                        $this->status_list[$key] = $is_summative ? static::STATUS_INCOMPLETED_SUMMATIVE : static::STATUS_INCOMPLETED_FORMATIVE;
                        break;
                }
            }
        }

        if (isset($this->status_list[$key])){
            $this->_add_score($score_type, $this->status_list[$key], $key, $tags);
            return $this->status_list[$key];
        }

        return null;

    }

    /**
     * @param array $activities
     * @param int   $userid
     * @param bool  $count_only_pp
     * @param bool  $use_kica
     * @param array|null $mm_list_data
     */
    public function check_activities_list($activities, $userid=0, $count_only_pp=false, $use_kica=null, $mm_list_data=null){
        global $USER;
        $userid = $userid ?: $USER->id;

        if (empty($activities)){
            return;
        }

        if (is_null($use_kica)){
            $use_kica = get_kica_if_exists($use_kica);
        }

        $mod = reset($activities);
        $options = ['courseid' => $mod->course, 'userid' => $userid];
        $activity_completion = get_course_modules_completion($options, true, true);

        foreach ($activities as $mod) {
            $this->check_activity($mod, $userid, $count_only_pp, $use_kica,
                $mm_list_data[$mod->id][$userid] ?? null, $activity_completion[$mod->id] ?? false);
        }
    }

    /**
     * @param      $course
     * @param      $user
     * @param bool $count_only_pp
     * @param null $use_kica
     */
    public function check_activities_by_course($course, $user=null, $count_only_pp=false, $use_kica=null){
        global $USER;
        if (is_numeric($user)){
            $user = \core_user::get_user($user);
        }
        $user = $user ?? $USER;

        $userid = $user->id;
        $courseid = $course->id;

        $MM = MM::get_MM_by_params(['course' => $course, 'set_students' => [$userid => $user]]);
        $filter = [MM::ST_ALL, MM::BY_ACTIVITY, MM::BY_USER, MM::GET_TAGS, MM::USE_DEADLINE, MM::USE_KICA];
        $MM->filter_hide_grades_before_midn_if_need($filter);
        $mm_data_list = $MM->get_data_all($filter);
        $activities = SH::get_important_activities_by_compare_list($courseid, $userid, $mm_data_list, true);

        if (empty($activities)){
            return;
        }

        if (is_null($use_kica)){
            $use_kica = get_kica_if_exists($courseid);
        }

        $this->check_activities_list($activities, $userid, $count_only_pp, $use_kica, $mm_data_list);
    }

    /**
     * @param      $score_type
     * @param      $status
     * @param      $key
     * @param      $tags
     * @param bool $from_add_score
     */
    protected function _add_score($score_type, $status, $key, $tags, $from_add_score = false){
        $this->fs_status_list[$key] = $score_type;
        $this->status_list[$key] = $status;
        $this->$status++;
        $this->all++;
        $this->all_tags +=  $tags;
        $this->tags_list[$key] = $tags;
        if (!$score_type){
            return;
        }

        $points = static::get_points_by_status($status, $score_type);
        $this->fs_activities[$score_type] += $points;

        if (!$from_add_score && isset($this->fs_activity_status[$score_type]) && $this->fs_activity_status[$score_type]){
            $this->fs_activity_status[$score_type]->_add_score($score_type, $status, $key, $tags, true);
            if (isset($this->offlineoverridestatus[$key])){
                $this->fs_activity_status[$score_type]->offlineoverridestatus[$key] = $this->offlineoverridestatus[$key];
            }
        }
    }

    /**
     * @param     $status
     * @param int $score_type
     *
     * @return int
     */
    static public function get_points_by_status($status, $score_type=USUAL_ACTIVITY){
        $points = 0;
        if ($score_type != SUMMATIVE_ACTIVITY && $score_type != FORMATIVE_ACTIVITY){
            return $points;
        }

        $category_status = static::get_category_status($status);
        if (in_array($category_status, static::PP_CATEGORIES)){
            $points = ($category_status == static::STATUS_INCOMPLETED && $score_type == FORMATIVE_ACTIVITY) ? -1 : 1;
        }

        return $points;
    }

    /**
     * @return array
     */
    public function get_list(){
        return [
            $this->get_all_completed(),
            $this->get_all_incompleted(),
            $this->draft,
            $this->notattempted,
            $this->get_all_waiting(),
            $this->status_list,
            $this->offlineoverridestatus
        ];
    }

    /**
     * @return array
     */
    public function get_empty_list(){
        return $this->empty_list;
    }

    /**
     * @return array
     */
    public function get_summative_list(){
        return $this->fs_activity_status[SUMMATIVE_ACTIVITY] ? $this->fs_activity_status[SUMMATIVE_ACTIVITY]->get_list() : $this->get_empty_list();
    }

    /**
     * @return array
     */
    public function get_formative_list(){
        return $this->fs_activity_status[FORMATIVE_ACTIVITY] ? $this->fs_activity_status[FORMATIVE_ACTIVITY]->get_list() : $this->get_empty_list();
    }

    /**
     * @return int
     */
    public function get_summative_score(){
        return max(0, $this->fs_activities[SUMMATIVE_ACTIVITY]);
    }

    /**
     * @return int
     */
    public function get_formative_score(){
        return max(0, $this->fs_activities[FORMATIVE_ACTIVITY]);
    }

    /**
     * @param $summative_activities
     * @param $formative_activities
     *
     * @return float|int
     */
    public static function calc_participation_power($summative_activities, $formative_activities){
        return ($summative_activities != 0) ? round($formative_activities / $summative_activities, 3) * 100 : 0;
    }

    /**
     * @return float|int
     */
    public function get_participation_power(){
        return self::calc_participation_power($this->get_summative_score(), $this->get_formative_score());
    }

    /**
     * @param int $participation_power
     *
     * @return string
     */
    public static function calc_high_low_result($participation_power){
        return $participation_power >= 200 ? 'high' : (($participation_power >= 100) ? 'medium' : 'low');
    }

    /**
     * @return string
     */
    public function get_high_low_result(){
        return self::calc_high_low_result($this->get_participation_power());
    }

    /**
     * Return summative_score, formative_score, participation_power, divided_score, text_status
     *
     * @return array
     */
    public function get_full_pp_data(){
        $summative_score = $this->get_summative_score();
        $formative_score = $this->get_formative_score();
        $participation_power = $this->get_participation_power();
        $divided_score = $participation_power/100;
        $text_status = static::calc_high_low_result($participation_power);
        return [$summative_score, $formative_score, $participation_power, $divided_score, $text_status];
    }

    /**
     * @param $key
     *
     * @return string|null
     */
    public function get_status($key){
        return $this->status_list[$key] ?? null;
    }

    /**
     * @param $key
     *
     * @return string|null
     */
    public function get_fs_status($key){
        return $this->fs_status_list[$key] ?? null;
    }

    /**
     * @return activity_status
     */
    public function get_summative_activity_status(){
        return $this->fs_activity_status[SUMMATIVE_ACTIVITY] ? $this->fs_activity_status[SUMMATIVE_ACTIVITY] : new self();
    }

    /**
     * @return activity_status
     */
    public function get_formative_activity_status(){
        return $this->fs_activity_status[FORMATIVE_ACTIVITY] ? $this->fs_activity_status[FORMATIVE_ACTIVITY] : new self();
    }

    /**
     * @return array
     */
    public function get_all_tags(){
        return $this->all_tags;
    }

    /**
     * @param $key
     *
     * @return array
     */
    public function get_tags($key){
        return $this->tags_list[$key] ?? [];
    }

    /**
     * @param $key
     * @param $tag_id
     *
     * @return bool
     */
    public function check_tag_by_id($key, $tag_id){
        return isset($this->tags_list[$key][$tag_id]);
    }

    /**
     * @param $key
     * @param $tag_name
     *
     * @return bool
     */
    public function check_tag_by_name($key, $tag_name){
        if (isset($this->tags_list[$key])){
            return in_array($tag_name, $this->tags_list[$key]) !== false;
        }
        return false;
    }

    /**
     * @param bool $include_kica
     *
     * @return int
     */
    public function get_all_completed($include_kica=true){
        $res = $this->completed + $this->passed;
        if ($include_kica){
            $res += $this->graded_kica + $this->graded_kica_summative + $this->graded_kica_formative;
        }
        return $res;
    }

    /**
     * @param bool $include_kica
     *
     * @return int
     */
    public function get_all_waiting($include_kica=true){
        $res = $this->waitingforgrade;
        if ($include_kica){
            $res += $this->ungraded_kica + $this->ungraded_kica_summative + $this->ungraded_kica_formative;
        }
        return $res;
    }

    /**
     * @param bool $include_kica
     *
     * @return int
     */
    public function get_all_incompleted($include_kica=true){
        $res = $this->incompleted + $this->incompleted_summative + $this->incompleted_formative + $this->ngc_zero;
        if ($include_kica){
            $res += $this->kica_zerograde;
        }
        return $res;
    }

    /**
     * @param bool $include_kica
     *
     * @return int
     */
    public function get_all_graded($include_kica=true){
        return $this->get_all_completed($include_kica) + $this->get_all_incompleted($include_kica);
    }

    /**
     * @param $status
     *
     * @return string
     */
    static function get_category_status($status){
        switch ($status){
            case static::STATUS_COMPLETED:
            case static::STATUS_PASSED:
            case static::STATUS_GRADED_KICA:
            case static::STATUS_GRADED_KICA_SUMMATIVE:
            case static::STATUS_GRADED_KICA_FORMATIVE:
                return static::STATUS_COMPLETED;

            case static::STATUS_WAITINGFORGRADE:
            case static::STATUS_UNGRADED_KICA:
            case static::STATUS_UNGRADED_KICA_SUMMATIVE:
            case static::STATUS_UNGRADED_KICA_FORMATIVE:
            case static::STATUS_DRAFT:
                return static::STATUS_WAITINGFORGRADE;

            case static::STATUS_INCOMPLETED:
            case static::STATUS_INCOMPLETED_SUMMATIVE:
            case static::STATUS_INCOMPLETED_FORMATIVE:
            case static::STATUS_KICA_ZEROGRADE:
            case static::STATUS_AWARD_ZERO:
                return static::STATUS_INCOMPLETED;
        }

        return $status;
    }

    /**
     * @param $status_to_check
     * @param $status_need
     *
     * @return bool
     */
    static function is_this_status($status_to_check, $status_need){
        if (!$status_to_check){
            return false;
        }
        if ($status_need === static::STATUS_ALL){
            return true;
        }
        return ($status_to_check === $status_need) || $status_need === static::get_category_status($status_to_check);
    }
}
