<?php
/**
 * Class course_users_status
 *
 * @package    block_ned_teacher_tools
 * @subpackage NED
 * @copyright  2020 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace block_ned_teacher_tools;
use block_ned_teacher_tools\shared_lib as SH;

defined('MOODLE_INTERNAL') || die();

/** @var \stdClass $CFG */
require_once($CFG->dirroot."/lib/completionlib.php");

/**
 * Class course_users_status
 *
 * @package block_ned_teacher_tools
 */
class course_users_status {
    const TABLE = 'block_ned_teacher_c_status';
    const STATUS_IN_PROGRESS = 1;
    const STATUS_WAITING_FOR_FINAL_GRADE = 2;
    const STATUS_PASS = 3;
    const STATUS_FAIL = 4;
    const STATUS_WAITING_FOR_INTERNAL_REVIEW = 5;

    const TAG_AUTO_SET_WAITING_FOR_GRADE = 'final exam';
    const COURSE_TYPE_AUTO_SET_WAITING_FOR_GRADE = 'Credit';
    const CUSTOM_FIELD_COURSE_TYPE_NAME = 'course_type';
    const ROLE_OT_SHORTNAME = 'online-teacher';
    const ROLE_CT_SHORTNAME = 'classroom-teacher';

    /**
     * Custom field badge code shortname
     */
    const BADGE_CODE_CUSTOM_FIELD = 'badge_code';

    protected $_courseid;
    protected $_error_type = 0;

    public function __construct($courseid){
        $this->_courseid = $courseid;
    }

    /**
     * @return integer
     */
    public function get_error_type(){
        return $this->_error_type;
    }

    /**
     * @return integer
     */
    public function get_courseid(){
        return $this->_courseid;
    }

    /**
     * @param integer $_courseid
     */
    public function set_courseid($_courseid){
        $this->_courseid = $_courseid;
    }

    /**
     * @param $userid
     * @param $status
     * @param $timecreated
     * @param $timeupdated
     *
     * @return \stdClass
     */
    public function get_c_status_obj($userid, $status, $timecreated, $timeupdated){
        global $USER;
        $obj = new \stdClass();
        $obj->userid = $userid;
        $obj->courseid = $this->_courseid;
        $obj->status = $status;
        $obj->timecreated = $timecreated;
        $obj->timeupdated = $timeupdated;
        $obj->examinator = $USER->id;
        return $obj;
    }

    /**
     * @return mixed
     */
    public function get_course_badge_code(){
        $badge_custom_field = $this->_load_course_badge_code();
        if (empty($badge_custom_field)) return '';

        $fc = new \customfield_select\field_controller($badge_custom_field->id);
        $options = $fc->get_options();
        $dc = new \customfield_select\data_controller($badge_custom_field->dataid, $badge_custom_field);

        return $options[$dc->get_value()];
    }

    /**
     * Load badge field from database
     *
     * @return false|\stdClass
     */
    protected function _load_course_badge_code(){
        $sql = 'SELECT cf.*, cd.id as dataid
                  FROM {customfield_field} cf
                  JOIN {customfield_data} cd ON cd.fieldid = cf.id
                 WHERE cd.instanceid = :courseid AND cf.shortname = :field_name';
        return SH::db()->get_record_sql($sql, ['courseid' => $this->_courseid, 'field_name' => static::BADGE_CODE_CUSTOM_FIELD]);
    }

    /**
     * @param $code
     * @param $userid
     *
     * @return array
     */
    public function get_user_badge_by_code($code, $userid){
        $sql = 'SELECT bi.id, bi.id AS badgeid 
                  FROM {badge} b
                  JOIN {badge_issued} bi ON bi.badgeid = b.id
                 WHERE b.version = :code AND bi.userid = :userid';
        return SH::db()->get_records_sql_menu($sql, ['code' => $code, 'userid' => $userid]);
    }

    /**
     * @param $code
     *
     * @return \stdClass|bool
     */
    public function get_badge_by_code($code){
        static $res = [];

        if (empty($code)) return false;

        if (!isset($res[$code])){
            $records = SH::db()->get_records('badge', ['version' => $code], 'timemodified DESC', '*', 0, 1);
            $res[$code] = reset($records);
        }

        return $res[$code];
    }

    /**
     * @param $code
     * @param $userid
     *
     * @return bool
     */
    public function delete_badge_by_version($code, $userid){
        $user_badges = $this->get_user_badge_by_code($code, $userid);
        foreach ($user_badges as $badge){
            SH::db()->delete_records('badge_issued', ['userid' => $userid, 'id' => $badge]);
        }
        return true;
    }

    /**
     * @param $userid
     * @param $status
     *
     * @return bool|int
     */
    public function add_user_status($userid, $status){
        global $USER;

        if (empty($userid)){
            throw new \moodle_exception('empty userid');
        }

        $c_status = SH::db()->get_record(self::TABLE, ['courseid' => $this->_courseid, 'userid' => $userid]);
        if (!empty($c_status) && $c_status->status == $status) return true;

        if ($status == static::STATUS_PASS && !is_siteadmin()){
            $courseobj = get_course($this->_courseid);
            $completioninfo = new \completion_info($courseobj);
            if (!$this->meets_course_criteria($USER->id)){
                $this->_error_type = 2;
                return false;
            }
            if (!$completioninfo->is_course_complete($userid)){
                $this->_error_type = 1;
                return false;
            }
        }

        if ($status == static::STATUS_PASS){
            $this->issue_badge($userid);
        }

        $time = time();
        if (!empty($c_status)){
            $old_status = $c_status->status;
            $c_status->status = $status;
            $c_status->timeupdated = $time;
            if (!is_siteadmin() && $old_status == static::STATUS_PASS && !$this->meets_course_criteria($USER->id)){
                $this->_error_type = 2;
                return false;
            }
            if ($old_status == static::STATUS_PASS){
                $badge_code = $this->get_course_badge_code();
                $this->delete_badge_by_version($badge_code, $userid);
            }

            if ($status == static::STATUS_IN_PROGRESS){
                return SH::db()->delete_records(static::TABLE, ['id' => $c_status->id]);
            }
            return SH::db()->update_record(static::TABLE, $c_status);
        }

        $c_status = $this->get_c_status_obj($userid, $status, $time, $time);
        if ($status == static::STATUS_IN_PROGRESS) return true;

        return SH::db()->insert_record(static::TABLE, $c_status);
    }

    /**
     * @param null $userid
     * @param bool $keyuserid
     *
     * @return array
     */
    public function get_users_statuses($userid=null, $keyuserid=true){
        $filter_params = ['courseid' => $this->_courseid];
        if (!empty($userid)){
            $filter_params['userid'] = $userid;
        }

        $fields = '';
        if ($keyuserid){
            $fields = 'userid, {'.static::TABLE.'}.*';
        }

        return SH::db()->get_records(static::TABLE, $filter_params, '', $fields);
    }

    /**
     * @param array $user_ids
     *
     * @return array|false
     */
    public function get_count_users_in_statuses_by_course($user_ids){
        if (empty($user_ids)){
            return false;
        }

        $where = [];
        $params = [];
        SH::sql_add_get_in_or_equal_options('bntcs.userid', $user_ids, $where, $params);
        SH::sql_add_equal('bntcs.courseid', $this->_courseid, $where, $params);
        $sql = SH::sql_generate("bntcs.STATUS, COUNT(bntcs.id)", [], static::TABLE, "bntcs", $where, "bntcs.status");
        $records = SH::db()->get_records_sql_menu($sql, $params);

        $records[static::STATUS_IN_PROGRESS] = count($user_ids);
        foreach ($records as $status => $count){
            if ($status == static::STATUS_IN_PROGRESS){
                continue;
            }
            $records[static::STATUS_IN_PROGRESS] -= $count;
        }
        return $records;
    }

    /**
     * @return bool
     */
    public function enable_on_course(){
        return course_cats_has_courseid(get_config(PLUGIN_NAME, 'coursecompletionstatuscontroller_coursecategories'), $this->_courseid);
    }

    /**
     * @param $userid
     *
     * @return bool
     */
    public function meets_course_criteria($userid){
        $user_roles = get_user_roles(SH::ctx($this->_courseid), $userid);
        $course_condition_manual_roles = SH::db()->get_records_menu('course_completion_criteria',
            ['course' => $this->_courseid, 'criteriatype' => COMPLETION_CRITERIA_TYPE_ROLE], '', 'role, id');

        foreach ($user_roles as $user_role){
            if (array_key_exists($user_role->roleid, $course_condition_manual_roles)){
                return true;
            }
        }
        return false;
    }

    /**
     * @param $userid
     */
    public function issue_badge($userid){
        $badge_code = $this->get_course_badge_code();
        $badge = $this->get_badge_by_code($badge_code);
        if (empty($badge)) return;

        $users_badges = $this->get_user_badge_by_code($badge_code, $userid);
        if (!empty($users_badges)) return;

        /** @var \core_badges\badge $badge */
        $badge = new \core_badges\badge($badge->id);
        $badge->issue($userid);
    }
}
