<?php
/**
 * @package    block_ned_teacher_tools
 * @category   deadline_manager
 * @copyright  2022 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace block_ned_teacher_tools;

use block_ned_teacher_tools\deadline_manager as DM;
use block_ned_teacher_tools\shared_lib as SH;

/**
 * Class deadline_manager_entity
 * @package block_ned_teacher_tools
 *
 * @property-read int $courseid
 * @property-read int $groupid
 * @property-read int $userid
 * @property-read int|string $dm_timezone
 * @property-read object $tt_config
 */
class deadline_manager_entity {
    use \local_ned_controller\base_get_set_for_class;

    const FIELD_TIME = 'deadline';
    const FIELD_OVERRULE = 'overrule';

    const FORM_DEADLINE_PREFIX = 'deadline_';
    const FORM_OVERRULE_PREFIX = 'overrule_';

    /** @var \block_ned_teacher_tools\mod\deadline_manager_mod[][] - [courseid => [$cmid => object]]  */
    static protected $_course_dm_modules = [];
    /** @var object[] - [cmid => object] with FIELD_* properties  */
    static protected $_activity_deadlines = [];
    /** @var bool[][] [courseid => [capability_name => bool]] */
    static protected $_course_capabilities = [];

    protected $_courseid = 0;
    protected $_groupid = 0;
    protected $_userid = 0;

    /** @var object[][] - [source => [cmid => object]] with FIELD_* properties  */
    protected $_dm_deadlines = [];
    /**
     * Original deadlines from the DB - please, don't change them without absolutely necessary
     * @var object[][] - [source => [cmid => object]] with FIELD_* properties
     */
    protected $_dm_original_deadlines = [];
    /** @var bool[] - [cmid => has_conflict] */
    protected $_cm_conflicts = null;
    /** @var bool[] - [source => bool] */
    protected $_manually_edited = [];
    /** @var int|string */
    protected $_dm_timezone;
    /** @var object */
    protected $_tt_config;

    /**
     * deadline_manager_entity constructor.
     * Normally you should provide at least group or user
     *
     * @param object|numeric $course_or_id
     * @param object|numeric $group_or_id (optional)
     * @param object|numeric $user_or_id  (optional)
     */
    public function __construct($course_or_id, $group_or_id=null, $user_or_id=null){
        [$courseid, $groupid, $userid] = SH::get_ids($course_or_id, $group_or_id, $user_or_id);
        if (empty($groupid) && !empty($userid)){
            $groupids = SH::get_user_groupids($courseid, $userid);
            $groupid = reset($groupids);
        }

        $this->_courseid = $courseid;
        $this->_groupid = $groupid;
        $this->_userid = $userid;
        $this->_tt_config = SH::get_tt_block_config($courseid);

        $timezone = SH::NED_TIMEZONE;
        if ($this->_groupid || $this->_userid){
            $school = $this->_userid ? SH::get_user_school($this->_userid) : DM::get_school_cohort($this->_groupid);
            if (class_exists('\local_schoolmanager\school') && $school instanceof \local_schoolmanager\school){
                $timezone = $school->get_timezone();
            } elseif (!empty($school->timezone)) {
                $timezone = $school->timezone;
            } else {
                $user = SH::get_user($this->_userid);
                if (!empty($user->timezone)){
                    $timezone = $user->timezone;
                }
            }
        }

        $this->_dm_timezone = $timezone;

        $this->_init();
    }

    //region Protected
    /**
     * Init all deadlines for current entity
     */
    protected function _init(){
        foreach (DM::DM_SOURCES as $source){
            $this->_dm_deadlines[$source] = [];
            $this->_dm_original_deadlines[$source] = [];
        }

        $dm_modules = static::get_course_dm_modules($this->_courseid);
        foreach ($dm_modules as $cmid => $dmm){
            if (!isset(static::$_activity_deadlines[$cmid])){
                static::$_activity_deadlines[$cmid] = static::make_dl_object($dmm->get_default_deadline());
            }

            if ($this->_groupid){
                $deadline = $dmm->get_override($this->_groupid, null, true);
                if (!is_null($deadline)){
                    $dl = static::make_dl_object($deadline, $dmm->get_overrule($this->_groupid));
                    $this->_dm_deadlines[DM::SOURCE_GROUP][$cmid] = $dl;
                    $this->_dm_original_deadlines[DM::SOURCE_GROUP][$cmid] = clone $dl;
                }
            }

            if ($this->_userid){
                $deadline = $dmm->get_override(null, $this->_userid, true);
                if (!is_null($deadline)){
                    $dl = static::make_dl_object($deadline, $dmm->get_overrule(null, $this->_userid));
                    $this->_dm_deadlines[DM::SOURCE_USER][$cmid] = $dl;
                    $this->_dm_original_deadlines[DM::SOURCE_USER][$cmid] = clone $dl;
                }

                $ext = $dmm->get_extension($this->_userid);
                if ($ext){
                    $dl = static::make_dl_object($ext->duedate, $ext->overrule);
                    $this->_dm_deadlines[DM::SOURCE_EXTENSION][$cmid] = $dl;
                    $this->_dm_original_deadlines[DM::SOURCE_EXTENSION][$cmid] = clone $dl;
                }
            }
        }
    }
    //endregion

    //region Base methods
    /**
     * Return tt block config, or its option, or null
     *
     * @param string|null $option - if option is set, return its value or null
     *
     * @return null|bool|object|mixed
     */
    public function get_tt_config($option=null){
        if ($option){
            return $this->_tt_config->$option ?? null;
        }
        return $this->_tt_config;
    }

    /**
     * Return all saved deadlines by source
     *
     * @param string $source   - one of the {@see \block_ned_teacher_tools\deadline_manager_entity::SOURCES},
     * @param bool   $original - (optional) if true, return data from the original deadline
     *
     * @return array|object[]
     */
    public function get_whole_deadline_data($source=DM::SOURCE_NONE, $original=false){
        switch($source){
            default:
            case DM::SOURCE_NONE:
                return [];
            case DM::SOURCE_ACTIVITY:
                return static::$_activity_deadlines;
            case DM::SOURCE_GROUP:
            case DM::SOURCE_USER:
            case DM::SOURCE_EXTENSION:
                if ($original){
                    return $this->_dm_original_deadlines[$source];
                } else {
                    return $this->_dm_deadlines[$source];
                }
        }
    }

    /**
     * Set current data manually
     *
     * @param object[] $data     - it should be an array of deadline objects {@see make_dl_object()} by cmids
     * @param string   $source   - one of the {@see \block_ned_teacher_tools\deadline_manager_entity::SOURCES},
     * @param bool     $merge    - (optional) if true, merge new data with old instead of replacing
     * @param bool     $original - (optional) if true, set data for original deadlines - DON'T use until absolutely necessary
     *
     * @return array - return $data, if it was inserted
     */
    public function set_whole_deadline_data($data, $source=DM::SOURCE_NONE, $merge=false, $original=false){
        if (!$original){
            $this->_manually_edited[$source] = true;
        }
        if ($merge){
            $data = array_merge($this->get_whole_deadline_data($source, $original), $data);
        }

        switch($source){
            default:
            case DM::SOURCE_NONE:
                return [];
            case DM::SOURCE_ACTIVITY:
                return static::$_activity_deadlines = $data;
            case DM::SOURCE_GROUP:
            case DM::SOURCE_USER:
            case DM::SOURCE_EXTENSION:
                if ($original){
                    return $this->_dm_original_deadlines[$source] = $data;
                } else {
                    return $this->_dm_deadlines[$source] = $data;
                }
        }
    }

    /**
     * Given a $time timestamp in GMT (seconds since epoch),
     * returns an array that represents the Gregorian date in the current DM timezone.
     *
     * @param int $datetime Timestamp in GMT
     *
     * @return array An array that represents the date in user time
     */
    public function dm_date($datetime){
        return usergetdate($datetime, $this->_dm_timezone);
    }

    /**
     * Given a time, return the GMT timestamp of the most recent midnight
     * with the current DM timezone.
     *
     * @param int $datetime Timestamp in GMT
     *
     * @return int Returns a GMT timestamp
     */
    public function dm_midnight($datetime){
        return usergetmidnight($datetime, $this->_dm_timezone);
    }

    /**
     * Check currents deadlines for rule conflicts and save result to the {@see _cm_conflicts}
     * @see has_date_conflict() for the cmid checking
     *
     * @param bool $force - recheck current conflicts, even if already checked them
     *
     * @return void
     */
    public function check_conflicts($force=false){
        if (!is_null($this->_cm_conflicts) && !$force) return;

        $this->_cm_conflicts = [];
        /**
         * Save here cmids by there midnight deadlines
         * Note: there will be not all deadlines/cmid,
         *  only for checking $x_days_between_dl_* and allow_quiz_other_dl_in_one_day rule
         *
         * @var array[] $cm_deadlines_quiz - deadline time => [cmids]
         * @var array[] $cm_deadlines_other - deadline time => [cmids]
         */
        $cm_deadlines_quiz = [];
        $cm_deadlines_other = [];

        $config = $this->get_tt_config();
        $quiz_seq_rule = !empty($config->forcequizsequence);
        $midterm_seq_rule = !empty($config->enablemidtermpoint);
        $x_days_between_quiz = $config->x_days_between_dl_quiz ?? SH::CONF_DM_X_DAYS_BETWEEN_DL_QUIZ;
        $x_days_between_other = $config->x_days_between_dl_other ?? SH::CONF_DM_X_DAYS_BETWEEN_DL_NON_QUIZ;
        $allow_quiz_other_in_one_day = $config->allow_quiz_other_dl_in_one_day ?? SH::CONF_DM_ALLOW_QUIZ_OTHER_DL_IN_ONE_DAY;
        $x_days_apply_to_all = !empty($config->x_days_apply_to_all);
        $is_admin = is_siteadmin();

        $quiz_last_dl = 0;
        $midterm_dl = 0;
        $midterm_cmid = null;
        $midterm_reached = false;
        if ($midterm_seq_rule){
            $midterm_dma = $this->get_midterm_module();
            if ($midterm_dma && $midterm_dma->is_enabled()){
                $midterm_dl = $midterm_dma->get_deadline();
            }
            if (empty($midterm_dl)){
                $midterm_seq_rule = false;
            } else {
                $midterm_cmid = $midterm_dma->cmid;
            }
        }

        $group_start = null;
        $group_end = null;
        if ($this->_groupid){
            $group = SH::get_group($this->_groupid);
            if (!empty($group->startdate)){
                $group_start = $this->dm_midnight($group->startdate);
            }
            if (!empty($group->enddate)){
                $group_end = $this->dm_midnight($group->enddate);
            }
        }

        $final_dl = $this->get_max_final_deadline();
        $final_dl = $final_dl ? $this->dm_midnight($final_dl) : $group_end;
        $latest_dl = $this->get_latest_deadline();

        $errors = [
            'quiz' => SH::str('assigneddateorderquizzeserror'),
            'midtermafter' => SH::str('error_midtermafter', DM::format_deadline($midterm_dl, $this->_dm_timezone)),
            'midtermbefore' => SH::str('error_midtermbefore', DM::format_deadline($midterm_dl, $this->_dm_timezone)),
            'startdeadline' => SH::str('error_startdeadline', DM::format_deadline($group_start, $this->_dm_timezone)),
            'finaldeadline_usual' => SH::str('error_finaldeadline_usual', DM::format_deadline($group_end, $this->_dm_timezone)),
            'finaldeadline_final' => SH::str('error_finaldeadline_final', DM::format_deadline($final_dl, $this->_dm_timezone)),
            'x_days_between_quiz' => SH::str('error_x_days_between_dl_quiz', $x_days_between_quiz),
            'x_days_between_other' => SH::str('error_x_days_between_dl_other', $x_days_between_other),
            'error_q_o_in_day' => SH::str('error_allow_quiz_other_dl_in_one_day'),
            'cant_changed' => SH::str('error_deadline_cant_changed'),
            'cant_removed' => SH::str('error_deadline_cant_removed'),
            'beforecurrent' => SH::str('error_deadline_beforecurrent'),
            'lastactivity' => SH::str('error_deadline_lastactivity'),
        ];

        /**
         * Check rules and save cmid and its deadline to the list
         *
         * @param array   $cm_dl_list    - list to save deadline and cmid value, UPDATED during function call
         * @param int     $cmid
         * @param int     $dl_midnight   - midnight of the deadline
         *
         * @return void - result saves in the $cm_dl_list
         */
        $f_cm_deadline2list = function(&$cm_dl_list, $cmid, $dl_midnight){
            if (empty($cm_dl_list[$dl_midnight])){
                $cm_dl_list[$dl_midnight] = [];
            }
            $cm_dl_list[$dl_midnight][] = $cmid;
        };

        /**
         * Save cmids conflicts with error text in $this->_cm_conflicts
         *
         * @param array[] $new_cm_conflicts - list of arrays(!) of cmids
         * @param string  $error_text       - translate error text
         *
         * @return void - result saves in the $this->_cm_conflicts
         */
        $f_save_new_conflicts = function($new_cm_conflicts, $error_text){
            if (empty($new_cm_conflicts)) return;

            $cm_conflicts_list = array_merge(...$new_cm_conflicts);
            foreach ($cm_conflicts_list as $cmid_conflict){
                /** @var numeric $cmid_conflict */
                $this->_cm_conflicts[$cmid_conflict] = $error_text;
            }
        };

        foreach ($this->iterate() as $cmid => $dma){
            if (!$dma->is_enabled()) continue;

            $conflict = false;
            if ($midterm_seq_rule && !$midterm_reached && $cmid == $midterm_cmid){
                $midterm_reached = true;
            }

            $source = $dma->get_source();
            $deadline = $dma->get_deadline($source);
            $tags = SH::cm_get_tags($cmid);

            if ($deadline && ($deadline != $dma->get_deadline($source, true)) && ($deadline < time())) {
                $conflict = $errors['beforecurrent'];
            } else if (in_array(SH::TAG_LAST_ACTIVITY, $tags) && $deadline && ($deadline < $latest_dl)) {
                $conflict = $errors['lastactivity'];
            } else {
                if ($dma->is_overruled() || $dma->is_proxy()) continue;
            }

            if (empty($deadline)){
                if (!$is_admin && $dma->get_source(true) == DM::SOURCE_GROUP && !in_array($source,DM::DM_SOURCES)){
                    $conflict = $errors['cant_removed'];
                } else {
                    continue;
                }
            }

            do {
                if ($conflict) break;

                if (!$dma->is_editable($source, true) && $deadline != $dma->get_deadline($source, true)){
                    $conflict = $errors['cant_changed'];
                    break;
                }

                $dl_midnight = $this->dm_midnight($deadline);

                if ($quiz_seq_rule && $dma->cm->modname == SH::QUIZ){
                    if ($deadline < $quiz_last_dl){
                        $conflict = $errors['quiz'];
                        break;
                    } else {
                        $quiz_last_dl = $deadline;
                    }
                }

                if ($midterm_seq_rule && $source != DM::SOURCE_EXTENSION){
                    if ($midterm_reached == ($midterm_dl > $deadline) && $deadline != $midterm_dl && !$dma->is_midterm()){
                        $conflict = $errors[$midterm_reached ? 'midtermafter' : 'midtermbefore'];
                        break;
                    }
                }

                if ($group_start && $dl_midnight < $group_start){
                    $conflict = $errors['startdeadline'];
                    break;
                }

                if ($dma->is_final_cm()){
                    if ($group_end && $dl_midnight > $group_end){
                        $conflict = $errors['finaldeadline_usual'];
                        break;
                    }
                } else {
                    if ($final_dl && $dl_midnight > $final_dl) {
                        $conflict = $errors['finaldeadline_final'];
                        break;
                    }
                }

                if ($x_days_apply_to_all || $source == DM::SOURCE_GROUP){
                    if ($dma->is_quiz()){
                        $f_cm_deadline2list($cm_deadlines_quiz, $cmid, $dl_midnight);
                    } else {
                        $f_cm_deadline2list($cm_deadlines_other, $cmid, $dl_midnight);
                    }
                }
            } while(false);

            if ($conflict){
                $this->_cm_conflicts[$cmid] = $conflict;
            }
        }


        /**
         * Check rules "Minimum days between..."
         *
         * @var $x_days_between_data - error key => [config option, array of cm deadlines]
         */
        $x_days_between_data = [
            'x_days_between_quiz' => [$x_days_between_quiz, &$cm_deadlines_quiz],
            'x_days_between_other' => [$x_days_between_other, &$cm_deadlines_other],
        ];
        foreach ($x_days_between_data as $x_error_key => [$x_days_config, &$cm_deadlines_list]){
            if ($x_days_config <= 0 || empty($cm_deadlines_list)) continue;

            $x_secs_between = $x_days_config*DAYSECS;
            ksort($cm_deadlines_list, SORT_NUMERIC);
            $prev_deadline = 0;
            $prev_cmids = [];
            $new_conflicts = [];

            foreach ($cm_deadlines_list as $cm_deadline => $cmids){
                if (($cm_deadline - $prev_deadline) < $x_secs_between){
                    $new_conflicts[] = $prev_cmids;
                    $new_conflicts[] = $cmids;
                } elseif (\count($cmids) > 1){
                    $new_conflicts[] = $cmids;
                }

                $prev_deadline = $cm_deadline;
                $prev_cmids = $cmids;
            }

            $f_save_new_conflicts($new_conflicts, $errors[$x_error_key]);
        }

        // Check rule "Allow quiz and non-quiz deadline on same day"
        if (!$allow_quiz_other_in_one_day && !empty($cm_deadlines_quiz) && !empty($cm_deadlines_other)){
            $new_conflicts = [];
            foreach ($cm_deadlines_quiz as $dl => $cmids){
                if (!empty($cm_deadlines_other[$dl])){
                    $new_conflicts[] = $cm_deadlines_other[$dl];
                    $new_conflicts[] = $cmids;
                }
            }

            $f_save_new_conflicts($new_conflicts, $errors['error_q_o_in_day']);
        }
    }

    /**
     * Return true, if data by this source was updated manually
     *
     * @param string $source
     *
     * @return bool
     */
    public function was_manually_edited($source=DM::SOURCE_NONE){
        return $this->_manually_edited[$source] ?? false;
    }

    /**
     * Iterate through all active dm modules, using dm_entity_activity for interaction
     * foreach ($this->iterate() as $cmid => $dma)
     *
     * @return \Generator<\block_ned_teacher_tools\support\dm_entity_activity[]>
     */
    public function iterate(){
        foreach (static::$_course_dm_modules[$this->_courseid] as $cmid => $dm_module){
            yield $cmid => new \block_ned_teacher_tools\support\dm_entity_activity($this, $cmid);
        }
    }
    //endregion

    //region is_*
    /**
     * @param numeric $cmid
     *
     * @return bool
     */
    public function is_proxy($cmid){
        if (empty($dm_module = $this->get_dm_module($cmid))) return null;

        return $dm_module->is_proxy();
    }

    /**
     * @param numeric $cmid
     *
     * @return bool
     */
    public function is_midterm($cmid){
        if (empty($this->get_dm_module($cmid))) return false;
        return SH::cm_is_midterm($cmid, $this->_courseid);
    }

    /**
     * @param numeric $cmid
     *
     * @return bool
     */
    public function is_final_cm($cmid){
        if (empty($this->get_dm_module($cmid))) return false;
        return SH::cm_is_final($cmid, $this->_courseid);
    }

    /**
     * @param numeric $cmid
     *
     * @return bool
     */
    public function is_quiz($cmid){
        if (empty($this->get_dm_module($cmid))) return false;

        $cm = SH::get_cm_by_cmid($cmid, $this->_courseid);
        return $cm && $cm->modname == SH::QUIZ;
    }

    /**
     * @param numeric $cmid
     *
     * @return bool
     */
    public function is_test_or_assignment($cmid){
        if (empty($this->get_dm_module($cmid))) return false;

        return SH::cm_is_test_or_assignment($cmid);
    }

    /**
     * Check does this deadline is editable
     * Note: normally, if you have already rewritten deadlines through {@see deadline_manager_entity::form_load_data()}
     *  and don't use $recalculate option, then you will get result based on previous deadline values (which is fine usually)
     *
     * @param numeric $cmid
     * @param string  $source   - one of the {@see \block_ned_teacher_tools\deadline_manager_entity::SOURCES},
     * @param bool    $original - (optional) if true, checked original deadline
     *
     * @return bool
     */
    public function is_editable($cmid, $source=DM::SOURCE_NONE, $original=false){
        if (!$source) $source = $this->get_source($cmid);

        if ($this->is_proxy($cmid)) return false;
        if ($source != DM::SOURCE_EXTENSION){
            if ($this->has_extension($cmid, $original)) return false;
            if ($this->is_uneditable_expired($cmid, $source, $original)) return false;
        }

        return true;
    }

    /**
     * Return true, if deadline by current source is overruled
     * Alias {@see get_overrule()}
     *
     * @param numeric $cmid
     * @param string  $source   - (optional) if null, use actual source
     * @param bool    $original - (optional) if true, checked original deadline
     *
     * @return bool
     */
    public function is_overruled($cmid, $source=null, $original=false){
        return $this->get_overrule($cmid, $source, $original);
    }

    /**
     * Return true, if deadline by current source is expired
     *
     * @param numeric $cmid
     * @param string  $source   - (optional) if null, use actual source
     * @param bool    $original - (optional) if true, checked original deadline
     *
     * @return bool
     */
    public function is_expired($cmid, $source=null, $original=false){
        $deadline = $this->get_deadline($cmid, $source, $original);
        return $deadline && $deadline < time();
    }

    /**
     * Return true, if deadline by current source is expired
     *  and current TT settings doesn't allow to edit such deadlines
     *
     * @param numeric $cmid
     * @param string  $source   - (optional) if null, use actual source
     * @param bool    $original - (optional) if true, checked original deadline
     *
     * @return bool
     */
    public function is_uneditable_expired($cmid, $source=null, $original=false){
        if ($this->get_tt_config('allowexpireddeadlineupdate')) return false;
        if (static::has_course_capability($this->_courseid, 'ignore_expired_dm_restriction')) return false;

        return $this->is_expired($cmid, $source, $original);
    }
    //endregion

    //region Getters
    /**
     * @param $cmid
     *
     * @return mod\deadline_manager_mod|null
     */
    public function get_dm_module($cmid){
        return static::$_course_dm_modules[$this->_courseid][$cmid] ?? null;
    }

    /**
     * Return actual source for the current deadline
     *
     * @param numeric $cmid
     * @param bool    $original - (optional) if true, get source of original deadline
     *
     * @return string - one of the {@see \block_ned_teacher_tools\deadline_manager_entity::SOURCES}
     */
    public function get_source($cmid, $original=false){
        foreach (DM::DM_SOURCES as $source){
            if ($original){
                if (isset($this->_dm_original_deadlines[$source][$cmid])) return $source;
            } else {
                if (isset($this->_dm_deadlines[$source][$cmid])) return $source;
            }
        }
        if (isset(static::$_activity_deadlines[$cmid])) return DM::SOURCE_ACTIVITY;

        return DM::SOURCE_NONE;
    }

    /**
     * Return deadline source with none-empty deadline before this $source
     *
     * @param numeric $cmid
     * @param string  $source   - (optional) if null, use actual source
     * @param bool    $original - (optional) get source of original deadline
     *
     * @return string - one of the {@see \block_ned_teacher_tools\deadline_manager_entity::SOURCES}
     */
    public function get_previous_source($cmid, $source=null, $original=false){
        if (!$source) $source = $this->get_source($cmid, $original);
        $source_reached = false;

        foreach (DM::SOURCES as $dm_source){
            if (!$source_reached){
                $source_reached = $dm_source == $source;
                continue;
            }

            $dl = $this->get_deadline($cmid, $dm_source, $original);
            if (!empty($dl)) return $dm_source;
        }

        return DM::SOURCE_NONE;
    }

    /**
     * Return current deadline object, {@see make_dl_object()} for structure
     *
     * @param numeric $cmid
     * @param string  $source   - (optional) if null, use actual source
     * @param bool    $original - (optional) if true, get original deadline object
     *
     * @return object|null - object with FIELD_* params
     */
    public function get_deadline_object($cmid, $source=null, $original=false){
        if (!$source) $source = $this->get_source($cmid);

        switch($source){
            default:
            case DM::SOURCE_NONE:
                return null;
            case DM::SOURCE_ACTIVITY:
                return static::$_activity_deadlines[$cmid] ?? null;
            case DM::SOURCE_GROUP:
            case DM::SOURCE_USER:
            case DM::SOURCE_EXTENSION:
                if ($original){
                    return $this->_dm_original_deadlines[$source][$cmid] ?? null;
                } else {
                    return $this->_dm_deadlines[$source][$cmid] ?? null;
                }
        }
    }

    /**
     * Get deadline by source (actual deadline by default)
     *
     * @param numeric $cmid
     * @param string  $source   - (optional) if null, use actual source
     * @param bool    $original - (optional) if true, get original deadline
     *
     * @return int|null
     */
    public function get_deadline($cmid, $source=null, $original=false){
        $object = $this->get_deadline_object($cmid, $source, $original);
        return $object->{static::FIELD_TIME} ?? null;
    }

    /**
     * Return true, if deadline by current source is overruled
     *
     * @param numeric $cmid
     * @param string  $source - (optional) if null, use actual source
     * @param bool    $original - (optional) if true, get data by original deadline
     *
     * @return bool
     */
    public function get_overrule($cmid, $source=null, $original=false){
        $object = $this->get_deadline_object($cmid, $source, $original);
        return $object->{static::FIELD_OVERRULE} ?? false;
    }

    /**
     * Alias to get grade through the dm module
     *
     * @param numeric $cmid
     *
     * @return string|null
     */
    public function get_grade($cmid){
        if (empty($this->_userid)) return null;
        if (empty($dm_module = $this->get_dm_module($cmid))) return null;

        return $dm_module->get_grade($this->_userid);
    }

    /**
     * Alias to get grade url through the dm module
     *
     * @param numeric $cmid
     *
     * @return string|null
     */
    public function get_grade_url($cmid){
        if (empty($this->_userid)) return null;
        if (empty($dm_module = $this->get_dm_module($cmid))) return null;

        return $dm_module->get_grade_url($this->_userid);
    }

    /**
     * Return true, if actual source is extension
     *
     * @param numeric $cmid
     * @param bool    $original - (optional) if true, checked original source
     *
     * @return bool
     */
    public function has_extension($cmid, $original=false){
        if (empty($this->_userid)) return false;
        return $this->get_source($cmid, $original) == DM::SOURCE_EXTENSION;
    }

    /**
     * Return full extension data
     * @see \block_ned_teacher_tools\deadline_manager::get_extension_course_data()
     *
     * @param numeric $cmid
     *
     * @return array($use_extension, $numberofextensions, $can_add_extension, $showextensionicon)
     */
    public function get_activity_extension_data($cmid){
        $def = [null, null, null, null];
        if (empty($this->_userid)) return $def;
        if (empty($this->get_dm_module($cmid))) return $def;

        return DM::get_activity_extension_data($cmid, $this->_userid, null, null, null, true, false, $this->courseid);
    }

    /**
     * Get formatted deadline to user-view format
     *
     * @param numeric $cmid
     * @param string  $source - (optional) if null, use actual source
     *
     * @return string
     */
    public function get_formatted_deadline($cmid, $source=null){
        $deadline = $this->get_deadline($cmid, $source);
        return DM::format_deadline($deadline, $this->_dm_timezone);
    }

    /**
     * Get translated source
     *
     * @param numeric $cmid
     * @param bool    $sub    - substr answer to one letter
     *
     * @return string
     */
    public function get_formatted_source($cmid, $sub=false){
        return DM::format_source($this->get_source($cmid), $sub);
    }

    /**
     * Return static counter for deadline by source
     *
     * @param numeric $cmid
     * @param string  $source - (optional) if null, use actual source
     *
     * @return string
     */
    public function get_deadline_counter($cmid, $source=null){
        $deadline = $this->get_deadline($cmid, $source);
        return SH::time_counter_to_str_max($deadline - time(), 2, true);
    }

    /**
     * Get time of submission by cm
     *
     * @param numeric $cmid
     *
     * @return int|false
     */
    public function get_submission_time($cmid){
        if (empty($this->_userid)) return false;
        if (empty($dm_module = $this->get_dm_module($cmid))) return false;

        return $dm_module->get_user_submission_time($this->_userid);
    }

    /**
     * Get number of graded students from the list by cmid
     *
     * @param numeric $cmid
     * @param array   $userids
     *
     * @return int
     */
    public function get_number_of_grades($cmid, $userids=[]){
        if (empty($userids)){
            if (!empty($this->_userid)){
                $userids = [$this->_userid];
            } else {
                return 0;
            }
        }
        if (empty($dm_module = $this->get_dm_module($cmid))) return 0;

        return $dm_module->get_number_of_grades($userids);
    }

    /**
     * Check actual deadline by the activity for the rule conflicts
     * Note: there is no way to check single deadline for conflicts, as they are all interconnected,
     *  so you need to check them all, even if you interested only in the single cmid
     *
     * @param numeric $cmid
     *
     * @return string|false - string about conflict or false
     */
    public function has_date_conflict($cmid){
        $this->check_conflicts();

        return $this->_cm_conflicts[$cmid] ?? false;
    }

    /**
     * Check actual deadline by the activity for the rule conflicts
     * Alias {@see has_date_conflict()}
     *
     * @param numeric $cmid
     *
     * @return string|false - string about conflict or false
     */
    public function get_date_conflict($cmid){
        return $this->has_date_conflict($cmid);
    }

    /**
     * Create and return dm_activity by cmid
     *
     * @param numeric $cmid
     *
     * @return \block_ned_teacher_tools\support\dm_entity_activity
     */
    public function get_dm_activity($cmid){
        return new \block_ned_teacher_tools\support\dm_entity_activity($this, $cmid);
    }

    /**
     * Get the first midterm module or null
     *
     * @return \block_ned_teacher_tools\support\dm_entity_activity|null
     */
    public function get_midterm_module(){
        if (empty(static::$_course_dm_modules[$this->_courseid])) return null;

        $midterms = SH::cmids_get_midterm($this->_courseid);
        $enable_midterms = array_intersect_key($midterms, static::$_course_dm_modules[$this->_courseid]);
        $cmid = reset($enable_midterms);
        if (!$cmid) return null;

        return $this->get_dm_activity($cmid);
    }

    /**
     * Get the max final deadline from the all final activities
     *
     * @param string $source - (optional) if null, use actual source
     *
     * @return int|null
     */
    public function get_max_final_deadline($source=null){
        if (empty(static::$_course_dm_modules[$this->_courseid])) return null;

        $final_cmids = SH::cmids_get_final($this->_courseid);
        $enable_final_cmids = array_intersect_key($final_cmids, static::$_course_dm_modules[$this->_courseid]);
        $deadlines = [];
        foreach ($enable_final_cmids as $cmid){
            $deadlines[] = $this->get_deadline($cmid, $source);
        }

        return SH::max($deadlines);
    }

    /**
     * Get the max final deadline from the all final activities
     *
     * @param string $source - (optional) if null, use actual source
     *
     * @return int|null
     */
    public function get_latest_deadline($source=null){
        if (empty(static::$_course_dm_modules[$this->_courseid])) return null;

        $cmids = array_keys(static::$_course_dm_modules[$this->_courseid]);
        $deadlines = [];
        foreach ($cmids as $cmid) {
            if (!$this->is_proxy($cmid)) {
                $deadlines[] = $this->get_deadline($cmid, $source);
            }
        }

        return SH::max($deadlines);
    }
    //endregion

    //region Form
    /**
     * Add deadline elements to the moodle form
     *
     * @param numeric          $cmid
     * @param \MoodleQuickForm $mform
     * @param string           $current_source - one of the {@see \block_ned_teacher_tools\deadline_manager_entity::SOURCES},
     *                                         but we really expect here only user or group source
     * @param bool $disabled - disable form elements
     *
     * @return void
     */
    public function form_add_element($cmid, $mform, $current_source=DM::SOURCE_NONE, $disabled=false){
        $dma = $this->get_dm_activity($cmid);

        $prev_dl = '';
        $prev_source = $dma->get_previous_source($current_source);
        if ($prev_source != DM::SOURCE_NONE){
            $prev_dl = $dma->get_formatted_deadline($prev_source);
        }
        $f_prev_source = SH::str('defaultdeadlinesource', DM::format_source($prev_source));
        $mform->addElement('html', SH::div(SH::span($f_prev_source, 'red').' '.$prev_dl, 'defaultdatelabel'));

        $real_source = $dma->get_source();
        $editable_source = $dma->has_extension() ? DM::SOURCE_EXTENSION : $current_source;
        $mform->addElement('html', SH::div(SH::span(DM::format_source($editable_source), 'red'), 'datetimelabel'));
        $mform->addElement('html', SH::div_start('datetime-wrapper d-flex align-items-center'));

        if ($dma->is_editable($current_source, true)){
            // date selector
            $key = static::FORM_DEADLINE_PREFIX.$cmid;
            $options = [
                'startyear' => +date('Y') - 5,
                'stopyear'  => +date('Y') + 5,
                'timezone' => $this->_dm_timezone,
                'optional'  => true
            ];

            $attributes = [];
            if ($disabled) {
                $attributes['disabled'] = 'disabled';
            }
            $mform->addElement('date_time_selector', $key, '', $options, $attributes);

            $dl = $dma->get_deadline() ?: $this->dm_midnight(time() + DAYSECS) - 1;
            if ($real_source == $current_source){
                $mform->setDefault($key, $dl);
            } else {
                $new_dl = $this->dm_date($dl);
                $def_date = [
                    'year'   => $new_dl['year'],
                    'month'  => $new_dl['mon'],
                    'day'    => $new_dl['mday'],
                    'hour'   => $new_dl['hours'],
                    'minute' => $new_dl['minutes'],
                ];
                $mform->setDefault($key, $def_date);
            }

            // overrule checkbox
            if (DM::can_override_restrictions($this->_courseid)){
                $key_or = static::FORM_OVERRULE_PREFIX.$cmid;
                $mform->addElement('html', SH::div_start('overrule-wrapper ml-5 d-flex align-items-center'));
                $mform->addElement('checkbox', $key_or, '');
                $mform->setDefault($key_or, $dma->is_overruled($current_source));
                $mform->disabledIf($key_or, $key.'[enabled]', 'notchecked');
                $mform->addElement('html', SH::span(SH::str('dm_overrule'), 'overrule-title mr-2'));
                $mform->addElement('html', SH::div_end()); // overrule-wrapper
            } else {
                if ($dma->is_overruled($current_source)){
                    $mform->addElement('html',  DM::icon_overruled());
                }
            }

        } else {
            // not editable
            $show_deadline = [$dma->get_formatted_deadline()];
            if ($dma->is_proxy()){
                $show_deadline[] = DM::icon_proxy();
            } else {
                if ($dma->is_uneditable_expired()){
                    $show_deadline[] = DM::icon_expired();
                }
                if ($dma->is_overruled()){
                    $show_deadline[] = DM::icon_overruled();
                }
            }
            $mform->addElement('html', SH::div($show_deadline, 'datetimelabel'));
        }

        $mform->addElement('html', SH::div_end()); // datetime-wrapper
    }

    /**
     * Load data from the form
     *
     * @param array|object $data array of ("fieldname"=>value) of submitted data
     * @param string $source - one of the {@see \block_ned_teacher_tools\deadline_manager_entity::SOURCES},
     *                                         but we really expect here only user or group source
     * @param bool $merge - if true, new data will be added to current (not replaced it)
     */
    public function form_load_data($data, $source=DM::SOURCE_NONE, $merge=false){
        $data = (array)$data;
        $new_data = $merge ? $this->get_whole_deadline_data($source) : [];

        $can_override = DM::can_override_restrictions($this->_courseid);
        foreach ($this->iterate() as $cmid => $dma){
            if (!$dma->is_enabled()) continue;
            if (!$dma->is_editable($source, true)){
                $new_data[$cmid] = $dma->get_deadline_object($source, true);
                continue;
            }

            $key_dl = static::FORM_DEADLINE_PREFIX.$cmid;
            if (empty($data[$key_dl])) continue;

            $dl = $data[$key_dl];
            if (is_array($dl)){
                $dl = ($dl['enabled'] ?? false) ?
                    make_timestamp($dl['year'], $dl['month'], $dl['day'], $dl['hour'], $dl['minute'], 0,
                        $this->_dm_timezone, true
                    ) : null;
            }

            if ($can_override){
                $key_or = static::FORM_OVERRULE_PREFIX.$cmid;
                $or = $data[$key_or] ?? false;
            } else {
                $or = $dma->get_overrule($source);
            }

            $new_data[$cmid] = static::make_dl_object($dl, $or);
        }

        $this->set_whole_deadline_data($new_data, $source);
    }

    /**
     * Return array with errors for the form validation
     *
     * @param string $source - one of the {@see \block_ned_teacher_tools\deadline_manager_entity::SOURCES}
     * @param bool  $recalculate - recheck all conflicts, if true
     *
     * @return array of "element_name"=>"error_description" if there are errors,
     *         or an empty array if everything is OK
     */
    public function form_get_validation($source=DM::SOURCE_NONE, $recalculate=false){
        $this->check_conflicts($recalculate);
        $errors = [];

        foreach ($this->iterate() as $cmid => $dma){
            if (!$dma->is_enabled()) continue;
            if (!$dma->is_editable($source)) continue;
            if ($dma->get_source() != $source && $dma->get_source(true) != $source) continue;

            if ($conflict = $dma->get_date_conflict()){
                $errors[static::FORM_DEADLINE_PREFIX.$cmid] = $conflict;
            }
        }

        return $errors;
    }
    //endregion

    //region DB
    /**
     * Save data from some source to the DB
     * Works only with sources as user or group
     *
     * @param string $source             - one of the {@see \block_ned_teacher_tools\deadline_manager_entity::SOURCES},
     *                                   but we really expect here only user or group source
     * @param bool   $check_capabilities - if true, require capability for editing
     *
     * @return bool
     */
    public function save_data($source=DM::SOURCE_NONE, $check_capabilities=true){
        $groupid = null;
        $userid = null;
        if ($source == DM::SOURCE_GROUP){
            if (empty($this->_groupid)) return false;
            if ($check_capabilities && !DM::can_edit_group_deadline($this->_courseid)) return false;

            $groupid = $this->_groupid;
        } elseif ($source == DM::SOURCE_USER){
            if (empty($this->_userid)) return false;
            if ($check_capabilities && !DM::can_edit_user_deadline($this->_courseid)) return false;

            $userid = $this->_userid;
        } else {
            return false;
        }

        $new_data = [];
        foreach ($this->iterate() as $cmid => $dma){
            if (!$dma->is_enabled() || !$dma->is_editable($source)) continue;

            if ($dma->dm_module->set_override($groupid, $userid, $dma->get_deadline($source), $dma->is_overruled($source))){
                $new_data[$cmid] = $dma->get_deadline_object($source);
            }
        }

        // we need to refresh original(!) deadlines which we have saved in the DB
        $this->set_whole_deadline_data($new_data, $source, true, true);

        return true;
    }
    //endregion

    //region Check methods
    /**
     * Check that any|all the deadlines more|less some time
     * Some examples:
     * User has some upcoming deadline - $compare_rule=SH::COMPARE_MORE, $logic_rule=SH::LOGIC_ANY
     * User has all passed deadlines - $compare_rule=SH::COMPARE_LESS, $logic_rule=SH::LOGIC_ALL
     *
     * @param int   $time         - timestamp for compare with, UNIX time; NOW by default
     * @param int   $compare_rule - check, that deadline more|less than $time; one of the {@see \local_ned_controller\shared\C::COMPARE_ML_OPTIONS}
     * @param int   $logic_rule   - ANY|ALL activities should have $wanted_status; one of the {@see \local_ned_controller\shared\C::LOGIC_ANY_OR_ALL}
     * @param array $cms_filter   - additional filter with cmids, if provided - check only them, otherwise check all activities from the DM
     *
     * @return bool
     */
    function compare_deadlines($time=null, $compare_rule=SH::COMPARE_MORE, $logic_rule=SH::LOGIC_ANY, $cms_filter=[]){
        $time = $time ?? time();
        $more = ($compare_rule ?? SH::COMPARE_MORE) == SH::COMPARE_MORE;
        $rule_any = ($logic_rule ?? SH::LOGIC_ANY) == SH::LOGIC_ANY;
        $rule_all = !$rule_any;
        if ($check_cm = !empty($cms_filter)){
            $cmids = SH::cm_get_cmids_filter($cms_filter);
        }

        foreach ($this->iterate() as $cmid => $dma){
            if ($check_cm && empty($cmids[$cmid])) continue;
            if (!$dma->is_enabled()) continue;

            $deadline = $dma->get_deadline();
            if (!$deadline) continue;

            $yes = $more ? ($deadline > $time) : ($deadline < $time);
            if ($yes){
                if ($rule_any) return true;
            } elseif ($rule_all) {
                return false;
            }
        }

        return $rule_all;
    }
    //endregion

    //region Static
    /**
     * Return all active deadline manager modules (dmm) for the course
     *
     * @param numeric $courseid
     *
     * @return \block_ned_teacher_tools\mod\deadline_manager_mod[] - [cmid => \block_ned_teacher_tools\mod\deadline_manager_mod]
     */
    static public function get_course_dm_modules($courseid){
        if (!isset(static::$_course_dm_modules[$courseid])){
            $res = [];
            // Ignores visible/access restrictions, get all activities and check only DM availability
            $cms = SH::get_course_cms($courseid);
            foreach ($cms as $cm){
                $module = DM::get_dmm_by_cm($cm, $courseid);
                if ($module && $module->is_enabled()){
                    $res[$cm->id] = $module;
                }
            }

            static::$_course_dm_modules[$courseid] = $res;
        }
        return static::$_course_dm_modules[$courseid];
    }

    /**
     * Check some course capability
     *
     * @param numeric $courseid
     * @param string  $capability the name of the capability to check. For example mod/forum:view
     *
     * @return bool
     */
    static public function has_course_capability($courseid, $capability){
        if (!isset(static::$_course_capabilities[$courseid][$capability])){
            static::$_course_capabilities[$courseid][$capability] = SH::has_capability($capability, SH::ctx($courseid));
        }
        return static::$_course_capabilities[$courseid][$capability];
    }

    /**
     * Make and return deadline object in format, which current class uses to store data
     *
     * @param int   $deadline
     * @param bool  $overrule
     *
     * @return object
     */
    static public function make_dl_object($deadline=0, $overrule=false){
        return (object)[
            static::FIELD_TIME => $deadline,
            static::FIELD_OVERRULE => $overrule,
        ];
    }

    /**
     * Check permission to award DL extension after last activity is submitted.
     *
     * @return bool
     */
    public function can_award_extension_after_last_activity() {
        if (!is_siteadmin() && SH::is_course_final_evaluation_completed($this->_courseid, $this->_userid)) {
            return false;
        }
        $issubmitted = false;
        foreach ($this->iterate() as $cmid => $dma) {
            if (!$dma->is_enabled()) continue;
            $tags = SH::cm_get_tags($cmid);
            if (in_array(SH::TAG_LAST_ACTIVITY, $tags)) {
                $dm_module = $this->get_dm_module($cmid);
                $issubmitted = $dm_module->is_submitted($this->_userid);
                break;
            }
        }
        if (!$issubmitted){
            return true;
        }
        return SH::has_capability('block/ned_teacher_tools:canawarddeadlineextensionafterlastactivity',
            SH::ctx($this->_courseid));
    }
    //endregion
}
