<?php
/**
 * @package    block_ned_teacher_tools
 * @subpackage NED
 * @copyright  2020 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @noinspection PhpUnused
 */

namespace block_ned_teacher_tools;
use block_ned_teacher_tools\output\grading_tracker_render as GTR;
use block_ned_teacher_tools\shared_lib as SH;
use local_ned_controller\marking_manager\marking_manager as MM;
use local_ned_controller\marking_manager\mm_data_by_activity_user;
use block_ned_teacher_tools\support\gt_attempt_data;

require_once(__DIR__ . '/../lib.php');
/** @var \stdClass $CFG */
require_once($CFG->dirroot . '/calendar/lib.php');

/**
 * Class grading_tracker
 *
 * @property-read bool $act;
 * @property-read int $userid;
 * @property-read int $courseid;
 * @property-read \context_course|\context $ctx;
 * @property-read int $cmid;
 * @property-read int $attempt = 0;
 * @property-read bool $latest - {@see grading_tracker::_latest}
 * @property-read object $gt_record;
 * @property-read object|mm_data_by_activity_user $mm_data;
 * @property-read object|gt_attempt_data $attempt_data;
 *
 * From GT record:
 * @property-read int id;
 * @property-read int timestart - UNIX time;
 * @property-read int timeend - UNIX time;
 * @property-read int timemodified - UNIX time;
 * @property-read int deadline - UNIX time;
 * @property-read int workdays;
 * @property-read int timegrade - UNIX time;
 * @property-read int graderid;
 * @property-read bool suspended;
 * @property-read bool proxy;
 * @property-read bool uncounted;
 * @property-read string uncounted_reason;
 * @property-read bool bug;
 * @property-read string bug_report;
 * @property-read bool hidden;
 * @property-read string note;
 */
class grading_tracker {
    const TABLE = 'block_ned_teacher_tools_grtr';
    const T_DEF_ZERO_FIELDS = ['timestart', 'timeend', 'deadline', 'timegrade', 'graderid',
        'hidden', 'workdays', 'attempt', 'suspended', 'proxy', 'bug', 'uncounted', 'timemodified', 'latest'];

    const MOD_ASSIGN = MM::MOD_ASSIGN;
    const MOD_QUIZ = MM::MOD_QUIZ;
    const MOD_FORUM = MM::MOD_FORUM;
    const MOD_ALL = MM::MOD_ALL;
    const MOD_TYPES = [self::MOD_ASSIGN, self::MOD_QUIZ, self::MOD_FORUM];

    const ROLE_OT_SHORTNAME = SH::ROLE_OT;
    const ROLE_STUDENT = SH::ROLE_STUDENT;

    const EXTRA_2_DAYS_SUBMISSIONS = 2;
    const EXTRA_5_DAYS_SUBMISSIONS = 5;

    const UPD_NONE = 'none';
    const UPD_ALL = 'all';
    const UPD_CHANGED = 'changed';
    const UPD_REMOVED = 'removed';
    const UPD_ADDED = 'added';
    const UPD_ERROR = 'error';
    const STATUSES_UPD = [self::UPD_NONE, self::UPD_ADDED, self::UPD_CHANGED, self::UPD_REMOVED, self::UPD_ALL, self::UPD_ERROR];

    const GT_RECORD_KEYS_ACCESS = ['id', 'timestart', 'timeend', 'deadline', 'workdays', 'timegrade', 'graderid',
       'suspended', 'proxy', 'uncounted', 'uncounted_reason', 'bug', 'bug_report', 'hidden', 'note',
    ];

    const CAP_CANT_VIEW = 0;
    const CAP_SEE_OWN = 1;
    const CAP_SEE_ALL = 2;

    const USE_CACHE_NONE = 0;
    const USE_CACHE_COURSE = 1;
    const USE_CACHE_USER = 2;

    const DATE_20210616 = 1623860000;

    static protected $_event_data = [];
    static protected $_config;
    static protected $_weekend_names = [];
    static protected $_weekend = [];
    static protected $_use_cache = self::USE_CACHE_NONE;
    static protected $_use_inday_cache = false;

    //region Static caches
    /** @var array|object[][][][] [courseid => [userid => [day_start => records of this day start period]]] */
    static protected $_cache_inday_data = [];
    /**
     * Normally here should be saved data by only one course
     * @var array|object[][][] [courseid => [userid => [cmid => $mm_record]]]
     */
    static protected $_cache_mm_data = [];
    /**
     * Normally here should be saved data by only one course
     * @var array|gt_attempt_data[][][][] [courseid => [userid => [cmid => [attempt => $gt_attempt_data]]]]
     */
    static protected $_cache_attempt_data = [];
    /**
     * Normally here should be saved data by only one course
     * @var array|int[][][] [courseid => [userid => [cmid => (int)max_attempt]]]
     */
    static protected $_cache_max_attempts = [];
    //endregion

    protected $_act = false;
    protected $_ctx;
    protected $_courseid;
    protected $_userid;
    protected $_cmid;
    protected $_attempt = 0;
    /** @var bool - this property shows that current attempt is really the latest for now, can be different from the DB value */
    protected $_latest = null;

    /** @var string - normally you should get its value through {@see grading_tracker::get_modname()} */
    protected $_modname;
    /** @var array|null */
    protected $_tags;

    /** @var object|null */
    protected $_gt_record;
    /**
     * Normally mm_data exists only for latest attempt
     * @var object|mm_data_by_activity_user|null
     */
    protected $_mm_data;
    /**
     * Normally gt_attempt_data exists only for previous (non-latest) attempts
     * @var object|gt_attempt_data|null
     */
    protected $_attempt_data;


    /**
     * grading_tracker constructor.
     *
     * @param                       $courseid
     * @param                       $cmid
     * @param                       $userid
     * @param int|null              $attempt - null means last attempt
     * @param string                $type
     * @param null|\context_course  $context
     * @param null|object           $record
     * @param null|object|mm_data_by_activity_user|gt_attempt_data $mm_or_attempt_data - mm_data for the latest attempt, or gt_attempt data for
     *                                                                                 previous attempts
     */
    public function __construct($courseid, $cmid, $userid, $attempt=null, $type=self::MOD_ALL, $context=null, $record=null, $mm_or_attempt_data=null){
        $this->_courseid = $courseid;
        $this->_cmid = $cmid;
        $this->_userid = $userid;
        $this->_act = false;

        if (empty($courseid) || empty($cmid) || empty($userid)) return;
        if (!static::is_gt_courseid($courseid)) return;
        if (!SH::user_is_student($userid, $courseid, null, null, null, false)) return;

        $this->_act = true;
        $this->_ctx = $context ?: SH::ctx($courseid);

        $mm_data = null;
        $attempt_data = null;
        if (!empty($mm_or_attempt_data)){
            if (!empty($mm_or_attempt_data->is_attempt_data)){
                $attempt_data = $mm_or_attempt_data;
            } else {
                $mm_data = $mm_or_attempt_data;
            }
        }

        if (is_null($attempt)){
            $this->_attempt = $this->get_actual_max_attempt();
            $this->_latest = true;
        } else {
            $this->_attempt = (int)$attempt;
            $this->_latest = $this->is_max_attempt();
        }

        if ($this->_latest){
            $this->_mm_data = $mm_data ?? static::get_mm_data($this->_courseid, $cmid, $this->_userid, $this->_ctx);
        } else {
            $this->_attempt_data = $attempt_data ?? static::get_attempt_data($this->_courseid, $cmid, $this->_userid, $this->_attempt);
        }

        $this->_gt_record = $record ?? static::get_record_by_cmid_userid_attempt($cmid, $userid, $this->_attempt);
        $this->_modname = null;
        if ($type && $type !== static::MOD_ALL){
            $this->_modname = $type;
        }
    }

    /**
     * @param $name
     *
     * @return null
     */
    public function __get($name){
        $pr_name = '_' . $name;
        if (property_exists($this, $pr_name)){
            return $this->$pr_name;
        }

        if (in_array($name, static::GT_RECORD_KEYS_ACCESS)){
            return $this->_gt_record->$name ?? null;
        }

        return $this->$name ?? null;
    }

    /**
     * @param $name
     *
     * @return bool
     */
    public function __isset($name){
        $val = $this->__get($name);
        return !is_null($val);
    }

    public function __destruct(){
        $this->_act = false;
        unset($this->_mm_data);
        unset($this->_attempt_data);
        unset($this->_gt_record);
    }

    /**
     * Debug for cli tasks
     *
     * @param string   $text
     * @param string[] $add_info
     */
    public function debug($text, $add_info=[]){
        if (!empty($this->_gt_record->id)){
            $id_info = $this->_gt_record->id;
        } else {
            $attempt = $this->_attempt ?? '?';
            $id_info = "cm.$this->_cmid:u.$this->_userid#$attempt";
        }
        static::cli_debug($text, $add_info, $id_info);
    }

    /**
     * @param bool|int|string $update_timemodified - update timemodified to current date if it's true, or for its value, if it's number
     *
     * @return bool|int
     */
    public function save($update_timemodified=true){
        return static::save_record($this->_gt_record, $update_timemodified);
    }

    /**
     * Save current mm_data for the cache
     * @see get_mm_data()
     *
     * @param \stdClass|mm_data_by_activity_user|null $data - if you wish save "nothing", use false, not null
     */
    public function save_mm_data_for_cache($data=null){
        if (!static::is_use_cache()) return;

        $mm_data = $data ?? $this->_mm_data ?? null;
        if (is_null($mm_data)) return;

        static::$_cache_mm_data[$this->_courseid][$this->_userid][$this->_cmid] = $mm_data;
    }

    /**
     * @return bool|null
     */
    public function remove_previous_ungraded_submissions(){
        if (!$this->act || !$this->gt_record || !$this->attempt) return null;

        $where = ["attempt < :maxattempt"];
        $params = ['maxattempt' => $this->_attempt];
        SH::sql_add_equal('cmid', $this->_cmid, $where, $params);
        SH::sql_add_equal('userid', $this->_userid, $where, $params);
        SH::sql_add_equal('timegrade', 0, $where, $params);

        return SH::db()->delete_records_select(static::TABLE, SH::sql_condition($where), $params);
    }

    /**
     * Return true, if successfully add record, false otherwise
     *
     * @param null|int $submission_time
     * @param null|int $timegrade
     * @param null|int $deadline
     * @param bool     $precheck
     *
     * @return bool
     */
    public function add_activity($submission_time=null, $timegrade=null, $deadline=null, $precheck=true){
        if (!$this->act){
            $this->debug("Can't add, as this GT is not act");
            return false;
        }

        if ($this->_gt_record){
            $this->debug("Can't add, as such record has already exist");
            return false;
        }

        $config = static::get_config();
        if ($config->gt_startdate > time()){
            $this->debug("GT start date in the future");
            return false;
        }

        $graderid = null;

        if ($precheck){
            if ($this->_latest){
                $mm_data = $this->mm_data;

                if(!$mm_data || !($mm_data->gt_shouldpass ?? false)){
                    $this->debug("Can't add, as no mm_data or it shouldn't pass");
                    return false;
                }

                if (!empty($mm_data->excluded)){
                    $this->debug("Excluded grade");
                    return false;
                }

                if (empty($mm_data->marked)){
                    $timegrade = 0;
                } else {
                    $timegrade = $mm_data->first_last_grade_time ?? $timegrade;
                    if ($timegrade){
                        $graderid = $mm_data->graderid ?? null;
                    }
                }

                $submission_time = $mm_data->submit_time ?? $submission_time;
            } else {
                $attempt_data = $this->attempt_data;
                if (empty($attempt_data->timegrade)){
                    $this->debug("Can't add, as it's previous attempt and there is no time grade");
                    return false;
                }

                $timegrade = $attempt_data->timegrade;
                $submission_time = $attempt_data->submit_time;
                $graderid = $attempt_data->graderid;
            }

            if (!$this->has_proper_groupname()){
                $this->debug("Can't add, have denied group name");
                return false;
            }

            if (!$this->has_config_gt_tag()){
                $this->debug("Can't add, as no necessary tags");
                return false;
            }
        }

        $submission_time = $submission_time ?? $this->get_submission_time(false, true);
        $deadline = $deadline ?? $this->get_deadline(false, true);

        // Shouldn't be such situation at the normal work process
        if (!$submission_time){
            if (!$deadline){
                $this->debug("Can't add, as no submission time and no deadline");
                return false;
            }

            if ($deadline > time()){
                $this->debug("Can't add, as no submission time and deadline in the future");
                return false;
            }

            if (!$timegrade && !SH::get_cm_visibility_by_user($this->_cmid, $this->_userid, false, false)){
                $this->debug("Can't add, as user hasn't access to the activity");
                return false;
            }
        }

        if (!empty($config->gt_startdate)){
            $gt_startdate = (int)$config->gt_startdate;
            if (max((int)$submission_time, (int)$timegrade) < $gt_startdate &&
                ($submission_time || $timegrade || (int)$deadline < $gt_startdate)){
                $this->debug("Can't add, as all record stats starts after GT start date");
                return false;
            }
        }

        static::set_use_inday_cache(false);
        $graderid = $graderid ?? $this->get_potential_graderid();
        $grade_time_start = $this->get_grade_time_start($submission_time, $deadline);
        $days = $this->get_time_to_grade($grade_time_start, $deadline);
        $grade_deadline = $this->get_grade_timeend($grade_time_start, $days, $deadline);
        $today_records = $this->get_same_day_records_not_forums($grade_time_start);
        $suspended = !static::is_student($this->_userid, $this->_courseid);

        // save new record before checking other records
        $params = ['courseid' => $this->_courseid, 'cmid' => $this->_cmid, 'userid' => $this->_userid, 'attempt' => $this->_attempt,
            'timestart' => $submission_time, 'timeend' => $grade_deadline, 'deadline' => $deadline,
            'graderid' => $graderid, 'workdays' => $days, 'suspended' => $suspended, 'proxy' => $this->is_cm_proxy(), 'latest' => $this->_latest];

        if ($timegrade){
            $params['timegrade'] = $timegrade;
        }

        $res = static::save_record($params);
        if ($res){
            $this->_gt_record = (object)$params;
            $this->_gt_record->id = ($this->_gt_record->id ?? 0) ?: $res;
            $this->save_mm_data_for_cache();

            $s_deadline = SH::ned_date($deadline);
            $latest = (int)$this->_latest;
            $debug_info = [
                "(courseid: $this->_courseid, cmid: $this->_cmid, userid: $this->_userid, " .
                "deadline: $s_deadline, attempt: $this->_attempt, latest: $latest)",
            ];
            $this->debug("Added!", $debug_info);
        } else {
            $this->debug("Can't add, problem with saving record in the DB");
        }

        // update other records, if there are a lot of them in this day from one user
        if ($res && !empty($today_records)){
            $in_day = count($today_records);
            if (empty($this->_gt_record->proxy)){
                $in_day++;
            }
            if ($in_day == static::EXTRA_2_DAYS_SUBMISSIONS || $in_day == static::EXTRA_5_DAYS_SUBMISSIONS){
                foreach ($today_records as $record){
                    static::record_check_update($record);
                }
            }
        }

        return (bool)$res;
    }

    /**
     * @return int|null
     */
    public function get_potential_graderid(){
        return SH::get_graderid_by_studentid($this->_userid, $this->_courseid, $this->_cmid);
    }

    /**
     * @return string
     */
    public function get_modname(){
        if (is_null($this->_modname)){
            $this->_modname = $this->_mm_data->modname ?? static::get_modname_by_cmid($this->_cmid);
        }
        return $this->_modname;
    }

    /**
     * @return int
     */
    public function get_school_students_count(){
        $schoolid = SH::get_user_school($this->userid, true);
        return SH::get_count_students_at_grader_school($this->courseid, $schoolid, $this->graderid);
    }

    /**
     * @return array - list of tag names
     */
    public function get_tags(){
        if (is_null($this->_tags)){
            if (isset($this->_mm_data->tags)){
                $this->_tags = explode(',', $this->_mm_data->tags);
            } else {
                $this->_tags = SH::cm_get_tags($this->_cmid);
            }
        }

        return $this->_tags;
    }

    /**
     * Return true, if object has at least one of the tags from config GT tags
     *
     * @return bool
     */
    public function has_config_gt_tag(){
        return static::is_gt_tags($this->get_tags());
    }

    /**
     * Return true, if current group allowed by GT settings
     *
     * @return bool
     */
    public function has_proper_groupname(){
        $group_ids = SH::get_user_groupids($this->courseid, $this->userid);
        return static::is_groupid_allowed(reset($group_ids));
    }

    /**
     * @param bool $set_time_if_null
     * @param bool $ignore_db_value
     *
     * @return int|null
     */
    public function get_submission_time($set_time_if_null=true, $ignore_db_value=false){
        $timestart = null;
        if (!$ignore_db_value && isset($this->_gt_record->timestart)){
            $timestart = $this->_gt_record->timestart;
        } elseif ($this->_latest){
            $timestart = $this->_mm_data->submit_time ?? $timestart;
        } else {
            $timestart = $this->_attempt_data->submit_time ?? $timestart;
        }
        if (is_null($timestart) && $set_time_if_null){
            $timestart = time();
            $this->_gt_record->timestart = $timestart;
        }
        return $timestart;
    }

    /**
     * @param bool $set_zero_if_null
     * @param bool $ignore_db_value
     *
     * @return int|null
     */
    public function get_deadline($set_zero_if_null=true, $ignore_db_value=false){
        if (!$ignore_db_value && isset($this->_gt_record->deadline)){
            $deadline = $this->_gt_record->deadline;
        } elseif (isset($this->_mm_data->duedate)){
            $deadline = $this->_mm_data->duedate;
        } else {
            $deadline = SH::get_deadline_by_cm($this->_cmid, $this->_userid, $this->_courseid) ?: null;
        }
        if (is_null($deadline) && $set_zero_if_null){
            $deadline = 0;
            if (!empty($this->_gt_record)){
                $this->_gt_record->deadline = $deadline;
            }
        }
        return $deadline;
    }

    /**
     * Return true, if course-module of record is active proxy activity
     * Checks data by activity, if you need value of the record, use $this->_gt_record->proxy
     *
     * @return bool
     */
    public function is_cm_proxy(){
        return $this->_mm_data->has_proxy ?? SH::dm_is_proxy_activity_enabled($this->cmid);
    }

    /**
     * @param null|int $submission_time
     * @param null|int $deadline
     *
     * @return int
     */
    public function get_grade_time_start($submission_time=null, $deadline=null){
        $submission_time = $submission_time ?? $this->get_submission_time(false);
        $deadline = $deadline ?? $this->get_deadline(false);

        if (!empty($this->_gt_record->proxy)){
            return (int)($submission_time ?: $deadline);
        }
        return (int)max($submission_time, $deadline);
    }

    /**
     * @param null|int $grade_time_start
     * @param null|int $deadline
     *
     * @return int
     */
    public function get_time_to_grade($grade_time_start=null, $deadline=null){
        $config = static::get_config();
        $days = 3;
        $deadline = $deadline ?? $this->get_deadline();
        $grade_time_start = $grade_time_start ?? $this->get_grade_time_start(null, $deadline);

        if (!empty(array_intersect($this->get_tags(), static::get_config_tags_5days()))){
            $days += 2;
        }

        if (!empty($config->gradingtracker_manystudents)){
            $students = $this->get_school_students_count();
            foreach ($config->gradingtracker_manystudents as $many_students){
                if (empty($many_students)) continue;

                if ($students >= $many_students){
                    $days++;
                } else {
                    break;
                }
            }
        }

        // Rule about adding day if there is no deadline - is not applying for the records after 16.06.2021
        if ($grade_time_start < static::DATE_20210616 && !$deadline){
            $days++;
        }

        $in_day = count($this->get_same_day_records_not_forums($grade_time_start));
        if (!isset($this->_gt_record->id) && !$this->is_cm_proxy()){
            $in_day++;
        }
        if ($in_day >= static::EXTRA_5_DAYS_SUBMISSIONS){
            $days += static::EXTRA_5_DAYS_SUBMISSIONS;
        } elseif ($in_day >= static::EXTRA_2_DAYS_SUBMISSIONS){
            $days += static::EXTRA_2_DAYS_SUBMISSIONS;
        }

        return $days;
    }

    /**
     * @param null $grade_time_start
     * @param null $days
     * @param null $deadline
     *
     * @return int
     */
    public function get_grade_timeend($grade_time_start=null, $days=null, $deadline=null){
        $deadline = $deadline ?? $this->get_deadline();
        $start_time = $grade_time_start ?? $this->get_grade_time_start(null, $deadline);
        $days = $days ?? $this->get_time_to_grade($grade_time_start, $deadline);
        if (static::is_day_off($start_time)){
            $days++;
        }

        $grade_deadline = $start_time;
        for($d = 0; $d < $days; $d++){
            $grade_deadline += DAYSECS;
            if (static::is_day_off($grade_deadline)){
                $days++;
            }
        }
        [, $end_grade_day] = SH::get_day_start_end($grade_deadline);

        return $end_grade_day;
    }

    /**
     * Get max attempt in current activity
     *
     * @return int|null - null, if it finds none attempt
     */
    public function get_actual_max_attempt(){
        return static::get_max_attempt($this->_courseid, $this->_cmid, $this->_userid);
    }

    /**
     * Check, does it max attempt for the GT
     *
     * @return bool
     */
    public function is_max_attempt(){
        return $this->_attempt >= $this->get_actual_max_attempt();
    }

    /**
     * Return all GT records make in the same day
     * We count as "the same day" all weekends + next work day (on usual week it wll be Saturday, Sunday and Monday)
     *
     * @param int $grade_time_start - time value
     *
     * @return array
     */
    public function get_same_day_records_not_forums($grade_time_start=null){
        if ($this->get_modname() == 'forum'){
            return [];
        }

        $grade_time_start = $grade_time_start ?? $this->get_grade_time_start();

        // We count as "the same day" all weekends + next work day (on usual week it wll be Saturday, Sunday and Monday)
        $start_day = $grade_time_start;
        while($start_day > 0 && static::is_day_off($prev_day = $start_day - DAYSECS)){
            $start_day = $prev_day;
        }

        $end_day = $grade_time_start;
        $c = 0;
        while(static::is_day_off($end_day) && $c < 1000){
            $end_day += DAYSECS;
            $c++;
        }

        if ($start_day == $end_day){
            [$daystart, $dayend] = SH::get_day_start_end($start_day);
        } else {
            [$daystart,] = SH::get_day_start_end($start_day);
            [, $dayend] = SH::get_day_start_end($end_day);
        }

        $res = static::is_use_inday_cache() ? (static::$_cache_inday_data[$this->_courseid][$this->_userid][$daystart] ?? null) : null;
        if (is_null($res)){
            $where = [
                'gt.courseid = :courseid',
                'gt.userid = :userid',
                'gt.proxy = 0',
                'm.name <> "forum"',
                "GREATEST(COALESCE(gt.timestart, 0), COALESCE(gt.deadline, 0)) BETWEEN :daystart AND :dayend",
            ];
            $params = [
                'courseid' => $this->_courseid,
                'userid' => $this->_userid,
                'daystart' => $daystart,
                'dayend' => $dayend,
            ];

            $res = static::get_gt_records($where, $params);
            if (static::is_use_inday_cache()){
                static::$_cache_inday_data[$this->_courseid][$this->_userid][$daystart] = $res;
            }
        }

        return $res;
    }

    /**
     * @param bool $all_attempts
     *
     * @return bool
     */
    public function delete_record($all_attempts=false){
        global $DB;

        if ($all_attempts){
            $params = ['cmid' => $this->_cmid, 'userid' => $this->_userid];
        } elseif (isset($this->_gt_record->id)) {
            $params = ['id' => $this->_gt_record->id];
        } else {
            return false;
        }

        return $DB->delete_records(static::TABLE, $params);
    }

    /* STATIC */

    /**
     * Debug for cli tasks
     *
     * @param string     $text
     * @param string[]   $add_info
     * @param string|int $record_id
     */
    static public function cli_debug($text, $add_info=[], $record_id=null){
        $rid = $record_id ? '['.$record_id.']' : '';
        SH::cli_debugging('GM'.$rid.' '.$text, $add_info);
    }

    /**
     * Checks for any GT capabilities
     *
     * @param \context|null         $context - by default uses context_system
     * @param numeric|object|null   $user_or_id - by default uses current user
     *
     * @return bool
     */
    static public function has_any_capabilities($context=null, $user_or_id=null){
        $cap = static::get_capability($context, $user_or_id);
        return $cap > static::CAP_CANT_VIEW;
    }

    /**
     * Return capability to view GT
     *
     * @param \context|null         $context - by default uses context_system
     * @param numeric|object|null   $user_or_id - by default uses current user
     *
     * @return int
     */
    static public function get_capability($context=null, $user_or_id=null){
        $context = $context ?? \context_system::instance();
        if (SH::has_capability('gradingtracker_seeall', $context, $user_or_id)){
            return static::CAP_SEE_ALL;
        } elseif (SH::has_capability('gradingtracker_seeown', $context, $user_or_id)){
            return static::CAP_SEE_OWN;
        } else {
            return static::CAP_CANT_VIEW;
        }
    }

    /**
     * @param      $userid
     * @param int  $courseid
     * @param int  $cmid
     * @param bool $is_active - if true, return true only if user enrolled AND active
     *
     * @return bool
     */
    static public function is_grader($userid, $courseid=0, $cmid=0, $is_active=true){
        return static::is_enrolled_user($userid, static::ROLE_OT_SHORTNAME, $courseid, $cmid, $is_active);
    }

    /**
     * @param      $userid
     * @param int  $courseid
     * @param bool $is_active - if true, return true only if user enrolled AND active
     *
     * @return bool
     */
    static public function is_student($userid, $courseid=0, $is_active=true){
        return static::is_enrolled_user($userid, static::ROLE_STUDENT, $courseid, null, $is_active);
    }

    /**
     * @param      $userid
     * @param      $rolename
     * @param int  $courseid
     * @param int  $cmid
     * @param bool $is_active - if true, return true only if user enrolled AND active
     *
     * @return bool
     */
    static public function is_enrolled_user($userid, $rolename, $courseid=0, $cmid=0, $is_active=true){
        global $DB;
        $where = ["ue.userid = :userid"];
        $params = ['userid' => $userid];
        if ($rolename == static::ROLE_STUDENT){
            // Normally, $cmid shouldn't be called with student role, so here some safety, if it does
            $cmid = null;
        }
        $sql = static::sql_user_enrolments('ue.id', $where, $params, $rolename, $courseid, $cmid, $is_active);
        return $DB->record_exists_sql($sql, $params);
    }

    /**
     * Return sql with FROM, JOIN and WHERE for getting user_enrolments
     *  if sent $params, necessary data will be saved here
     * How exactly checking rules works, @see \local_ned_controller\shared\db_util::sql_user_enrolments()
     *
     * @param string             $select    - by default return user_enrolments id
     *                                      use "ra_cm.id" if you need to get cm->id data (instead of "cm.id"), when $cmid is not NULL
     * @param array|string       $where     - array of conditions for AND
     * @param array              $params    - array for query params, new params also will be saved here
     * @param string|array       $rolenames - shortname(s) from the role table
     * @param numeric|false|null $courseid  - course id, null to check nothing, false to check all, or id
     * @param numeric|bool|null  $cmid      - course module id, null to check nothing, false to check all, true - check all by course, or id
     * @param bool|int           $is_active - if false - get all, if true, return only active users, if === -1 - return only inactive
     * @param string             $other_sql - any other sql string to add after WHERE conditions
     * @param string|array       $add_join  - add other tables to join, if == 'group' - will add group table to join
     * @param string             $prefix    - prefix to sql params
     *
     * @return string
     */
    static public function sql_user_enrolments($select='ue.id', $where=[], &$params=[], $rolenames=self::ROLE_STUDENT,
        $courseid=SH::ALL, $cmid=null, $is_active=true, $other_sql='', $add_join='', $prefix='ue_'){
        return SH::sql_user_enrolments($select, $where, $params, $rolenames, $courseid, SH::ALL, $cmid, SH::ALL,
            $is_active, $other_sql, $add_join, $prefix);
    }

    /**
     * @param     $userid
     * @param int $courseid
     * @param int $cmid
     *
     * @return bool
     */
    static public function has_such_grader($userid, $courseid=0, $cmid=0){
        global $DB;
        static $_data = [];
        if (!isset($_data[$userid][$courseid][$cmid])){
            $params = ['graderid' => $userid];
            if ($courseid){
                $params['courseid'] = $courseid;
            }
            if ($cmid){
                $params['cmid'] = $cmid;
            }

            $_data[$userid][$courseid][$cmid] = $DB->record_exists(static::TABLE, $params);
        }

        return $_data[$userid][$courseid][$cmid];
    }

    /**
     * @param      $userid
     * @param int  $courseid
     * @param int  $cmid
     * @param null $active - return only active is true, or only suspended if false, or all if null
     *
     * @return bool
     */
    static public function has_such_student($userid, $courseid=0, $cmid=0, $active=null){
        global $DB;
        static $_data = [];
        $active_key = is_null($active) ? 0 : ($active ? 1 : 2);
        if (!isset($_data[$userid][$courseid][$cmid][$active_key])){
            $params = ['userid' => $userid];
            if ($courseid){
                $params['courseid'] = $courseid;
            }
            if ($cmid){
                $params['cmid'] = $cmid;
            }
            if (!is_null($active)){
                $params['suspended'] = (int)(!$active);
            }

            $_data[$userid][$courseid][$cmid][$active_key] = $DB->record_exists(static::TABLE, $params);
        }

        return $_data[$userid][$courseid][$cmid][$active_key];
    }

    /**
     * @return array
     */
    static public function get_all_gt_students(){
        global $DB;
        static $_data = null;
        if (is_null($_data)){
            $t = static::TABLE;
            /** @noinspection PhpUnnecessaryCurlyVarSyntaxInspection */
            $sql = "SELECT DISTINCT gt.userid, u.* 
                FROM {{$t}} gt
                JOIN {user} u ON u.id = gt.userid
            ";
            $_data = $DB->get_records_sql($sql);
        }

        return $_data;
    }

    /**
     * @param $userid
     *
     * @return \stdClass|null
     */
    static public function get_user($userid){
        return SH::get_user($userid);
    }

    /**
     * Return records by user and cm, sorted by attempt, or only one record for the last attempt
     * @param \cm_info|object|numeric   $cm_or_id
     * @param object|numeric            $user_or_id
     * @param bool                      $only_last - optional, if true - return only one record for last attempt (or null)
     *
     * @return array|object[]|object|null - if $only_last is true, return object or null, otherwise return array
     */
    static public function get_record_by_cmid_userid($cm_or_id, $user_or_id, $only_last=false){
        $params = ['cmid' => SH::get_id($cm_or_id), 'userid' => SH::get_id($user_or_id)];
        if ($only_last){
            $params['latest'] = 1;
        }

        $records = SH::db()->get_records(static::TABLE, $params);
        if ($only_last){
            return reset($records) ?: null;
        }

        return $records;
    }

    /**
     * @param      $cmid
     * @param      $userid
     * @param int  $attempt
     *
     * @return \stdClass|false
     */
    static public function get_record_by_cmid_userid_attempt($cmid, $userid, $attempt=0){
        global $DB;
        static $data = [];
        if (!isset($data[$cmid][$userid][$attempt])){
            $params = ['cmid' => $cmid, 'userid' => $userid, 'attempt' => $attempt];
            $data[$cmid][$userid][$attempt] = $DB->get_record(static::TABLE, $params);
        }

        return $data[$cmid][$userid][$attempt];
    }

    /**
     * @param $gtid
     *
     * @return \stdClass|false
     */
    static public function get_record_by_gtid($gtid){
        global $DB;
        static $data = [];
        if (!isset($data[$gtid])){
            $data[$gtid] = $DB->get_record(static::TABLE, ['id' => $gtid]);
        }

        return $data[$gtid];
    }

    /**
     * @param $cmid
     *
     * @return string|null
     */
    static public function get_modname_by_cmid($cmid){
        global $DB;
        $select = ['m.name AS `module`'];
        $from = ["{course_modules} cm
        JOIN {modules} m
            ON cm.module = m.id
        LEFT JOIN {assign} a
            ON m.name = 'assign'
            AND a.id = cm.instance
        LEFT JOIN {forum} f
            ON m.name = 'forum'
            AND f.id = cm.instance
        LEFT JOIN {quiz} q
            ON m.name = 'quiz'
            AND q.id = cm.instance
        "];
        $where = ['cm.id = :cmid'];
        $params = ['cmid' => $cmid];

        $select = "SELECT ". join(', ', $select);
        $from = "\nFROM ".join("\n", $from);
        $where = !empty($where) ? ("\nWHERE (" . join(') AND (', $where) . ')') : '';

        $record = $DB->get_record_sql($select.$from.$where, $params);
        return $record->module ?? null;
    }

    /**
     * Simple get records from GT table
     *
     * @param array $conditions optional array $fieldname=>requestedvalue with AND in between
     * @param string $sort an order to sort the results in (optional, a valid SQL ORDER BY parameter).
     * @param string $fields a comma separated list of fields to return (optional, by default all fields are returned).
     *                          The first field will be used as key for the array so must be a unique field such as 'id'.
     * @param int $limitfrom return a subset of records, starting at this point (optional).
     * @param int $limitnum return a subset comprising this many records in total (optional, required if $limitfrom is set).
     *
     * @return array An array of Objects indexed by first column.
     */
    static public function get_records($conditions=[], $sort='', $fields='*', $limitfrom=0, $limitnum=0){
        return SH::db()->get_records(static::TABLE, $conditions, $sort, $fields, $limitfrom, $limitnum) ?: [];
    }

    /**
     * Get records from GT table which match a particular WHERE clause.
     *
     * @param string|array $conditions A fragment of SQL to be used in a where clause in the SQL call.
     * @param array        $params     An array of sql parameters
     * @param string       $sort       An order to sort the results in (optional, a valid SQL ORDER BY parameter).
     * @param string       $fields     A comma separated list of fields to return
     *                                 (optional, by default all fields are returned). The first field will be used as key for the
     *                                 array so must be a unique field such as 'id'.
     * @param int          $limitfrom  return a subset of records, starting at this point (optional).
     * @param int          $limitnum   return a subset comprising this many records in total (optional, required if $limitfrom is set).
     *
     * @return array An array of Objects indexed by first column.
     */
    static public function get_records_select($conditions='', $params=[], $sort='', $fields='*', $limitfrom=0, $limitnum=0){
        if (is_array($conditions)){
            $conditions = SH::sql_condition($conditions);
        }
        return SH::db()->get_records_select(static::TABLE, $conditions, $params, $sort, $fields, $limitfrom, $limitnum) ?: [];
    }

    /**
     * @param array  $where
     * @param array  $params
     * @param string $sql_after_where
     *
     * @return array
     */
    static public function get_gt_records($where=[], $params=[], $sql_after_where=''){
        global $DB;
        $select = ["gt.*", 'm.name AS `module`'];
        $from = ["{".static::TABLE."} gt
        JOIN {course} c
            ON c.id = gt.courseid
        JOIN {course_modules} cm
            ON cm.id = gt.cmid
        JOIN {modules} m
            ON cm.module = m.id
        LEFT JOIN {assign} a
            ON m.name = 'assign'
            AND a.id = cm.instance
        LEFT JOIN {forum} f
            ON m.name = 'forum'
            AND f.id = cm.instance
        LEFT JOIN {quiz} q
            ON m.name = 'quiz'
            AND q.id = cm.instance
        "];

        $select = "SELECT ". join(', ', $select);
        $where = !empty($where) ? ("\nWHERE (" . join(') AND (', $where) . ')') : '';

        if (strpos($select, 'grp.') !== FALSE || strpos($where, 'grp.') !== FALSE){
            $from[] = "
                LEFT JOIN (
                    SELECT grp.id, g_m.userid, grp.courseid, grp.name
                    FROM {groups} grp
                    JOIN {groups_members} g_m
                       ON g_m.groupid = grp.id
                    GROUP BY grp.courseid, g_m.userid
                ) grp  
                    ON grp.courseid = gt.courseid
                    AND grp.userid = gt.userid
            ";
        }

        $from = "\nFROM ".join("\n", $from);
        $sql_after_where = "\n".$sql_after_where;

        $records = $DB->get_records_sql($select.$from.$where.$sql_after_where, $params);
        return $records ?: [];
    }

    /* for events */

    /**
     * @param          $courseid
     * @param          $cmid
     * @param          $graderid
     * @param          $userid
     * @param int|null $attempt
     * @param string   $type
     * @param null     $context
     * @param int|null $maxattempt
     */
    static public function grade_activity($courseid, $cmid, $graderid, $userid, $attempt=null, $type=self::MOD_ALL, $context=null, $maxattempt=null){
        if (isset(static::$_event_data[$cmid][$userid][$attempt])){
            return;
        }
        static::$_event_data[$cmid][$userid][$attempt] = true;

        $GT = new static($courseid, $cmid, $userid, $attempt, $type, $context);
        if (!$GT->act || !$GT->gt_record || ($GT->gt_record->timegrade ?? 0)){
            return;
        }

        $mm_data = $GT->mm_data;
        $maxattempt = $maxattempt ?? $GT->get_actual_max_attempt();
        if ($maxattempt == $GT->attempt){
            if(!($mm_data->marked ?? false) && !($mm_data->reopened ?? false)){
                return;
            }

            $GT->remove_previous_ungraded_submissions();
            $GT->gt_record->timegrade = $mm_data->first_last_grade_time ?? time();
        } else {
            $GT->gt_record->timegrade = time();
        }

        $GT->gt_record->graderid = $graderid;
        $GT->save();
    }

    /**
     * Check $userid as grader in GT, for records, which should change
     *  if $new == $old, check all cases (when user can become grade, and stop becoming grader)
     *
     * @param      $userid
     * @param int  $courseid
     * @param int  $cmid
     * @param bool $new
     * @param bool $old
     *
     * @noinspection PhpUnusedParameterInspection
     */
    static public function check_updated_grader($userid, $courseid=0, $cmid=0, $new=false, $old=false){
        // Turn off due to the PIPE-83
        return;
        /** @noinspection PhpUnreachableStatementInspection */
        $new = (bool)$new;
        $old = (bool)$old;
        $old_or_new = ($new == $old);

        if (static::has_such_student($userid, $courseid, $cmid)){
            // may be $userid - student, who have new grader or lose old grader
            static::update_gt_graders($courseid, $cmid, null, null, null, $userid);
        }

        if ($old_or_new){
            static::update_gt_graders($courseid, $cmid, null, null, $userid);
        } elseif ($new){
            if (!static::is_grader($userid, $courseid, $cmid)){
                return;
            }
            static::update_gt_graders($courseid, $cmid, 0, $userid);
        } else {
            // old
            if (!static::has_such_grader($userid, $courseid, $cmid)){
                return;
            }
            static::update_gt_graders($courseid, $cmid, $userid);
        }
    }

    /**
     * Check student records for suspend
     *
     * @param      $userid
     * @param int  $courseid
     * @param int  $cmid
     * @param null $set_suspend
     */
    static public function check_updated_student($userid, $courseid=0, $cmid=0, $set_suspend=null){
        if (!static::has_such_student($userid, $courseid, $cmid, $set_suspend)){
            return;
        }

        if (is_null($set_suspend)){
            static::update_gt_suspend_students($courseid, $userid, $cmid);
        } else {
            static::set_suspended_user($userid, $courseid, $cmid, $set_suspend);
        }
    }

    /* update records */

    /**
     * Add activity by MM data
     * Return number of successfully added records
     *
     * @param object|mm_data_by_activity_user[]|array $mm_data_or_list
     * @param numeric                                 $courseid
     * @param null|\context_course                    $context
     * @param bool                                    $precheck
     *
     * @return int
     */
    static public function add_activity_by_mm_data($mm_data_or_list, $courseid, $context=null, $precheck=false){
        if (empty($mm_data_or_list) || !$courseid){
            return false;
        }

        $mm_data_or_list = SH::val2arr($mm_data_or_list);
        $res = 0;
        $prev_cache = static::is_use_cache();
        static::set_use_cache(static::USE_CACHE_NONE);

        foreach ($mm_data_or_list as $mm_data){
            $new_gt = new static($courseid, $mm_data->cmid, $mm_data->userid, $mm_data->attempt ?? 0,
                $mm_data->modname, $context, false, $mm_data);
            $res += (int)$new_gt->add_activity($mm_data->submit_time ?? 0, $mm_data->first_last_grade_time ?? 0,
                $mm_data->duedate ?? 0, $precheck);
            unset($new_gt);
        }

        unset($mm_data_or_list);
        static::set_use_cache($prev_cache);
        return $res;
    }

    /**
     * Add activity by attempt data
     * Return number of successfully added records
     *
     * @param object|array|gt_attempt_data[] $attempt_data_or_list
     * @param numeric                        $courseid
     * @param null|\context_course           $context
     * @param bool                           $precheck
     *
     * @return int
     */
    static public function add_activity_by_attempt_data($attempt_data_or_list, $courseid, $context=null, $precheck=false){
        if (empty($attempt_data_or_list) || !$courseid){
            return false;
        }

        $attempt_data_or_list = SH::val2arr($attempt_data_or_list);
        $res = 0;
        $prev_cache = static::is_use_cache();
        static::set_use_cache(static::USE_CACHE_NONE);

        foreach ($attempt_data_or_list as $attempt_data){
            /** @var gt_attempt_data|object $attempt_data */
            $new_gt = new static($courseid, $attempt_data->cmid, $attempt_data->userid, $attempt_data->attempt,
                static::MOD_ALL, $context, false, $attempt_data);
            $res += (int)$new_gt->add_activity($attempt_data->submit_time, $attempt_data->timegrade, null, $precheck);
            unset($new_gt);
        }

        unset($attempt_data_or_list);
        static::set_use_cache($prev_cache);
        return $res;
    }

    /**
     * @param                 $data
     * @param bool|int|string $update_timemodified - update timemodified to current date if it's true, or for its value, if it's number
     *
     * @return bool|int
     */
    static public function save_record($data, $update_timemodified=true){
        $data = (object)$data;
        if ($update_timemodified){
            $data->timemodified = is_numeric($update_timemodified) ? (int)$update_timemodified : time();
        }

        foreach (static::T_DEF_ZERO_FIELDS as $field){
            if (property_exists($data, $field) && is_null($data->$field)){
                $data->$field = 0;
            }
        }

        if ($data->id ?? false){
            return SH::db()->update_record(static::TABLE, $data);
        } else {
            return SH::db()->insert_record(static::TABLE, $data);
        }
    }

    /**
     * Set new value for all records, which fit some params
     *
     * @param string       $field - name of changed column
     * @param \mixed       $newvalue - new value for this column
     * @param array        $where_params - where keys - columns of TABLE, and values - checked params
     * @param array|string $where - you can add sql where conditions here
     * @param array        $params - array of sql parameters
     *
     * @return bool
     */
    static public function update_records_value($field, $newvalue, $where_params=[], $where=[], $params=[]){
        global $DB;
        $where  = SH::val2arr($where);
        $params = SH::val2arr($params);
        $has_field = false;

        $col_infos = $DB->get_columns(static::TABLE);
        foreach ($col_infos as $column => $col_info){
            if (isset($where_params[$column]) && !(empty($where_params[$column]) && is_array($where_params[$column]))){
                SH::sql_add_get_in_or_equal_options($column, $where_params[$column], $where, $params, 'gt_');
            }

            $has_field = $has_field || $column == $field;
        }

        if (!$has_field || empty($where)){
            // we don't want change nothing or change all the table
            return false;
        }

        $where = SH::sql_condition($where);

        return $DB->set_field_select(static::TABLE, $field, $newvalue, $where, $params);
    }

    /**
     * Set timemodified value for some amount of GT records (by id)
     *
     * @param array|string|int $ids  - id(s) which should change
     * @param int|null  $value       - timemodified value, if null, uses 'now'
     * @param int|null  $before_time - if set, update values only with timemodified earlier than $before_time
     *
     * @return bool
     */
    static public function update_timemodified_value($ids=[], $value=null, $before_time=null){
        if (empty($ids)){
            return false;
        }

        $value = $value ?? time();
        $where_params  = [];
        $where = [];
        $params = [];

        if ($ids === '*'){
            // change full table, use with caution!
            $where[] = SH::SQL_TRUE_COND;
        } else {
            $where_params['id'] = SH::val2arr($ids);
        }

        if ($before_time){
            $where[] = 'timemodified < :before_time';
            $params['before_time'] = $before_time;
        }

        return static::update_records_value('timemodified', $value, $where_params, $where, $params);
    }

    /**
     * @param int|string|null $cmid
     * @param int|string|null $userid
     * @param int|string|null $attempt
     * @param int|string|null $id
     *
     * @return bool
     */
    static public function remove_records($cmid=null, $userid=null, $attempt=null, $id=null){
        global $DB;

        $params = [];
        if (!is_null($cmid)){
            $params['cmid'] = $cmid;
        }
        if (!is_null($userid)){
            $params['userid'] = $userid;
        }
        if (!is_null($attempt)){
            $params['attempt'] = $attempt;
        }
        if (!is_null($id)){
            $params['id'] = $id;
        }

        if (empty($params)){
            return false;
        }

        return $DB->delete_records(static::TABLE, $params);
    }


    /**
     * Get mm_data for the GT object
     *
     * @param object|numeric  $course_or_id
     * @param numeric         $cmid
     * @param numeric         $userid
     * @param \context_course $context
     *
     * @return mm_data_by_activity_user|null
     */
    static public function get_mm_data($course_or_id, $cmid, $userid, $context=null){
        $course = SH::get_chosen_course($course_or_id, false);
        $courseid = $course->id;
        $context = $context ?? SH::ctx($courseid);
        $mm_params = ['course' => $course, 'context' => $context, 'type' => static::MOD_TYPES];
        $filter = [MM::ST_ALL, MM::GET_TAGS, MM::USE_DEADLINE, MM::BY_ACTIVITY_USER, MM::USE_GM,
                   MM::GET_PROXY_ENABLED, MM::USE_COURSE_STUDENTS_ONLY];
        $is_use_cache = static::is_use_cache();
        if ($is_use_cache){
            switch ($is_use_cache){
                default:
                case static::USE_CACHE_COURSE:
                    $cache_data_loaded = isset(static::$_cache_mm_data[$courseid]);
                    break;
                case static::USE_CACHE_USER:
                    $cache_data_loaded = isset(static::$_cache_mm_data[$courseid][$userid]);
                    break;
            }

            if (!$cache_data_loaded){
                switch ($is_use_cache){
                    default:
                    case static::USE_CACHE_COURSE:
                        $mm_params['set_students'] = static::get_all_gt_students();
                        static::$_cache_mm_data[$courseid] = [];
                        break;
                    case static::USE_CACHE_USER:
                        $mm_params['set_students'] = [$userid => SH::get_user($userid)];
                        static::$_cache_mm_data[$courseid][$userid] = [];
                        break;
                }

                $MM = MM::get_MM_by_params($mm_params);
                $filter[] = MM::ONLY_GM;
                $data = $MM->get_raw_data($filter);

                foreach ($data as $mm_record){
                    static::$_cache_mm_data[$courseid][$mm_record->userid][$mm_record->cmid] = $mm_record;
                }
            }

            $mm_user_data = static::$_cache_mm_data[$courseid][$userid][$cmid] ?? null;
        } else {
            $mm_params['set_students'] = [$userid => SH::get_user($userid)];
            $MM = MM::get_MM_by_params($mm_params);
            $filter[MM::COURSEMODULE_IDS] = $cmid;
            $data = $MM->get_raw_data($filter);
            $mm_user_data = $data[$cmid.'_'.$userid] ?? null;
        }

        return $mm_user_data;
    }

    /**
     * Get attempt data for the GT object
     *
     * @param object|numeric $course_or_id
     * @param numeric        $cmid
     * @param numeric        $userid
     * @param numeric        $attempt
     *
     * @return gt_attempt_data|null
     */
    static public function get_attempt_data($course_or_id, $cmid, $userid, $attempt){
        $courseid = SH::get_id($course_or_id);
        $is_use_cache = static::is_use_cache();
        if ($is_use_cache){
            switch ($is_use_cache){
                default:
                case static::USE_CACHE_COURSE:
                    $cache_data_loaded = isset(static::$_cache_attempt_data[$courseid]);
                    break;
                case static::USE_CACHE_USER:
                    $cache_data_loaded = isset(static::$_cache_attempt_data[$courseid][$userid]);
                    break;
            }

            if (!$cache_data_loaded){
                $gt_cmids = static::get_gt_cmids($courseid);
                if (empty($gt_cmids)){
                    static::$_cache_attempt_data[$courseid][$userid] = [];
                    return null;
                }

                switch ($is_use_cache){
                    default:
                    case static::USE_CACHE_COURSE:
                        $course_data = gt_attempt_data::get_attempt_data($courseid, $gt_cmids, null, true);
                        static::$_cache_attempt_data[$courseid] = $course_data[$courseid] ?? [];
                        break;
                    case static::USE_CACHE_USER:
                        $user_data = gt_attempt_data::get_attempt_data($courseid, $gt_cmids, $userid, true);
                        static::$_cache_attempt_data[$courseid][$userid] = $user_data[$courseid][$userid] ?? [];
                        break;
                }
            }

            $attempt_user_data = static::$_cache_attempt_data[$courseid][$userid][$cmid][$attempt] ?? null;
        } else {
            $attempt_user_data = gt_attempt_data::get_attempt_data($courseid, $cmid, $userid, null, null, $attempt, true);
        }

        return $attempt_user_data;
    }

    /**
     * Get max attempt per user/activity
     *
     * @param object|numeric $course_or_id
     * @param numeric        $cmid
     * @param numeric        $userid
     *
     * @return int|null - null, if it finds none attempt
     */
    static public function get_max_attempt($course_or_id, $cmid, $userid){
        $courseid = SH::get_id($course_or_id);
        $is_use_cache = static::is_use_cache();
        if ($is_use_cache){
            switch ($is_use_cache){
                default:
                case static::USE_CACHE_COURSE:
                    $cache_data_loaded = isset(static::$_cache_max_attempts[$courseid]);
                    break;
                case static::USE_CACHE_USER:
                    $cache_data_loaded = isset(static::$_cache_max_attempts[$courseid][$userid]);
                    break;
            }

            if (!$cache_data_loaded){
                $gt_cmids = static::get_gt_cmids($courseid);
                if (empty($gt_cmids)){
                    static::$_cache_max_attempts[$courseid][$userid] = [];
                    return null;
                }

                $set_userid = false;
                switch ($is_use_cache){
                    default:
                    case static::USE_CACHE_COURSE:
                        static::$_cache_max_attempts[$courseid] = [];
                        break;
                    case static::USE_CACHE_USER:
                        static::$_cache_max_attempts[$courseid][$userid] = [];
                        $set_userid = true;
                        break;
                }

                $records = gt_attempt_data::get_max_attempts($courseid, $gt_cmids, $set_userid ? $userid : null);
                foreach ($records as $record){
                    static::$_cache_max_attempts[$record->courseid][$record->userid][$record->cmid] = $record->max_attempt;
                }
            }

            $res = static::$_cache_max_attempts[$courseid][$userid][$cmid] ?? null;
        } else {
            $res = SH::get_attempt_by_cm($cmid, $userid);
        }

        return $res;
    }

    /**
     * It can change or delete record in the DB, if $save == true
     * @noinspection PhpCastIsUnnecessaryInspection
     *
     * @param object $record
     *
     * @return array [$upd_status, $record]
     */
    static public function record_check_update($record){
        $upd_status = static::UPD_NONE;
        $debug_info = [];
        $change = false;
        $delete = false;
        $old_record = clone($record);
        $delete_all_attempts = false;
        // useful functions
        $check_diff = function($new_record, &$info=[]) use ($old_record) {
            if (!CLI_SCRIPT) return $info;

            foreach ($old_record as $key => $item){
                $new_item = $new_record->$key ?? null;
                if ($item != $new_item){
                    if ($key == 'timestart' || $key == 'timeend' || $key == 'timegrade' || $key == 'deadline'){
                        $f = '%d.%m.%Y @ %T';
                        $item = ned_date($item, null, $f);
                        $new_item = ned_date($new_item, null, $f);
                    }
                    $info[] = "$key:\t from '$item' to '$new_item'";
                }
            }

            return $info;
        };

        $GT = new static($record->courseid, $record->cmid, $record->userid, $record->attempt, static::MOD_ALL, null, $record);
        $gt_act = $GT->act && $GT->gt_record;
        $config = static::get_config();
        do{
            if (!$gt_act){
                $debug_info[] = 'GM is missing or not active now';
                $delete = true;
                break;
            }

            if ($GT->latest){
                // Checks here only for last (relevant) attempt
                /** @var \local_ned_controller\stdClass2|mm_data_by_activity_user $mm_data */
                $mm_data = SH::stdClass2($GT->mm_data);
                $NGC = SH::$ned_grade_controller;

                if ($mm_data->excluded != 0){
                    $debug_info[] = 'Excluded grade';
                    $delete = true;
                    break;
                }

                if ($mm_data->ngc_reason == $NGC::REASON_SUBMISSION && $mm_data->ngc_grade_type == $NGC::GT_AWARD_ZERO &&
                    $mm_data->ngc_status == $NGC::ST_DONE){
                    // NGC affects on the all attempts
                    $debug_info[] = 'Record has active deadline Zero from NGC';
                    $delete = true;
                    break;
                }

                if (!$GT->mm_data){
                    $debug_info[] = 'There are no necessary (MM) data';
                    $delete = true;
                    break;
                } elseif (!$mm_data->gt_shouldpass){
                    $debug_info[] = 'It became activity (forum?), which can be passed';
                    $delete = true;
                    break;
                }

                if ((int)$mm_data->attempt < (int)$record->attempt){
                    $debug_info[] = "It's old attempt, which doesn't exist anymore";
                    $debug_info[] = "mm_data->attempt: $mm_data->attempt, record->attempt: $record->attempt";
                    $delete = true;
                    break;
                }

                if ($record->timegrade){
                    if (!$mm_data->marked || $mm_data->unmarked){
                        $record->timegrade = 0;
                        $change = true;
                    } elseif ((int)$record->timegrade == (int)$mm_data->grade_time && $mm_data->first_last_grade_time &&
                        (int)$record->timegrade != (int)$mm_data->first_last_grade_time){
                        // fix some old records
                        $record->timegrade = (int)$mm_data->first_last_grade_time;
                        $change = true;
                    }

                    // For empty timegrade, grader check will be later
                    if ($record->timegrade){
                        if (!empty($mm_data->graderid) && (int)$record->graderid != (int)$mm_data->graderid){
                            $record->graderid = (int)$mm_data->graderid;
                            $change = true;
                        }
                    }

                } else {
                    if ($mm_data->marked && !$mm_data->unmarked && $mm_data->first_last_grade_time){
                        $record->timegrade = (int)$mm_data->first_last_grade_time;
                        $change = true;
                    }
                }

                $deadline = $GT->get_deadline(true, true);
                if ((int)$record->deadline != (int)$deadline){
                    $record->deadline = (int)$deadline;
                    $change = true;
                }

                if (empty($record->latest)){
                    $record->latest = true;
                    $change = true;
                }
            } else {
                // Checks here only for previous (non-last) attempt
                $attempt_data = $GT->attempt_data;
                if (empty($attempt_data)){
                    $debug_info[] = "It previous attempt now, but we haven't data for it.";
                    $delete = true;
                    break;
                }

                if (empty($attempt_data->timegrade)){
                    $debug_info[] = "It previous attempt now, and it doesn't graded.";
                    $delete = true;
                    break;
                }

                if ((int)$record->timegrade != (int)$attempt_data->timegrade){
                    $record->timegrade = (int)$attempt_data->timegrade;
                    $change = true;
                }

                if (!empty($attempt_data->graderid) && (int)$record->graderid != (int)$attempt_data->graderid){
                    $record->graderid = (int)$attempt_data->graderid;
                    $change = true;
                } elseif (empty($record->graderid)){
                    $record->graderid = $GT->get_potential_graderid();
                    $change = true;
                }

                if (!empty($record->latest)){
                    $record->latest = false;
                    $change = true;
                }
            }

            if (!$GT->has_proper_groupname()){
                $debug_info[] = 'Have denied group name';
                $delete = true;
                break;
            }

            if (!$GT->has_config_gt_tag()){
                $debug_info[] = 'There are no necessary tags';
                $delete = true;
                break;
            }

            $timestart = (int)$GT->get_submission_time(false, true);
            if (!$timestart && !$record->timegrade &&
                !SH::get_cm_visibility_by_user($record->cmid, $record->userid, false, false)){
                $debug_info[] = 'Student haven\'t access to this activity';
                $delete = true;
                break;
            }

            if ((int)$record->timestart != $timestart){
                $record->timestart = $timestart;
                $change = true;
            }

            // Should we check && !$record->timegrade here too?..
            if (!$record->timestart && !$record->deadline){
                $debug_info[] = 'There is no submission or deadline now';
                $delete = true;
                break;
            }

            if (!$record->timestart && (int)$record->deadline > time()){
                $debug_info[] = 'There is no submission and deadline in the future';
                $delete = true;
                break;
            }

            $gt_timestart = (int)$config->gt_startdate;
            if (max($timestart , (int)$record->timegrade) < $gt_timestart &&
                ($timestart || $record->timegrade || (int)$record->deadline < $gt_timestart)){
                $debug_info[] = 'Record starts after GT start date';
                $delete = true;
                break;
            }

            if ($GT->latest){
                $deadline = $deadline ?? $GT->get_deadline(true, true);
                $grade_time_start = (int)$GT->get_grade_time_start($timestart, $deadline);
                $days = (int)$GT->get_time_to_grade($grade_time_start, $deadline);
                $grade_timeend = (int)$GT->get_grade_timeend($grade_time_start, $days, $deadline);

                if ($grade_timeend != (int)$record->timeend || (int)$record->workdays != $days) {
                    $change = true;
                    $record->workdays = $days;
                    $record->timeend = $grade_timeend;
                }

                // For non-empty timegrade there was fixing grader in previous "_latest" condition
                if (empty($record->timegrade)){
                    $graderid = $GT->get_potential_graderid();
                    if ((int)$record->graderid != (int)$graderid){
                        $change = true;
                        $record->graderid = $graderid;
                    }
                }
            }

            $is_proxy = $GT->is_cm_proxy();
            if ($record->proxy != $is_proxy){
                $change = true;
                $record->proxy = $is_proxy;
            }

            if (!static::is_use_cache()){
                // with cache using, more effective check should be already done
                $is_active = $mm_data->is_active ?? static::is_student($record->userid, $record->courseid, true);
                if ($is_active == $record->suspended){
                    $change = true;
                    $record->suspended = (int)(!$is_active);
                }
            }
        } while (false);

        if ($delete){
            $s_deadline = SH::ned_date($record->deadline);
            $debug_info[] = "(courseid: $record->courseid, cmid: $record->cmid, userid: $record->userid, ".
                "deadline: $s_deadline, attempt: $record->attempt, latest: $record->latest)";

            if (!$gt_act){
                static::remove_records(null, null, null, $record->id);
            } else {
                $check_diff($GT->gt_record, $debug_info);
                $GT->delete_record($delete_all_attempts);
            }

            static::cli_debug('DELETE: ', $debug_info, $record->id);
            $record = null;
            $upd_status = static::UPD_REMOVED;
        } elseif ($change){
            $check_diff($record, $debug_info);
            $GT->debug("CHANGE: \n\t", $debug_info);
            $GT::save_record($record);
            $GT->save_mm_data_for_cache();
            $upd_status = static::UPD_CHANGED;
        }

        return [$upd_status, $record];
    }

    /**
     * Check and update some GT rules, which can be easily done for the whole GT-table at once
     * You can additionally filter update by course or users, but normally you shouldn't use them for optimization goal
     *
     * @param numeric|object|null $course_or_id - if set, check records only with this course
     * @param numeric|int[]|null  $userids - if set, check only for this user(s)
     */
    static public function check_and_update_global_gt($course_or_id=null, $userids=null){
        $courseid = SH::get_courseid_or_global($course_or_id);
        $userids = SH::val2arr($userids);
        $where_filter = $where_delete = $params = [];

        if (!empty($userids)){
            SH::sql_add_get_in_or_equal_options('userid', $userids, $where_filter, $params);
        }

        if (!empty($courseid) && $courseid != SITEID){
            SH::sql_add_equal('courseid', $courseid, $where_filter, $params);

            if (!static::is_gt_courseid($courseid)){
                SH::db()->delete_records_select(static::TABLE, SH::sql_condition($where_filter), $params);
                return;
            }
        } else {
            $gt_courses = static::get_gt_all_courseids();
            if (empty($gt_courses)){
                SH::db()->delete_records_select(static::TABLE, SH::sql_condition($where_filter), $params);
                return;
            }

            if (is_array($gt_courses)){
                // remove all records with courseid from not-GT courses
                SH::sql_add_get_in_or_equal_options('courseid', $gt_courses, $where_delete, $params, null, false);
            }
        }

        $gt_startdate = static::get_config('gt_startdate');
        if ($gt_startdate){
            // remove all records with date before gt_startdate
            $where_delete[] = "GREATEST(timestart, timegrade, IF(timestart = 0 AND timegrade = 0, deadline, 0)) < :gt_startdate";
            $params['gt_startdate'] = $gt_startdate;
        }

        if (!empty($where_delete)){
            $where_filter[] = SH::sql_condition($where_delete, "OR");

            SH::db()->delete_records_select(static::TABLE, SH::sql_condition($where_filter), $params);
        }
    }

    /**
     * Update users "suspended" status by course
     * You can additionally filter update by users, but normally you shouldn't use them for optimization goal
     *
     * @param numeric|object|null $course_or_id
     * @param numeric|int[]|null  $userids - if set, check only for this user(s)
     */
    static public function check_and_update_course_suspended($course_or_id=null, $userids=null){
        $courseid = SH::get_courseid_or_global($course_or_id);
        if (empty($courseid) || $courseid == SITEID || !static::is_gt_courseid($courseid)) return;

        $userids = SH::val2arr($userids);
        $active_users = SH::get_course_students_by_role($courseid, null, null, true, false);
        $active_userids = array_keys($active_users);
        unset($active_users);

        $user_updates = [true, false];
        foreach ($user_updates as $make_active){
            $where = $params = [];
            if (!empty($userids)){
                if ($make_active){
                    $users2update = array_intersect($userids, $active_userids);
                } else {
                    $users2update = array_diff($userids, $active_userids);
                }
                if (empty($users2update)) continue;

                SH::sql_add_get_in_or_equal_options('userid', $users2update, $where, $params);
            } else {
                if (empty($active_userids)){
                    if ($make_active) continue;
                } else {
                    SH::sql_add_get_in_or_equal_options('userid', $active_userids, $where, $params, null, $make_active);
                }
            }

            SH::sql_add_equal('courseid', $courseid, $where, $params);
            SH::sql_add_equal('suspended', $make_active, $where, $params);

            SH::db()->set_field_select(static::TABLE, 'suspended', !$make_active, SH::sql_condition($where), $params);
        }
    }

    /**
     * Check existing and missing records by course
     * If it is the only course to check, you may need to use {@see grading_tracker::check_and_update_global_gt()} first
     * If non-empty $userids are provided, then also using {@see grading_tracker::check_and_update_course_suspended()} for them
     *
     * Return array of statistic with keys from {@see grading_tracker::STATUSES_UPD}
     *  UPD_ALL => all checked records
     *  UPD_NONE => count of records without changes
     *  UPD_CHANGED => changed and resaved records
     *  UPD_REMOVED => count of removed records
     *  UPD_ADDED => count of new (added) records
     *  UPD_ERROR => string error
     *
     * @param numeric|object|null $course_or_id
     * @param numeric|int[]|null  $userids           - if set, check only for this user(s)
     * @param array|string        $additional_filter - additional SQL condition(s) to filter GT records for updating
     * @param array               $params            - SQL params for $additional_filter
     * @param callable|null       $debug             - debug function for accepting debug messages
     *
     * @return array
     */
    static public function check_and_update_course($course_or_id=null, $userids=null, $additional_filter=[], $params=[], $debug=null){
        $courseid = SH::get_courseid_or_global($course_or_id);
        [$userids, $where, $params] = SH::val2arr_multi(true, $userids, $additional_filter, $params);
        $res = array_fill_keys(static::STATUSES_UPD, 0);

        if ($debug && is_callable($debug)){
            $d = $debug;
        } else {
            $debug = null;
            $d = function(...$args){};
        }
        $gt_name = SH::str(GTR::TITLE_KEY);

        if (empty($courseid) || $courseid == SITEID){
            $msg = "$gt_name can't check/update global site";
            $d($msg);
            $res[static::UPD_ERROR] = $msg;
            return $res;
        }

        $course_name = SH::q_course_link($courseid, true, true)."[$courseid]";
        if (!static::is_gt_courseid($courseid)){
            /**
             * we don't delete these records by course here,
             * because normally it should happen in the {@see grading_tracker::check_and_update_global_gt()}
             */
            $msg = "Course $course_name is not in the $gt_name, pass";
            $d($msg);
            $res[static::UPD_ERROR] = $msg;
            return $res;
        }

        $userid = null;
        if (!empty($userids) && count($userids) == 1){
            $userid = reset($userids);
            $d("Checking single user with id '$userid'");
        }

        if ($debug){
            $msg = "Start check $course_name";
            if (!empty($userids)){
                $msg.= ' for ';
                if ($userid){
                    $msg.= SH::q_user_link($userid, null, false, true)."[$userid]";
                } else {
                    $msg.= 'users ['.join(', ', $userids).']';
                }
            }
            $d($msg);
        }

        SH::sql_add_equal('courseid', $courseid, $where, $params);
        if (!empty($userids)){
            SH::sql_add_get_in_or_equal_options('userid', $userids, $where, $params);

            $d('Checking users records for active/suspending status');
            static::check_and_update_course_suspended($courseid, $userids);
        }

        // Update existing records
        $records = static::get_records_select($where, $params);
        if (!empty($records)){
            $c = count($records);
            $d("Found $c record(s) to check");
            $upd_cache = true;
            if ($userid){
                static::set_use_inday_cache(true);
                static::set_use_cache(static::USE_CACHE_USER);
                $upd_cache = false;
            }

            $st = static::update_gt_records($records, $upd_cache);
            $msg = [];
            foreach ($st as $st_key => $st_value){
                if (empty($st_value)) continue;

                $res[$st_key] += $st_value;
                if ($st_key != static::UPD_ALL && $st_key != static::UPD_NONE){
                    $msg[] = $st_key.' - '.$st_value;
                }
            }

            if (empty($msg)){
                $d('There are no record updates');
            } else {
                $d('Result of update: records '.join(', ', $msg));
            }
        } else {
            $d("There are no records to check");
        }
        unset($records);

        // Check for the new records
        $d("Check for the new records");
        $added = static::check_missed_records_by_course($courseid, $userids, $debug);
        if ($added){
            $res[static::UPD_ADDED] += $added;
            $d("Save $added new record(s)");
        } else {
            $d("There are no new records to save");
        }

        // Check for the lost previous attempts records
        $d("Check for the lost previous attempts");
        $added = static::check_missed_previous_attempts_by_course($courseid, $userids, $debug);
        if ($added){
            $res[static::UPD_ADDED] += $added;
            $d("Save $added new previous attempt(s)");
        } else {
            $d("There are no new attempts to save");
        }

        return $res;
    }

    /**
     * Update records with the new data
     * Return array with updated statistic:
     *  UPD_ALL => all checked records
     *  UPD_NONE => count of records without changes
     *  UPD_CHANGED => changed and resaved records
     *  UPD_REMOVED => count of removed records
     *
     * @param array|\moodle_recordset|object|true $records - data to check and update,
     *                                                     if === true, will update the ENTIRE table, use it with caution
     * @param bool       $change_cache_settings - if true, can turn on class cache for the current situation,
     *                                          otherwise will work with cache as it have been already set
     *
     * @return array
     */
    static public function update_gt_records($records=null, $change_cache_settings=true){
        $is_recordset = false;
        $upd_statuses = [static::UPD_NONE => 0, static::UPD_ALL => 0, static::UPD_CHANGED => 0, static::UPD_REMOVED => 0];

        if (empty($records)) return $upd_statuses;

        if ($records === true){
            $records = SH::db()->get_recordset(static::TABLE);
        }
        if ($records instanceof \moodle_recordset){
            if (!$records->valid()) return $upd_statuses;

            $is_recordset = true;
        } else {
            $records = SH::val2arr($records);
        }

        if ($change_cache_settings){
            $record_count = empty($records) ? 0 : count($records);
            $prev_use_cache = static::is_use_cache();
            static::set_use_inday_cache(true);
            if ($is_recordset || ($record_count >= 2000)){
                SH::load_all_user_groups();
                static::set_use_cache(static::USE_CACHE_COURSE);
            }
        }

        $t_start = time();
        $checked_ids = [];
        foreach ($records as $record) {
            [$upd_status, $record] = static::record_check_update($record);
            $upd_statuses[$upd_status]++;
            $upd_statuses[static::UPD_ALL]++;
            if ($record->id ?? false){
                $checked_ids[] = $record->id;
            }
        }
        if ($is_recordset){
            $records->close();
        }

        if (!empty($checked_ids)){
            static::update_timemodified_value($checked_ids, $t_start, $t_start);
        }

        if ($change_cache_settings && isset($prev_use_cache)){
            static::set_use_cache($prev_use_cache);
        }

        return $upd_statuses;
    }

    /**
     * Check missed GT records by course:
     * • with passed deadlines (as it can be added in other way)
     * • all which not added (by any reason) - if record should be in GT, but there is no
     * Return number of successfully added (and saved) records
     *
     * @param numeric       $course_or_id
     * @param array|null    $users_or_ids
     * @param callable|null $debug - debug function for accepting debug messages
     *
     * @return int
     */
    static public function check_missed_records_by_course($course_or_id, $users_or_ids=null, $debug=null){
        if ($debug && is_callable($debug)){
            $d = $debug;
        } else {
            $d = function(...$args){};
        }

        $course = SH::get_chosen_course($course_or_id, false);
        if (!$course){
            $d('[check_missed_records] There are no such course, pass');
            return 0;
        }

        $courseid = $course->id;
        if (!static::is_gt_courseid($courseid)){
            $d("[check_missed_records] Course $courseid is not in the GT, pass");
            return 0;
        }

        $config = static::get_config();
        $filter = [MM::ST_GM_POSSIBLE, MM::GET_TAGS, MM::BY_ACTIVITY_USER, MM::USE_DEADLINE,
            MM::GET_PROXY_ENABLED, MM::USE_GM_ATTEMPT, MM::GM_NGC_FILTER, MM::GM_SHOULDPASS, MM::NOT_EXCLUDED,
            MM::GM_STARTDATE => $config->gt_startdate, MM::USE_COURSE_STUDENTS_ONLY,
            MM::ONLY_NOT_GM => true];
        $filter[MM::TAG_NAME_LIST_HAVE_ANY] = static::get_gt_tags();
        if (empty($filter[MM::TAG_NAME_LIST_HAVE_ANY])){
            $d('[check_missed_records] There are no GT tags, pass');
            return 0;
        }

        $deny_group_postfixes = $config->gradingtracker_deny_group_postfixes;
        if (!empty($deny_group_postfixes)){
            $filter[MM::GROUP_NOT_REGEXP] = SH::preg_quote_list($deny_group_postfixes, true).'$';
        }

        $context = SH::ctx($courseid, null, null, IGNORE_MISSING);
        if (!$context){
            $d("[check_missed_records] There is no context for course $courseid, pass");
            return 0;
        }

        $mm_params = [
            'course' => $course, 'context' => $context, 'type' => static::MOD_TYPES,
            'groupid' => 0, 'set_students' => '*',
        ];

        if (!empty($users_or_ids)){
            $users_or_ids = SH::val2arr($users_or_ids);
            $users = [];
            foreach ($users_or_ids as $user_or_id){
                if (is_object($user_or_id)){
                    $users[$user_or_id->id] = $user_or_id;
                } else {
                    $user = SH::get_user($user_or_id);
                    if ($user){
                        $users[$user->id] = $user;
                    }
                }
            }
            if (!empty($users)){
                $mm_params['set_students'] = $users;
            }
        }

        $MM = MM::get_MM_by_params($mm_params);
        $mm_data_list = $MM->get_raw_data($filter) ?: [];
        $d('[check_missed_records] Found '.count($mm_data_list).' record(s)');

        if (empty($mm_data_list)) return 0;

        return static::add_activity_by_mm_data($mm_data_list, $courseid, $context, true);
    }

    /**
     * Check missed GT previous attempts by course
     * Return number of successfully added (and saved) records
     *
     * @param numeric       $course_or_id
     * @param array|null    $users_or_ids
     * @param callable|null $debug - debug function for accepting debug messages
     *
     * @return int
     */
    static public function check_missed_previous_attempts_by_course($course_or_id, $users_or_ids=null, $debug=null){
        if ($debug && is_callable($debug)){
            $d = $debug;
        } else {
            $d = function(...$args){};
        }

        $courseid = SH::get_id($course_or_id);
        if (!static::is_gt_courseid($courseid)){
            $d("[check_missed_previous_attempts] Course $courseid is not in the GT, pass");
            return 0;
        }

        $gt_cmids = static::get_gt_cmids($courseid);
        if (empty($gt_cmids)){
            $d("[check_missed_previous_attempts] There are no GT cmids in the course $courseid, pass");
            return 0;
        }

        $userids = [];
        if (!empty($users_or_ids)){
            $users_or_ids = SH::val2arr($users_or_ids);
            foreach ($users_or_ids as $user_or_id){
                $userids[] = SH::get_id($user_or_id);
            }
        }

        $attempt_data_list = gt_attempt_data::get_attempt_data($courseid, $gt_cmids, $userids, false, static::get_config('gt_startdate'));
        $attempt_data_list = SH::arr_multi2one($attempt_data_list);
        $d('[check_missed_previous_attempts] Found '.count($attempt_data_list).' previous attempt(s)');

        if (empty($attempt_data_list)) return 0;

        return static::add_activity_by_attempt_data($attempt_data_list, $courseid, SH::ctx($courseid), true);
    }

    /**
     * Check & update GT records with new graders
     *
     * @param int   $courseid
     * @param int   $cmid
     * @param null  $old_grader
     * @param null  $new_grader
     * @param null  $check_grader
     * @param null  $check_student
     * @param array $where
     * @param array $params
     *
     */
    static public function update_gt_graders($courseid=0, $cmid=0, $old_grader=null, $new_grader=null, $check_grader=null, $check_student=null,
        $where=[], $params=[]){
        global $DB;
        $w = [
            "gt.timegrade = 0", // update only records, which are not graded
        ];
        $p = [];
        if ($courseid){
            $w[] = 'gt.courseid = :courseid';
            $p['courseid'] = $courseid;
        }
        if ($cmid){
            $w[] = 'gt.cmid = :cmid';
            $p['cmid'] = $cmid;
        }
        if (!is_null($old_grader)){
            $w[] = 'gt.graderid = :old_graderid';
            $p['old_graderid'] = $old_grader;
        }
        if (!is_null($new_grader)){
            $w[] = 'ot.userid = :new_graderid';
            $p['new_graderid'] = $new_grader;
        }
        if (!is_null($check_grader)){
            $w[] = '(gt.graderid = :old_graderid OR ot.userid = :new_graderid)';
            $p['old_graderid'] = $p['new_graderid'] = $check_grader;
        }
        if (!is_null($check_student)){
            $w[] = 'gt.userid = :student';
            $p['student'] = $check_student;
        }

        $table = static::TABLE;

        $where = array_merge($w, $where);
        $params = array_merge($p, $params);
        $where = !empty($where) ? ("\nWHERE (" . join(') AND (', $where) . ')') : '';
        $sql_ue = static::sql_user_enrolments('ue.userid, e.courseid, COALESCE(gr.id, 0) AS groupid', [], $params,
            static::ROLE_OT_SHORTNAME, $courseid, $cmid, true, 'GROUP BY e.courseid, groupid', 'group');
        /** @noinspection PhpUnnecessaryCurlyVarSyntaxInspection */
        $sql = "
            UPDATE {{$table}} gt
            JOIN (
                SELECT gt.userid, gt.courseid, COALESCE(gr.id, 0) AS groupid
                FROM {block_ned_teacher_tools_grtr} gt
                LEFT JOIN (
                    SELECT grp.id, g_m.userid, grp.courseid
                    FROM {groups} grp
                    JOIN {groups_members} g_m
                       ON g_m.groupid = grp.id
                    GROUP BY grp.courseid, g_m.userid
                ) gr  
                    ON gr.courseid = gt.courseid
                    AND gr.userid = gt.userid   
                GROUP BY gt.courseid, gt.userid 
            ) student 
                ON student.courseid = gt.courseid
                AND student.userid = gt.userid
            
            LEFT JOIN (
                $sql_ue 
            ) ot 
                ON ot.courseid = gt.courseid
                AND ot.groupid = student.groupid
                
            SET gt.graderid = COALESCE(ot.userid, 0)
            $where    
        ";
        $DB->execute($sql, $params);
    }

    /**
     * Check & update GT records for suspended status
     *
     * @param int               $courseid
     * @param null              $userid
     * @param numeric|bool|null $cmid - course module id, null to check nothing, false to check all, true - check all by course, or id
     * @param array             $where
     * @param array             $params
     *
     */
    static public function update_gt_suspend_students($courseid=0, $userid=null, $cmid=false, $where=[], $params=[]){
        global $DB;
        $w = [];
        $p = [];
        if ($userid){
            $w[] = 'gt.userid = :userid';
            $p['userid'] = $userid;
        }

        $select = ["ue.userid", "e.courseid"];
        $cm_join_condition = "";
        if (!is_null($cmid)){
            $select[] = "ra_cm.id AS cmid";
            $cm_join_condition = "AND (students.cmid = gt.cmid OR students.cmid IS NULL)";

            if ($cmid && !is_bool($cmid)){
                $w[] = 'gt.cmid = :cmid';
                $p['cmid'] = $cmid;

                if (empty($courseid)){
                    $courseid = SH::get_courseid_by_cmorid($cmid);
                }
            }
        }
        if ($courseid){
            $w[] = 'gt.courseid = :courseid';
            $p['courseid'] = $courseid;
        }

        $table = static::TABLE;

        $where = array_merge($w, $where);
        $params = array_merge($p, $params);
        $where = !empty($where) ? ("\nWHERE (" . join(') AND (', $where) . ')') : '';

        $sql_ue = static::sql_user_enrolments($select, [], $params,
            static::ROLE_STUDENT, $courseid, $cmid, true, 'GROUP BY e.courseid, ue.userid');
        /** @noinspection PhpUnnecessaryCurlyVarSyntaxInspection */
        $sql = "
            UPDATE {{$table}} gt
            LEFT JOIN (
                $sql_ue
            ) students 
                ON students.courseid = gt.courseid
                AND students.userid = gt.userid
                $cm_join_condition
                
            SET gt.suspended = IF(students.userid IS NULL, 1, 0)
            $where    
        ";
        $DB->execute($sql, $params);
    }

    /**
     * Update all gt records by groupid, requires at least $courseid or $cmid
     *  return update status or null, if there are nothing update
     *
     * @param     $groupid
     * @param int $courseid
     * @param int $cmid
     *
     * @return array|null
     */
    static public function update_group($groupid, $courseid=0, $cmid=0){
        if (!$groupid || (!$courseid && !$cmid)){
            return null;
        }

        $where = ["grp.id = :groupid"];
        $params = ['groupid' => $groupid];
        if ($courseid){
            $where[] = 'gt.courseid = :courseid';
            $params['courseid'] = $courseid;
        }

        if ($cmid){
            $where[] = 'gt.cmid = :cmid';
            $params['cmid'] = $cmid;
        }

        $records = static::get_gt_records($where, $params);
        if (!empty($records)){
            return static::update_gt_records($records);
        }
        return null;
    }

    /**
     * Update all gt records by $userid
     *  return update status or null, if there are nothing update
     *
     * @param     $userid
     * @param int $courseid
     * @param int $cmid
     *
     * @return array|null
     */
    static public function update_user_records($userid, $courseid=0, $cmid=0){
        if (!$userid){
            return null;
        }

        $params = ['userid' => $userid];
        if ($courseid){
            $params['courseid'] = $courseid;
        }

        if ($cmid){
            $params['cmid'] = $cmid;
        }

        $records = static::get_records($params);
        if (!empty($records)){
            return static::update_gt_records($records);
        }
        return null;
    }

    /**
     * Check GT teachers, and sent notifications to them, if need it.
     *
     * @param numeric|null  $graderid - if not set, check all of them
     * @param callable|null $print - function to print debug or log text
     *  @see task\adhoc_crongt_notify::print()
     *
     * @return int[]|array - list($all_checked, $notifies) - all checked teachers, and how much get notifications
     */
    static public function check_gt_notifications($graderid=null, $print=null){
        $p = function($text, $use_time=false) use ($print){
            if ($print){
                $print($text, $use_time);
            }
        };

        $params = [];
        if ($graderid){
            $params[SH::PAR_GRADER] = $graderid;
            $p("Function was called to check only grader with id '$graderid'");
        }

        $noemail = SH::cfg('noemailever');
        if ($noemail){
            $p('Not sending notification due to global config setting!');
        }

        $data = GTR::GTR_get_overview_GT_data(true, $params);
        $notifications = 0;
        foreach ($data as $grader_record){
            $graderid = $grader_record->graderid ?? false;
            if (!$graderid) continue;

            if ($grader_record->overdue > 0){
                $gr_name = $grader_record->gradername .'(id: '.$graderid.')';
                $p("$gr_name - $grader_record->overdue overdue.");
                if ($noemail){
                    continue;
                }

                static::send_overview_notification($graderid);
                $notifications++;
            }
        }

        return [count($data), $notifications];
    }

    /**
     * Send notification to teacher about GT overview
     * About messaging see
     * @link https://docs.moodle.org/dev/Message_API
     *
     * @param numeric   $graderid - id of teacher
     */
    static public function send_overview_notification($graderid){
        if (!static::has_any_capabilities(null, $graderid)){
            return;
        }

        $gt_url = GTR::get_url([GTR::PAR_VIEW => GTR::VIEW_OVERVIEW]);
        $message = new \stdClass();
        $message->component = SH::$PLUGIN_NAME;
        $message->name = 'gt_overdue'; // from message.php
        $message->userfrom = \core_user::get_noreply_user();
        $message->userto = SH::get_user($graderid);
        $message->subject = SH::str('gt_overdue:title');
        $message->fullmessage = $message->smallmessage = SH::str('gt_overdue:message');
        $message->fullmessageformat = FORMAT_HTML;
        $message->fullmessagehtml = SH::str('gt_overdue:message_html', $gt_url->out(false));
        $message->notification = 1;
        $message->contexturl = $gt_url;
        $message->contexturlname = SH::str(GTR::TITLE_KEY);
        $additional_content = ['*' => [
            'header' => SH::get_site_logo(),
            'footer' => SH::str('pleasenotreply'),
        ]];

        $msg = new \core\message\message();
        foreach ($message as $key => $item){
            $msg->$key = $item;
        }
        $msg->set_additional_content('email', $additional_content);
        message_send($msg);
    }

    /**
     * Set suspended status just on/off
     *  If you don't know, what status should be, use update_gt_students function
     *
     * @param      $userid
     * @param int  $courseid
     * @param int  $cmid
     * @param bool $suspend
     *
     */
    static public function set_suspended_user($userid, $courseid=0, $cmid=0, $suspend=true){
        global $DB;
        $params = ['userid' => $userid];
        if ($courseid){
            $params['courseid'] = $courseid;
        }
        if ($cmid){
            $params['cmid'] = $cmid;
        }
        $DB->set_field(static::TABLE, 'suspended', ($suspend ? 1 : 0), $params);
    }

    /* config data */

    /**
     * @param string $name (optional)
     *
     * @return object|null|\mixed
     */
    static public function get_config($name=null){
        if (is_null(static::$_config)){
            static::$_config = clone(SH::get_config());

            $weekends = explode("\n", (static::$_config->gradingtracker_weekend ?? ''));
            foreach ($weekends as $weekend){
                $weekend = trim($weekend);
                if (!empty($weekend)){
                    static::$_weekend_names[] = $weekend;
                }
            }

            if (empty(static::$_config->gradingtracker_manystudents)){
                static::$_config->gradingtracker_manystudents = [];
            } else {
                $option = static::$_config->gradingtracker_manystudents;
                try {
                    $option = str_replace(' ', '', $option);
                    static::$_config->gradingtracker_manystudents = explode(',', $option) ?: [];
                } catch (\Exception $e) {
                    static::$_config->gradingtracker_manystudents = [];
                }
            }

            if (empty(static::$_config->gradingtracker_deny_group_postfixes)){
                static::$_config->gradingtracker_deny_group_postfixes = [];
            } else {
                $option = static::$_config->gradingtracker_deny_group_postfixes;
                try {
                    $option = str_replace(' ', '', $option);
                    static::$_config->gradingtracker_deny_group_postfixes = explode(';', $option) ?: [];
                } catch (\Exception $e) {
                    static::$_config->gradingtracker_deny_group_postfixes = [];
                }
            }

            static::$_config->gt_startdate = static::get_date_from_text(static::$_config->gradingtracker_startdate ?? '');
        }

        if ($name){
            return static::$_config->$name ?? null;
        }
        return static::$_config;
    }

    /**
     * @return array
     */
    static public function get_config_tags_3days(){
        static $data = null;

        if (is_null($data)){
            $data = static::get_config_tags('gradingtracker_activitiestags3days');
        }

        return $data;
    }

    /**
     * @return array
     */
    static public function get_config_tags_5days(){
        static $data = null;

        if (is_null($data)){
            $data = static::get_config_tags('gradingtracker_activitiestags5days');
        }

        return $data;
    }

    /**
     * @param $configname
     *
     * @return array - list of tag names
     */
    static public function get_config_tags($configname){
        global $DB, $CFG;
        $config = static::get_config();
        if (empty($config->$configname)){
            return [];
        }
        $tag_ids = explode(',', $config->$configname);
        $tagcollid = \core_tag_area::get_collection('core', 'course_modules');
        $namefield = empty($CFG->keeptagnamecase) ? 'name' : 'rawname';
        $all_tags = $DB->get_records_menu('tag', ['isstandard' => 1, 'tagcollid' => $tagcollid], $namefield, 'id,' . $namefield);
        $tags = [];
        foreach ($all_tags as $tag_id => $tag_name){
            if (in_array($tag_id, $tag_ids)){
                $tags[] = $tag_name;
            }
        }
        return $tags;
    }

    /**
     * @return array - return list of all GT tag names
     */
    static public function get_gt_tags(){
        return array_merge(static::get_config_tags_3days(), static::get_config_tags_5days());
    }

    /**
     * Return all GT cmids by course
     * Note: result of this method is not cached anywhere
     *
     * @param object|numeric $course_or_id
     *
     * @return array
     */
    static public function get_gt_cmids($course_or_id=null){
        return SH::cmids_get_by_tags(static::get_gt_tags(), [], $course_or_id);
    }

    /**
     * @param array|string $tags - list of tags or tag name
     *
     * @return bool - true, if at least one of the tags from GT tags
     */
    static public function is_gt_tags($tags){
        if (empty($tags)){
            return false;
        }

        if (!is_array($tags)){
            $tags = [$tags];
        }

        return !empty(array_intersect($tags, static::get_gt_tags()));
    }

    /**
     * Check, that group with $groupid allowed by GT settings
     *
     * @param int|string $groupid Group id
     *
     * @return bool
     */
    static public function is_groupid_allowed($groupid){
        if (empty($groupid)) return true;

        $deny_group_postfixes = static::get_config('gradingtracker_deny_group_postfixes');
        if (empty($deny_group_postfixes)) return true;

        $groupname = SH::get_groupname($groupid);
        return !SH::str_ends_with($groupname, $deny_group_postfixes, false);
    }

    /**
     * Return array of the GT course ids, or true, if all courses - are GT courses
     *
     * @return array|int[]|bool - if return true, then all courses are GT courses
     */
    static public function get_gt_all_courseids(){
        static $_courses = null;
        if (is_null($_courses)){
            $config = static::get_config();
            $_courses = SH::course_cats_has_courseid($config->gradingtracker_coursecategories ?? []);
        }

        return $_courses;
    }

    /**
     * Return true, if TT config has such course id in GT settings
     *
     * @param $courseid
     *
     * @return bool
     */
    static public function is_gt_courseid($courseid){
        if (!$courseid){
            return false;
        }

        $_courses = static::get_gt_all_courseids();
        return $_courses === true || isset($_courses[$courseid]);
    }

    /**
     *
     * Return UNIX time from the "DD/MM/YYYY" string
     *
     * @param $text
     *
     * @return int
     */
    static public function get_date_from_text($text){
        do {
            if (empty($text)) break;

            /** @noinspection RegExpSimplifiable */
            $data = preg_split('/([\\\\]|[\/])/', trim($text));
            if (count($data) != 3) break;

            [$day, $month, $year] = $data;
            if (!checkdate((int)$month, (int)$day, (int)$year)) break;

            return (int)mktime(0, 0, 0, $month, $day, $year);
        } while (false);

        return 0;
    }

    /* timing */

    /**
     * @param $time
     *
     * @return bool
     */
    static public function is_day_off($time){
        static::get_config(); // init weekend_names
        if (empty(static::$_weekend_names)){
            return false;
        }

        [$t, $t_end] = SH::get_day_start_end($time);
        if (!isset(static::$_weekend[$t])){
            $w = date('N', $t);
            if ($w > 5){
                $res = true;
            } else {
                $events = calendar_get_events($t, $t_end, false, false, 1, true, false);
                $res = false;
                foreach ($events as $event){
                    foreach (static::$_weekend_names as $weekend_name){
                        if (strpos($event->name, $weekend_name) !== FALSE){
                            $res = true;
                            break;
                        }
                    }
                    if ($res){
                        break;
                    }
                }
            }
            static::$_weekend[$t] = $res;
        }

        return static::$_weekend[$t];
    }

    /**
     * Calculate count of work days before time1 & time2
     *
     * @param $time1
     * @param $time2
     *
     * @return int
     */
    static public function calc_work_days($time1, $time2){
        $days = 0;
        while ($time1 <= $time2){
            if (!static::is_day_off($time1)){
                $days++;
            }
            $time1 += DAYSECS;
        }

        return $days;
    }

    /**
     * Return difference between time1 & time2 more, then $longerh (or not)
     *
     * @param      $time1
     * @param      $longerh
     * @param null $time2
     *
     * @return bool
     */
    static public function time_longer_h($time1, $longerh, $time2=null){
        $time2 = is_null($time2) ? time() : $time2;
        $diff = abs($time2 - $time1);
        $h = floor($diff/HOURSECS);
        return $h >= $longerh;
    }

    /**
     * Set static $_use_cache
     * If you changed type of cache in the middle of some process,
     *  you have to reset existing cache through {@see grading_tracker::reset_caches()}
     *
     * @param int $value
     */
    static public function set_use_cache($value=self::USE_CACHE_NONE){
        static::$_use_cache = (int)$value;
    }

    /**
     * Return static $_use_cache
     *
     * @return int
     */
    static public function is_use_cache(){
        return static::$_use_cache;
    }

    /**
     * Set static $_use_inday_cache
     *
     * @param bool $value
     */
    static public function set_use_inday_cache($value){
        static::$_use_inday_cache = (bool)$value;
    }

    /**
     * Return static $_use_inday_cache
     *
     * @return bool
     */
    static public function is_use_inday_cache(){
        return static::$_use_inday_cache;
    }

    /**
     * Reset static cache storages

     * @param numeric|array|null $courseids - if set, reset cache only for chosen course ids,
     *                                      otherwise purge it all
     */
    static public function reset_caches($courseids=null){
        $courseids = SH::val2arr($courseids);

        // Sometimes loop-way is better to clear memory
        $mm_keys = $courseids ?: array_keys(static::$_cache_mm_data) ?: [];
        foreach ($mm_keys as $key){
            unset(static::$_cache_mm_data[$key]);
        }

        $attempt_keys = $courseids ?: array_keys(static::$_cache_attempt_data) ?: [];
        foreach ($attempt_keys as $key){
            unset(static::$_cache_attempt_data[$key]);
        }

        $max_attempt_keys = $courseids ?: array_keys(static::$_cache_max_attempts) ?: [];
        foreach ($max_attempt_keys as $key){
            unset(static::$_cache_max_attempts[$key]);
        }

        $inday_keys = $courseids ?: array_keys(static::$_cache_inday_data) ?: [];
        foreach ($inday_keys as $key){
            unset(static::$_cache_inday_data[$key]);
        }

        if (empty($courseids)){
            static::$_cache_mm_data = [];
            static::$_cache_attempt_data = [];
            static::$_cache_max_attempts = [];
            static::$_cache_inday_data = [];
        }
    }

    /* EVENTS */

    /* add record */

    /**
     * @param \stdClass | \core\event\base | \mod_assign\event\assessable_submitted | \mod_forum\event\post_created | \mod_forum\event\discussion_created | \mod_quiz\event\attempt_submitted $event
     * @param string $type
     */
    static public function add_activity_by_event($event, $type=self::MOD_ALL){
        global $DB;
        $cmid = $event->contextinstanceid;
        $userid = $event->relateduserid ?? $event->userid;
        $courseid = $event->courseid;
        $component = $event->component ?? '';

        $get_attempt = false;
        if ($component == 'mod_assign'){
            $type = static::MOD_ASSIGN;
            $get_attempt = 'attemptnumber';
        } elseif ($component == 'mod_quiz'){
            $type = static::MOD_QUIZ;
            $get_attempt = 'attempt';
        }

        $attempt = 0;
        if ($get_attempt){
            $attempt = $DB->get_field($event->objecttable, $get_attempt, ['id' => $event->objectid]) ?: 0;
        }

        if ($type == static::MOD_QUIZ && $attempt > 0){
            $attempt--;
        }

        if (isset(static::$_event_data[$cmid][$userid][$attempt])){
            return;
        }
        static::$_event_data[$cmid][$userid][$attempt] = $event;
        static::set_use_cache(false);
        $GT = new static($courseid, $cmid, $userid, $attempt, $type, $event->get_context());
        $GT->add_activity();
    }

    /* grade record */

    /**
     * @param \stdClass | \core\event\base | \mod_assign\event\assessable_submitted | \mod_forum\event\post_created | \mod_forum\event\discussion_created | \mod_quiz\event\attempt_submitted $event
     */
    static public function grade_activity_by_event($event){
        global $DB;
        $userid = $event->relateduserid;
        $graderid = $event->userid;
        $courseid = $event->courseid;
        $gg_id = $event->objectid;

        $sql = "
        SELECT cm.id, m.name 
        FROM {course_modules} cm
        JOIN {grade_grades} gg
            ON gg.id = :ggid
            AND gg.userid = :userid
        JOIN {grade_items} gi
            ON gi.id = gg.itemid
            AND gi.courseid = cm.course
            AND gi.iteminstance = cm.instance
        JOIN {modules} m
            ON m.name = gi.itemmodule 
        WHERE cm.course = :courseid
        GROUP BY cm.id
        ";

        $cms = $DB->get_records_sql($sql, ['ggid' => $gg_id, 'userid' => $userid, 'courseid' => $courseid]);
        $cm = reset($cms);
        if (!$cm){
            return;
        }

        $type = $cm->name;
        $cmid = $cm->id;

        static::grade_activity($courseid, $cmid, $graderid, $userid, null, $type, $event->get_context());
    }

    /**
     * @param \stdClass | \local_ned_controller\event\submission_graded | \mod_assign\event\submission_graded $event
     */
    static public function grade_activity_by_ned_assign_event($event){
        $cmid = $event->contextinstanceid;
        $courseid = $event->courseid;
        $graderid = $event->userid;
        $type = static::MOD_ASSIGN;
        $other = $event->other;
        $userid = $event->relateduserid;
        $maxattempt = $other['maxattempt'] ?? 0;
        $attempt = $other['attempt'] ?? 0;

        static::grade_activity($courseid, $cmid, $graderid, $userid, $attempt, $type, $event->get_context(), $maxattempt);
    }

    /**
     * @param \stdClass | \mod_quiz\event\question_manually_graded $event
     */
    static public function question_manually_graded($event){
        global $DB;
        $cmid = $event->contextinstanceid;
        $courseid = $event->courseid;
        $graderid = $event->userid;
        $type = static::MOD_QUIZ;
        // don't try to  get relateduserid from $event
        $other = $event->other;
        $attemptid = $other['attemptid'] ?? 0;
        $quizid = $other['quizid'] ?? 0;
        if (!$attemptid || !$quizid){
            return;
        }
        $quiz_attempts = $DB->get_record('quiz_attempts', ['id' => $attemptid]);
        if (!$quiz_attempts || is_null($quiz_attempts->sumgrades ?? null)){
            return;
        }
        $userid = $quiz_attempts->userid;
        $attempt = $quiz_attempts->attempt - 1;

        static::grade_activity($courseid, $cmid, $graderid, $userid, $attempt, $type, $event->get_context());
    }

    /* delete record */

    /**
     * @param \mod_forum\event\post_deleted | \mod_forum\event\discussion_deleted $event
     */
    static public function check_forum_submission_to_delete_by_event($event){
        $cmid = $event->contextinstanceid;
        $graderid = $event->userid; // who delete
        // if relateduserid is null, then it was user post
        $userid = $event->relateduserid ?? $graderid;
        $courseid = $event->courseid;
        $type = static::MOD_FORUM;
        $attempt = 0;

        $GT = new static($courseid, $cmid, $userid, $attempt, $type, $event->get_context());
        if (!$GT->act || !$GT->_gt_record){
            return;
        }

        $mm_data = $GT->mm_data;
        if($mm_data->submitted ?? false){
            return;
        }

        static::remove_records($cmid, $userid, $attempt);
    }

    /**
     * @param \stdClass | \core\event\base | \local_ned_controller\event\submission_removed $event
     */
    static public function remove_activity_by_event($event){
        global $DB;
        $params = static::_get_assign_event_params($event);
        $DB->delete_records(static::TABLE, $params);
    }

    /**
     * @param \stdClass | \core\event\base | \local_ned_controller\event\submission_remove_grades $event
     */
    static public function remove_grade_by_event($event){
        global $DB;
        $params = static::_get_assign_event_params($event);
        $DB->set_field(static::TABLE, 'timegrade', 0, $params);
    }
    /* user (students & graders) update */

    /**
     * @param \core\event\base | \core\event\user_updated $event
     */
    static public function user_updated_event($event){
        $graderid = $event->userid; // who update
        // if relateduserid is null, then user updated himself
        $userid = $event->relateduserid ?? $graderid;
        $courseid = $event->courseid ?? 0;
        static::check_updated_grader($userid, $courseid);
        static::check_updated_student($userid, $courseid);
    }

    /**
     * @param \core\event\base | \core\event\user_deleted $event
     */
    static public function user_deleted_event($event){
        $graderid = $event->userid; // who update
        // if relateduserid is null, then user updated himself
        $userid = $event->relateduserid ?? $graderid;
        $courseid = $event->courseid ?? 0;
        static::check_updated_grader($userid, $courseid, 0, false, true);
        static::check_updated_student($userid, $courseid, 0, true);
    }

    /**
     * @param \core\event\base | \core\event\user_enrolment_updated $event
     */
    static public function user_enrolment_updated_event($event){
        $graderid = $event->userid; // who update
        // if relateduserid is null, then user updated himself
        $userid = $event->relateduserid ?? $graderid;
        $courseid = $event->courseid ?? 0;
        static::check_updated_grader($userid, $courseid);
        static::check_updated_student($userid, $courseid);
    }

    /**
     * @param \core\event\base | \core\event\user_enrolment_created $event
     */
    static public function user_enrolment_created_event($event){
        $graderid = $event->userid; // who update
        // if relateduserid is null, then user updated himself
        $userid = $event->relateduserid ?? $graderid;
        $courseid = $event->courseid ?? 0;
        static::check_updated_grader($userid, $courseid, 0,true);
        static::check_updated_student($userid, $courseid, 0, false);
    }

    /**
     * @param \core\event\base | \core\event\user_enrolment_deleted $event
     */
    static public function user_enrolment_deleted_event($event){
        $graderid = $event->userid; // who update
        // if relateduserid is null, then user updated himself
        $userid = $event->relateduserid ?? $graderid;
        $courseid = $event->courseid ?? 0;
        static::check_updated_grader($userid, $courseid, 0, false, true);
        static::check_updated_student($userid, $courseid, 0, true);
    }

    /**
     * @param \core\event\base | \core\event\group_member_added $event
     */
    static public function group_member_added_event($event){
        $graderid = $event->userid; // who update
        // if relateduserid is null, then user updated himself
        $userid = $event->relateduserid ?? $graderid;
        $courseid = $event->courseid ?? 0;
        //$groupid = $event->objectid ?? 0;

        static::check_updated_grader($userid, $courseid);
    }

    /**
     * @param \core\event\base | \core\event\group_member_removed $event
     */
    static public function group_member_removed_event($event){
        $graderid = $event->userid; // who update
        // if relateduserid is null, then user updated himself
        $userid = $event->relateduserid ?? $graderid;
        $courseid = $event->courseid ?? 0;
        //$groupid = $event->objectid ?? 0;

        static::check_updated_grader($userid, $courseid);
    }

    /**
     * Return data for functions by role_* events
     *
     * @param \core\event\base | \core\event\role_assigned | \core\event\role_unassigned $event
     *
     * @return array($userid, $courseid, $cmid, $rolename)
     */
    static protected function _get_role_event_data($event){
        global $DB;
        $graderid = $event->userid; // who update
        // if relateduserid is null, then user updated himself
        $userid = $event->relateduserid ?? $graderid;
        $courseid = $event->courseid ?? 0;
        $cmid = 0;
        if ($event->contextlevel == CONTEXT_MODULE){
            $cmid = $event->contextinstanceid;
        }

        $roleid = $event->objectid;
        $role = $DB->get_record('role', ['id' => $roleid], 'shortname');
        $rolename = $role->shortname ?? '';

        return [$userid, $courseid, $cmid, $rolename];
    }

    /**
     * Return params for DB functions by mod_assign events
     *
     * @param \core\event\base $event
     *
     * @return array
     */
    static protected function _get_assign_event_params($event){
        $cmid = $event->contextinstanceid;
        $userid = $event->relateduserid ?? $event->userid;
        $courseid = $event->courseid;
        $params = ['cmid' => $cmid, 'userid' => $userid, 'courseid' => $courseid];
        if (isset($event->other['submissionattempt'])){
            $params['attempt'] = $event->other['submissionattempt'];
        }

        return $params;
    }

    /**
     * @param \core\event\base | \core\event\role_assigned $event
     */
    static public function role_assigned_event($event){
        [$userid, $courseid, $cmid, $rolename] = static::_get_role_event_data($event);
        switch ($rolename){
            case static::ROLE_OT_SHORTNAME:
                static::check_updated_grader($userid, $courseid, $cmid, true);
                break;
            case static::ROLE_STUDENT:
                static::check_updated_student($userid, $courseid, $cmid, false);
                break;
        }
    }

    /**
     * @param \core\event\base | \core\event\role_unassigned $event
     */
    static public function role_unassigned_event($event){
        [$userid, $courseid, $cmid, $rolename] = static::_get_role_event_data($event);
        switch ($rolename){
            case static::ROLE_OT_SHORTNAME:
                static::check_updated_grader($userid, $courseid, $cmid, false, true);
                break;
            case static::ROLE_STUDENT:
                static::check_updated_student($userid, $courseid, $cmid, true);
                break;
        }
    }

    /**
     * @param \core\event\base | \core\event\course_deleted $event
     */
    static public function course_deleted_event($event){
        global $DB;
        $DB->delete_records(static::TABLE, ['courseid' => $event->courseid]);
    }

    /**
     * @param \core\event\base | \core\event\course_module_updated $event
     */
    static public function course_module_updated_event($event){
        $cmid = $event->contextinstanceid;
        $records = static::get_records(['cmid' => $cmid]);
        if (!empty($records)){
            static::update_gt_records($records);
        }
    }

    /**
     * @param \core\event\base | \core\event\course_module_deleted $event
     */
    static public function course_module_deleted_event($event){
        static::remove_records($event->contextinstanceid);
    }

    /**
     * @param \core\event\base | \core\event\group_deleted $event
     */
    static public function group_deleted_event($event){
        // some group was deleted, but we don't know, wat users were in it
        // so, check all students in this course
        $courseid = $event->courseid;
        if ($courseid){
            $records = static::get_records(['courseid' => $courseid]);
            if (!empty($records)){
                static::update_gt_records($records);
            }
        }
    }

    /**
     * @param \core\event\base | \mod_assign\event\group_override_created | \mod_assign\event\group_override_updated | \mod_assign\event\group_override_deleted | \mod_quiz\event\group_override_created | \mod_quiz\event\group_override_updated | \mod_quiz\event\group_override_deleted  $event
     */
    static public function group_override_updated_event($event){
        $groupid = $event->other['groupid'] ?? 0;
        $courseid = $event->courseid;
        $cmid = $event->contextinstanceid ?? 0;
        static::update_group($groupid, $courseid, $cmid);
    }

    /**
     * @param \core\event\base | \mod_assign\event\user_override_created | \mod_assign\event\user_override_updated | \mod_assign\event\user_override_deleted | \mod_quiz\event\user_override_created | \mod_quiz\event\user_override_updated | \mod_quiz\event\user_override_deleted  $event
     */
    static public function user_override_updated_event($event){
        $graderid = $event->userid; // who update
        // if relateduserid is null, then user updated himself
        $userid = $event->relateduserid ?? $graderid;
        $courseid = $event->courseid;
        $cmid = $event->contextinstanceid ?? 0;
        static::update_user_records($userid, $courseid, $cmid);
    }

    /* Calling modal_form */

    /**
     * @param bool      $set
     * @param string    $bool_field
     * @param string    $text_field
     * @param \context  $context
     * @param array     $formdata
     * @param array     $data
     * @param bool      $confirm
     * @param string    $description
     * @param string    $placeholder
     *
     * @return array|string|null
     */
    static protected function _bool_text_form_call($set, $bool_field, $text_field,
        $context, $formdata=[], $data=[], $confirm=false,
        $description='', $placeholder='')
    {
        $F = SH::$form;
        $raw_ids = SH::val2arr($data['gtid'] ?? 0);
        if (!static::has_any_capabilities($context) || empty($raw_ids)){
            return null;
        }

        $f_params = [
            'description' => $description,
        ];
        $form = null;
        $save = $confirm;

        if ($set){
            $f_params['placeholder'] = $placeholder.'...';
            $f_params['required'] = true;
            $form = $F::$text_simple_form::create(null, $f_params,'post', '', null, true, $formdata);
            if ($save){
                $save = $form->get_data();
            }
        }

        if ($save){
            $params = $data['urlparams'] ?? [];
            $gt_ids = GTR::GTR_check_gt_ids($raw_ids, $params);
            if (empty($gt_ids)){
                return SH::notification('There is no such record(s)!', SH::NOTIFY_ERROR);
            }

            if ($set){
                $bool_value = 1;
                $text_value = $save->text ?? '';
            } else {
                $bool_value = 0;
                $text_value = null;
            }

            if (count($gt_ids) == 1){
                $gtid = reset($gt_ids);
                $gt = static::get_record_by_gtid($gtid);

                $gt->$bool_field = $bool_value;
                $gt->$text_field = $text_value;

                static::save_record($gt, false);

                return [
                    'replace_id' => $gtid,
                    'replace' => GTR::GTR_get_rendered_row($gtid, $params),
                ];
            } else {
                $gt_params = ['id' => $gt_ids];
                static::update_records_value($bool_field, $bool_value, $gt_params);
                static::update_records_value($text_field, $text_value, $gt_params);

                return ['reload' => 1];
            }
        }

        if (!$set){
            $form = $F::$confirm_simple_form::create(null, $f_params,'post', '', null, true, $formdata);
        }

        return $form->draw();
    }

    /**
     * @param       $context
     * @param array $formdata
     * @param array $data
     * @param bool  $confirm
     *
     * @return array|string|null
     */
    static public function form_reportbug($context, $formdata=[], $data=[], $confirm=false){
        return static::_bool_text_form_call(true, 'bug', 'bug_report', $context, $formdata, $data, $confirm,
            str('askbugreport'), str('bugreport'));
    }

    /**
     * @param       $context
     * @param array $formdata
     * @param array $data
     * @param bool  $confirm
     *
     * @return array|string|null
     */
    static public function form_unreportbug($context, $formdata=[], $data=[], $confirm=false){
        return static::_bool_text_form_call(false, 'bug', 'bug_report', $context, $formdata, $data, $confirm,
            str('askunsetbugreport'));
    }

    /**
     * @param       $context
     * @param array $formdata
     * @param array $data
     * @param bool  $confirm
     *
     * @return array|string|null
     */
    static public function form_setuncounted($context, $formdata=[], $data=[], $confirm=false){
        return static::_bool_text_form_call(true, 'uncounted', 'uncounted_reason', $context, $formdata, $data, $confirm,
            str('asksetuncounted'), str('reason'));
    }

    /**
     * @param       $context
     * @param array $formdata
     * @param array $data
     * @param bool  $confirm
     *
     * @return array|string|null
     */
    static public function form_setcounted($context, $formdata=[], $data=[], $confirm=false){
        return static::_bool_text_form_call(false, 'uncounted', 'uncounted_reason', $context, $formdata, $data, $confirm,
            str('asksetcounted'));
    }

    /**
     * @param       $context
     * @param array $formdata
     * @param array $data
     * @param bool  $confirm
     *
     * @return array|string|null
     */
    static public function form_gthide($context, $formdata=[], $data=[], $confirm=false){
        return static::_bool_text_form_call(true, 'hidden', 'note', $context, $formdata, $data, $confirm,
            str('askgthide'), str('reason'));
    }

    /**
     * @param       $context
     * @param array $formdata
     * @param array $data
     * @param bool  $confirm
     *
     * @return array|string|null
     */
    static public function form_gtshow($context, $formdata=[], $data=[], $confirm=false){
        return static::_bool_text_form_call(false, 'hidden', 'note', $context, $formdata, $data, $confirm,
            str('askgtshow'));
    }

    /**
     * @param       $context
     * @param array $formdata
     * @param array $data
     * @param bool  $confirm
     *
     * @return string|array|null
     *
     * @noinspection PhpUnusedParameterInspection
     */
    static public function form_refreshnow($context, $formdata=[], $data=[], $confirm=false){
        $raw_ids = SH::val2arr($data['gtid'] ?? 0);
        $params = $data['urlparams'] ?? [];
        if (!static::has_any_capabilities($context) || empty($raw_ids)){
            return null;
        }

        $gt_ids = GTR::GTR_check_gt_ids($raw_ids, $params);
        if (empty($gt_ids)){
            return SH::notification('There is no such record(s)!', SH::NOTIFY_ERROR);
        }

        if (count($gt_ids) == 1){
            $gt_id = reset($gt_ids);
            $gt = static::get_record_by_gtid($gt_id);
            static::update_gt_records($gt, false);

            return [
                'replace_id' => $gt_id,
                'replace' => GTR::GTR_get_rendered_row($gt_id, $params),
                'notifications' => ['message' => SH::str('done'), 'type' => SH::NOTIFY_SUCCESS],
            ];
        }

        return null;
    }
}
