<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Deadline Manager Header
 *
 * @package    block_ned_teacher_tools
 * @copyright  Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */


namespace block_ned_teacher_tools\mod;

use block_ned_teacher_tools as NED;
use block_ned_teacher_tools\shared_lib as SH;

defined('MOODLE_INTERNAL') || die();

/** @var \stdClass $CFG */
require_once($CFG->dirroot . '/mod/quiz/locallib.php');
require_once($CFG->dirroot . '/mod/quiz/lib.php');
require_once($CFG->dirroot . '/lib/gradelib.php');
require_once(__DIR__ . '/../../lib.php');

/**
 * Class deadline_manager_quiz
 * @package block_ned_teacher_tools\mod
 */
class deadline_manager_quiz extends deadline_manager_mod {
    const TABLE = 'quiz_overrides';
    const MODNAME = 'quiz';
    const FIELD_MOD = 'quiz';
    const FIELD_TIME = 'timeclose';

    /**
     * deadline_manager_quiz constructor
     *
     * @param \cm_info $cm
     *
     * @throws \dml_exception
     */
    public function __construct(\cm_info $cm){
        parent::__construct($cm);
        $this->_module = $this->_instance;
    }

    /**
     * Return instance object to update by moodle
     *
     * @return object
     */
    public function process_mod_instance_before_update(){
        $instance = parent::process_mod_instance_before_update();
        $instance->quizpassword = $this->_instance->password;

        return $instance;
    }


    /**
     * Process $override in mod classes
     * It should also call @see _update_override_object() inside
     *
     * @param object $override
     *
     * @return object - processed override
     */
    protected function _process_override($override){
        $instanceid = $this->_cm->instance;
        $override->id = $this->_update_override_object($override);

        if (empty($override->id)){
            return null;
        }

        quiz_update_open_attempts(['quizid' => $instanceid]);
        if ($override->groupid ?? false){
            // Priorities may have shifted, so we need to update all of the calendar events for group overrides.
            quiz_update_events($this->_module);
        } else {
            // User override. We only need to update the calendar event for this user override.
            quiz_update_events($this->_module, $override);
        }

        return $override;
    }

    /**
     * Process override deleting in mod classes
     *
     * @param object $override
     *
     * @return bool - true if $override was deleted
     */
    protected function _process_delete_override($override){
        $quiz = $this->_module;
        // Set the course module id before calling quiz_delete_override().
        $quiz->cmid = $this->_cm->id;
        quiz_delete_override($quiz, $override->id);

        return true;
    }


    /**
     * Process all overrides deleting in mod classes
     * Called from the @see delete_all_overrides()
     *
     * @return bool
     */
    protected function _process_delete_all_overrides(){
       $quiz = $this->_module;
        // Set the course module id before calling quiz_delete_all_overrides().
        $quiz->cmid = $this->_cm->id;
        quiz_delete_all_overrides($quiz);

        return true;
    }

    /**
     * @param $userid
     * @return mixed|object
     */
    public function get_user_effective_access($userid){
        $instance =  quiz_update_effective_access(clone $this->_instance, $userid);
        return $instance->{static::FIELD_TIME};
    }

    /**
     * @return mixed|void
     * @throws \coding_exception
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function cron(){
        global $DB;

        $timeend = time();
        $timestart = $timeend - DAYSECS;
        $sql = "SELECT q.*
                  FROM {quiz} q 
                 WHERE (q.timeclose > ? AND q.timeclose < ?) 
                    OR (SELECT COUNT(1) FROM {quiz_overrides} qo WHERE qo.timeclose > ? AND qo.timeclose < ? AND qo.quiz = q.id) > 0";

        if ($quizzes = $DB->get_records_sql($sql, [$timestart, $timeend, $timestart, $timeend])) {
            foreach ($quizzes as $instance){
                $course = get_course($instance->course);
                $cm = get_coursemodule_from_instance('quiz', $instance->id, 0, false, MUST_EXIST);
                $cmid = $cm->id;
                $gradeitem = \grade_item::fetch(['itemtype' => 'mod', 'iteminstance' => $instance->id, 'itemmodule' => 'quiz']);

                if ($users = NED\get_enrolled_users($course->id, \context_course::instance($instance->course), 'mod/quiz:attempt')) {
                    foreach ($users as $user){
                        $modinfo = get_fast_modinfo($course, $user->id);
                        $cm = $modinfo->get_cm($cmid);
                        if (!$cm->uservisible){
                            continue;
                        }
                        $quiz = new \quiz($instance, $cm, $course);

                        $instance = quiz_update_effective_access($instance, $user->id);

                        $deadline = $instance->timeclose;

                        if (empty($deadline) || $deadline > time()) {
                            continue;
                        }

                        // If attempted before deadline.
                        $sql = "SELECT qa.userid
                                  FROM {quiz_attempts} qa
                                 WHERE qa.quiz = ?
                                   AND qa.userid = ? 
                                   AND qa.timefinish <= ?
                                   AND qa.state = ?";

                        if ($DB->record_exists_sql($sql, [$instance->id, $user->id, $deadline, 'finished'])) {
                            continue;
                        }

                        if (!$grade = \grade_grade::fetch(['userid' => $user->id, 'itemid' => $gradeitem->id])) {
                            $grade = new \grade_grade();
                            $grade->userid = $user->id;
                            $grade->itemid = $gradeitem->id;
                            $grade->rawgrademax = $gradeitem->grademax;
                            $grade->timemodified = time();
                            $grade->feedbackformat = FORMAT_PLAIN;
                            $grade->feedback = null;
                            $grade->finalgrade = null;
                        }

                        // Skip if graded.
                        if (!is_null($grade->finalgrade)) {
                            continue;
                        }

                        // Give minimum grade.
                        $grade->rawgrade = $gradeitem->grademin;

                        $status = grade_update('mod/quiz', $gradeitem->courseid,
                            $gradeitem->itemtype, $gradeitem->itemmodule,
                            $gradeitem->iteminstance, $gradeitem->itemnumber,
                            $grade);

                        if ($status == GRADE_UPDATE_OK){
                            if ($quizgrade = $DB->get_record('quiz_grades', array('quiz' => $instance->id, 'userid' => $user->id))) {
                                $quizgrade->grade = $gradeitem->grademin;
                                $quizgrade->timemodified = time();
                                $DB->update_record('quiz_grades', $quizgrade);

                            } else {
                                $quizgrade = new \stdClass();
                                $quizgrade->quiz = $instance->id;
                                $quizgrade->userid = $user->id;
                                $quizgrade->grade = $gradeitem->grademin;
                                $quizgrade->timemodified = time();
                                $DB->insert_record('quiz_grades', $quizgrade);
                            }
                        }
                    }
                }
            }
        }
    }

    /**
     * @param $userid
     * @return bool
     * @throws \dml_exception
     */
    public function get_mod_grade($userid){
        if ($grades = quiz_get_user_grades($this->_instance, $userid)) {
            $grade = array_shift($grades);
            return $grade->rawgrade;
        }
        return false;
    }

    /**
     * @param $userid
     * @return bool
     * @throws \dml_exception
     */
    public function get_user_submission($userid){
        global $DB;

        $sql = "SELECT qa.*
                  FROM {quiz_attempts} qa
                 WHERE qa.quiz = ?
                   AND qa.userid = ? 
                   AND qa.state = ?
              ORDER BY qa.attempt DESC";

        return $DB->get_record_sql($sql, [$this->_instance->id, $userid, 'finished'], IGNORE_MULTIPLE);
    }

    /**
     * Set time limit for all groups/users in the current quiz
     *
     * @param int|null $value - Unix timestamp, 0 - no limit, null - do not override it
     *
     * @return void
     */
    public function set_timelimit($value=null){
        if (!$this->is_enabled()) return;

        SH::db()->set_field(static::TABLE, 'timelimit', $value, ['quiz' => $this->_cm->instance]);
    }
}