<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Deadline Manager Header
 *
 * @package    block_ned_teacher_tools
 * @copyright  Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */


namespace block_ned_teacher_tools\output;

defined('MOODLE_INTERNAL') || die();

use block_ned_teacher_tools as NED;
use block_ned_teacher_tools\deadline_manager as DM;
use block_ned_teacher_tools\output\menu_bar as MB;
use block_ned_teacher_tools\shared_lib as SH;

/**
 * Class deadline_manager_report_overview - overview for all groups
 *
 * @package block_ned_teacher_tools\output
 */
class deadline_manager_report_overview extends deadline_manager_report {

    /**
     * Export page contents for template
     * @param \renderer_base $output
     * @return \stdClass
     */
    public function export_for_template(\renderer_base $output) {
        $data = new \stdClass();
        $data->groups = [];
        $deadlinemanager = new DM($this->course->id);
        $schedulestr = DM::get_schedule_options();
        $show_inactive = NED\check_show_inactive_with_cache($this->_show_inactive, $this->course->id);

        $data->canmanagegroups = SH::has_capability('manage_group_requirements', $this->_context);
        $groups = $this->_groups ?: [];

        foreach ($groups as $group) {
            if (!$show_inactive && empty($group->users)) continue;

            $row = new \stdClass();
            $row->groupname = $group->name;
            if ($data->canmanagegroups) {
                // Schedule.
                $tmplschedule = new \core\output\inplace_editable(NED\PLUGIN_NAME, 'schedule', $group->id,
                    true, $schedulestr[$group->schedule], $group->schedule);
                $tmplschedule->set_type_select($schedulestr);
                $row->schedule = SH::render($tmplschedule);

                // Start Date.
                $startdate = self::strdate($group->startdate);
                $tmplstartdate = new \core\output\inplace_editable(NED\PLUGIN_NAME, 'startdate', $group->id,
                    true, $startdate, $startdate);
                $row->startdate = SH::render($tmplstartdate);

                // End Date.
                $row->raw_enddate = $group->enddate;
                $enddate = self::strdate($group->enddate) ;
                $tmplenddate = new \core\output\inplace_editable(NED\PLUGIN_NAME, 'enddate', $group->id,
                    true, $enddate, $enddate);
                $row->enddate = SH::render($tmplenddate);
                $row->groupmemberaddurl = (new \moodle_url('/group/members.php', ['group' => $group->id]))->out();
            } else {
                if ($group->schedule == DM::SCHEDULE_NONE) {
                    unset($groups[$group->id]);
                    continue;
                }
            }

            $row->id = "G".$group->id;
            $row->groupurl = $this->make_url(MB::PAGE_DM, [MB::PAR_SUBTYPE => DM::PAGE_GROUP, MB::PAR_GROUP => $group->id]);
            $row->onlineteacher = $this->get_users_by_role(SH::ROLE_OT, $group);
            $row->classroomteacher = $this->get_users_by_role(SH::ROLE_CT, $group);
            // Missing deadline warning.
            $row->warningicon = $deadlinemanager->has_missed_schedule($group);
            // Students
            $row->studentsurl = $this->make_url(MB::PAGE_DM,
                [MB::PAR_SUBTYPE => DM::PAGE_USER, MB::PAR_GROUP => $group->id, MB::PAR_USER => 0]);

            $students = $group->users ?: [];
            $row->students = count($students);
            $row->extensionsgranted = 0;
            foreach ($students as $student){
                $row->extensionsgranted += DM::get_number_of_extensions_in_course($student, $this->course, true);
            }

            $data->groups[] = $row;
        }
        $data->hasgroups = !empty($data->groups);
        return $data;
    }
}
