<?php
/**
 * @package    block_ned_teacher_tools
 * @subpackage output
 * @copyright  2020 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace block_ned_teacher_tools\output;

use block_ned_teacher_tools as NED;
use block_ned_teacher_tools\deadline_manager as DM;
use \local_ned_controller\marking_manager\marking_manager as MM;
use \block_ned_teacher_tools\shared_lib as SH;

defined('MOODLE_INTERNAL') || die();
require_once (__DIR__ .'/../deadline_manager.php');

/**
 * Class menu_bar
 *
 * @package block_ned_teacher_tools\output
 *
 * @property-read string $page = '';
 * @property-read string $prevpage = '';
 * @property-read string $pagetype = '';
 * @property-read string $pagesubtype = '';
 * @property-read string $base_url = '';
 * @property-read array $pageparams = [];
 * @property-read bool $is_DM = false;
 * @property-read bool $act = false;
 * @property-read bool $external = false;
 * @property-read bool $isteacher = false;
 *
 * @property-read int $courseid;
 * @property-read int $show_inactive;
 * @property-read int $cm_students;
 * @property-read int $groupid;
 * @property-read int $userid;
 * @property-read int $setuserid;
 * @property-read int $lock_user;
 *
 * @property-read \stdClass $course;
 * @property-read \context|\context_course $context;
 * @property-read \stdClass $config;
 * @property-read \stdClass $tt_config;
 *
 * @property-read array $allgroups;
 * @property-read \stdClass $group;
 * @property-read array $allstudents;
 * @property-read array $students;
 * @property-read \stdClass $user;
 * @property-read \stdClass $teacher;
 * @property-read int $teacherid;
 *
 * @property-read bool $kicainstalled;
 * @property-read bool $use_class_progress;
 * @property-read bool $use_student_progress;
 * @property-read bool $use_moodle_gradebook;
 * @property-read bool $use_kica_gradebook;
 * @property-read bool $use_deadline_manager;
 * @property-read bool $has_deadline_manager_groups;
 * @property-read string $tag_id;
 * @property-read array $tag_options = [];
 *
 * @property-read \block_ned_teacher_tools\output\renderer $renderer
 */
class menu_bar implements \renderable, \templatable
{
    const DEF = 'default';
    const TYPE = 'type';
    const PAR = 'param';
    const DEF_USER = 0;
    // Page parameter names
    const PAR_COURSE = 'courseid';
    const PAR_ID = 'id'; // it's course id sometime
    const PAR_GROUP = 'group';
    const PAR_USER = 'user';
    const PAR_SETUSER = 'setuser';
    const PAR_SHOWINACTIVE = 'show_inactive';
    const PAR_CMSTUDENTS = 'cm_students';
    const PAR_LOCKUSER = 'lock_user';
    const PAR_PREVPAGE = 'prevpage';
    const PAR_SHOWINACTIVE_BLOCK = NED\PLUGIN_NAME . '_' . 'show_inactive';
    const PAR_GROUP_BLOCK = NED\PLUGIN_NAME . '_' . 'groupid';
    const PAR_SUBTYPE = 'p';
    const PAR_PR_SHOW = 'prshow';

    const PAR_TAG = 'tag';
    const PAR_SORT = 'sort';

    const TAGS_ALL = MM::TAGS_ALL;
    const TAGS_NONE = MM::TAGS_NONE;

    // Pages
    //  PAGE_* - all pages, on which MB can be
    const PAGE_MM = 'marking_manager';
    const PAGE_SP = 'student_progress';
    const PAGE_CP = 'progress_report';
    const PAGE_RAPS = 'raps';
    const PAGE_KICA = '/local/kica/grade_user';
    const PAGE_GB = '/grade/report/grader/index';
    const PAGE_DM = 'deadline_manager';
    const PAGE_CTA = 'ctatracker';
    const PAGE_COURSE = '/course/view';
    const PAGE_PARTICIPANTS = '/user/index';

    const PAGE_DM_EDIT = 'deadline_manager_edit';
    const PAGE_KICA_SETTINGS = '/local/kica/coursesettings';
    const PAGE_KICA_ACTIVITIES = '/local/kica/kicaactivities';
    const PAGE_KICA_ACTIVITIES_EDIT = '/local/kica/kicaactivities_edit';
    const PAGE_KICA_GRADE_ACTIVITY = '/local/kica/grade_activity';
    const PAGE_KICA_NOTIFICATIONS = '/local/kica/notifications';
    const PAGE_KICA_ZEROGRADES = '/local/kica/zerogrades';

    // MENU_* - values the same to render static methods
    const MENU_STUDENT = 'student_selector';
    const MENU_GROUP = 'group_selector';
    const MENU_SHOWINACTIVE = 'show_inactive_checkbox';
    const MENU_CMSTUDENTS = 'show_cmstudents_checkbox';

    // Links for menu bar
    const PAGE_LINKS = [
        self::PAGE_CP => 'simplegradebook',
        self::PAGE_SP => 'activitycompletion',
        self::PAGE_MM => 'markingmanager',
        self::PAGE_KICA => 'kicagradebook',
        self::PAGE_GB => 'gradebook',
        self::PAGE_DM => 'deadlinemanager',
        self::PAGE_CTA => 'ctaactivities',
        self::PAGE_RAPS => 'raps',
    ];

    // here should be all PAGE_*
    const PAGE_TYPES = [
        self::PAGE_MM   => self::PAGE_MM,
        self::PAGE_SP   => self::PAGE_SP,
        self::PAGE_CP   => self::PAGE_CP,
        self::PAGE_RAPS   => self::PAGE_RAPS,
        self::PAGE_KICA => self::PAGE_KICA,
        self::PAGE_GB   => self::PAGE_GB,

        self::PAGE_DM       => self::PAGE_DM,
        self::PAGE_DM_EDIT  => self::PAGE_DM,

        self::PAGE_CTA => self::PAGE_CTA,

        self::PAGE_KICA_SETTINGS        => self::PAGE_KICA,
        self::PAGE_KICA_ACTIVITIES      => self::PAGE_KICA,
        self::PAGE_KICA_GRADE_ACTIVITY  => self::PAGE_KICA,
        self::PAGE_KICA_NOTIFICATIONS   => self::PAGE_KICA,
        self::PAGE_KICA_ZEROGRADES      => self::PAGE_KICA,
        self::PAGE_KICA_ACTIVITIES_EDIT => self::PAGE_KICA,

        self::PAGE_COURSE => self::PAGE_COURSE,
        self::PAGE_PARTICIPANTS => self::PAGE_PARTICIPANTS,
    ];

    const PARAM_SETTINGS = [
        self::PAR_USER =>           [self::DEF => self::DEF_USER, self::TYPE => PARAM_INT, self::PAR => '_userid'],
        self::PAR_SETUSER =>        [self::DEF => self::DEF_USER, self::TYPE => PARAM_INT, self::PAR => '_setuserid'],
        self::PAR_LOCKUSER =>       [self::DEF => false, self::TYPE => PARAM_BOOL, self::PAR => '_lock_user'],
        self::PAR_CMSTUDENTS =>     [self::DEF => null, self::TYPE => PARAM_INT, self::PAR => '_cm_students'],
        self::PAR_SHOWINACTIVE_BLOCK => [self::DEF => null, self::TYPE => PARAM_INT, self::PAR => '_show_inactive_block'],
        self::PAR_SHOWINACTIVE =>   [self::DEF => null, self::TYPE => PARAM_INT, self::PAR => '_show_inactive'],
        self::PAR_GROUP_BLOCK =>    [self::DEF => null, self::TYPE => PARAM_INT, self::PAR => '_groupid_block'],
        self::PAR_GROUP =>          [self::DEF => null, self::TYPE => PARAM_INT, self::PAR => '_groupid'],
        self::PAR_SUBTYPE =>        [self::DEF => null, self::TYPE => PARAM_TEXT, self::PAR => '_pagesubtype'],
        self::PAR_TAG =>            [self::DEF => self::TAGS_ALL, self::TYPE => PARAM_TEXT, self::PAR => '_tag_id'],
        self::PAR_PREVPAGE =>       [self::DEF => '', self::TYPE => PARAM_TEXT, self::PAR => '_prevpage'],
    ];

    // Other
    const COURSE_URL = '/course/view.php';
    const PARTICIPANTS_URL = '/user/index.php';

    const SORT_GRADE = 'grade';
    const SORT_GROUP = 'group';
    const SORT_NAME = 'studentname';
    const SORT_WAITINGFORGRADE = 'waitingforgrade';
    const SORT_COMPLETED = 'completed';

    const SELECT_SORT = [self::SORT_GRADE, self::SORT_GROUP, self::SORT_NAME, self::SORT_COMPLETED, self::SORT_WAITINGFORGRADE];

    /**
     * @var \block_ned_teacher_tools\output\renderer
     */
    protected static $_renderer = null;

    /**
     * @var static
     */
    protected static $_last_this = null;

    /**
     * @var string
     */
    protected $_page = '';
    /**
     * @var string
     */
    protected $_prevpage = '';
    /**
     * @var string
     */
    protected $_pagetype = '';
    /**
     * @var string
     */
    protected $_pagesubtype = null;
    /**
     * @var string
     */
    protected $_base_url = '';
    /**
     * @var array
     */
    protected $_pageparams = [];
    /**
     * @var bool
     */
    protected $_is_DM = false;
    /**
     * Show, need something to count or not
     * @var bool
     */
    protected $_act = false;
    /**
     * External or not current page
     * @var bool
     */
    protected $_external = false;
    /**
     * @var bool
     */
    protected $_isteacher = false;

    /**
     * @var int
     */
    protected $_courseid;
    /**
     * @var null|bool
     */
    protected $_show_inactive;
    /**
     * @var null|bool
     */
    protected $_show_inactive_block;
    /**
     * @var null|bool
     */
    protected $_cm_students;
    /**
     * @var int
     */
    protected $_groupid;
    /**
     * @var int
     */
    protected $_groupid_block;
    /**
     * @var int
     */
    protected $_userid;
    /**
     * @var int
     */
    protected $_setuserid;
    /**
     * @var bool
     */
    protected $_lock_user = false;

    /**
     * @var \stdClass
     */
    protected $_course;
    /**
     * @var \context|\context_course
     */
    protected $_context;
    /**
     * @var \stdClass
     */
    protected $_config;
    /**
     * @var \stdClass
     */
    protected $_tt_config;

    /**
     * @var array
     */
    protected $_allgroups;
    /**
     * @var \stdClass
     */
    protected $_group;
    /**
     * @var array
     */
    protected $_allstudents;
    /**
     * @var array
     */
    protected $_students;
    /**
     * @var \stdClass
     */
    protected $_user;
    /**
     * @var \stdClass
     */
    protected $_teacher;
    /**
     * @var int
     */
    protected $_teacherid;
    /**
     * @var bool
     */
    protected $_use_class_progress;
    /**
     * @var bool
     */
    protected $_use_raps;
    /**
     * @var bool
     */
    protected $_use_student_progress;
    /**
     * @var bool
     */
    protected $_kicainstalled;
    /**
     * @var bool
     */
    protected $_use_moodle_gradebook;
    /**
     * @var bool
     */
    protected $_use_kica_gradebook;
    /**
     * @var bool
     */
    protected $_use_deadline_manager;
    /**
     * @var bool
     */
    protected $_has_deadline_manager_groups;
    /**
     * @var bool
     */
    protected $_use_cta_tracker;

    protected $_tag_id;
    protected $_tag_options = [];

    /**
     * menu_bar constructor.
     *
     * @param string $page
     * @param array  $pageparams
     * @param null   $courseid - if null, loads form page params as self::PAR_COURSE
     * @param bool   $ignore_teacher_capability - set to true, if want MB works with any capabilities
     */
    public function __construct($page='', $pageparams=[], $courseid=null, $ignore_teacher_capability=false)
    {
        $this->_page = static::get_page_path($page);
        $this->_external = static::is_external($this->_page);
        $this->_pagetype = NED\isset2(static::PAGE_TYPES, [$page], $page);
        $this->_base_url = static::get_page_url($page);
        $this->_is_DM = $this->_pagetype == static::PAGE_DM;
        $this->_pageparams = [];
        $this->_load_page_parameters($courseid);
        $this->_check_parameters_and_load_data($ignore_teacher_capability);
        $this->_init_static();
        $this->_pageparams = array_merge($this->_pageparams, $pageparams);
    }

    /**
     * @param $name
     *
     * @return \mixed
     */
    public function __get($name){
        $pr_name = '_' . $name;
        if (property_exists($this, $pr_name)){
            return ($this::${$pr_name} ?? $this->$pr_name) ?? null;
        } elseif (property_exists($this, $name)){
            return ($this::${$name} ?? $this->$name) ?? null;
        }
        return null;
    }

    /**
     * @param $name
     * @param $value
     *
     * @return \mixed
     */
    public function __set($name, $value)
    {
        $private_name = '_' . $name;
        if (property_exists($this, $private_name) || ($name[0] == '_' && property_exists($this, $name))){
            return null;
        }
        $this->$name = $value;
        return $value;
    }

    /**
     * Init static functions
     */
    protected function _init_static(){
        global $PAGE;
        if (!static::$_last_this){
            static::$_last_this = &$this;
            static::$_renderer = $PAGE->get_renderer(NED\PLUGIN_NAME);
        }
    }

    /**
     * Called by __construct
     *
     * @param null $courseid - if null, loads form page params as self::PAR_COURSE
     *
     */
    protected function _load_page_parameters($courseid=null){
        global $COURSE;
        $courseid = $courseid ?: optional_param(static::PAR_COURSE, null, PARAM_INT);
        if (!$courseid){
            $courseid = optional_param(SH::PAR_COURSE,null, PARAM_INT);
        }
        if (!$courseid){
            $courseid = $COURSE->id ?? null;
        }
        if (!$courseid){
            $courseid = optional_param(static::PAR_ID,null, PARAM_INT);
        }
        if (!$courseid){
            SH::print_error('missingparam', '', '', static::PAR_COURSE);
        }
        $this->_courseid = $courseid;

        foreach (static::PARAM_SETTINGS as $par_name => $setting){
            $par_value = $setting[static::PAR] ?? '';
            if (empty($par_value)){
                continue;
            }
            $def = $setting[static::DEF] ?? null;
            $type = $setting[static::TYPE] ?? PARAM_RAW;
            $this->$par_value = optional_param($par_name, $def, $type);
        }

        $this->_show_inactive = $this->_show_inactive_block ?? $this->_show_inactive;
        $this->_groupid = $this->_groupid_block ?? $this->_groupid;
        $this->_lock_user = $this->_lock_user && $this->_userid != static::DEF_USER;

        if ($this->_setuserid){
            $this->_userid = $this->_setuserid;
            $this->_groupid = $this->_groupid ?? NED\GROUP_ALL;
        } else {
            if (!static::is_same_page($this->_page, $this->_prevpage) && !$this->_lock_user){
                $this->_userid = static::DEF_USER;
            }
        }
    }

    /**
     * Called by __construct
     *
     * @param bool $ignore_teacher_capability - set to true, if want MB works with any capabilities
     *
     */
    protected function _check_parameters_and_load_data($ignore_teacher_capability=false){
        global $DB, $USER, $PAGE;
        if (!$course = $DB->get_record('course', ['id' => $this->_courseid])) {
            SH::print_error('Course ID was incorrect');
        }
        $this->_course = $course;

        if ($PAGE->course->id != $this->_courseid && empty($PAGE->get_where_theme_was_initialised())){
            require_login($course);
        }

        $this->_context = \context_course::instance($course->id);
        $this->_isteacher = is_siteadmin() || can_access_course($course, $USER, SH::CAPABILITY_COURSE_GRADER, true);

        if (!$this->_isteacher && !$this->_external && !$ignore_teacher_capability){
            return;
        }

        $this->_config = get_config(NED\PLUGIN_NAME);
        $this->_tt_config = NED\get_block_config($this->_courseid, NED\PLUGIN);
        $show_inactive = $this->_show_inactive;
        $this->_cm_students = $this->is_show_cmstudents_checkbox() ? NED\check_load_cm_students_with_cache($this->_cm_students, $course->id) : null;
        $groupid = $this->_groupid;
        [$this->_show_inactive, $this->_allgroups, $this->_group, $this->_groupid, $this->_allstudents, $this->_students,
            $this->_user, $this->_userid]
        = static::get_showinactive_groups_group_groupid_allstudents_students_user($course, $show_inactive, $groupid,
            $this->_userid, $this->_is_DM, $this->_setuserid, $this->_cm_students);

        static::update_cach($this->_courseid, $this->_show_inactive, $this->_groupid, $this->_cm_students);

        $this->_act = !is_null($this->_groupid) && $this->_groupid != NED\GROUP_NONE;

        $this->_teacher = $USER;
        $this->_teacherid = $USER->id;

        $this->_use_class_progress = $this->_config->showgradeslink;
        $this->_use_student_progress = $this->_config->showactivitycompletions;
        $this->_kicainstalled = has_capability('block/ned_teacher_tools:viewkicagradebook', $this->_context)
            && NED\utils::kica_gradebook_enabled($this->_courseid);
        $this->_use_raps = has_capability('block/ned_teacher_tools:canviewrapsreport', $this->_context);
        if ($this->_tt_config->showlinktogradebook ?? true){
            $this->_use_moodle_gradebook = ($this->_config->showgradebook &&
                ($this->_config->gradebooklink == 0 || $this->_config->gradebooklink == 2 || !$this->_kicainstalled));
            $this->_use_kica_gradebook = ($this->_config->showgradebook && $this->_kicainstalled &&
                ($this->_config->gradebooklink == 1 || $this->_config->gradebooklink == 2));
        }

        // DM
        $this->_use_deadline_manager = false;
        $this->_has_deadline_manager_groups = false;
        $deadline_aag = false;

        if ($this->_tt_config->enabledeadlinemanager ?? false){
            $deadline_aag = has_capability(NED\PLUGIN_CAPABILITY.'view_all_groups', $this->_context);
            $this->_use_deadline_manager = $deadline_aag ||
                has_any_capability([NED\PLUGIN_CAPABILITY.'view_own_groups', NED\PLUGIN_CAPABILITY.'deadlineviewonly'], $this->_context);
        }

        if ($this->_use_deadline_manager){
            if ($this->_is_DM || ($deadline_aag || $this->_course->groupmode == VISIBLEGROUPS)){
                $this->_has_deadline_manager_groups = !empty($this->_allgroups);
            } else {
                $this->_has_deadline_manager_groups = false;
                foreach ($this->_allgroups as $group){
                    if ($group->schedule == NED\deadline_manager::SCHEDULE_FULL) {
                        $this->_has_deadline_manager_groups = true;
                        break;
                    }
                }
            }
        }

        // CTA
        $this->_use_cta_tracker = \block_ned_teacher_tools\cta_tracker::can_view_active_tracker($this->_courseid, $this->_teacher);

        $this->_pageparams[static::PAR_COURSE] = $this->_courseid;
        $this->_pageparams[static::PAR_USER] = $this->_userid;

        if (!is_null($show_inactive) && $this->is_show_selector(static::MENU_SHOWINACTIVE)){
            $this->_pageparams[static::PAR_SHOWINACTIVE] = $this->_show_inactive;
        }
        if ($this->is_show_cmstudents_checkbox()){
            $this->_pageparams[static::PAR_CMSTUDENTS] = $this->_cm_students;
        }
        if (!is_null($groupid) && $this->is_show_selector(static::MENU_GROUP)){
            $this->_pageparams[static::PAR_GROUP] = $this->_groupid;
        }
        if (!is_null($this->_pagesubtype)){
            $this->_pageparams[static::PAR_SUBTYPE] = $this->_pagesubtype;
        }
        $this->_pageparams[static::PAR_PREVPAGE] = $this->_page;
        $this->_pageparams[static::PAR_LOCKUSER] = $this->_lock_user;

        $this->_tag_options = $this->get_tag_options($this->_context);
        $this->_tag_id = NED\isset_key($this->_tag_options, $this->_tag_id);
        $this->_pageparams[static::PAR_TAG] = $this->_tag_id;

        $this->_check_DM_subpage();
    }

    /**
     * Check and change _pagesubtype for DM pages
     *
     */
    protected function _check_DM_subpage(){
        if (!$this->_is_DM){
            return;
        }

        $pst = $this->_pagesubtype;

        if (is_null($pst)){
            $pst = DM::PAGE_OVERVIEW;
        }

        if ($pst == DM::PAGE_GROUP && is_null($this->_group)){
            $pst = DM::PAGE_OVERVIEW;
        } elseif ($pst == DM::PAGE_OVERVIEW && $this->_groupid != NED\GROUP_NONE && $this->_groupid != NED\GROUP_ALL) {
            $pst = DM::PAGE_GROUP;
        }

        if ($pst == DM::PAGE_USER && $this->_userid < 1){
            $pst = DM::PAGE_USER_OVERVIEW;
        } elseif ($pst == DM::PAGE_USER_OVERVIEW && $this->_userid > 0){
            $pst = DM::PAGE_USER;
        }

        $this->_pagesubtype = $pst;
    }

    /**
     * Return should we show link on this page in menu bar
     *
     * @param $page
     *
     * @return bool
     */
    public function is_show_page($page){
        switch ($page){
            case static::PAGE_DM: return $this->_use_deadline_manager && $this->_has_deadline_manager_groups;
            case static::PAGE_KICA: return $this->_use_kica_gradebook;
            case static::PAGE_GB: return $this->_use_moodle_gradebook;
            case static::PAGE_CP: return $this->_use_class_progress;
            case static::PAGE_RAPS: return $this->_use_raps;
            case static::PAGE_SP: return $this->_use_student_progress;
            case static::PAGE_CTA: return $this->_use_cta_tracker;
        }
        return true;
    }

    /**
     * Return should we show this selector in menu bar
     *
     * @param $selector
     *
     * @return bool
     */
    public function is_show_selector($selector){
        switch ($selector){
            case static::MENU_STUDENT:
                if (!$this->_act){
                    return false;
                }
                if ($this->_is_DM){
                    switch ($this->_pagesubtype){
                        case DM::PAGE_USER_OVERVIEW:
                        case DM::PAGE_USER:
                        case DM::PAGE_USER_EDIT:
                            return true;
                    }
                    return false;
                }
                break;
            case static::MENU_SHOWINACTIVE: return !is_null($this->_show_inactive);
            case static::MENU_CMSTUDENTS: return $this->is_show_cmstudents_checkbox();
        }

        return true;
    }

    /**
     * Return pageparam by name, if exists, null otherwise
     *
     * @param $name
     *
     * @return mixed|null
     */
    public function get_pageparam($name){
        return NED\isset2($this->_pageparams, [$name], null);
    }

    /**
     * Set pageparam
     *
     * @param $name
     * @param $value
     *
     * @return mixed|null
     */
    public function set_pageparam($name, $value){
        return $this->_pageparams[$name] = $value;
    }

    /**
     * Union new page params with the current
     *
     * @param array|object $new_pageparams
     * @param bool         $replace_all - if true, clear old page params first
     *
     * @return array
     */
    public function update_pageparams($new_pageparams=[], $replace_all=false){
        $new_pageparams = SH::val2arr($new_pageparams);
        if ($replace_all){
            $this->_pageparams = $new_pageparams;
        } else {
            $this->_pageparams = array_merge($this->_pageparams, $new_pageparams);
        }
        return $this->_pageparams;
    }

    /**
     * @param array $options
     *
     * @return array
     */
    public function parameters_for_MM($options=[]){
        $res = [
            'course' => $this->_course,
            'context' => $this->_context,
            'show_inactive' => $this->_show_inactive,
            'cmstudents' => $this->_cm_students,
            'groupid' => $this->_groupid,
            'studuserid' => $this->_userid,
            'load_users' => true,
            'set_user' => $this->_setuserid,
            'set_students' => false
        ];

        return array_merge($res, $options);
    }

    /**
     * Export page contents for template
     *
     * @param \renderer_base $output
     * @return \stdClass
     */
    public function export_for_template(\renderer_base $output)
    {
        global $PAGE, $DB, $USER;
        $data = new \stdClass();
        $data->id = NED\PLUGIN_NAME . '_menu_bar';
        $data->class = 'menu-bar';
        $data->add_body_class = 'path-blocks-NED';
        $data->apps = [];

        $base_url = $this->get_url();
        $base_url->remove_params('showttclasses');

        // Course page.
        $add_class = '';
        if ($PAGE->has_set_url() && $PAGE->url->get_path(true) == static::COURSE_URL){
            $add_class = 'active';
            $icon_course = 'course_white';
        } else {
            $icon_course = 'course_grey';
        }
        $img_course = NED\img($icon_course, '', 'block_ned_teacher_tools',
            ['title' => NED\str('returnto', $this->_course->shortname)]
        );
        $data->apps[] = NED\link([static::COURSE_URL, ['id' => $this->_courseid]], $img_course, $add_class);

        // List all Users.
        $add_class = '';
        if ($PAGE->has_set_url() && $PAGE->url->get_path(true) == static::PARTICIPANTS_URL){
            $add_class = 'active';
            $icon_users = 'users_white';
        } else {
            $icon_users = 'users_grey';
        }
        $img_users = NED\img($icon_users, '', 'block_ned_teacher_tools',
            ['title' => NED\str('listallusers')]
        );

        $data->apps[] = NED\link(['/user/index.php', ['contextid' => $this->context->id, 'group' => $this->group->id ?? 0,]], $img_users, $add_class);

        foreach (static::PAGE_LINKS as $page => $page_name){
            if (!$this->is_show_page($page)){
                continue;
            }
            $val = $this->get_menu_link($page);
            if (!empty($val)){
                $data->apps[] = $val;
            }
        }

        $selectors = [
            static::MENU_GROUP => [$base_url, $this->_allgroups, $this->_groupid, static::PAR_GROUP, null, $this->_show_inactive, $this->_is_DM],
            static::MENU_STUDENT => [$base_url, $this->_students, $this->_userid, $this->_lock_user, static::PAR_SETUSER],
            static::MENU_SHOWINACTIVE => [$base_url, $this->_show_inactive],
            static::MENU_CMSTUDENTS => [$base_url, $this->_cm_students],
        ];

        $data->selectors = [];
        foreach ($selectors as $selector_render => $args){
            if (!$this->is_show_selector($selector_render)){
                continue;
            }
            $val = static::$selector_render(...$args);
            if (!empty($val)){
                $data->selectors[]= $val;
            }
        }

        // Toggle
        if ($viewclassesmenubar = has_capability('block/ned_teacher_tools:viewclassesmenubar', $this->context)) {
            $pref = (int)get_user_preferences('block_ned_teacher_tools_showclassesbar', 0);
            $showclsbar = optional_param('showttclasses', $pref, PARAM_INT);
            if ($showclsbar != $pref) {
                set_user_preference('block_ned_teacher_tools_showclassesbar', $showclsbar);
            }
            $data->school_link_class = (empty($showclsbar)) ? '' : 'active';
            $icon_name = (empty($showclsbar)) ? 'rps_grey' : 'rps_blue';
            $icon = NED\img($icon_name, 'icon', NED\PLUGIN_NAME,
                ['title' => SH::$C::str('schools')]
            );
            $data->school_link = NED\link([$base_url, ['showttclasses' => (!empty($showclsbar)) ? 0 : 1]], $icon);
        } else {
            $data->school_link_class = 'nobar';
        }

        if ($viewclassesmenubar && !empty($showclsbar) && ($this->_groupid > 0 || $school = SH::get_user_school($USER->id, false))) {
            if ($PAGE->has_set_url() && $PAGE->url->get_path(true) == static::PARTICIPANTS_URL) {
                $classurl = new \moodle_url('/user/index.php', ['contextid' => $this->_context->id, 'group' => 0]);
            } else {
                $classurl = $this->get_url();
            }

            list($classes, $courses, $school) = $this->get_school_classes($school->code ?? null);

            if ($school) {
                $data->displayschool = true;
                $data->schoolcode = $school->code;
                $data->schoollink = (new \moodle_url('/local/schoolmanager/view.php', ['schoolid' => $school->id, 'view' => 'students']))->out(false);
            }

            if ($courses) {
                $data->courses = [];
                $path = $classurl->get_path();
                foreach ($courses as $courseid => $course) {
                    $urlparams = ['courseid' => $courseid,'group' => $course['groupid']];
                    if ($path && $path === static::COURSE_URL) {
                        $urlparams['id'] = $courseid;
                    }
                    $data->courses[] = [
                        'name' => $course['shortname'],
                        'url' => (new \moodle_url($classurl, $urlparams))->out(false),
                        'active' => $this->_courseid == $courseid
                    ];
                }
            }

            if ($classes) {
                $data->showclassesbar = true;
                $data->classes = [];
                foreach ($classes as $classid => $class) {
                    $data->classes[] = [
                        'name' => $class['name'],
                        'url' => (new \moodle_url($classurl, ['courseid' => $class['courseid'],  'group' => $classid]))->out(false),
                        'active' => $this->_groupid == $classid
                    ];
                }
            }
            $data->all_classes_link = (new \moodle_url('/local/schoolmanager/view.php',
                ['schoolid' => $school->id, 'view' => 'classes']))->out(false);
        }

        $data->course_link = '';

        if (is_siteadmin()) {
            // Action menu.
            $menu = new \local_ned_controller\output\ned_action_menu();
            $menu->attributessecondary['class'] .= ' ned-user-profile-context-menu user-profile-context-menu';
            $triggerextraclasses = [];
            $triggerextraclasses[] = 'dimmed';
            $action_attributes = ['class' => 'context-menu-link', 'target' => '_blank'];

            $menu_add = function($url_page, $url_params=[], $text='', $add_class=[]) use (&$menu, $action_attributes){
                $url = new \moodle_url($url_page, $url_params);
                if (!empty($add_class)){
                    $action_attributes['class'] = ($action_attributes['class'] ?? '').' '. SH::arr2str($add_class);
                }
                $menu->add_secondary_action(new \action_link($url, SH::fa('custom').$text, null, $action_attributes));
            };

            $groupid = $this->group->id ?? 0;

            // Proxy Submissions.
            $url_params = [
                'id' => $this->_courseid,
                'group' => $groupid,
            ];
            $menu_add(SH::PLUGIN_DIRS[SH::PROXY_LOCAL] . '/overview.php', $url_params, SH::str('proxysubmissions'));

            // Deadline Extensions.
            if ($groupid) {
                $cls = '';
                if (!$DB->record_exists('block_ned_teacher_tools_exte', ['groupid' => $groupid])) {
                    $cls = 'dimmed_text';
                }
                $url_params = ['filterclass' => $groupid];
                $menu_add('/report/ghs/ghs_activity_extension.php', $url_params, SH::str('deadlineextensions'), $cls);
            }

            // Academic Integrity Infractions:
            if ($groupid && SH::is_ai_exists()){
                $infractions = new \local_academic_integrity\output\datatable_infractions();
                $infractions->init(false, ['course' => $this->_courseid]);
                $igroups = $infractions->get_group_options();

                $cls = '';
                if (!$igroups || empty($igroups[$groupid])) {
                    $cls = 'dimmed_text';
                }
                $url_params = [
                    'course' => $this->_courseid,
                    'group' => $groupid,
                ];
                $menu_add('/local/academic_integrity/infractions.php', $url_params, SH::str('academicintegrityinfractions'), $cls);
            }

            // Grade Penalties.
            if ($groupid) {
                $url_params = [
                    'course' => $this->_courseid,
                    'group' => $groupid,
                    'reason' => 0,
                ];

                $cls = '';
                if (!$PAGE->has_set_url() || $PAGE->url->get_path(true) != static::COURSE_URL) {
                    $gradecontoller = new \local_ned_controller\output\ned_grade_controller_render();
                    $gradecontoller->init(false, $url_params);
                    if (isset($infractions)){
                        $ggroups = $infractions->get_group_options();

                        if (!$ggroups || empty($ggroups[$groupid])) {
                            $cls = 'dimmed_text';
                        }
                    }
                }

                $menu_add('/local/ned_controller/grade_controller.php', $url_params, SH::str('gradepenalties'), $cls);
            }

            // Teacher Tools Settings.
            $config = block_ned_teacher_tools_get_block_config($this->_courseid);
            if (!empty($config->block_id)) {
                $url_params = [
                    'id' => $this->_courseid,
                    'bui_editid' => $config->block_id,
                ];
                $menu_add(SH::PLUGIN_DIRS[SH::TT] . '/block_setting.php', $url_params, SH::str('ttsettings'));
            }

            // CT Submissions
            if (SH::is_ctsubmission_exists()){
                $menu_add(SH::PLUGIN_DIRS[SH::CTSUBMISSION] . '/overview.php',
                    ['courseid' => $this->_courseid, 'groupid' => $groupid],
                    get_string('ctsubmission', SH::CTSUBMISSION));
            }

            // Manage activity tags.
            $menu_add('/blocks/ned_teacher_tools/manage_activities_settings.php', ['id' => $this->_courseid], SH::str('manage_activities_tags'));

            $menu->attributes['class'] = 'float-right ml-1';
            $menu->set_menu_trigger(' ', implode(' ', $triggerextraclasses));

            $data->course_link .=  SH::render($menu);
        }

        $data->hasapps = !empty($data->apps);
        $data->hasselectors = !empty($data->selectors);

        return $data;
    }

    /**
     * @return array|false
     * @throws \dml_exception
     */
    public function get_school_classes($code) {
        global $DB;

        if (empty($this->_groupid) && empty($code)) {
            return false;
        }

        if (empty($code)) {
            $name = explode('/', $this->group->name);

            if (strlen($name[0]) != 4) {
                return false;
            }
            $code = $name[0];
        }

        $school = $DB->get_record('local_schoolmanager_school', ['code' => $code]);

        $classes = [];
        foreach ($this->_allgroups as $group){
            if (!$this->_show_inactive && empty($group->users)){
                continue;
            }
            if (strpos($group->name, $code) === 0) {
                $classes[$group->id] = ['courseid' => $this->_courseid, 'name' => format_string($group->name)];
            }
        }

        // Courses
        $courses = [];
        $params = [];
        $insql = '';

        if (!is_siteadmin()) {
            if (!$enrolled = enrol_get_my_courses()) {
                return [$classes, $courses, $school];
            }
            list($insql, $params) = $DB->get_in_or_equal(array_keys($enrolled), SQL_PARAMS_NAMED);

            $insql = "AND c.id {$insql}";
        }

        $like = $DB->sql_like('g.name', ':name');
        $params['name'] = $code . '%';
        $params['enddate'] = time();

        $sql = "SELECT c.*, g.id groupid
                  FROM {groups} g
                  JOIN {course} c ON g.courseid = c.id
                 WHERE {$like}
                       {$insql}
                   AND g.schedule = 2 AND g.enddate > :enddate
              GROUP BY c.id";
        if ($courses = $DB->get_records_sql($sql, $params)) {
            foreach ($courses as $course) {
                $courses[$course->id] = [
                    'shortname' => $course->shortname,
                    'groupid' => $course->groupid
                ];
            }
        }

        return [$classes, $courses, $school];
    }

    /**
     * Check capabilities an show error if necessary
     *
     * @param array $check_other_capabilities_all - all of this capabilities should be
     * @param array $check_other_capabilities_any - any of this capabilities should be
     * @param bool  $ignore_teacher_capability - check or not base teacher capability
     */
    public function show_error_for_not_teacher($check_other_capabilities_all=[], $check_other_capabilities_any=[], $ignore_teacher_capability=false){
        $pr_error = function(){
            SH::print_error('nopermissions', 'error', '', get_string('checkpermissions', 'core_role'));
        };
        $ctx = $this->_context;

        if (!$ignore_teacher_capability && !$this->_isteacher){
            $pr_error();
        }
        if (!empty($check_other_capabilities_all) && !has_all_capabilities($check_other_capabilities_all, $ctx)){
            $pr_error();
        }
        if (!empty($check_other_capabilities_any) && !has_any_capability($check_other_capabilities_any, $ctx)){
            $pr_error();
        }

        if ($this->_is_DM){
            if (!$this->_use_deadline_manager){
                redirect(new \moodle_url('/course/view.php',['id' => $this->_courseid]), NED\str('deadlinemanagerisnotenabled'));
            }
            if (!$this->_has_deadline_manager_groups){
                redirect(new \moodle_url('/course/view.php',['id' => $this->_courseid]), NED\str('norequiredgroupfound'));
            }
        }
    }

    /**
     * @return bool
     */
    public function is_show_cmstudents_checkbox(){
        global $DB;
        static $data = [];
        $is_dm = (int)$this->is_DM;
        if (!isset($data[$this->courseid][$is_dm])){
            [$allowedgroups, $ignore_grouping] = NED\get_groups($this->course, $this->is_DM);

            $f_check_other_stidents = function($groupid) use ($DB) {
                $params = [];
                $where = ["(r.id <> role_course.id AND r.id = role_cm.id)"];
                $sql = NED\get_sql_user_enrolments(
                    "DISTINCT u.id, u.*, gr.id AS groupid, role_course.shortname AS course_role",
                    $where, $params, NED\ROLE_STUDENT,
                    $this->_courseid,
                    $groupid, 0, $this->_show_inactive, 'GROUP BY u.id ORDER BY u.firstname, u.lastname');
                return $DB->record_exists_sql($sql, $params);
            };

            $res = false;
            if ($ignore_grouping){
                $res = $f_check_other_stidents(0);
            } else {
                foreach ($allowedgroups as $group){
                    $groupid = $group->id ?? 0;
                    if ($groupid){
                        if ($f_check_other_stidents($group->id)){
                            $res = true;
                            break;
                        }
                    }
                }
            }
            $data[$this->courseid][$is_dm] = $res;
        }

        return $data[$this->courseid][$is_dm];
    }

    /**
     * Return true, if selector has chosen any group(s) other than NONE
     *
     * @return bool
     */
    public function has_group_selected(){
        return $this->_groupid != SH::GROUP_NONE;
    }

    /**
     * Return true, if selector has chosen any group other than NONE or ALL
     *
     * @return bool
     */
    public function has_specific_group_selected(){
        return $this->_groupid > SH::GROUP_ALL;
    }

    /**
     * Is page name look like external one
     *
     * @param $page
     *
     * @return bool
     */
    public static function is_external($page){
        return isset($page[0]) && $page[0] == '/';
    }

    /**
     * Create moodle_url to this (other) page, using class (and merged) page params
     *
     * @param null  $other_page
     * @param array $merged_params
     *
     * @return \moodle_url
     */
    public function get_url($other_page=null, $merged_params=[]){
        $page = $other_page ? $other_page : $this->_page;
        $params = array_merge($this->_pageparams, $merged_params);
        return static::get_page_moodle_url($page, $params);
    }

    /**
     * Return link to page by its name (without additional params)
     *
     * @param $page
     *
     * @return string
     */
    public static function get_page_url($page){
        if (empty($page)){
            return '#';
        }

        if (!isset(static::PAGE_TYPES[$page])){
            return $page;
        }

        if (static::is_external($page)){
            return $page . '.php';
        }

        return NED\PLUGIN_URL . "$page.php";
    }

    /**
     * Create moodle_url by page name and page params
     *
     * @param string            $page
     * @param array|\stdClass   $params
     *
     * @return \moodle_url
     */
    public static function get_page_moodle_url($page, $params=null){
        if (!empty($params)){
            $params = (array)$params;
            foreach ($params as $par_name => $par_value){
                if (empty(static::PARAM_SETTINGS[$par_name])){
                    continue;
                }

                $def = static::PARAM_SETTINGS[$par_name][static::DEF] ?? null;
                if ($par_value == $def){
                    unset($params[$par_name]);
                }
            }
        }
        return new \moodle_url(static::get_page_url($page), $params);
    }

    /**
     * Get html link to page for Menu Bar
     *
     * @param $page
     *
     * @return string
     */
    public function get_menu_link($page){
        $pageparams = (array)$this->_pageparams;
        if (isset($pageparams['id'])) {
        unset($pageparams['id']);
        }
        $add_class = '';
        if ($page == $this->_pagetype){
            $add_class = 'active';
        } else {
            foreach ($pageparams as $par_name => $par_value){
                if ($par_name == static::PAR_COURSE || $par_name == static::PAR_ID){
                    continue;
                }

                if (empty(static::PARAM_SETTINGS[$par_name])){
                    unset($pageparams[$par_name]);
                }
            }
        }
        return static::get_page_link($page, $pageparams, $add_class);
    }

    /**
     * Create html link by page name
     *
     * @param        $page
     * @param array  $pageparams
     * @param string $link_class
     * @param null   $other_text
     * @param array  $link_attr
     *
     * @return string
     */
    public static function get_page_link($page, $pageparams=[], $link_class='', $other_text=null, $link_attr=[]){
        if (!$other_text && isset(static::PAGE_LINKS[$page])){
            $text = SH::$C::str_check(static::PAGE_LINKS[$page], null, SH::str_check(static::PAGE_LINKS[$page]));
        } else {
            $text = $other_text ? $other_text : $page;
        }

        if ($page == static::PAGE_GB && isset($pageparams['courseid'])){
            $pageparams['id'] = $pageparams['courseid'];
            unset($pageparams['courseid']);
        }

        return NED\link(static::get_page_moodle_url($page, $pageparams), $text, $link_class, $link_attr);
    }

    /**
     * Return html group selector
     *
     * @param        $urlroot
     * @param        $groups
     * @param        $active_groupid
     * @param string $name
     * @param null   $formid
     * @param bool   $show_inactive
     * @param bool   $force_render_menu
     *
     * @return string
     */
    public static function group_selector($urlroot, $groups, $active_groupid, $name=self::PAR_GROUP, $formid=null, $show_inactive=false,
        $force_render_menu=false){
        $output = '';
        if (empty($groups)){
            return $output;
        }

        $formid = $formid ? $formid : 'select-' . $name;
        $select_label = NED\img('i/users', '');
        $groupsmenuoptions = [];
        foreach ($groups as $group){
            if (!$show_inactive && empty($group->users)){
                continue;
            }
            $groupsmenuoptions[$group->id] = format_string($group->name).' ('.count($group->users).')';
        }
        if (empty($groupsmenuoptions)){
            return $output;
        }

        if (count($groups) > 1 || $force_render_menu){
            $groupsmenu = [];
            $groupsmenu[NED\GROUP_NONE] = get_string('none');
            $groupsmenu[NED\GROUP_ALL] = NED\str('allclasses');
            $groupsmenu += $groupsmenuoptions;
            $output = SH::single_select($urlroot, $name, $groupsmenu, $active_groupid, $select_label, [], $formid);
        } else {
            $output = $select_label . \html_writer::span(reset($groupsmenuoptions), 'singleselect');
        }
        $output = \html_writer::div($output, 'groupuserselector group-selector');

        return $output;
    }

    /**
     * Return html user selector
     *
     * @param        $urlroot
     * @param        $students
     * @param        $active_studentid
     * @param bool   $lock_user
     * @param string $name
     * @param null   $formid
     *
     * @return string
     */
    public static function student_selector($urlroot, $students, $active_studentid, $lock_user=null, $name=self::PAR_SETUSER, $formid=null){
        $output = '';

        if (empty($students)){
            return $output;
        }

        $formid = $formid ? $formid : 'select-' . $name;
        $select_label = NED\img('i/user', '');
        $div_class = ['groupuserselector', 'student-selector'];

        if (count($students) > 1){
            $studentopts = [0 => NED\str('allparticipants')];
            foreach ($students as $student) {
                $studentopts[$student->id] = fullname($student);
            }

            $url = new \moodle_url($urlroot);
            $url->remove_params(static::PAR_USER);
            $output = SH::single_select($url, $name, $studentopts, $active_studentid, $select_label, [], $formid);

            if ($active_studentid > 0 && !is_null($lock_user)){
                $output .= static::lock_user_checkbox($urlroot, $lock_user);
                $div_class[] = $lock_user ? 'locked' : 'unlocked';
            }
        } else {
            $output = $select_label . \html_writer::span(fullname(reset($students)), 'selectuser');
        }

        $output = \html_writer::div($output, join(' ', $div_class));

        return $output;
    }

    /**
     * @param \context_course | \context_module | \context $context
     *
     * @return array
     */
    public function get_tag_options($context){
        static $results = [];
        if (!isset($results[$context->id])){
            $is_MM = $this->_page == static::PAGE_MM;
            $hide_cta = $is_MM && \block_ned_teacher_tools\cta_tracker::is_tracker_on($this->_courseid);
            $contexts = $context->get_child_contexts();
            $contexts[] = $context;
            $tag_options = [static::TAGS_ALL => SH::$C::str(static::TAGS_ALL), static::TAGS_NONE => SH::$C::str(static::TAGS_NONE)];
            $tags = \core_tag_tag::get_tags_by_area_in_contexts('core', 'course_modules', $contexts);
            foreach ($tags as $tag){
                if ($hide_cta && $tag->rawname == SH::TAG_CTA) continue;

                $tag_options[$tag->id] = $tag->get_display_name(true);
            }
            $results[$context->id] = $tag_options;
        }

        return $results[$context->id];
    }

    /**
     * @param string $label
     * @param bool   $div
     * @param string $div_class
     * @param string $surround
     *
     * @return \single_select|string
     */
    public function render_tag_selector($label='', $div=false, $div_class='', $surround=''){
        $tag_options = $this->_tag_options;
        if ($surround && $tag_options) {
            foreach ($tag_options as $index => $item) {
                $tag_options[$index] = $surround . $item . $surround;
            }
        }
        $selector = SH::single_select($this->get_url(), static::PAR_TAG, $tag_options, $this->_tag_id, $label);
        if ($div){
            return \html_writer::div($selector, $div_class);
        } else {
            return $selector;
        }
    }

    /**
     * Return html sort selector
     *
     * @param \moodle_url $url
     * @param string      $chosen_sort
     *
     * @return string
     */
    public static function sort_selector(\moodle_url $url, $chosen_sort=self::SORT_GRADE){
        $url = new \moodle_url($url);
        $url->remove_params(static::PAR_SORT);

        $sort_options = [];
        foreach (static::SELECT_SORT as $sort){
            $sort_options[$sort] = NED\str($sort);
        }
        $select = SH::single_select($url, static::PAR_SORT, $sort_options, $chosen_sort, NED\str('choosesort'), [],'selectsort');
        $output = \html_writer::div($select, 'groupselector sort-selector');

        return $output;
    }

    /**
     * Return html show_inactive checkbox
     *
     * @param \moodle_url $url
     * @param             $value - if null, return empty string
     * @param string      $name
     *
     * @return string
     */
    public static function show_inactive_checkbox(\moodle_url $url, $value, $name=self::PAR_SHOWINACTIVE){
        if (is_null($value)){
            return '';
        }
        $icon = \html_writer::div(NED\fa('fa-user') . NED\fa('fa-folder'), 'tt-icon-user-folder');

        return SH::single_checkbox2($url, $value, $name, $icon, $icon, 'show-inactive-checkbox tt-box-group',
            NED\str('hideinactiveusers'), NED\str('showinactiveusers'));
    }

    /**
     * Return html show_cmstudents checkbox
     *
     * @param \moodle_url $url
     * @param             $value - if null, return empty string
     * @param string      $name
     *
     * @return string
     */
    public static function show_cmstudents_checkbox(\moodle_url $url, $value, $name=self::PAR_CMSTUDENTS){
        if (is_null($value) || !SH::has_capability('viewcttoggle')){
            return '';
        }

        $icon = SH::div('', 'tt-icon-cmstudents');
        return SH::single_checkbox2($url, $value, $name, $icon, $icon, 'show-cmstudents-checkbox',
            SH::str('hideclassroomteachers'), SH::str('showclassroomteachers'));
    }

    /**
     * Return html lock_user checkbox
     *
     * @param \moodle_url $url
     * @param             $value - if null, return empty string
     * @param string      $name
     *
     * @return string
     */
    public static function lock_user_checkbox(\moodle_url $url, $value, $name=self::PAR_LOCKUSER){
        if (is_null($value)){
            return '';
        }

        return SH::single_checkbox2($url, $value, $name, NED\fa('fa-lock'), NED\fa('fa-unlock-alt'), 'lock-user-checkbox',
            NED\str('unlockuser'), NED\str('lockuser'));
    }

    /**
     * Return params (from function name), checked cash, where need it
     *  it caches results, if data is not $for_deadline_manager
     *
     * @param \stdClass $course
     * @param null      $show_inactive
     * @param null      $groupid
     * @param null      $userid
     * @param bool      $for_deadline_manager
     * @param null      $setuserid
     * @param null      $load_cm_students
     *
     * @return array
     */
    public static function get_showinactive_groups_group_groupid_allstudents_students_user($course=null, $show_inactive=null, $groupid=null,
        $userid=null, $for_deadline_manager=false, $setuserid=null, $load_cm_students=null){
        static $load_data = [];
        $courseid = $course->id;

        if (!isset($load_data[$courseid][$load_cm_students][$for_deadline_manager])){
            $show_inactive = NED\check_show_inactive_with_cache($show_inactive, $courseid);
            [$allstudents, $groups] = NED\get_users_and_groups($course, $for_deadline_manager, !$show_inactive, $load_cm_students);
            $groupid = NED\check_group_with_cache($groupid, $courseid, $groups);

            $load_data[$courseid][$load_cm_students][$for_deadline_manager] = [$show_inactive, $groups, $groupid, $allstudents];
        } else {
            [$show_inactive, $groups, $groupid, $allstudents] = $load_data[$courseid][$load_cm_students][$for_deadline_manager];
        }

        if ($setuserid && !isset($allstudents[$userid])){
            $user = \core_user::get_user($userid);
            if ($user){
                $user_groups = SH::get_all_user_course_groups($courseid, $user->id);
                $user->group = reset($user_groups);
                $allstudents[$userid] = $user;
            }
        }

        $group = null;
        $groups = $groups ?: [];
        if (count($groups) == 1 && !$for_deadline_manager){
            $group = reset($groups);
            $groupid = $group->id;
            $enrolledusers = $group->users;
        } elseif ($groupid == NED\GROUP_NONE){
            $enrolledusers = [];
        } elseif($groupid == NED\GROUP_ALL){
            $enrolledusers = $allstudents;
        } else {
            $group = isset($groups[$groupid]) ? $groups[$groupid] : null;
            $enrolledusers = $group ? $group->users : [];
        }

        $enrolledusers = $enrolledusers ?: [];
        if (count($enrolledusers) == 1){
            $user = reset($enrolledusers);
            $userid = $user->id;
        } elseif (isset($enrolledusers[$userid])){
            $user = $enrolledusers[$userid];
        } elseif ($setuserid && isset($allstudents[$userid])){
            $user = $allstudents[$userid];
        } else {
            $user = null;
            $userid = 0;
        }

        if ($setuserid && $user){
            // set group of user as chosen group
            $set_groupid = $user->group->id ?? NED\GROUP_ALL;
            if ($set_groupid != $groupid){
                if (!empty($groups[$set_groupid]->users)){
                    $groupid = $set_groupid;
                    $group = $groups[$groupid];
                    $enrolledusers = $group ? $group->users : [];
                } else {
                    $groupid = NED\GROUP_ALL;
                    $enrolledusers = $allstudents;
                }
            }
        }

        return [$show_inactive, $groups, $group, $groupid, $allstudents, $enrolledusers, $user, $userid];
    }

    /**
     * Update params in moodle cash
     *
     * @param $courseid
     * @param $show_inactive
     * @param $groupid
     * @param $cm_students
     */
    public static function update_cach($courseid, $show_inactive=null, $groupid=null, $cm_students=null){
        global $SESSION;
        NED\set_cache_group($courseid, $groupid);
        NED\set_cache_show_inactive($courseid, $show_inactive);
        NED\set_cache_load_cm_students($courseid, $cm_students);
        $SESSION->currentgroup[$courseid] = $groupid;
    }

    /**
     * @param $path
     *
     * @return mixed|string
     */
    public static function get_page_path($path){
        global $SCRIPT;
        $path = empty($page) ? $SCRIPT : $path;
        if (strpos($path, NED\PLUGIN_URL) !== FALSE){
            $data = explode('/', $path);
        } else {
            $data = explode('//', $path, 1);
        }
        $fname = end($data);
        $data = explode('.', $fname);

        return reset($data);
    }

    /**
     * Get page name for prevpage from page path
     * @param $path
     *
     * @return string
     */
    public static function path2page($path){
        return empty($path) ? '-' : explode('.', basename($path))[0];
    }

    /**
     * @param $page1
     * @param $page2
     *
     * @return bool
     */
    public static function is_same_page($page1, $page2){
        if ($page1 == $page2){
            return true;
        }

        if (empty($page1) || empty($page2)){
            return false;
        }

        return strpos($page1, $page2) !== false || strpos($page2, $page1) !== false;
    }

    /**
     * Renders menu_bar and returns the HTML to display it.
     *
     * @param bool $return - echo rendered object, if false
     *
     * @return string
     */
    public function render($return=false){
        return static::render_sth($this, $return);
    }

    /**
     * Renders the provided widget through ned_teacher_tools render and returns the HTML to display it.
     *
     * @param \renderable $widget instance with renderable interface
     * @param bool $return - echo rendered object, if false
     *
     * @return string
     */
    public static function render_sth($widget, $return=false){
        $out = static::$_renderer->render($widget);

        if (!$return){
            echo $out;
        }
        return $out;
    }

    /**
     * Return last example of menu_bar
     *
     * @param string $page
     * @param array  $pageparams
     * @param null   $courseid                  - if null, loads form page params as self::PAR_COURSE
     * @param bool   $ignore_teacher_capability - set to true, if want MB works with any capabilities
     *
     * @return static|null
     */
    public static function get_menu_bar($page='', $pageparams=[], $courseid=null, $ignore_teacher_capability=true){
        if (!static::$_last_this){
            return new static($page, $pageparams, $courseid, $ignore_teacher_capability);
        }

        return static::$_last_this;
    }

    /**
     * Return last example of menu_bar
     *
     * @param        $courseid
     * @param string $page
     * @param array  $pageparams
     * @param bool   $ignore_teacher_capability - set to true, if want MB works with any capabilities
     *
     * @return static|null
     */
    public static function get_menu_bar_by_courseid($courseid, $page='', $pageparams=[], $ignore_teacher_capability=true){
        return static::get_menu_bar($page, $pageparams, $courseid, $ignore_teacher_capability);
    }
}
