<?php
/**
 * @package    block_ned_teacher_tools
 * @subpackage support
 * @category   NED
 * @copyright  2022 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace block_ned_teacher_tools\support;

use block_ned_teacher_tools\grading_tracker;
use block_ned_teacher_tools\shared_lib as SH;

SH::require_file('/mod/quiz/attemptlib.php');

defined('MOODLE_INTERNAL') || die();

/**
 * Class gt_attempt_data
 * Designed to represent previous GT attempts
 * Also has some methods for loading relevant data
 *
 * @package block_ned_teacher_tools\support
 */
class gt_attempt_data {
    use \local_ned_controller\base_empty_class;

    public $courseid;
    public $cmid;
    public $attempt;
    public $submit_time;
    public $userid;
    /**
     * Can be null, even if $timegrade is present, when grader is unknown
     * @var int|null
     */
    public $graderid;
    public $timegrade;
    public $is_attempt_data;

    /**
     * gt_attempt_data constructor.
     *
     * @param object $record
     * @param string $prefix
     */
    public function __construct($record, $prefix=''){
        $this->import($record, true, null, $prefix);
        $this->is_attempt_data = true;
    }

    /**
     * Return SQL filters ($where, $params, $user_join_condition) for other methods with DB calls
     *
     * @param numeric|null       $courseid
     * @param numeric|array|null $cmids
     * @param numeric|array|null $userids
     *
     * @return array - [$where, $params, $user_join_condition]
     */
    static protected function _create_sql_filters($courseid=null, $cmids=null, $userids=null){
        $where = [];
        $params = [];

        if ($courseid){
            SH::sql_add_equal('cm.course', $courseid, $where, $params);
        }

        $cmids = SH::val2arr($cmids);
        if (!empty($cmids)){
            SH::sql_add_get_in_or_equal_options('cm.id', $cmids, $where, $params);
        }

        $userids = SH::val2arr($userids);
        if (!empty($userids)){
            $users_join_by = [];
            SH::sql_add_get_in_or_equal_options('u.id', $userids, $users_join_by, $params);
            $user_join_condition = SH::sql_condition($users_join_by);
        } else {
            $user_join_condition = '1 = 1';
        }

        return [$where, $params, $user_join_condition];
    }

    /**
     * Return graded, previous (non-latest) attempts for assigns and quizzes
     *
     * @param numeric|null       $courseid
     * @param numeric|array|null $cmids
     * @param numeric|array|null $userids
     * @param numeric|null       $attempt
     * @param null|bool          $gt_exist   - if null, do not check GT table, true/false - return only if records exists/not-exists in GT
     * @param numeric            $start_date - UNIX time, if provided, record stats (submit & grade time) should be after this date
     * @param bool               $only_one   - if true, return single object/record, otherwise return array
     *
     * @return array|static[]|static|null - single object or array(courseid => [userid => [cmid => [attempt => $gt_attempt_data]]])
     */
    static public function get_attempt_data($courseid=null, $cmids=null, $userids=null,
        $gt_exist=null, $start_date=null, $attempt=null, $only_one=false){
        if (empty($courseid) && empty($cmids)){
            SH::debugging('gt_attempt_data::get_attempt_data: You need to provide course or course-module ID');
            return $only_one ? null : [];
        }

        // assign attempts starts from 0, quiz attempts starts from 1 => gt attempts starts from 0
        $sql_attempt = 'COALESCE(a_g.attemptnumber, q_a.attempt-1, 0)';
        $sql_timegrade = 'COALESCE(a_g.timemodified, q_a.timemodified, 0)';

        /** @var $sql_assign_submit_time - {@see \local_ned_controller\marking_manager\marking_manager_assign::SQL_SUBMIT_TIME} */
        $sql_assign_submit_time = "IF(a_s.status = 'submitted', a_s.timemodified, NULL)";
        /** @var $sql_quiz_submit_time - {@see \local_ned_controller\marking_manager\marking_manager_quiz::SQL_SUBMIT_TIME} */
        $sql_quiz_submit_time = "IF(q_a.state = '".\quiz_attempt::FINISHED."', q_a.timefinish, NULL)";
        $sql_submit_time = "COALESCE($sql_assign_submit_time, $sql_quiz_submit_time, 0)";

        [$where, $params, $user_join_condition] = static::_create_sql_filters($courseid, $cmids, $userids);
        if (!is_null($attempt)){
            SH::sql_add_equal($sql_attempt, $attempt, $where, $params, 'attempt');
        }

        if ($start_date){
            $where[] = "GREATEST($sql_submit_time, $sql_timegrade) >= :start_date";
            $params['start_date'] = $start_date;
        }

        $select = [
            "CONCAT(cm.id, '_', u.id, '_', $sql_attempt) AS uniqid",
            "cm.course AS courseid",
            "u.id AS userid",
            "cm.id AS cmid",
            "$sql_attempt AS attempt",
            "$sql_submit_time AS submit_time",
            "$sql_timegrade AS timegrade",
            "IF(a_g.grader < 0, 0, a_g.grader) AS graderid",
        ];
        $joins = ["
            JOIN {modules} m
                ON m.id = cm.module
            JOIN {user} u
                ON $user_join_condition
                
            LEFT JOIN {assign_grades} a_g
                ON m.name = :assign
                AND a_g.assignment = cm.instance
                AND a_g.userid = u.id
                AND COALESCE(a_g.grade, -1) <> -1
            LEFT JOIN {assign_submission} a_s
                ON a_s.assignment = a_g.assignment
                AND a_s.userid = a_g.userid
                AND a_s.attemptnumber = a_g.attemptnumber
                AND a_s.latest = 0
                
            LEFT JOIN {quiz_attempts} q_a
                ON m.name = :quiz
                AND q_a.quiz = cm.instance   
                AND q_a.userid = u.id
                AND q_a.sumgrades IS NOT NULL
            LEFT JOIN (
                SELECT quiz, userid, MAX(attempt) AS max_attempt 
                FROM {quiz_attempts}  
                GROUP BY quiz, userid
            ) q_a_max 
                ON q_a_max.quiz = q_a.quiz
                AND q_a_max.userid = q_a.userid
                AND q_a_max.max_attempt = q_a.attempt
        "];

        $where[] = "COALESCE(a_s.id, q_a.id) IS NOT NULL";
        $where[] = "q_a_max.max_attempt IS NULL";
        $params['assign'] = SH::MOD_ASSIGN;
        $params['quiz'] = SH::MOD_QUIZ;

        if (!is_null($gt_exist)){
            $joins[] = "
                LEFT JOIN {".grading_tracker::TABLE."} gt
                    ON gt.cmid = cm.id
                    AND gt.userid = u.id
                    AND gt.attempt = $sql_attempt
            ";
            $where[] = 'gt.id IS '. ($gt_exist ? 'NOT NULL' : 'NULL');
        }

        $sql = SH::sql_generate($select, $joins, 'course_modules', 'cm', $where);
        $records = SH::db()->get_records_sql($sql, $params);
        if (empty($records)) return $only_one ? null : [];

        if ($only_one){
            return new static(reset($records));
        }

        $objects = [];
        foreach ($records as $record){
            $obj = new static($record);
            $objects[$obj->courseid][$obj->userid][$obj->cmid][$obj->attempt] = $obj;
        }

        return $objects;
    }

    /**
     * Return max attempts per user/activity for assigns and quizzes
     * If return an array, objects in it will have keys: courseid, userid, cmid, max_attempt
     *
     * @param numeric|null       $courseid
     * @param numeric|array|null $cmids
     * @param numeric|array|null $userids
     * @param bool               $single_result - if true, return single int value (max attempt) or null, otherwise return array
     *
     * @return array|object[]|int|null - single int value or array with records (and "max_attempt" key in them)
     */
    static public function get_max_attempts($courseid=null, $cmids=null, $userids=null, $single_result=false){
        if (empty($courseid) && empty($cmids)){
            SH::debugging('gt_attempt_data::get_max_attempts: You need to provide course or course-module ID');
            return $single_result ? null : [];
        }

        [$where, $params, $user_join_condition] = static::_create_sql_filters($courseid, $cmids, $userids);

        $select = [
            "CONCAT(cm.id, '_', u.id) AS uniqid",
            "cm.course AS courseid",
            "u.id AS userid",
            "cm.id AS cmid",
        ];
        // assign attempts starts from 0, quiz attempts starts from 1 => gt attempts starts from 0
        $select[] = "COALESCE(a_s.attemptnumber, q_a.attempt-1) AS max_attempt";

        $joins = ["
            JOIN {modules} m
                ON m.id = cm.module
            JOIN {user} u
                ON $user_join_condition
                
            LEFT JOIN {assign_submission} a_s
                ON m.name = :assign
                AND a_s.assignment = cm.instance
                AND a_s.userid = u.id
                AND a_s.latest = 1
             
            LEFT JOIN {quiz_attempts} q_a
                ON m.name = :quiz
                AND q_a.quiz = cm.instance  
                AND q_a.userid = u.id 
            LEFT JOIN {quiz_attempts} q_a_fake
                ON q_a_fake.quiz = q_a.quiz   
                AND q_a_fake.userid = q_a.userid
                AND q_a_fake.attempt > q_a.attempt
        "];

        $where[] = "COALESCE(a_s.id, q_a.id) IS NOT NULL";
        $where[] = "q_a_fake.id IS NULL";
        $params['assign'] = SH::MOD_ASSIGN;
        $params['quiz'] = SH::MOD_QUIZ;

        $sql = SH::sql_generate($select, $joins, 'course_modules', 'cm', $where);
        $records = SH::db()->get_records_sql($sql, $params);

        if (empty($records)) return $single_result ? null : [];
        if ($single_result) return reset($records)->max_attempt ?? null;
        return $records;
    }
}
