<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * UTILS
 *
 * @package block_ned_teacher_tools
 * @copyright 2019 Michael Gardener
 * @license http://www.gnu.org/copyleft/gpl.html GNU Public License
 */

namespace block_ned_teacher_tools;

use block_ned_teacher_tools\shared_lib as SH;
use html_writer;

defined('MOODLE_INTERNAL') || die();

/** @var \stdClass $CFG */
require_once($CFG->dirroot.'/blocks/ned_teacher_tools/lib.php');

class utils {
    public static function kica_gradebook_enabled($courseid=0) {
        if ($courseid){
            $kica = get_kica_if_exists($courseid);
            return $kica->enabled ?? false;
        } else {
            return is_kica_exists();
        }
    }

    /**
     * @param $inputseconds
     * @return array
     */
    public static function secondstotime($inputseconds) {

        $secondsInAMinute = 60;
        $secondsInAnHour  = 60 * $secondsInAMinute;
        $secondsInADay    = 24 * $secondsInAnHour;

        // extract days
        $days = floor($inputseconds / $secondsInADay);

        // extract hours
        $hourSeconds = $inputseconds % $secondsInADay;
        $hours = floor($hourSeconds / $secondsInAnHour);

        // extract minutes
        $minuteSeconds = $hourSeconds % $secondsInAnHour;
        $minutes = floor($minuteSeconds / $secondsInAMinute);

        // extract the remaining seconds
        $remainingSeconds = $minuteSeconds % $secondsInAMinute;
        $seconds = ceil($remainingSeconds);

        // return the final array
        $obj = array(
            'd' => (int) $days,
            'h' => (int) $hours,
            'm' => (int) $minutes,
            's' => (int) $seconds,
        );
        return $obj;
    }

    public static function get_user_course_grade_percentage($userid, $courseid) {
        global $DB;

        $sql = "SELECT ROUND((100 * gg.finalgrade / gg.rawgrademax), 0) courseaverage 
                  FROM {grade_grades} gg
                  JOIN {grade_items} gi 
                    ON gi.id = gg.itemid
                 WHERE gi.courseid = ? 
                   AND gg.userid = ? 
                   AND gi.itemtype = 'course'";

        return $DB->get_field_sql($sql, array($courseid, $userid));
    }

    public static function deadline_manager_groups_print_select($course, $urlroot, $return=false, $activegroup=false, $groups=false,
        $for_deadline_manager=true, $show_inactive=false) {
        global $USER, $OUTPUT;

        if (!$groupmode = $course->groupmode) {
            if ($return) {
                return '';
            } else {
                return;
            }
        }

        $context = \context_course::instance($course->id);
        $aag = $for_deadline_manager ? has_capability('block/ned_teacher_tools:view_all_groups', $context) :
            has_capability('moodle/site:accessallgroups', $context);

        $usergroups = array();
        $groupsmenu = array();

        if ($groups === false) {
            list($users, $allowedgroups) = block_ned_teacher_tools_get_users_and_groups($course, true, !$show_inactive);
        } else {
            $allowedgroups = $groups;
        }

        if ($groupmode == VISIBLEGROUPS or $aag) {
            $usergroups = groups_get_all_groups($course->id, $USER->id, $course->defaultgroupingid);
        }

        if ($activegroup === false) {
            $activegroup = groups_get_course_group($course, true, $allowedgroups);
        }

        $groupsmenuoptions = groups_sort_menu_options($allowedgroups, $usergroups);
        $numberofgroups = count($groupsmenuoptions);

        if (!empty($groupsmenuoptions)) {
            if ($numberofgroups > 1) {
                if (!$allowedgroups or $groupmode == VISIBLEGROUPS or $groupmode == SEPARATEGROUPS or $aag) {
                    $groupsmenu[0] = get_string('allgroups', 'block_ned_teacher_tools');
                }

                $groupsmenu += groups_sort_menu_options($allowedgroups, $usergroups);

                if ($groupmode == VISIBLEGROUPS) {
                    $grouplabel = get_string('groupsvisible');
                } else {
                    $grouplabel = get_string('groupsseparate');
                }

                if ($aag and $course->defaultgroupingid) {
                    if ($grouping = groups_get_grouping($course->defaultgroupingid)) {
                        $grouplabel = $grouplabel . ' (' . format_string($grouping->name) . ')';
                    }
                }

                $groupsmenunew = [GROUP_NONE => get_string('none')];
                $stroverriden = get_string('overriden', 'block_ned_teacher_tools');
                $strnoyoverriden = get_string('notoverriden', 'block_ned_teacher_tools');

                foreach ($groupsmenu as $index => $groupsmenu_item) {
                    if (!$index) {
                        $groupsmenunew[$index] = $groupsmenu_item;
                    } else {
                        if (\block_ned_teacher_tools\deadline_manager::is_overridden_group($index)) {
                            $groupsmenunew['overriden'][$stroverriden][$index] = $groupsmenu_item;
                        } else {
                            $groupsmenunew['notoverriden'][$strnoyoverriden][$index] = $groupsmenu_item;
                        }
                    }
                }

                $select = new \single_select(new \moodle_url($urlroot), 'group', $groupsmenunew, $activegroup, null, 'selectgroup');
                $select->label = \html_writer::img($OUTPUT->image_url('i/users'), '');
                $output = $OUTPUT->render($select);
            } else {
                $output = \html_writer::span(\html_writer::img($OUTPUT->image_url('i/users'), '') . ' '. reset($groupsmenuoptions), 'single-group-container mr-3');
            }
        } else {
            $output = '';
        }

        $output = '<div class="groupselector">'.$output.'</div>';

        if ($return) {
            return $output;
        } else {
            echo $output;
        }
    }

    /**
     * @param $fromform
     * @return array
     */
    public static function get_activity_cmids_from_form($fromform, $name = 'activity') {
        $cmids = [];
        foreach ($fromform as $key => $value) {
            if (strpos($key, $name) === 0) {
                if ((int)$value > 0) {
                    $cmids[$value] = $value;
                }
            }
        }
        return $cmids;
    }

    /**
     * @param int        $totalcount
     * @param \moodle_url $baseurl
     * @param string     $pagevar
     * @param bool       $return
     *
     * @return string
     */
    public static function perpage_bar($totalcount, $baseurl, $pagevar = 'page', $return=false) {

        if (!isset($totalcount) || is_null($totalcount)) {
            throw new \coding_exception('perpage_bar requires a totalcount value.');
        }
        if (empty($baseurl)) {
            throw new \coding_exception('perpage_bar requires a baseurl value.');
        }

        $perpages = array();

        if ($totalcount > 200) {
            $perpages = array(20, 50, 100, 200, 500);
        } else if ($totalcount > 100) {
            $perpages = array(20, 50, 100, 200);
        } else if ($totalcount > 50) {
            $perpages = array(20, 50, 100);
        }

        $currentperpage = $baseurl->get_param('perpage');

        if (!empty($perpages)) {

            $perpagelink = get_string('show') . ': ';

            $links = [];

            foreach ($perpages as $perpage) {
                if ($perpage == $currentperpage) {
                    $links[] = \html_writer::span($perpage, 'current-perpage');
                } else {
                    $links[] = \html_writer::link(new \moodle_url($baseurl, array($pagevar => 0, 'perpage' => $perpage)), $perpage);
                }
            }
            if (isset($perpage) && $perpage == $totalcount) {
                $links[] = \html_writer::span(get_string('all'), 'current-perpage');
            } else {
                $links[] = \html_writer::link(new \moodle_url($baseurl, array($pagevar => 0, 'perpage' => $totalcount)), get_string('all'));
            }


            $perpagelink .= implode('&nbsp;|&nbsp;', $links);

            if ($return){
                return \html_writer::div($perpagelink, 'perpagebar');
            } else {
                echo \html_writer::div($perpagelink, 'perpagebar');
            }
        }
        return '';
    }

    /**
     * @return array
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function get_assignable_course_roles() {
        global $DB;
        $roles = get_roles_for_contextlevels(CONTEXT_COURSE);
        list($insql, $params) = $DB->get_in_or_equal($roles);
        return $DB->get_records_select_menu('role', "id {$insql}", $params, 'shortname', 'id,shortname');
    }

    /**
     * @return false|string[]|void
     * @throws \dml_exception
     */
    public static function get_school_years() {
        if ($currentschoolyears = get_config('block_ned_teacher_tools','currentschoolyears')) {
            return explode('-', $currentschoolyears);
        }
        return false;
    }

    /**
     * @param $cm
     * @param $studentid
     * @param $type
     * @return array|null
     * @throws \dml_exception
     */
    public static function render_copyleaks_score($cm, $studentid, $type, $checkeligibility = false, $history = false) {
        global $DB;
        $none = ['-', []];

        if (!isset($DB->get_tables(true)['plagiarism_copyleaks_config'])) return $none;

        if ($checkeligibility) {
            if ($cm->modname != 'assign' || !$DB->record_exists('plagiarism_copyleaks_config', ['cm' => $cm->id, 'name' => 'plagiarism_copyleaks_enable', 'value' => 1])) {
                return $none;
            }

            $tags = SH::cm_get_tags($cm);

            if (!$tags || !in_array(SH::TAG_SUMMATIVE, $tags)) {
                return $none;
            }
        }

        if ($cm->modname == 'quiz' || $history) {
            $limitnum = 0;
        } else {
            $limitnum = 1;
        }

        $submittedfiles = $DB->get_records('plagiarism_copyleaks_files', [
            'cm' => $cm->id, 'userid' => $studentid, 'statuscode' => 'success'
        ], 'lastmodified DESC', '*', 0, $limitnum);

        if (empty($submittedfiles)) {
            return $none;
        }
        $plagiarismdetails = [];
        $scores = [];

        foreach ($submittedfiles as $submittedfile) {
            // Plagiarism Score level class.
            $scores[] = $submittedfile->{$type};
            $scorelevelclass = '';
            if ($submittedfile->{$type} <= 40) {
                $scorelevelclass = "cls-plag-score-level-low";
            } else if ($submittedfile->{$type} <= 80) {
                $scorelevelclass = "cls-plag-score-level-mid";
            } else {
                $scorelevelclass = "cls-plag-score-level-high";
            }
            if ($cm->modname == 'quiz') {
                $url = new \moodle_url('/mod/quiz/review.php', [
                    'attempt' => $submittedfile->itemid,
                ]);
            } else if ($cm->modname == 'assign') {
                $url = new \moodle_url('/mod/assign/view.php', [
                    'id' => $cm->id,
                    'course' => $cm->course,
                    'action' => 'grader',
                    'userid' => $studentid
                ]);
            }
            $plagiarismdetails[] = html_writer::tag(
                'div',
                html_writer::tag(
                    'div',
                    (isset($submittedfile->{$type}) ?
                        html_writer::tag('span', '', ['class' => "score-level $scorelevelclass"]) .
                        html_writer::link($url ?? '#', $submittedfile->{$type}, ['target' => '_blank']) : '-'),
                    array('class' => 'cls-score-container')
                ),
                array('class' => 'cls-result-item')
            );
        }

        return [html_writer::tag('div',implode(' ', $plagiarismdetails),['class' => 'cls-details-content']), $scores];
    }

    public static function raps_row_filter($range, $instances, $numofaccurance, $minscore, $gfilterpassed, &$similarityscore, &$displayrow, &$occurance) {
        switch ($range) {
            case 0:
                if ($gfilterpassed) {
                    $displayrow = true;
                    $occurance++;
                } else {
                    $similarityscore = '';
                }
                break;
            case 1:
                if ($minscore >= 40 && $gfilterpassed) {
                    $displayrow = true;
                    $occurance++;
                } else {
                    $similarityscore = '';
                }
                break;
            case 2:
                if ($minscore >= 60 && $gfilterpassed) {
                    $displayrow = true;
                    $occurance++;
                } else {
                    $similarityscore = '';
                }
                break;
            case 3:
                if ($minscore >= 80 && $gfilterpassed) {
                    $displayrow = true;
                    $occurance++;
                } else {
                    $similarityscore = '';
                    $occurance++;
                }
                break;
            case 4:
                if ($minscore >= 90 && $gfilterpassed) {
                    $displayrow = true;
                    $occurance++;
                } else {
                    $similarityscore = '';
                }
                break;
            case 5:
                if ($minscore == 100 && $gfilterpassed) {
                    $displayrow = true;
                    $occurance++;
                } else {
                    $similarityscore = '';
                }
                break;
        }

        if ($instances > 0 && $numofaccurance > $occurance) {
            $displayrow = false;
        }
    }
}