<?php
/**
 * @package    block_ned_teacher_tools
 * @subpackage NED
 * @copyright  2020 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

use block_ned_teacher_tools\shared_lib as SH;
use local_ned_controller\tt_config_manager as CM;
use block_ned_teacher_tools\grading_tracker as GT;

/**
 * All upgrades checks from Moodle 3
 *
 * @param int $oldversion
 */
function block_ned_teacher_tools_moodle3_upgrades($oldversion): void{
    global $DB, $CFG;
    $debug = function($text, $add_line=false, $show_time=true){
        if ($show_time){
            $t = date("H:i:s");
            $text = "[$t] $text";
        }
        $class = 'text-monospace';
        if ($add_line){
            $class .= ' m-b-1';
        }
        echo html_writer::div($text, $class);
    };

    $dbman = $DB->get_manager();
    $GT_table = \block_ned_teacher_tools\grading_tracker::TABLE;

    if ($oldversion < 2019082802) {

        $table = new xmldb_table('groups');
        $field = new xmldb_field('schedule', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, 0);

        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        // Main savepoint reached.
        upgrade_block_savepoint(true, 2019082802, 'ned_teacher_tools');
    }

    if ($oldversion < 2019101302) {
        $systemcontext = context_system::instance();

        update_capabilities('block_ned_teacher_tools');

        // Online Teacher [OT].
        if ($onlineteacherrole = $DB->get_record( 'role', array( 'shortname' => 'online-teacher') )) {
            $onlineteacherrolecaps = array(
                'block/ned_teacher_tools:view_all_groups',
                'block/ned_teacher_tools:manage_group_requirements',
                'block/ned_teacher_tools:manage_group_deadlines',
                'block/ned_teacher_tools:manage_user_requirements',
                'block/ned_teacher_tools:manage_user_deadlines',
                'block/ned_teacher_tools:add_extension',
                'block/ned_teacher_tools:view_extension_detail',
                'block/ned_teacher_tools:manage_extensions'
            );
            foreach ($onlineteacherrolecaps as $cap) {
                assign_capability($cap, CAP_ALLOW, $onlineteacherrole->id, $systemcontext->id);
            }
        }

        // Classroom Teacher [CT].
        if ($classroomteacherrole = $DB->get_record( 'role', array( 'shortname' => 'classroom-teacher') )) {
            $classroomteacherrolecaps = array(
                'block/ned_teacher_tools:view_own_groups',
                'block/ned_teacher_tools:manage_group_deadlines',
                'block/ned_teacher_tools:add_extension',
                'block/ned_teacher_tools:view_extension_detail'
            );
            foreach ($classroomteacherrolecaps as $cap) {
                assign_capability($cap, CAP_ALLOW, $classroomteacherrole->id, $systemcontext->id);
            }
        }

        // School Admin [SA].
        if ($rsarole = $DB->get_record( 'role', array( 'shortname' => 'r-sa') )) {
            $rsarolecaps = array(
                'block/ned_teacher_tools:view_own_groups',
                'block/ned_teacher_tools:view_extension_detail'
            );
            foreach ($rsarolecaps as $cap) {
                assign_capability($cap, CAP_ALLOW, $rsarole->id, $systemcontext->id);
            }
        }

        // Define table block_ned_teacher_tools_exte to be created.
        $table = new xmldb_table('block_ned_teacher_tools_exte');

        // Adding fields to table block_ned_teacher_tools_exte.
        $table->add_field('id', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, XMLDB_SEQUENCE, null);
        $table->add_field('courseid', XMLDB_TYPE_INTEGER, '10', null, null, null, null);
        $table->add_field('cmid', XMLDB_TYPE_INTEGER, '10', null, null, null, '0');
        $table->add_field('modname', XMLDB_TYPE_CHAR, '30', null, null, null, null);
        $table->add_field('instance', XMLDB_TYPE_INTEGER, '10', null, null, null, '0');
        $table->add_field('overridenby', XMLDB_TYPE_INTEGER, '10', null, null, null, '0');
        $table->add_field('userid', XMLDB_TYPE_INTEGER, '10', null, null, null, '0');
        $table->add_field('duedate', XMLDB_TYPE_INTEGER, '10', null, null, null, '0');
        $table->add_field('reason', XMLDB_TYPE_TEXT, null, null, null, null, null);
        $table->add_field('timecreated', XMLDB_TYPE_INTEGER, '10', null, null, null, '0');

        // Adding keys to table block_ned_teacher_tools_exte.
        $table->add_key('id', XMLDB_KEY_PRIMARY, ['id']);

        // Conditionally launch create table for block_ned_teacher_tools_exte.
        if (!$dbman->table_exists($table)) {
            $dbman->create_table($table);
        }

        // Ned_teacher_tools savepoint reached.
        upgrade_block_savepoint(true, 2019101302, 'ned_teacher_tools');
    }

    if ($oldversion < 2019101700) {

        if ($settings = $DB->get_records('config_plugins', ['plugin' => 'block_net_teacher_tools'])) {
            foreach ($settings as $setting) {
                $name = str_replace('dm_activities', 'dm_extensions', $setting->name);
                set_config($name, $setting->value, $setting->plugin);
            }
        }

        // Ned_teacher_tools savepoint reached.
        upgrade_block_savepoint(true, 2019101700, 'ned_teacher_tools');
    }

    if ($oldversion < 2019123000) {

        $table = new xmldb_table('groups');
        $field = new xmldb_field('startdate', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, 0);
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);

            $field = new xmldb_field('enddate', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, 0);
            if (!$dbman->field_exists($table, $field)) {
                $dbman->add_field($table, $field);
            }

            // Main savepoint reached.
            upgrade_block_savepoint(true, 2019123000, 'ned_teacher_tools');
        }
    }

    if ($oldversion < 2020011400) {

        if ($settings = $DB->get_records('config_plugins', ['plugin' => 'block_net_teacher_tools'])) {
            foreach ($settings as $setting) {
                set_config($setting->name, $setting->value, 'block_ned_teacher_tools');
            }
        }
        $DB->delete_records('config_plugins', ['plugin' => 'block_net_teacher_tools']);

        // Ned_teacher_tools savepoint reached.
        upgrade_block_savepoint(true, 2020011400, 'ned_teacher_tools');
    }

    if ($oldversion < 2020072400) {

        // Define table block_ned_teacher_tools_usrd to be created.
        $table = new xmldb_table('block_ned_teacher_tools_usrd');

        // Adding fields to table block_ned_teacher_tools_usrd.
        $table->add_field('id', XMLDB_TYPE_INTEGER, '11', null, XMLDB_NOTNULL, XMLDB_SEQUENCE, null);
        $table->add_field('userid', XMLDB_TYPE_INTEGER, '11', null, null, null, null);
        $table->add_field('courseid', XMLDB_TYPE_INTEGER, '11', null, null, null, null);
        $table->add_field('startdate', XMLDB_TYPE_INTEGER, '11', null, null, null, '0');
        $table->add_field('enddate', XMLDB_TYPE_INTEGER, '11', null, null, null, '0');
        $table->add_field('timecreated', XMLDB_TYPE_INTEGER, '11', null, null, null, '0');
        $table->add_field('timemodified', XMLDB_TYPE_INTEGER, '11', null, null, null, '0');

        // Adding keys to table block_ned_teacher_tools_usrd.
        $table->add_key('id', XMLDB_KEY_PRIMARY, ['id']);

        // Adding indexes to table block_ned_teacher_tools_usrd.
        $table->add_index('ix_dm_usr_cour', XMLDB_INDEX_NOTUNIQUE, ['userid', 'courseid']);

        // Conditionally launch create table for block_ned_teacher_tools_usrd.
        if (!$dbman->table_exists($table)) {
            $dbman->create_table($table);
        }

        // Ned_teacher_tools savepoint reached.
        upgrade_block_savepoint(true, 2020072400, 'ned_teacher_tools');
    }

    if ($oldversion < 2020080800) {

        // Define table block_ned_teacher_tools_cued to be created.
        $table = new xmldb_table('block_ned_teacher_tools_cued');

        // Adding fields to table block_ned_teacher_tools_cued.
        $table->add_field('id', XMLDB_TYPE_INTEGER, '11', null, XMLDB_NOTNULL, XMLDB_SEQUENCE, null);
        $table->add_field('submitter', XMLDB_TYPE_INTEGER, '11', null, null, null, null);
        $table->add_field('userid', XMLDB_TYPE_INTEGER, '11', null, null, null, '0');
        $table->add_field('groupid', XMLDB_TYPE_INTEGER, '11', null, null, null, '0');
        $table->add_field('courseid', XMLDB_TYPE_INTEGER, '11', null, null, null, '0');
        $table->add_field('enddateorig', XMLDB_TYPE_INTEGER, '11', null, null, null, '0');
        $table->add_field('enddate', XMLDB_TYPE_INTEGER, '11', null, null, null, '0');
        $table->add_field('reason', XMLDB_TYPE_TEXT, null, null, null, null, null);
        $table->add_field('timecreated', XMLDB_TYPE_INTEGER, '11', null, null, null, '0');

        // Adding keys to table block_ned_teacher_tools_cued.
        $table->add_key('id', XMLDB_KEY_PRIMARY, ['id']);

        // Adding indexes to table block_ned_teacher_tools_cued.
        $table->add_index('mdl_blocnedteactoolusrd_use_ix', XMLDB_INDEX_NOTUNIQUE, ['userid', 'courseid']);

        // Conditionally launch create table for block_ned_teacher_tools_cued.
        if (!$dbman->table_exists($table)) {
            $dbman->create_table($table);
        }

        // Ned_teacher_tools savepoint reached.
        upgrade_block_savepoint(true, 2020080800, 'ned_teacher_tools');
    }

    if ($oldversion < 2020080801) {

        // Define field changetype to be added to block_ned_teacher_tools_cued.
        $table = new xmldb_table('block_ned_teacher_tools_cued');
        $field = new xmldb_field('changetype', XMLDB_TYPE_CHAR, '30', null, null, null, null, 'submitter');

        // Conditionally launch add field changetype.
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        // Ned_teacher_tools savepoint reached.
        upgrade_block_savepoint(true, 2020080801, 'ned_teacher_tools');
    }

    if ($oldversion < 2020081101) {

        // Define field cohortid to be added to block_ned_teacher_tools_cued.
        $table = new xmldb_table('block_ned_teacher_tools_cued');
        $field = new xmldb_field('cohortid', XMLDB_TYPE_INTEGER, '11', null, null, null, '0', 'submitter');

        // Conditionally launch add field cohortid.
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        // Ned_teacher_tools savepoint reached.
        upgrade_block_savepoint(true, 2020081101, 'ned_teacher_tools');
    }

    if ($oldversion < 2020082000) {

        // Define field cohortid to be added to block_ned_teacher_tools_exte.
        $table = new xmldb_table('block_ned_teacher_tools_exte');
        $field = new xmldb_field('cohortid', XMLDB_TYPE_INTEGER, '18', null, null, null, '0', 'id');

        // Conditionally launch add field cohortid.
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        // Define field groupid to be added to block_ned_teacher_tools_exte.
        $table = new xmldb_table('block_ned_teacher_tools_exte');
        $field = new xmldb_field('groupid', XMLDB_TYPE_INTEGER, '18', null, null, null, '0', 'courseid');

        // Conditionally launch add field groupid.
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        // Define field duedateorig to be added to block_ned_teacher_tools_exte.
        $table = new xmldb_table('block_ned_teacher_tools_exte');
        $field = new xmldb_field('duedateorig', XMLDB_TYPE_INTEGER, '18', null, null, null, '0', 'userid');

        // Conditionally launch add field duedateorig.
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        // Ned_teacher_tools savepoint reached.
        upgrade_block_savepoint(true, 2020082000, 'ned_teacher_tools');
    }

    if ($oldversion < 2020090303) {
        if ($blockinstances = $DB->get_records('block_instances', ['blockname' => 'ned_teacher_tools'])) {
            foreach ($blockinstances as $blockinstance) {
                if (!empty($blockinstance->configdata)) {
                    $configdata = unserialize(base64_decode($blockinstance->configdata));
                    if (isset($configdata->donotallowunordered)) {
                        $configdata->allowunordered = (!empty($configdata->donotallowunordered)) ? 0 : 1;
                        unset($configdata->donotallowunordered);

                        $configdata = base64_encode(serialize($configdata));

                        $rec = new \stdClass();
                        $rec->id = $blockinstance->id;
                        $rec->configdata = $configdata;

                        $DB->update_record('block_instances', $rec);
                    }
                }
            }
        }

        // Ned_teacher_tools savepoint reached.
        upgrade_block_savepoint(true, 2020090303, 'ned_teacher_tools');
    }

    if ($oldversion < 2020091000) {

        $table = new xmldb_table($GT_table);

        // Adding fields to table block_ned_teacher_tools_cued.
        $table->add_field('id', XMLDB_TYPE_INTEGER, '20', null, XMLDB_NOTNULL, XMLDB_SEQUENCE);
        $table->add_field('courseid', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null);
        $table->add_field('cmid', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null);
        $table->add_field('userid', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null);
        $table->add_field('timestart', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, '0');
        $table->add_field('timeend', XMLDB_TYPE_INTEGER, '11', null, XMLDB_NOTNULL, null, '0');
        $table->add_field('timegrade', XMLDB_TYPE_INTEGER, '11', null, XMLDB_NOTNULL, null, '0');
        $table->add_field('graderid', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null);
        $table->add_field('note', XMLDB_TYPE_TEXT, null, null, null, null, null);
        $table->add_field('hidden', XMLDB_TYPE_INTEGER, '1', null, XMLDB_NOTNULL, null, '0');

        // Adding keys to table
        $table->add_key('id', XMLDB_KEY_PRIMARY, ['id']);
        $table->add_key('courseid', XMLDB_KEY_FOREIGN, ['courseid'], 'course', ['id']);
        $table->add_key('cmid', XMLDB_KEY_FOREIGN, ['cmid'], 'course_modules', ['id']);
        $table->add_key('userid', XMLDB_KEY_FOREIGN, ['userid'], 'user', ['id']);
        $table->add_key('graderid', XMLDB_KEY_FOREIGN, ['graderid'], 'user', ['id']);

        // Conditionally launch create table
        if (!$dbman->table_exists($table)) {
            $dbman->create_table($table);
        }

        // Ned_teacher_tools savepoint reached.
        upgrade_block_savepoint(true, 2020091000, 'ned_teacher_tools');
    }

    if ($oldversion < 2020092600) {
        $table = new xmldb_table($GT_table);
        $field = new xmldb_field('workdays', XMLDB_TYPE_INTEGER, '4', null, XMLDB_NOTNULL, null, 0);

        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        $DB->execute('DELETE grtr
        FROM {block_ned_teacher_tools_grtr} grtr
        LEFT JOIN {role} r 
            ON r.shortname = "student"
        LEFT JOIN {role_assignments} ra
          ON ra.roleid = r.id
          AND ra.userid = grtr.userid
        WHERE ra.id IS NULL OR grtr.userid = grtr.graderid');

        upgrade_block_savepoint(true, 2020092600, 'ned_teacher_tools');
    }

    if ($oldversion < 2020092900){
        $course_ids = $DB->get_records($GT_table, null, '', 'DISTINCT courseid');
        foreach ($course_ids as $courseid => $record){
            $context = \context_course::instance($courseid);
            $DB->execute('DELETE grtr 
            FROM {block_ned_teacher_tools_grtr} grtr
            LEFT JOIN {role_assignments} ra
                ON ra.userid = grtr.userid
                AND ra.contextid = :contextid
            LEFT JOIN {role} r 
                ON r.id = ra.roleid
                AND r.shortname = "student"
            WHERE grtr.courseid = :courseid AND r.id IS NULL
            ', ['contextid' => $context->id, 'courseid' => $courseid]);
        }

        upgrade_block_savepoint(true, 2020092900, 'ned_teacher_tools');
    }

    if ($oldversion < 2020100201){
        $course_ids = $DB->get_records('course', null, '', 'id');
        foreach ($course_ids as $courseid => $record){
            block_ned_teacher_tools_remove_inactive_deadlines($courseid);
        }

        upgrade_block_savepoint(true, 2020100201, 'ned_teacher_tools');
    }

    if ($oldversion < 2020100700){
        $c = 0;
        $i = 0;
        $type = '';
        $debug_s = function($set_type, $count) use (&$i, &$c, &$debug, &$type){
            $type = $set_type;
            $c = $count;
            $debug("Required to update $c $type records:");
            $i = 0;
        };
        $debug_i = function() use (&$i, &$c, &$type, &$debug){
            $i++;
            if ($i % 100 == 0){
                $debug("• updating $i/$c $type record...");
            }
        };
        $debug_f = function() use (&$i, &$c, &$type, &$debug){
            $debug("$i/$c $type records have been updated.", true);
        };
        $assign_records = $DB->get_records_sql("
            SELECT DISTINCT a.*, cm.id as cmid 
            FROM {assign} a
            JOIN {assign_overrides} ao
              ON ao.assignid = a.id
            JOIN {course_modules} cm
              ON cm.instance = a.id 
            JOIN {modules} m
                ON m.id = cm.module
                AND m.name = 'assign'
        ");

        $debug_s('assign', count($assign_records));
        foreach ($assign_records as $assign_record){
            $debug_i();
            block_ned_teacher_tools_assign_update_events($assign_record);
        }
        $debug_f();

        require_once($CFG->dirroot . '/mod/quiz/lib.php');
        $quizzes = $DB->get_records_sql("
            SELECT DISTINCT q.*, cm.id AS coursemodule 
            FROM {quiz} q
            JOIN {quiz_overrides} qo
              ON qo.quiz = q.id
            JOIN {course_modules} cm
              ON cm.instance = q.id 
            JOIN {modules} m
                ON m.id = cm.module
                AND m.name = 'quiz'
        ");
        $debug_s('quiz', count($quizzes));
        foreach ($quizzes as $quiz){
            $debug_i();
            quiz_update_events($quiz);
        }
        $debug_f();

        upgrade_block_savepoint(true, 2020100700, 'ned_teacher_tools');
    }

    if ($oldversion < 2020100801) {
        if ($blockinstances = $DB->get_records('block_instances', ['blockname' => 'ned_teacher_tools'])) {
            foreach ($blockinstances as $blockinstance) {
                if (!empty($blockinstance->configdata)) {
                    $configdata = unserialize(base64_decode($blockinstance->configdata));
                    if (isset($configdata->allowunordered)) {
                        $configdata->forcequizsequence = (!empty($configdata->allowunordered)) ? 0 : 1;
                        unset($configdata->allowunordered);

                        $configdata = base64_encode(serialize($configdata));

                        $rec = new \stdClass();
                        $rec->id = $blockinstance->id;
                        $rec->configdata = $configdata;

                        $DB->update_record('block_instances', $rec);
                    }
                }
            }
        }

        // Ned_teacher_tools savepoint reached.
        upgrade_block_savepoint(true, 2020100801, 'ned_teacher_tools');
    }

    if ($oldversion < 2020101401){

        $DB->execute("
            UPDATE {block_ned_teacher_tools_grtr} gt
            JOIN {course_modules} cm
                ON cm.id = gt.cmid
            JOIN {modules} m
                ON cm.module = m.id
            JOIN {grade_items} gi
                ON gi.courseid = cm.course
                AND gi.iteminstance = cm.instance
                AND gi.itemmodule = m.name
            LEFT JOIN {grade_grades} gg
                ON gg.itemid = gi.id
                AND gg.userid = gt.userid
            LEFT JOIN {grade_grades_history} ggh
                ON ggh.itemid = gi.id
                AND ggh.userid = gt.userid
                AND ggh.rawgrade IS NOT NULL 
                AND ggh.source <> 'system'
                AND ggh.usermodified <> ggh.userid
            LEFT JOIN {grade_grades_history} ggh2
                ON ggh2.itemid = gi.id
                AND ggh2.userid = gt.userid
                AND ggh2.rawgrade IS NOT NULL 
                AND ggh2.source <> 'system'
                AND ggh2.usermodified <> ggh.userid
                AND ggh2.timemodified > ggh.timemodified
            SET gt.graderid = 
              IF(gg.usermodified IS NOT NULL AND gg.usermodified <> gt.userid, gg.usermodified,
              IF(ggh.usermodified IS NOT NULL AND ggh.usermodified <> gt.userid, ggh.usermodified,
              gt.graderid))
            WHERE gt.timegrade > 0 AND ggh2.id IS NULL
        ");

        upgrade_block_savepoint(true, 2020101401, 'ned_teacher_tools');
    }

    if ($oldversion < 2020101503) {
        $table = new xmldb_table('block_ned_teacher_c_status');
        $table->add_field('id', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, XMLDB_SEQUENCE, null);
        $table->add_field('courseid', XMLDB_TYPE_INTEGER, '10', null, true, null, null);
        $table->add_field('userid', XMLDB_TYPE_INTEGER, '10', null, true, null, null);
        $table->add_field('status', XMLDB_TYPE_INTEGER, '1', null, false, null, null);
        $table->add_field('examinator', XMLDB_TYPE_INTEGER, '10', null, true, null, null);
        $table->add_field('timeupdated', XMLDB_TYPE_INTEGER, '10', null, null, null, 0);
        $table->add_field('timecreated', XMLDB_TYPE_INTEGER, '10', null, null, null, 0);

        $table->add_key('id', XMLDB_KEY_PRIMARY, ['id']);

        $table->add_index('courseid_userid', XMLDB_INDEX_UNIQUE, ['courseid', "userid"]);
        $table->add_index('userid_courseid', XMLDB_INDEX_UNIQUE, ['userid', "courseid"]);

        if (!$dbman->table_exists($table)) {
            $dbman->create_table($table);
        }

        upgrade_block_savepoint(true, 2020101503, 'ned_teacher_tools');
    }

    if ($oldversion < 2020102600) {

        $table = new xmldb_table($GT_table);
        $field = new xmldb_field('attempt', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, 0);

        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
            $table->add_index('uniqueattemptsubmission', XMLDB_INDEX_UNIQUE, ['cmid', 'userid', 'attempt']);
        }

        block_ned_teacher_tools_insert_resubmissions_assign();
        block_ned_teacher_tools_insert_resubmissions_quiz();

        // Main savepoint reached.
        upgrade_block_savepoint(true, 2020102600, 'ned_teacher_tools');
    }

    if ($oldversion < 2020102700) {
        block_ned_teacher_tools_update_grades_gt();
        block_ned_teacher_tools_delete_empty_forum_posts_form_gt();
        upgrade_block_savepoint(true, 2020102700, 'ned_teacher_tools');
    }

    if ($oldversion < 2020110900) {

        $table = new xmldb_table($GT_table);
        $field = new xmldb_field('suspended', XMLDB_TYPE_INTEGER, '1', null, XMLDB_NOTNULL, null, 0);

        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        $table = new xmldb_table('block_ned_teacher_tools_exte');
        $field = new xmldb_field('number', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, 1);

        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        $DB->execute('DELETE grtr
            FROM {block_ned_teacher_tools_grtr} grtr
            WHERE grtr.userid = grtr.graderid
        ');

        block_ned_teacher_tools_update_gt_graders();
        block_ned_teacher_tools_update_gt_students();

        // Define field timezone to be added to cohort.
        $table = new xmldb_table('cohort');
        $field = new xmldb_field('timezone', XMLDB_TYPE_CHAR, '100', null, null, null, null, 'component');

        // Conditionally launch add field timezone.
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        // Define field location to be added to cohort.
        $table = new xmldb_table('cohort');
        $field = new xmldb_field('location', XMLDB_TYPE_CHAR, '255', null, null, null, null, 'timezone');

        // Conditionally launch add field location.
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        upgrade_block_savepoint(true, 2020110900, 'ned_teacher_tools');
    }

    if ($oldversion < 2020111100) {

        $table = new xmldb_table($GT_table);
        $field = new xmldb_field('bug', XMLDB_TYPE_INTEGER, '1', null, XMLDB_NOTNULL, null, 0);

        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        $field = new xmldb_field('bug_report', XMLDB_TYPE_TEXT, null, null, null, null, null);
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        upgrade_block_savepoint(true, 2020111100, 'ned_teacher_tools');
    }

    if ($oldversion < 2020111400) {
        $t = strtotime("1 September 2020");
        block_ned_teacher_tools_insert_lost_submissions_assign($t);
        block_ned_teacher_tools_insert_lost_submissions_quiz($t);

        upgrade_block_savepoint(true, 2020111400, 'ned_teacher_tools');
    }

    if ($oldversion < 2020111700) {

        $table = new xmldb_table($GT_table);
        $field = new xmldb_field('uncounted', XMLDB_TYPE_INTEGER, '1', null, XMLDB_NOTNULL, null, 0);

        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        $field = new xmldb_field('uncounted_reason', XMLDB_TYPE_TEXT, null, null, null, null, null);
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        upgrade_block_savepoint(true, 2020111700, 'ned_teacher_tools');
    }

    if ($oldversion < 2020112000) {
        block_ned_teacher_tools_delete_previous_submissions_of_graded_attempts();
        upgrade_block_savepoint(true, 2020112000, 'ned_teacher_tools');
    }

    if ($oldversion < 2020120200) {
        block_ned_teacher_tools_delete_not_graded_cm_from_gt();
        upgrade_block_savepoint(true, 2020120200, 'ned_teacher_tools');
    }

    if ($oldversion < 2021031900) {

        $table = new xmldb_table($GT_table);
        $field = new xmldb_field('deadline', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, 0, 'timeend');

        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        upgrade_block_savepoint(true, 2021031900, 'ned_teacher_tools');
    }

    if ($oldversion < 2021052800) {

        $table = new xmldb_table($GT_table);
        $field = new xmldb_field('timemodified', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, 0);

        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        upgrade_block_savepoint(true, 2021052800, 'ned_teacher_tools');
    }

    if ($oldversion < 2021081000) {

        // Define index ix_usr_cor (not unique) to be added to block_ned_teacher_tools_exte.
        $table = new xmldb_table('block_ned_teacher_tools_exte');
        $index = new xmldb_index('ix_usr_cor', XMLDB_INDEX_NOTUNIQUE, ['courseid', 'userid']);

        // Conditionally launch add index ix_usr_cor.
        if (!$dbman->index_exists($table, $index)) {
            $dbman->add_index($table, $index);
        }

        // Ned_teacher_tools savepoint reached.
        upgrade_block_savepoint(true, 2021081000, 'ned_teacher_tools');
    }

    if ($oldversion < 2021090700){
        // update assign deadlines
        block_ned_teacher_tools_update_dma();

        // Ned_teacher_tools savepoint reached.
        upgrade_block_savepoint(true, 2021090700, 'ned_teacher_tools');
    }

    if ($oldversion < 2021121101){
        // We need to show admin, that setting was changed
        $DB->delete_records('config_plugins', ['plugin' => SH::$PLUGIN_NAME, 'name' => 'gradingtracker_manystudents']);

        // Change old GT name
        $old_name = 'Grading Tracker';
        $new_name = 'Grading Monitor';

        $records = $DB->get_records_select('config_plugins', "`plugin` = 'local_boostnavigation' AND `value` LIKE '%$old_name%'");
        foreach ($records as $record){
            $record->value = str_replace($old_name, $new_name, $record->value);
            $DB->update_record('config_plugins', $record);
        }

        $records = $DB->get_records_select('label', "`name` LIKE '%$old_name%' OR `intro` LIKE '%$old_name%'");
        foreach ($records as $record){
            $record->name = str_replace($old_name, $new_name, $record->name);
            $record->intro = str_replace($old_name, $new_name, $record->intro);
            $DB->update_record('label', $record);
        }

        // Ned_teacher_tools savepoint reached.
        upgrade_block_savepoint(true, 2021121101, 'ned_teacher_tools');
    }

    if ($oldversion >= 2019090800 && $oldversion < 2022071500){
        $table = new xmldb_table('assign_overrides');
        $field = new xmldb_field('extension');
        if ($dbman->field_exists($table, $field)) {
            $dbman->drop_field($table, $field);
        }

        $table = new xmldb_table('quiz_overrides');
        $field = new xmldb_field('extension');
        if ($dbman->field_exists($table, $field)) {
            $dbman->drop_field($table, $field);
        }

        upgrade_block_savepoint(true, 2022071500, 'ned_teacher_tools');
    }

    if ($oldversion < 2022081500){
        $field = new xmldb_field('overrule', XMLDB_TYPE_INTEGER, '2', null, XMLDB_NOTNULL, null, '0');
        $tables = ['assign_overrides', 'quiz_overrides', 'block_ned_teacher_tools_exte'];
        foreach ($tables as $table_name){
            $table = new xmldb_table($table_name);
            if (!$dbman->field_exists($table, $field)) {
                $dbman->add_field($table, $field);
            }
        }

        upgrade_block_savepoint(true, 2022081500, 'ned_teacher_tools');
    }

    if ($oldversion < 2022082800){
        // Define table block_ned_teacher_tools_cfg to be created.
        $table = new xmldb_table('block_ned_teacher_tools_cfg');

        // Adding fields to table block_ned_teacher_tools_cfg.
        $table->add_field('id', XMLDB_TYPE_INTEGER, '11', null, XMLDB_NOTNULL, XMLDB_SEQUENCE);
        $table->add_field('courseid', XMLDB_TYPE_INTEGER, '11', null, XMLDB_NOTNULL, null);
        $table->add_field('cmid', XMLDB_TYPE_INTEGER, '11', null, XMLDB_NOTNULL, null);
        $table->add_field('resubmission', XMLDB_TYPE_INTEGER, '1', null, null, null, '0');
        $table->add_field('deadline_manager', XMLDB_TYPE_INTEGER, '1', null, null, null, '0');
        $table->add_field('extensions', XMLDB_TYPE_INTEGER, '1', null, null, null, '0');

        // Adding keys to table block_ned_teacher_tools_cfg.
        $table->add_key('primary', XMLDB_KEY_PRIMARY, ['id']);
        $table->add_key('courseid', XMLDB_KEY_FOREIGN, ['courseid'], 'course', ['id']);
        $table->add_key('cmid', XMLDB_KEY_FOREIGN, ['cmid'], 'course_modules', ['id']);

        // Adding indexes to table block_ned_teacher_tools_cfg.
        $table->add_index('courseid-cmid', XMLDB_INDEX_NOTUNIQUE, ['courseid', 'cmid']);
        $table->add_index('cmid-deadline_manager-extensions', XMLDB_INDEX_NOTUNIQUE, ['cmid', 'deadline_manager','extensions']);

        // Conditionally launch create table for block_ned_teacher_tools_cfg.
        if (!$dbman->table_exists($table)) {
            $dbman->create_table($table);

            /** @see block_ned_teacher_tools_update_tt_activities_settings() */
            local_ned_controller\task\disposable_adhoc_task::add_new_job(
                'block_ned_teacher_tools_update_tt_activities_settings',
                '', '', [], 0, null, __DIR__ . '/upgradelib.php'
            );
            SH::ctrl_admin_alert_add('warn_tt_updating', SH::NOTIFY_WARNING);
        }

        upgrade_block_savepoint(true, 2022082800, 'ned_teacher_tools');
    }

    if ($oldversion < 2022112800){
        $table = new xmldb_table($GT_table);
        $field = new xmldb_field('latest', XMLDB_TYPE_INTEGER, '1', null, XMLDB_NOTNULL, null, 0);

        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);

            $sql = "
                UPDATE {{$GT_table}} gt
                LEFT JOIN {{$GT_table}} gt2
                    ON gt2.cmid = gt.cmid
                    AND gt2.courseid = gt.courseid
                    AND gt2.userid = gt.userid
                    AND gt2.attempt > gt.attempt
                SET gt.latest = '1'
                WHERE gt2.id IS NULL
            ";
            $DB->execute($sql);
        }

        // delete old tasks
        $where = $params = [];
        $tasks = ['\\block_ned_teacher_tools\\task\\adhoc_crongt_course', '\\block_ned_teacher_tools\\task\\adhoc_crongt_update'];
        SH::sql_add_get_in_or_equal_options('classname', $tasks, $where, $params);
        $DB->delete_records_select('task_adhoc', SH::sql_condition($where), $params);

        upgrade_block_savepoint(true, 2022112800, 'ned_teacher_tools');
    }

    if ($oldversion < 2024012500){
        $table = new xmldb_table($GT_table);
        $field = new xmldb_field('proxy', XMLDB_TYPE_INTEGER, '1', null, XMLDB_NOTNULL, null, 0, 'suspended');

        if (!$dbman->field_exists($table, $field)){
            $dbman->add_field($table, $field);

            // update proxy field values
            $proxy_cmids = SH::cmids_get_by_tags(SH::TAG_PROXY);
            [$col_sql, $col_params] = SH::db()->get_in_or_equal($proxy_cmids, SQL_PARAMS_NAMED, 'proxy_', true);
            SH::db()->set_field_select($GT_table, 'proxy', 1, "cmid $col_sql", $col_params);
        }

        upgrade_block_savepoint(true, 2024012500, 'ned_teacher_tools');
    }
}

/**
 * This function updates the events associated to the assign.
 *  // function from /mod/assign/lib.php: assign_update_events
 *
 * @param \stdClass $assign_record the assign record plus 'cmid' value.
 *
 */
function block_ned_teacher_tools_assign_update_events($assign_record) {
    global $CFG, $DB;
    require_once($CFG->dirroot . '/mod/assign/lib.php');
    require_once($CFG->dirroot . '/calendar/lib.php');
    $override = null;

    // Load the old events relating to this assign.
    $conds = array('modulename' => 'assign', 'instance' => $assign_record->id);
    if (!empty($override)) {
        // Only load events for this override.
        if (isset($override->userid)) {
            $conds['userid'] = $override->userid;
        } else if (isset($override->groupid)) {
            $conds['groupid'] = $override->groupid;
        } else {
            // This is not a valid override, it may have been left from a bad import or restore.
            $conds['groupid'] = $conds['userid'] = 0;
        }
    }
    $oldevents = $DB->get_records('event', $conds, 'id ASC');

    // Now make a to-do list of all that needs to be updated.
    if (empty($override)) {
        // We are updating the primary settings for the assignment, so we need to add all the overrides.
        $overrides = $DB->get_records('assign_overrides', array('assignid' => $assign_record->id), 'id ASC');
        // It is necessary to add an empty stdClass to the beginning of the array as the $oldevents
        // list contains the original (non-override) event for the module. If this is not included
        // the logic below will end up updating the wrong row when we try to reconcile this $overrides
        // list against the $oldevents list.
        array_unshift($overrides, new stdClass());
    } else {
        // Just do the one override.
        $overrides = array($override);
    }

    $cmid = $assign_record->cmid;

    foreach ($overrides as $current) {
        $groupid   = isset($current->groupid) ? $current->groupid : 0;
        $userid    = isset($current->userid) ? $current->userid : 0;
        $duedate = isset($current->duedate) ? $current->duedate : $assign_record->duedate;

        // Only add 'due' events for an override if they differ from the assign default.
        $addclose = empty($current->id) || !empty($current->duedate);

        $event = new stdClass();
        $event->type = CALENDAR_EVENT_TYPE_ACTION;
        $event->description = format_module_intro('assign', $assign_record, $cmid, false);
        $event->format = FORMAT_HTML;
        // Events module won't show user events when the courseid is nonzero.
        $event->courseid    = ($userid) ? 0 : $assign_record->course;
        $event->groupid     = $groupid;
        $event->userid      = $userid;
        $event->modulename  = 'assign';
        $event->instance    = $assign_record->id;
        $event->timestart   = $duedate;
        $event->timeduration = 0;
        $event->timesort    = $event->timestart + $event->timeduration;
        $event->visible     = instance_is_visible('assign', $assign_record);
        $event->eventtype   = ASSIGN_EVENT_TYPE_DUE;
        $event->priority    = null;

        // Determine the event name and priority.
        if ($groupid) {
            // Group override event.
            $params = new stdClass();
            $params->assign = $assign_record->name;
            $params->group = groups_get_group_name($groupid);
            if ($params->group === false) {
                // Group doesn't exist, just skip it.
                continue;
            }
            $eventname = get_string('overridegroupeventname', 'assign', $params);
            // Set group override priority.
            if (isset($current->sortorder)) {
                $event->priority = $current->sortorder;
            }
        } else if ($userid) {
            // User override event.
            $params = new stdClass();
            $params->assign = $assign_record->name;
            $eventname = get_string('overrideusereventname', 'assign', $params);
            // Set user override priority.
            $event->priority = CALENDAR_EVENT_USER_OVERRIDE_PRIORITY;
        } else {
            // The parent event.
            $eventname = $assign_record->name;
        }

        if ($duedate && $addclose) {
            if ($oldevent = array_shift($oldevents)) {
                $event->id = $oldevent->id;
            } else {
                unset($event->id);
            }
            $event->name      = $eventname.' ('.get_string('duedate', 'assign').')';
            calendar_event::create($event, false);
        }
    }

    // Delete any leftover events.
    foreach ($oldevents as $badevent) {
        $badevent = calendar_event::load($badevent);
        $badevent->delete();
    }
}

function block_ned_teacher_tools_insert_resubmissions_assign(){
    $sql = "SELECT CONCAT(cm.id, a_s.userid, a_s.attemptnumber) as uniqid, 
            cm.id AS cmid, cm.course AS courseid, a_s.userid, a_s.attemptnumber AS attempt, a_s.timemodified AS submission_time, a_g.timemodified AS timegrade
            FROM {course_modules} cm
            JOIN {modules} m
              ON cm.module = m.id
            JOIN {assign} a
              ON m.name = 'assign'
              AND a.id = cm.instance
            JOIN {block_ned_teacher_tools_grtr} gt 
              ON gt.cmid = cm.id 
              AND gt.attempt = 0
            JOIN {assign_submission} a_s
              ON a_s.assignment = a.id 
              AND a_s.userid = gt.userid
              AND a_s.status = 'submitted'
            LEFT JOIN {assign_grades} a_g 
              ON a_g.assignment = a_s.assignment 
              AND a_g.userid = a_s.userid
              AND a_g.attemptnumber = a_s.attemptnumber
              AND a_g.grade > -1
            LEFT JOIN {block_ned_teacher_tools_grtr} gt2
              ON gt2.cmid = cm.id
              AND gt2.userid = a_s.userid
              AND gt2.attempt = a_s.attemptnumber
            WHERE gt2.id IS NULL
    ";

    block_ned_teacher_tools_insert_sql_records_in_gt($sql, $type=GT::MOD_ASSIGN);
}

function block_ned_teacher_tools_insert_resubmissions_quiz(){
    $sql = "SELECT CONCAT(cm.id, q_a.userid, q_a.attempt) as uniqid, 
            cm.id AS cmid, cm.course AS courseid, q_a.userid, (q_a.attempt-1) AS attempt, q_a.timefinish AS submission_time, 
            IF(q_a.sumgrades IS NOT NULL, q_a.timemodified, 0) AS timegrade
            FROM {course_modules} cm
            JOIN {modules} m
              ON cm.module = m.id
            JOIN {quiz} q
              ON m.name = 'quiz'
              AND q.id = cm.instance
            JOIN {block_ned_teacher_tools_grtr} gt 
              ON gt.cmid = cm.id 
              AND gt.attempt = 0
            JOIN {quiz_attempts} q_a
              ON q_a.quiz = q.id 
              AND q_a.userid = gt.userid
              AND q_a.state = 'finished'
            LEFT JOIN {block_ned_teacher_tools_grtr} gt2
              ON gt2.cmid = cm.id
              AND gt2.userid = q_a.userid
              AND gt2.attempt = (q_a.attempt-1)
            WHERE gt2.id IS NULL
    ";

    block_ned_teacher_tools_insert_sql_records_in_gt($sql, $type=GT::MOD_QUIZ);
}

function block_ned_teacher_tools_insert_sql_records_in_gt($sql, $type=GT::MOD_ALL, $params=[]){
    global $DB;

    $records = $DB->get_records_sql($sql, $params) ?: [];
    foreach ($records as $r){
        $gt = new GT($r->courseid, $r->cmid, $r->userid, $r->attempt, $type);
        $gt->add_activity($r->submission_time, $r->timegrade);
    }
}

function block_ned_teacher_tools_update_grades_gt(){
    global $DB;

    $sql = "UPDATE {block_ned_teacher_tools_grtr} gt
      JOIN {course_modules} cm
        ON cm.id = gt.cmid
      JOIN {modules} m
        ON m.id = cm.module
        
      -- assign
      LEFT JOIN {assign_submission} a_s
        ON m.name = 'assign'
        AND a_s.assignment = cm.instance
        AND a_s.userid = gt.userid
        AND a_s.status = 'submitted'
        AND a_s.attemptnumber = gt.attempt
      LEFT JOIN {assign_grades} a_g 
        ON a_g.assignment = a_s.assignment 
        AND a_g.userid = a_s.userid
        AND a_g.attemptnumber = a_s.attemptnumber
        AND (a_g.grade IS NOT NULL AND a_g.grade > -1)
        
      -- quiz
      LEFT JOIN {quiz_attempts} q_a
        ON m.name = 'quiz'
        AND q_a.quiz = cm.instance 
        AND q_a.userid = gt.userid
        AND q_a.attempt = (gt.attempt+1)
        AND q_a.state = 'finished'
        AND q_a.sumgrades IS NOT NULL
        
      -- other
      LEFT JOIN {grade_items} gi
        ON (m.name <> 'assign' AND m.name <> 'quiz')
        AND gi.itemmodule = m.name
        AND gi.courseid = cm.course
        AND gi.iteminstance = cm.instance
      LEFT JOIN {grade_grades} gg
        ON gg.itemid = gi.id
        AND gg.userid = gt.userid
        AND gt.attempt = 0
        AND (gg.rawgrade IS NOT NULL AND gg.rawgrade > -1)
        
      SET gt.timegrade = COALESCE(a_g.timemodified, q_a.timemodified, gg.timemodified, 0)
      WHERE gt.timegrade = 0 AND COALESCE(a_g.timemodified, q_a.timemodified, gg.timemodified, 0) > 0
    ";

    $DB->execute($sql);
}

function block_ned_teacher_tools_delete_empty_forum_posts_form_gt(){
    global $DB;

    $sql = "DELETE gt FROM {block_ned_teacher_tools_grtr} gt
      JOIN {course_modules} cm
        ON cm.id = gt.cmid
      JOIN {modules} m
        ON m.id = cm.module
      JOIN {forum} f 
        ON f.id = cm.instance
        AND m.name = 'forum'
      LEFT JOIN (
        SELECT fd.forum AS forumid, 
            fp.userid AS userid, 
            MAX(fd.timemodified) AS timemodified,
            SUM(fp.id IS NOT NULL) as posts
        FROM {forum_discussions} fd
        LEFT JOIN {forum_posts} fp 
            ON fp.discussion = fd.id
        GROUP BY fp.userid, fd.forum
      ) p ON p.userid = gt.userid AND p.forumid = f.id
      WHERE f.id IS NOT NULL AND COALESCE(p.posts, 0) = 0
    ";

    $DB->execute($sql);
}

function block_ned_teacher_tools_update_gt_graders(){
    GT::update_gt_graders();
}

function block_ned_teacher_tools_update_gt_students(){
    GT::update_gt_suspend_students();
}

function block_ned_teacher_tools_insert_lost_submissions_assign($t=0){
    $params = [];
    $sql_students = GT::sql_user_enrolments("ue.userid, e.courseid, ra_cm.id AS cmid", [],$params,
    GT::ROLE_STUDENT, 0, null, true, 'GROUP BY e.courseid, u.id, ra_cm.id');
    $where = "gt.id IS NULL";
    if ($t){
        $where .= " AND a_s.timemodified > :t";
        $params['t'] = $t;
    }

    $sql = "SELECT CONCAT(cm.id, a_s.userid, a_s.attemptnumber) as uniqid, 
            cm.id AS cmid, cm.course AS courseid, a_s.userid, 
            a_s.attemptnumber AS attempt, a_s.timemodified AS submission_time, a_g.timemodified AS timegrade
            FROM {course_modules} cm
            JOIN {modules} m
              ON cm.module = m.id
            JOIN {assign} a
              ON m.name = 'assign'
              AND a.id = cm.instance
            JOIN {assign_submission} a_s
              ON a_s.assignment = a.id 
              AND a_s.status = 'submitted'
            JOIN ($sql_students) students
              ON students.courseid = cm.course
              AND students.userid = a_s.userid
              AND (students.cmid = cm.id OR students.cmid IS NULL)
            LEFT JOIN {assign_grades} a_g 
              ON a_g.assignment = a_s.assignment 
              AND a_g.userid = a_s.userid
              AND a_g.attemptnumber = a_s.attemptnumber
              AND a_g.grade > -1
            LEFT JOIN {block_ned_teacher_tools_grtr} gt
              ON gt.cmid = cm.id
              AND gt.userid = a_s.userid
              AND gt.attempt = a_s.attemptnumber
            WHERE $where
    ";

    block_ned_teacher_tools_insert_sql_records_in_gt($sql, $type=GT::MOD_ASSIGN, $params);
}

function block_ned_teacher_tools_insert_lost_submissions_quiz($t=0){
    $params = [];
    $sql_students = GT::sql_user_enrolments("ue.userid, e.courseid, ra_cm.id AS cmid", [],$params,
        GT::ROLE_STUDENT, 0, null, true, 'GROUP BY e.courseid, u.id, ra_cm.id');
    $where = "gt.id IS NULL";
    if ($t){
        $where .= " AND q_a.timefinish > :t";
        $params['t'] = $t;
    }
    $sql = "SELECT CONCAT(cm.id, q_a.userid, q_a.attempt) as uniqid, 
            cm.id AS cmid, cm.course AS courseid, q_a.userid, (q_a.attempt-1) AS attempt, q_a.timefinish AS submission_time, 
            IF(q_a.sumgrades IS NOT NULL, q_a.timemodified, 0) AS timegrade
            FROM {course_modules} cm
            JOIN {modules} m
              ON cm.module = m.id
            JOIN {quiz} q
              ON m.name = 'quiz'
              AND q.id = cm.instance
            JOIN {quiz_attempts} q_a
              ON q_a.quiz = q.id 
              AND q_a.state = 'finished'
            JOIN ($sql_students) students
              ON students.courseid = cm.course
              AND students.userid = q_a.userid
              AND (students.cmid = cm.id OR students.cmid IS NULL)
            LEFT JOIN {block_ned_teacher_tools_grtr} gt
              ON gt.cmid = cm.id
              AND gt.userid = q_a.userid
              AND gt.attempt = (q_a.attempt-1)
            WHERE $where
    ";

    block_ned_teacher_tools_insert_sql_records_in_gt($sql, $type=GT::MOD_QUIZ, $params);
}

function block_ned_teacher_tools_delete_previous_submissions_of_graded_attempts(){
    global $DB;

    $sql = "DELETE grtr_prev 
        FROM {block_ned_teacher_tools_grtr} grtr
        JOIN (
            SELECT cmid, userid, MAX(attempt) AS maxattempt 
            FROM {block_ned_teacher_tools_grtr}  
            GROUP BY cmid, userid
        ) grtr_max 
            ON grtr_max.cmid = grtr.cmid
            AND grtr_max.userid = grtr.userid
            AND grtr_max.maxattempt = grtr.attempt
            AND grtr.timegrade > 0
        JOIN {block_ned_teacher_tools_grtr} grtr_prev
            ON grtr_prev.cmid = grtr.cmid
            AND grtr_prev.userid = grtr.userid
            AND grtr_prev.attempt < grtr_max.maxattempt
            AND grtr_prev.timegrade = 0
    ";

    $DB->execute($sql);
}

function block_ned_teacher_tools_delete_not_graded_cm_from_gt(){
    global $DB, $CFG;
    $tagfield = empty($CFG->keeptagnamecase) ? 'name' : 'rawname';
    $params = [
        'tiitemtype' => 'course_modules',
        'ticomponent' => 'core',
    ];

    $gt_tags = GT::get_gt_tags();
    [$tag_sql, $tag_params] = $DB->get_in_or_equal($gt_tags, SQL_PARAMS_NAMED);
    $params = array_merge($params, $tag_params);

    $sql = "DELETE gt 
        FROM {block_ned_teacher_tools_grtr} gt
        LEFT JOIN {course_modules} cm
          ON cm.id = gt.cmid
        LEFT JOIN {modules} m
          ON m.id = cm.module
        LEFT JOIN {assign} a
          ON m.name = 'assign'
          AND a.id = cm.instance
          AND a.grade <> 0
        LEFT JOIN {quiz} q
          ON m.name = 'quiz'
          AND q.id = cm.instance
          AND q.grade <> 0
        LEFT JOIN {forum} f
          ON m.name = 'forum'
          AND f.id = cm.instance
          AND f.assessed <> 0
          
        LEFT JOIN (
            SELECT tg.id, ti.itemid AS itemid 
            FROM {tag} tg 
            LEFT JOIN {tag_instance} ti
                ON ti.tagid = tg.id
                AND ti.itemtype = :tiitemtype 
                AND ti.component = :ticomponent
            LEFT JOIN {course_modules} cm
                ON cm.id = ti.itemid
            WHERE tg.$tagfield $tag_sql
            GROUP BY ti.itemid
        ) AS tag
            ON tag.itemid = cm.id
            
        WHERE cm.id IS NULL OR m.id IS NULL OR
          (a.id IS NULL AND q.id IS NULL AND f.id IS NULL) OR 
          tag.id IS NULL          
    ";

    $DB->execute($sql, $params);
}

/**
 * Update DMA table for two type of deadlines
 *
 * @return void
 */
function block_ned_teacher_tools_update_dma(){
    global $DB;
    $table = \block_ned_teacher_tools\mod\deadline_manager_assign::TABLE;
    $params = ['cutoffdate_delay' => SH::CUTOFFDATE_DELAY];
    $sql = "UPDATE {{$table}}
            SET cutoffdate = (duedate + :cutoffdate_delay)
            WHERE COALESCE(cutoffdate, 0) > 0 AND cutoffdate = duedate
    ";
    $DB->execute($sql, $params);
}

/**
 * Delete extensions where store old, useless data.
 */
function block_ned_teacher_tools_delete_wrong_extensions(){
    $table = block_ned_teacher_tools\deadline_manager::TABLE_EXTENSION;
    $select = ['e.id'];
    $join = [
        'LEFT JOIN {groups} g ON g.id = e.groupid',
        'LEFT JOIN {course_modules} cm ON cm.id = e.cmid',
        'LEFT JOIN {course} c ON c.id = e.courseid',
        'LEFT JOIN {user} u ON u.id = e.userid',
    ];
    $where = [
        'g.name is NULL AND e.groupid > 0',
        'cm.module is NULL',
        'u.firstname is NULL OR u.deleted = 1 OR u.suspended = 1',
        'c.shortname is NULL'
    ];
    $sql = SH::sql_generate($select, $join, $table, 'e', SH::sql_condition($where, 'OR'), [], [], '500');
    while ($data = SH::db()->get_records_sql($sql)){
        [$where, $params] = SH::sql_get_in_or_equal_options(['id' => array_keys($data)]);
        SH::db()->delete_records_select($table, $where, $params);
    }
}

/**
 * Set up DM/RM settings for the new table and remove them from block|plugin configs (deprecated logic)
 */
function block_ned_teacher_tools_update_tt_activities_settings(){
    block_ned_teacher_tools_delete_wrong_extensions();

    $checked_courses = [];
    $blocks = SH::db()->get_records_select('block_instances', "blockname = :block" , ['block' => SH::TT_NAME]);
    foreach ($blocks as $block){
        $ctx = SH::ctx(null, null, $block->parentcontextid);
        if (empty($ctx) || $ctx->id == SITEID) continue;

        $courseid = $ctx->instanceid;
        if ($checked_courses[$courseid] ?? false) continue;

        $dm_activities_conf = SH::get_config('dm_activities_'.$courseid);
        if (!empty($dm_activities_conf)){
            $dm_activities = explode(',', $dm_activities_conf);
            $dm_activities = array_combine($dm_activities, $dm_activities);
        }

        $dm_extensions_conf = SH::get_config('dm_extensions_'.$courseid);
        if (!empty($dm_extensions_conf)){
            $dm_extensions = explode(',', $dm_extensions_conf);
            $dm_extensions = array_combine($dm_extensions, $dm_extensions);
        }

        $block_config = SH::get_block_config($courseid, SH::TT_NAME);
        $rm_activities = $block_config->resubmission_assignments ?? [];
        if (!empty($rm_activities)){
            $rm_activities = array_keys(array_filter($rm_activities));
            $rm_activities = array_combine($rm_activities, $rm_activities);
        }

        $cms = [];
        if (!empty($dm_activities) || !empty($dm_extensions) || !empty($rm_activities)){
            $cms = SH::get_course_cms($courseid);
        }

        foreach ($cms as $cm){
            $cmid = $cm->id;
            $data = [];

            if ($dm_activities[$cmid] ?? false){
                if (CM::dm_is_cm_enabled_by_type($cm)){
                    $data[CM::CONFIG_DM_FIELD] = true;
                    if ($dm_extensions[$cmid] ?? false){
                        $data[CM::CONFIG_EXT_FIELD] = true;
                    }
                }
            }

            if ($rm_activities[$cmid] ?? false){
                if (CM::rm_is_cm_enabled_by_type($cm)){
                    $data[CM::CONFIG_RM_FIELD] = true;
                }
            }

            if (!empty($data)){
                CM::check_and_save_record($data, $cmid, $courseid);
            }
        }

        // Remove old settings
        if (isset($block_config->resubmission_assignments) || isset($block_config->numberofactivitieshavesamedue)){
            unset($block_config->resubmission_assignments);
            unset($block_config->numberofactivitieshavesamedue);
            SH::save_block_config($courseid, SH::TT_NAME, $block_config);
        }

        SH::unset_config('dm_activities_'.$courseid);
        SH::unset_config('dm_extensions_'.$courseid);

        $checked_courses[$courseid] = true;
    }

    SH::ctrl_admin_alert_rem('warn_tt_updating');
}
