<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * @package    block_ned_teacher_tools
 * @copyright  Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
use block_ned_teacher_tools as NED;
use block_ned_teacher_tools\shared_lib as SH;

require_once($CFG->dirroot.'/rating/lib.php');

if (! $forum = $DB->get_record("forum", array("id" => $iid))) {
    SH::print_error("Forum ID was incorrect or no longer exists");
}

$course_ctx = SH::ctx($course->id);
$isteacheredit = has_capability('moodle/course:update', $course_ctx);
$cm = $cm ?? get_coursemodule_from_instance('forum', $forum->id, $course->id);
$o = '';

$mod_ctx = SH::ctx(0, $cm->id);
$isteacher = has_any_capability(['moodle/rating:rate', 'mod/forum:rate', 'mod/forum:grade'], $mod_ctx);

// Find out current groups mode.
$currentgroup = groups_get_activity_group($cm, true);
$groupmode = groups_get_activity_groupmode($cm);

if ($forum->type == "teacher") {
    if (!$isteacheredit  || !$isteacher) {
        SH::print_error("You must be a course teacher to view this forum");
    }
}

if ($groupmode && ($currentgroup === false) && !$isteacheredit) {
    $OUTPUT->heading(get_string("notingroup", "forum"));
    exit;
}

// Print settings and things in a table across the top.
$forum->intro = trim($forum->intro);

if (isset($MM)){
    $data = $MM->get_activity_data($cm->id);
    $studentidlist = implode(',', array_keys($data['forum']));
    $students = $MM->students;
} else {
    $studentidlist = implode(',', array_keys($students));
}

if (empty($studentidlist)){
    $studentidlist = -1;
}

// Get students from forum_posts.
$itemnumber_rating = SH::GRADE_ITEMNUMBER_FORUM_RATING;
$itemnumber_forum = SH::GRADE_ITEMNUMBER_FORUM_WHOLE;
$user_grades = $DB->get_records_sql("
  SELECT u.id,
    IF(gg_rating.overridden > 0, gg_rating.finalgrade, gg_rating.rawgrade) AS grade_rating,
    IF(gg_forum.overridden > 0, gg_forum.finalgrade, gg_forum.rawgrade) AS grade_forum,
    COUNT(DISTINCT ufd.id) AS numberofposts,
    COUNT(DISTINCT fp.id) AS numberofreplies
  FROM {user} u 
  JOIN {forum} f 
    ON f.id = :forumid
  LEFT JOIN {grade_items} gi_rating
    ON gi_rating.iteminstance = f.id 
    AND gi_rating.courseid = f.course
    AND gi_rating.itemmodule = 'forum'
    AND gi_rating.itemtype = 'mod'
    AND gi_rating.itemnumber = $itemnumber_rating
  LEFT JOIN {grade_grades} gg_rating
    ON gg_rating.itemid = gi_rating.id AND gg_rating.userid = u.id
  LEFT JOIN {grade_items} gi_forum
    ON gi_forum.iteminstance = f.id 
    AND gi_forum.courseid = f.course
    AND gi_forum.itemmodule = 'forum'
    AND gi_forum.itemtype = 'mod'
    AND gi_forum.itemnumber = $itemnumber_forum
  LEFT JOIN {grade_grades} gg_forum
    ON gg_forum.itemid = gi_forum.id AND gg_forum.userid = u.id
  LEFT JOIN {forum_discussions} ufd -- only user discussions
    ON ufd.forum = f.id AND ufd.userid = u.id
  LEFT JOIN {forum_discussions} fd -- all discussions, where user can reply
    ON fd.forum = f.id
  JOIN {forum_posts} fp
    ON fp.discussion = fd.id AND fp.userid = u.id
  
  WHERE u.id IN ($studentidlist)
  GROUP BY f.id, u.id
  ", ['forumid' => $forum->id]);


// Show all the unsubmitted users.
if ($show == 'unsubmitted'){
    $o .= NED\render_marking_manager_not_submitted($students, $user_grades, $cm, $forum);
} elseif (($show == 'unmarked') || ($show == 'marked')) {
    // Show all the marked or unmarked users.
    if (!empty($user_grades)){
        $rm = new rating_manager();
        $scale_menu_rate = $scale_menu_grade = null;
        $use_rating = !empty($forum->assessed);
        $use_fgrading = !empty($forum->grade_forum);

        $forum_info0 = $forum_info1 = $forum_info2 = [];
        $f_add_forum_info = function($label='', $value='-'){
            return SH::div($label.': '.SH::span($value, 'forum-header-data'));
        };
        $f_add_forum_info_str = function($label_key='', $value='-') use ($f_add_forum_info) {
            return $f_add_forum_info(SH::str($label_key), $value);
        };
        $f_get_grade_text_class = function($grade_val, $scale_menu, $max_val){
            $cellclass = '';
            if (is_null($grade_val)){
                $gr_str = '-';
                $cellclass .= 'fn-highlighted';
            } else {
                if (!empty($scale_menu)){
                    $grade = round($grade_val, 0);
                    $grade = min($grade, $max_val);
                    $gr_str = $scale_menu[$grade] ?? '-';
                } else {
                    $gr_str = round($grade_val, 0).'/'.$max_val;
                }
            }

            return [$gr_str, $cellclass];
        };

        $forum_info0[] = $f_add_forum_info_str('type', forum_get_forum_types()[$forum->type] ?? '-');

        if ($use_rating){
            $forum_info1[] = $f_add_forum_info_str('aggregatetype', $rm->get_aggregate_types()[$forum->assessed] ?? '-');

            if ($forum->scale < 0){
                $scale_rate = $DB->get_record('scale', ['id' => -$forum->scale]);
                $scaleinfo = $scale_rate->name ?? '';
                $scale_menu_rate = make_grades_menu($forum->scale);
                // If the scale was changed after ratings were submitted some ratings may have a value above the current maximum.
                // We can't just do count($scale_menu_rate) - 1 as custom scales start at index 1, not 0.
                $max_rating = $scale_menu_rate ? max(array_keys($scale_menu_rate)) : 0;
            } else {
                $max_rating = $forum->scale;
                $scaleinfo = '/'.$max_rating;
            }
            $forum_info1[] = $f_add_forum_info_str('scale', $scaleinfo);

            $rolenames = get_role_names_with_caps_in_context($mod_ctx, ['moodle/rating:rate', 'mod/forum:rate']);
            $forum_info1[] = $f_add_forum_info_str('whocanrate', implode(', ', $rolenames));
        } else {
            $forum_info0[] = $f_add_forum_info_str('aggregatetype', $rm->get_aggregate_types()[0] ?? '-');
            $max_rating = 0;
        }

        if ($use_fgrading){
            if ($forum->grade_forum > 0){
                $forum_info2[] = $f_add_forum_info(get_string('grade_forum_header', 'forum'), get_string('modgradetypepoint', 'grades'));
                $max_grade = $forum->grade_forum;
                $scaleinfo = '/'.$max_grade;
            } else {
                $forum_info2[] = $f_add_forum_info(get_string('grade_forum_header', 'forum'), get_string('modgradetypescale', 'grades'));

                $scale_grade = $DB->get_record('scale', ['id' => -$forum->grade_forum]);
                $scaleinfo = $scale_grade->name ?? '';
                $scale_menu_grade = make_grades_menu($forum->grade_forum);
                // If the scale was changed after ratings were submitted some ratings may have a value above the current maximum.
                // We can't just do count($scale_menu_rate) - 1 as custom scales start at index 1, not 0.
                $max_grade = $scale_menu_grade ? max(array_keys($scale_menu_grade)) : 0;
            }
            $forum_info2[] = $f_add_forum_info_str('scale', $scaleinfo);
            $rolenames = get_role_names_with_caps_in_context($mod_ctx, ['mod/forum:grade']);
            $forum_info2[] = $f_add_forum_info_str('whocangrade', implode(', ', $rolenames));
        } else {
            $forum_info0[] = $f_add_forum_info(get_string('grade_forum_header', 'forum'), get_string('modgradetypenone', 'grades'));
            $max_grade = 0;
        }

        $forum_info = [];
        $forum_info_cols = [$forum_info0, $forum_info1, $forum_info2];
        foreach ($forum_info_cols as $forum_info_col){
            if (empty($forum_info_col)) continue;

            $forum_info[] = SH::div($forum_info_col);
        }

        $o .= SH::div($forum_info, 'fn_forum_header d-flex px-5');

        $forum_header = SH::div(SH::str('student_have_posted'));
        if ($cm->uservisible){
            $forum_header .= SH::link(['/mod/forum/view.php', ['id' => $cm->id]], SH::str($show == 'marked' ? 'seeallrateddiscussion' : 'openforum'),
                'btn btn-secondary', ['target' => '_blank']);
        }

        if (class_exists('\local_quickgrader\helper')) {
            if (\local_quickgrader\helper::has_quick_grader_tag($cm->id)) {
                $btnlabel = SH::str('quickfeedback');
            } else {
                $btnlabel = SH::str('openforumgrader');
            }
            if (SH::cm_can_grade_cm($cm->id)){
                $forum_header .= SH::link(['/local/quickgrader/forum.php', ['course' => $cm->course, 'forum' => $cm->instance, 'group' => $groupid]], $btnlabel,
                    'btn btn-secondary', ['target' => '_blank', 'style' => 'margin: 0 0 0 5px;']);
            }
        }

        $o .= SH::div($forum_header, 'fn_forum_header_post');
        $table = SH::html_table('generaltable', 'fn-forum-grading');
        $table->head = [
            SH::cell(SH::str('student'),'fn-student'),
            SH::cell(SH::str('posts'),'fn-posts'),
            SH::cell(SH::str('replies'),'fn-replies'),
        ];

        if ($use_rating){
            $table->head[] = SH::cell(SH::str('rating'),'fn-rating');
        }
        if ($use_fgrading){
            $table->head[] = SH::cell(SH::str('fgrading'),'fn-fgrade');
        }

        foreach ($students as $student) {
            $user_grade = $user_grades[$student->id] ?? null;
            if (!$user_grade) continue;

            $row = SH::row();
            $user_pic = SH::O()->user_picture($student, ['link' => true, 'includefullname' => true]);

            $row->cells[] = SH::cell($user_pic, 'fn-student');
            $row->cells[] = SH::cell($user_grade->numberofposts, 'fn-posts');
            $row->cells[] = SH::cell($user_grade->numberofreplies, 'fn-replies');
            if ($use_rating){
                [$gr_str, $cellclass] = $f_get_grade_text_class($user_grade->grade_rating, $scale_menu_rate, $max_rating);
                $row->cells[] = SH::cell($gr_str, "fn-rating $cellclass");

            }
            if ($use_fgrading){
                [$gr_str, $cellclass] = $f_get_grade_text_class($user_grade->grade_forum, $scale_menu_grade, $max_grade);
                $row->cells[] = SH::cell($gr_str, "fn-fgrade $cellclass");

            }

            $table->data[] = $row;
        }

        $o .= SH::render_table($table);
    } else {
        $o .= $OUTPUT->box(SH::str('nostudentwithpost'), 'box generalbox generalboxcontent boxaligncenter', 'intro');
    }
}

return $o;
