<?php
/**
 * @package    block_ned_teacher_tools
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace block_ned_teacher_tools;
use block_ned_teacher_tools\shared_lib as SH;
use local_kica as KICA;

require_once(__DIR__ . '/common_lib.php');

/**
 * @param                      $all_students
 * @param array                $submitted_students
 * @param \cm_info | \stdClass $cm
 * @param \stdClass            $instance
 *
 * @return string
 */
function render_marking_manager_not_submitted($all_students, $submitted_students, $cm, $instance=null){
    global $OUTPUT, $DB, $PAGE, $MB;
    if (!isset($MB)){
        $MB = new output\menu_bar($PAGE->url->out());
    }
    if (is_kica_exists() && $MB->kicainstalled){
        $kica = SH::get_kica($MB->courseid);
        $kicaitem = SH::ki_get_by_cm($cm);
    }
    if (empty($all_students)){
        return $OUTPUT->box(str('nostudents'), 'box generalbox generalboxcontent boxaligncenter', 'intro');
    }

    $o = '';
    if (is_null($instance)){
        $instance = $DB->get_record($cm->modname, ['id' => $cm->instance]);
    }

    $users_duedates = get_users_duedates_override($cm, $all_students);
    $now = $mod_duedate = null;
    if (isset($instance->duedate)){
        $duedate_exist = true;
        $mod_duedate = $instance->duedate;
    } elseif (isset($instance->timeclose)){
        $duedate_exist = true;
        $mod_duedate = $instance->timeclose;
    } else {
        $duedate_exist = !empty($users_duedates);
    }
    $str_mod_duedate = ned_date($mod_duedate);
    $o .= SH::div(get_string('modulename', $cm->modname).': '.SH::mod_link($cm, 16), 'unsubmitted_header');

    $table_not_subm = new \html_table();
    $table_not_subm->attributes['class'] = 'submit-table not-submitted';
    $table_subm = new \html_table();
    $table_subm->attributes['class'] = 'submit-table submitted';
    $head = [get_string('name'), get_string('group')];
    if (isset($kica) && isset($kicaitem) && $kicaitem->id){
        $head[] = str('kicagrade');
    }
    $head[] = get_string('lastlogin');
    if ($duedate_exist){
        $head[] =  str('duedate');
        SH::$C::js_call_amd('add_sorter', 'add_sort', ['table.submit-table']);
        $PAGE->requires->js_call_amd(PLUGIN_NAME . '/marking_manager_not_submitted', 'init');

        $filter_div = \html_writer::div(
            \html_writer::span(str('duedates') . ':', 'filter-header').
            \html_writer::checkbox('past', true, true, str('past')).
            \html_writer::checkbox('missing', true, true, str('missing')).
            \html_writer::checkbox('upcoming', true, true, str('upcoming')).
            $OUTPUT->help_icon('notsubmittedfilter', PLUGIN_NAME),
            'not-submitted-filter'
        );
        $o .= $filter_div;

        $now = time();
    }
    $table_subm->head = $table_not_subm->head = $head;
    $page_url = new \moodle_url($PAGE->url);

    foreach ($all_students as $student) {
        $add_class = '';
        $cells = [];
        $name = fullname($student);
        $cells[] = cell($OUTPUT->user_picture($student) . $name, 'username', ['data-sort-value' => $name]);
        $groupsort = 'zzzzzzz';
        $grouplink = '-';
        if (isset($student->group)){
            $groupsort = $student->group->name;
            $page_url->param('group', $student->group->id);
            $grouplink = link($page_url, $groupsort);
        }
        $cells[] = cell($grouplink, 'groupname', ['data-sort-value' => $groupsort]);
        if (isset($kica) && isset($kicaitem) && $kicaitem->id){
            $kicagrade = SH::kg_get_by_userid_itemid($student, $kicaitem);
            if (!empty($kicagrade->id)) {
                $kicagradepercentage = $kicagrade->get_grade_percentage(true, 2, true);
            } else {
                $kicagradepercentage = '-';
            }
            $cells[] =  cell($kicagradepercentage, 'kica-grade', ['data-sort-value' => $kicagradepercentage]);
        }
        $cells[] = cell(ned_date($student->lastlogin), 'lastlogin', ['data-sort-value' => $student->lastlogin]);
        if ($duedate_exist){
            if (isset($users_duedates[$student->id])){
                $duedate = $users_duedates[$student->id];
                $str_duedate = ned_date($duedate);
            } else {
                $duedate = $mod_duedate;
                $str_duedate = $str_mod_duedate;
            }

            if (!$duedate){
                $add_class .= ' missing ';
            } elseif($duedate <= $now){
                $add_class .= ' past ';
            } else {
                $add_class .= ' upcoming';
            }
            $cells[] = cell($str_duedate, 'duedate', ['data-sort-value' => $duedate]);
        }
        $row = row($cells, $add_class);
        if (!is_array($submitted_students) || !array_key_exists($student->id, $submitted_students)) {
            $table_not_subm->data[] = $row;
        } else {
            $table_subm->data[] = $row;
        }
    }

    $add_links = [];
    if (has_all_capabilities(['gradereport/singleview:view', 'moodle/grade:viewall', 'moodle/grade:edit'], $MB->context)){
        $gi_id = $DB->get_field('grade_items', 'id',
            ['courseid' => $MB->courseid, 'itemtype' => 'mod', 'itemmodule' => $cm->modname, 'iteminstance' => $cm->instance]);
        if ($gi_id){
            $add_links[] = link(['/grade/report/singleview/index.php',
                ['id' => $MB->courseid, 'item' => 'grade', 'group' => $MB->groupid, 'itemid' => $gi_id]],
                str('opengradeoverride'), '', ['target' => '_blank']);
        }
    }
    if (!empty($add_links)){
        $o .= \html_writer::div(join($add_links), 'help-links');
    }

    $o .= \html_writer::tag('h4', str('unsubmitted'), ['class' => 'subm_header unsubmitted_msg']).
        \html_writer::table($table_not_subm).
        \html_writer::tag('h4', str('submitted'), ['class' => 'subm_header submitted_msg']).
        \html_writer::table($table_subm);
    return $o;
}

/**
 * @param \cm_info | \stdClass $cm
 * @param array                $users
 *
 * @return array
 */
function get_users_duedates_override($cm, $users=[]){
    global $DB;

    if ($cm->modname != 'quiz' && $cm->modname != 'assign'){
        return [];
    }

    $add_where = '';
    if (!empty($users)){
        if (!isset($users[0])){
            $users = array_keys($users);
        }
        $add_where = 'AND u.id in (' . join(',', $users) . ')';
    }

    if ($cm->modname == 'assign'){
        $sql = "SELECT u.id AS userid,
          COALESCE(v.userclose, v.groupclose, 0) AS duedate
          FROM (
              SELECT u.id as userid,
                MAX(overr.duedate) AS userclose, MAX(g_overr.duedate) AS groupclose
               FROM {user} u
          LEFT JOIN {assign} assign ON assign.id = :assignid
          LEFT JOIN {assign_overrides} overr ON assign.id = overr.assignid AND overr.userid = u.id
          LEFT JOIN {groups_members} gm ON gm.userid = u.id
          LEFT JOIN {assign_overrides} g_overr ON assign.id = g_overr.assignid AND g_overr.groupid = gm.groupid
          WHERE (overr.duedate IS NOT NULL OR g_overr.duedate IS NOT NULL)
          $add_where
          GROUP BY u.id) v
               JOIN {user} u ON u.id = v.userid
        ";
        $records = $DB->get_records_sql_menu($sql, ['assignid' => $cm->instance]);
    } else { // quiz
        $sql = "SELECT u.id AS userid,
          COALESCE(v.userclose, v.groupclose, 0) AS duedate
          FROM (
              SELECT u.id as userid,
                MAX(overr.timeclose) AS userclose, MAX(g_overr.timeclose) AS groupclose
               FROM {user} u
          LEFT JOIN {quiz} quiz ON quiz.id = :quizid
          LEFT JOIN {quiz_overrides} overr ON quiz.id = overr.quiz AND overr.userid = u.id
          LEFT JOIN {groups_members} gm ON gm.userid = u.id
          LEFT JOIN {quiz_overrides} g_overr ON quiz.id = g_overr.quiz AND g_overr.groupid = gm.groupid
          WHERE (overr.timeclose IS NOT NULL OR g_overr.timeclose IS NOT NULL)
          $add_where
          GROUP BY u.id) v
               JOIN {user} u ON u.id = v.userid
        ";
        $records = $DB->get_records_sql_menu($sql, ['quizid' => $cm->instance]);
    }

    return empty($records) ? [] : $records;
}

/**
 * @param null $courseid
 *
 * @return array
 */
function get_courses_with_exam_mod_id($courseid = null){
    global $DB;

    $id_course_type = get_id_course_type();
    if (!$id_course_type){
        return [];
    }

    list($course_type_field_id, $course_type_id, $is_course_type_default) = $id_course_type;

    $filter = [
        'fieldid' => $course_type_field_id,
        'value' => $course_type_id,
        'name' => course_users_status::TAG_AUTO_SET_WAITING_FOR_GRADE,
        'criteriatype' => COMPLETION_CRITERIA_TYPE_ACTIVITY
    ];

    $where = [];

    if(!empty($courseid)){
        $filter['courseid'] = $courseid;
        $where[] = 'c.id = :courseid';
    }

    $customfield_data_where = 'cd.value = :value';
    if ($is_course_type_default){
        $customfield_data_where .= " OR cd.value IS NULL";
    }
    $where[] = $customfield_data_where;

    if (!empty($where)){
        $where = 'WHERE (' . join(') AND (', $where) . ')';
    } else {
        $where = '';
    }

    return $DB->get_records_sql(
        "SELECT c.id, GROUP_CONCAT(ccc.moduleinstance) AS exammodid
                    FROM {course} c
                    LEFT JOIN {customfield_data} cd ON cd.instanceid = c.id AND cd.fieldid = :fieldid
                    JOIN {course_completion_criteria} ccc ON ccc.course = c.id AND ccc.criteriatype = :criteriatype AND ccc.module = 'assign'
                    JOIN {tag_instance} ti ON  ti.itemid = ccc.moduleinstance
                    JOIN {tag} t ON t.name = :name AND t.id = ti.tagid
                    $where 
                    GROUP BY c.id" , $filter
    );
}

/**
 * @return bool|array
 */
function get_id_course_type(){
    static $res = null;
    if (is_null($res)){
        $record = \core_customfield\field::get_record(['shortname' => course_users_status::CUSTOM_FIELD_COURSE_TYPE_NAME]);
        if ($record){
            $id = $record->get('id');
            /** @var \customfield_select\field_controller $field */
            $field = \core_customfield\field_controller::create($id);
            $defaultvalue = $field->get_configdata_property('defaultvalue');
            $choices = $field->get_options();
            $res = [$id, array_search(course_users_status::COURSE_TYPE_AUTO_SET_WAITING_FOR_GRADE, $choices),
                course_users_status::COURSE_TYPE_AUTO_SET_WAITING_FOR_GRADE == $defaultvalue];
        } else {
            $res = false;
        }
    }
    return $res;
}

/**
 * @param $statuses
 * @return array
 */
function get_statuses_users_by_statuses($statuses) {
    global $DB;
    list($sql, $params) = $DB->get_in_or_equal($statuses, SQL_PARAMS_QM, 'statuses');
    $user_statuses = $DB->get_records_sql("
            SELECT CONCAT(cust.id, c.id), cust.courseid, cust.userid, cust.`status`, cust.timeupdated, 
              GROUP_CONCAT(NULLIF(c.idnumber, '') SEPARATOR ', ') AS 'schoolid', cr.shortname AS 'coursecode'
            FROM {" . course_users_status::TABLE . "} AS cust
            JOIN {cohort_members} cm ON cm.userid = cust.userid
            JOIN {cohort} c ON c.id = cm.cohortid
            JOIN {course} cr ON cr.id = cust.courseid
            WHERE cust.status {$sql}
            GROUP BY cust.userid
            ORDER BY cust.timeupdated DESC ",
        $params);

    return $user_statuses;
}

/**
 * @param        $courseid
 * @param        $context
 * @param        $userid
 * @param string $rolename
 *
 * @return array|false
 */
function get_group_teachers_by_course_context_userid($courseid, $context, $userid, $rolename=course_users_status::ROLE_OT_SHORTNAME) {
    global $DB;
    $params = [
        'courseid' => $courseid,
        'context' => $context,
        'userid' => $userid,
        'rolename' => $rolename,
    ];
    return $DB->get_records_sql("SELECT DISTINCT u2.*
                FROM {user} u 
                JOIN {groups_members} gm ON gm.userid = u.id
                JOIN {groups} g ON g.id = gm.groupid
                JOIN {groups_members} gm1 ON gm1.groupid = g.id
                JOIN {role_assignments} ra ON ra.userid = gm1.userid
                JOIN {role} r ON r.id = ra.roleid
                JOIN {user} u2 ON u2.id = ra.userid
                WHERE u.id = :userid AND r.shortname = :rolename AND g.courseid = :courseid AND ra.contextid = :context
            ", $params);
}

/**
 * @param        $courseid
 * @param        $context
 * @param int    $groupid
 * @param string $rolename
 *
 * @return array
 */
function get_group_teachers_by_course_context($courseid, $context, $groupid = 0, $rolename=course_users_status::ROLE_OT_SHORTNAME) {
    global $DB;
    $params = [
        'courseid' => $courseid,
        'context' => $context,
        'rolename' => $rolename,
    ];
    $where = '';
    if (!empty($groupid) && $groupid > 0){
        $params['groupid'] = $groupid;
        $where = ' and g.id = :groupid';
    }
    return $DB->get_records_sql("SELECT DISTINCT u.*
                FROM {user} u 
                JOIN {role_assignments} ra ON ra.userid = u.id
                JOIN {role} r ON r.id = ra.roleid
                LEFT JOIN {groups_members} gm ON gm.userid = u.id
                LEFT JOIN {groups} g ON g.id = gm.groupid AND g.courseid = :courseid
                WHERE r.shortname = :rolename AND ra.contextid = :context {$where}
            ", $params);
}



/**
 * @param $data1
 * @param $date2
 * @param array $array_text
 *
 * @return string
 */
function get_diff_dates($data1,$date2,$array_text = []){
    $origin = new \DateTime(date('m/d/Y H:i:s', $data1));
    $target = new \DateTime(date('m/d/Y H:i:s', $date2));
    $interval = $origin->diff($target);
    $day = $interval->format('%a');
    $r = $array_text[$day == 1 ? 0 : 1];
    return "$day $r";
}

/**
 * @return array
 */
function get_tabs_for_course_completion_status_manager() {
    global $PAGE;
    $tabs = [
        'course_completion_status_manager' => get_string('course'),
        'course_completion_status_manager_users' => get_string('users')
    ];
    $url = $PAGE->url;
    $current_tab = optional_param('active', 'course_completion_status_manager', PARAM_TEXT);
    $active_url = '';
    $output = \html_writer::start_tag('ul', ['class' => 'nav nav-tabs']);
    foreach ($tabs as $tab_url => $tab_name) {
        $active_class = '';
        if ($current_tab == $tab_url) {
            $active_url = $tab_url;
            $active_class = ' active';
        }
        $url->param('active', $tab_url);
        $output .= \html_writer::start_tag('li', ['class' => 'nav-item']);
        $output .= \html_writer::link($url, $tab_name, ['class' => 'nav-link ' . $active_class]);
        $output .= \html_writer::end_tag('li');
    }
    $output .= \html_writer::end_tag('ul');
    return [$output, basename($active_url, '.php')];
}

/**
 * Return course role names, or one name, if $rolename specifying
 *
 * @param        $courseid
 * @param string $rolename = null
 *
 * @return array|string|bool
 */
function get_role_names_by_course_and_rolename($courseid, $rolename=null) {
    global $DB;
    $course_context = \context_course::instance($courseid);
    $params = ['contextid' => $course_context->id];
    $select = 'rn.roleid';
    $join = '';
    if ($rolename) {
        $join = 'JOIN {role} r ON r.id = rn.roleid AND r.shortname = :rolename';
        $params['rolename'] = $rolename;
        $select = 'r.shortname';
    }
    $result = $DB->get_records_sql_menu(" SELECT {$select},rn.name
                FROM {role_names} rn
                {$join}
                WHERE rn.contextid = :contextid
                ", $params);
    if ($rolename){
        if (is_array($result)){
            return reset($result);
        }
        return $result;
    }
    return $result;
}

/**
 * @param $courseid
 * @param $status
 * @return bool
 */
function get_check_permission_view_grade($courseid, $status) {
    $course_metadata = block_ned_teacher_tools_get_course_metadata($courseid);
    $coursetype = $course_metadata['course_type']['export_value'] ?? '';
    if (has_capability(PLUGIN_CAPABILITY . 'seecoursegradewhilewaitingforfinalgrade', \context_course::instance($courseid))) {
        if ($coursetype == course_users_status::COURSE_TYPE_AUTO_SET_WAITING_FOR_GRADE) {
            if (in_array($status, [course_users_status::STATUS_WAITING_FOR_FINAL_GRADE, course_users_status::STATUS_WAITING_FOR_INTERNAL_REVIEW])) {
                return true;
            }
        }
    }
    return false;
}

/**
 * Return saved $load_cm_students from block settings
 * @param $courseid
 *
 * @return false|int
 */
function get_cache_load_cm_students($courseid){
    $cache = \cache::make(PLUGIN_NAME, 'block_load_cm_students');
    return $cache->get($courseid);
}

/**
 * Saved $load_cm_students in cache
 * @param $courseid
 * @param $load_cm_students
 */
function set_cache_load_cm_students($courseid, $load_cm_students){
    if (is_null($load_cm_students)){
        return;
    }
    $cache = \cache::make(PLUGIN_NAME, 'block_load_cm_students');
    $cache->set($courseid, (int)$load_cm_students);
}

/**
 * Check $load_cm_students an replace it with the cache one if necessary
 *  Return 1/0 as $load_cm_students value
 * @param $load_cm_students
 * @param $courseid
 *
 * @return null|int
 */
function check_load_cm_students_with_cache($load_cm_students, $courseid){
    if (is_null($load_cm_students)){
        $load_cm_students = get_cache_load_cm_students($courseid);
    }
    return (int)$load_cm_students;
}
