<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * NED Filter
 *
 * @package    filter
 * @subpackage NED
 * @copyright  2017 Michael Gardener mgardener@cissq.com
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace filter_ned;

defined('MOODLE_INTERNAL') || die();

class shortcode {

    protected static $shortcodetags = array();

    protected function __construct() {
        // Nothing here.
    }

    public static function add($shortcode, $callbackfunction) {
        // Redefine vars.
        $shortcode = (string) $shortcode;
        // Add new shortcode.
        if (is_callable($callbackfunction)) {
            self::$shortcodetags[$shortcode] = $callbackfunction;
        }
    }

    public static function delete($shortcode) {
        // Redefine vars.
        $shortcode = (string) $shortcode;
        // Delete shortcode.
        if (self::exists($shortcode)) {
            unset(self::$shortcodetags[$shortcode]);
        }
    }

    public static function clear() {
        self::$shortcodetags = array();
    }

    public static function exists($shortcode) {
        // Redefine vars.
        $shortcode = (string) $shortcode;
        // Check shortcode.
        return array_key_exists($shortcode, self::$shortcodetags);
    }

    public static function parse($content) {
        if (! self::$shortcodetags) {
            return $content;
        }
        $shortcodes = implode('|', array_map('preg_quote', array_keys(self::$shortcodetags)));
        $pattern    = "/(.?)\{([$shortcodes]+)(.*?)(\/)?\}(?(4)|(?:(.+?)\{\/\s*\\2\s*\}))?(.?)/s";

        return preg_replace_callback($pattern, '\filter_ned\shortcode::_handle', $content);
    }

    protected static function _handle($matches) {
        $prefix    = $matches[1];
        $suffix    = $matches[6];
        $shortcode = $matches[2];
        // Allow for escaping shortcodes by enclosing them in [[shortcode]].
        if ($prefix == '[' && $suffix == ']') {
            return substr($matches[0], 1, -1);
        }
        $attributes = array(); // Parse attributes into into this array.
        if (preg_match_all('/(\w+) *= *(?:([\'"])(.*?)\\2|([^ "\'>]+))/', $matches[3], $match, PREG_SET_ORDER)) {
            foreach ($match as $attribute) {
                if (! empty($attribute[4])) {
                    $attributes[strtolower($attribute[1])] = $attribute[4];
                } else if (! empty($attribute[3])) {
                    $attributes[strtolower($attribute[1])] = $attribute[3];
                }
            }
        }

        // Check if this shortcode realy exists then call user function else return empty string.
        return (isset(self::$shortcodetags[$shortcode])) ? $prefix .  call_user_func(self::$shortcodetags[$shortcode], $attributes, $matches[5], $shortcode) . $suffix : $matches[0];
    }
}