<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Infractions
 *
 * @package    local_academic_integrity
 * @copyright  2021 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
namespace local_academic_integrity\output;

defined('MOODLE_INTERNAL') || die();

use local_academic_integrity\ai_flag;
use local_academic_integrity\output\datatable_infractions as DI;
use local_academic_integrity\shared_lib as NED;

/** @var \stdClass $CFG */
require_once($CFG->dirroot . '/local/academic_integrity/lib.php');

/**
 * Infractions
 *
 * @package    local_academic_integrity
 * @copyright  2021 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class datatable_student_infractions implements \renderable, \templatable {

    const PAR_SCHOOL_YEAR = NED::PAR_PROFILE_SCHOOL_YEAR;

    /**
     * @var int user id.
     */
    public $userid;

    /**
     * @var int course id.
     */
    public $courseid;

    public function __construct(int $userid, int $courseid) {
        $this->userid = $userid;
        $this->courseid = $courseid;
    }

    /**
     * Export this data, so it can be used as the context for a mustache template.
     *
     * @param \renderer_base $output
     *
     * @return object
     */
    public function export_for_template(\renderer_base $output) {
        global $PAGE;

        $pageparams = [
            NED::PAR_COURSE => $this->courseid,
            NED::PAR_RETURN_URL => ($PAGE->url)->out_as_local_url(),
        ];

        $data = new \stdClass();
        $data->form = [];
        $status = ai_flag::status($this->userid);
        if ($status){
            $data->flag = ai_flag::flag($this->userid);
            $data->status = ai_flag::str_status($status);
        }

        $can_view = NED::can_view_user_ai($this->userid);
        $can_add = $can_view && NED::can_add_user_ai($this->userid, true);
        $can_view_hidden = NED::cap_view_unapproved();
        if ($can_add){
            $data->add_capability = true;
            $data->addnewinfractionurl = NED::url('~/infraction.php', $pageparams);
            $data->addnewinfractionurl->param(NED::PAR_USER, $this->userid);
            $data->addnewinfraction = NED::str('addnewinfraction');
        }

        $infractions = [];
        if ($can_view){
            $school_year = DI::SCHOOL_YEAR_CURRENT;
            $school_year_options = static::get_school_year_options();
            if (!empty($school_year_options)){
                $school_year = optional_param(static::PAR_SCHOOL_YEAR, $school_year, PARAM_INT);
                $school_year = NED::isset_key($school_year_options, $school_year, DI::SCHOOL_YEAR_CURRENT);

                $data->form[] = static::r_school_year_selector($school_year);
            }
            $infractions = DI::get_student_records($this->userid, $school_year, null, $can_view_hidden);
        }

        if (!empty($infractions)) {
            $data->showtable = 1;
            $data->can_view_hidden = $can_view_hidden;
            $data->tabledata = [];
            $header = DI::get_user_profile_columns();

            foreach ($infractions as $infraction){
                if (NED::grade_is_hidden_now_before_midn($infraction->cmid, $infraction->student)) continue;

                $row = new \stdClass();
                foreach ($header as $column) {
                    $column = $columns_header[$column] ?? $column;
                    $row->$column = DI::get_row_data($infraction, $column, $counter, $pageparams);
                }
                $data->tabledata[] = $row;
            }
        }

        $data->has_form = !empty($data->form);

        return $data;
    }

    /**
     * Get school options, like for datatable_infractions table
     * @see datatable_infractions::get_school_year_options()
     *
     * @return array
     */
    static public function get_school_year_options(){
        return is_siteadmin() ? DI::SCHOOL_YEAR_OPTIONS : [];
    }

    /**
     * Get school options selector, like for datatable_infractions table
     * @see datatable_infractions::r_school_year_selector()
     *
     * @return string
     */
    static public function r_school_year_selector($school_year=DI::SCHOOL_YEAR_CURRENT){
        $options = static::get_school_year_options();
        if (!empty($options)){
            return NED::single_select(NED::page()->url, static::PAR_SCHOOL_YEAR, DI::get_list_as_options($options),
                $school_year, NED::$C::str('ned_school_year_title'));
        }
        return '';
    }
}
