<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Infractions
 *
 * @package    local_academic_integrity
 * @copyright  2021 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
namespace local_academic_integrity\output;

use \local_academic_integrity\infraction;
use \local_academic_integrity\shared_lib as NED;

defined('MOODLE_INTERNAL') || die();

/** @var \stdClass $CFG */
require_once($CFG->dirroot . '/local/academic_integrity/lib.php');

/**
 * Infractions
 *
 * @package    local_academic_integrity
 * @copyright  2021 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class infraction_view implements \renderable, \templatable {

    /**
     * @var int id.
     */
    public $id;

    /**
     * @var int course id.
     */
    public $courseid;

    /**
     * @var string return url
     */
    public $returnurl;

    /**
     * @var \context
     */
    public $context;

    /**
     * infraction_view constructor.
     *
     * @param int|string         $id
     * @param int|string         $courseid
     * @param string|\moodle_url $returnurl
     * @param \context           $ctx
     */
    public function __construct($id, $courseid=0, $returnurl='', $ctx=null) {
        $this->id = $id;
        $this->courseid = $courseid;
        $this->returnurl = $returnurl;
        $this->context = $ctx ?? \context_system::instance();
    }

    /**
     * Export this data, so it can be used as the context for a mustache template.
     *
     * @param \renderer_base $output
     * @return \stdClass
     */
    public function export_for_template(\renderer_base $output) {
        $infraction = new \local_academic_integrity\infraction($this->id);
        $infraction->show_error_if_can_not_view();

        $data = clone $infraction->instance;

        foreach (datatable_infractions::get_table_columns(false) as $column){
            $export = false;
            if ($column == 'note'){
                $export = true;
            }
            $data->$column = datatable_infractions::get_row_data($infraction->instance, $column, $counter, null, $export);
        }

        $fs = get_file_storage();
        /** @var \stored_file[] $files */
        $files = $fs->get_directory_files(NED::ctx()->id, NED::$PLUGIN_NAME, 'uploadedfiles', $this->id, '/', false, false);
        $data->files = [];
        if (!empty($files)){
            foreach ($files as $file){
                $data->files[] = NED::file_get_link($file, true, true);
            }
        }
        $data->has_files = !empty($data->files);

        $data->editurl = NED::url('~/infraction.php',
            [NED::PAR_ID => $this->id, NED::PAR_COURSE => $this->courseid, NED::PAR_RETURN_URL => $this->returnurl]);
        $data->closeurl = new \moodle_url($this->returnurl);
        $data->edit_capability = NED::cap_manage_violations();
        NED::save_author_and_editor_to_obj($data, $data);

        $NGC = NED::$ned_grade_controller;
        $ngc_record = $NGC::get_record_by_related_id($data->id);
        if (!empty($ngc_record) && $NGC::can_see_record($ngc_record)){
            $data->ngcurl = NED::ngc_record_view_url($ngc_record->id, $this->context->contextlevel == CONTEXT_COURSE);
        }

        return $data;
    }
}
