<?php
/**
 * @package    local_academic_integrity
 * @category   NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_academic_integrity;

defined('MOODLE_INTERNAL') || die();

/**
 * Class shared_lib
 *
 * @package local_academic_integrity
 */
class shared_lib extends \local_ned_controller\shared\base_class {
    use \local_ned_controller\shared\base_trait;

    const DEFAULT_ROLE_FIELD = 'default_role';

    /**
     * @param array              $page_params
     * @param string             $title
     * @param string|\moodle_url $url
     * @param \context|null      $context
     * @param bool               $skip_capability_check
     *
     * @return \moodle_url[]|array($thispageurl, $redirecturl)
     */
    static public function setup_ai_page($page_params=[], $title='', $url=null, $context=null, $skip_capability_check=false){
        global $PAGE, $SITE;
        $url = $url ?: '~/infraction.php';
        $thispageurl = static::url($url, $page_params);
        $returnurl = $page_params[static::PAR_RETURN_URL] ?? null;
        if ($returnurl) {
            $redirecturl = new \moodle_url($returnurl);
        } else {
            $redirecturl = static::get_current_url() ?: new \moodle_url($thispageurl);
        }

        $PAGE->set_url($thispageurl);
        $courseid = $page_params[static::PAR_COURSE] ?? null;
        if (static::course_can_view_course_info($courseid)){
            require_login($courseid);
            $context = $context ?? \context_course::instance($courseid);
            $PAGE->set_pagelayout('course');
        } else {
            require_login();
            $context = $context ?? \context_system::instance();
            $PAGE->set_pagelayout('admin');
        }
        $PAGE->set_context($context);

        $name = static::str_check($title);
        // Breadcrumb.
        $PAGE->navbar->add(static::str('pluginname'));
        $PAGE->navbar->add(static::str('academicintegrityinfractions'), static::url('~/infractions.php'));
        $PAGE->navbar->add($name, $thispageurl);

        $PAGE->set_title($name);
        $PAGE->set_heading($SITE->fullname);
        $PAGE->add_body_class('academicintegrity');

        if (!$skip_capability_check){
            static::require_any_capabilities(['viewownschool', 'viewallschools']);
        }

        return array($thispageurl, $redirecturl);
    }

    /**
     * @return bool
     */
    static public function cap_view_all_schools(){
        return static::has_capability('viewallschools');
    }

    /**
     * @return bool
     */
    static public function cap_view_own_schools(){
        return static::has_capability('viewownschool');
    }

    /**
     * Add permission
     *
     * @return bool
     */
    static public function cap_add_violations(){
        return static::has_capability('addviolations');
    }

    /**
     * Edit (but not add) permission
     *
     * @return bool
     */
    static public function cap_manage_violations(){
        return static::has_capability('manageviolations');
    }

    /**
     * @return bool
     */
    static public function cap_view_unapproved(){
        return static::has_capability('viewunapproved');
    }

    /**
     * @return bool
     */
    static public function cap_edit_state(){
        return static::has_capability('changestate');
    }

    /**
     * Function checks the permissions to see user academic integrity
     * Can get academic integrity of target user if:
     *  1. User is getting own value and "Show Academic Integrity Flags to students" is On
     *  2. User has AI:viewallschools capability.
     *  3. User getting value is in same school as target user and has AI:viewownschool capability.
     *
     * @param numeric|object $user_or_id - if null, uses global USER
     *
     * @return bool
     */
    static public function can_view_user_ai($user_or_id=null){
        if (static::cap_view_all_schools()) return true;

        $userid = static::get_userid_or_global($user_or_id);
        if ($userid == static::get_userid_or_global()){
            return (bool)static::get_config('showflagtostudent');
        }

        if (!static::cap_view_own_schools()) return false;

        return static::is_in_same_school($user_or_id);
    }

    /**
     * Function checks the permissions to add (don't mess up with edit/manage) user academic integrity
     * Can add academic integrity of target user if:
     * - user can view AI
     * - has AI:addviolations capability
     *
     * @param numeric|object $user_or_id      - if null, uses global USER
     * @param bool           $skip_view_check - if true, doesn't check view capabilities
     *
     * @return bool
     */
    static public function can_add_user_ai($user_or_id=null, $skip_view_check=false){
        if (!$skip_view_check && !static::can_view_user_ai($user_or_id)) return false;

        return static::cap_add_violations();
    }

    /**
     * Function checks the permissions to view unapproved user academic integrity records
     *
     * @return bool
     */
    static public function can_view_unapproved(){
        return static::cap_view_unapproved();
    }

    /**
     * Function checks the permissions to view hidden user academic integrity records
     *
     * @return bool
     */
    static public function can_view_hidden(){
        return static::cap_view_unapproved() || static::cap_manage_violations();
    }

    /**
     * Function checks the permissions to edit state of the user academic integrity records
     *
     * @return bool
     */
    static public function can_edit_state(){
        return static::cap_edit_state();
    }

    /**
     * Function checks the permissions to edit (don't mess up with add) user academic integrity
     * Can edit academic integrity of target user if:
     * - user can view AI
     * - has AI:manageviolations capability
     *
     * @param numeric|object $user_or_id      - if null, uses global USER
     * @param bool           $skip_view_check - if true, doesn't check view capabilities
     *
     * @return bool
     */
    static public function can_edit_user_ai($user_or_id=null, $skip_view_check=false){
        if (!$skip_view_check && !static::can_view_user_ai($user_or_id)) return false;

        return static::cap_manage_violations();
    }

    /**
     * Return true, if user is student for AI plugin
     *
     * @param numeric|object|null $user_or_id - if null, uses global USER
     *
     * @return bool
     */
    static public function ai_is_student($user_or_id=null){
        return static::role_is_user_default_student($user_or_id);
    }
}

shared_lib::init();
