<?php
/**
 * @package    local_academic_integrity
 * @subpackage NED
 * @copyright  2024 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

use \local_academic_integrity\shared_lib as NED;
use \local_academic_integrity\infraction as INF;

/**
 * All upgrades checks from Moodle 3
 *
 * @param int $oldversion
 */
function local_academic_integrity_moodle3_upgrades($oldversion): void{
    global $DB;

    $dbman = $DB->get_manager();
    $AI_TABLE = 'local_academic_integrity_inf';

    if ($oldversion < 2021042601) {

        // Define table local_academic_integrity_inf to be created.
        $table = new xmldb_table($AI_TABLE);

        // Adding fields to table local_academic_integrity_inf.
        $table->add_field('id', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, XMLDB_SEQUENCE, null);
        $table->add_field('student', XMLDB_TYPE_INTEGER, '10', null, null, null, '0');
        $table->add_field('infractiondate', XMLDB_TYPE_INTEGER, '10', null, null, null, '0');
        $table->add_field('grader', XMLDB_TYPE_INTEGER, '10', null, null, null, null);
        $table->add_field('course', XMLDB_TYPE_CHAR, '255', null, null, null, null);
        $table->add_field('activitycode', XMLDB_TYPE_CHAR, '255', null, null, null, null);
        $table->add_field('activitytype', XMLDB_TYPE_CHAR, '255', null, null, null, null);
        $table->add_field('note', XMLDB_TYPE_TEXT, null, null, null, null, null);
        $table->add_field('timecreated', XMLDB_TYPE_INTEGER, '10', null, null, null, '0');
        $table->add_field('timemodified', XMLDB_TYPE_INTEGER, '10', null, null, null, '0');

        // Adding keys to table local_academic_integrity_inf.
        $table->add_key('id', XMLDB_KEY_PRIMARY, ['id']);

        // Conditionally launch create table for local_academic_integrity_inf.
        if (!$dbman->table_exists($table)) {
            $dbman->create_table($table);
        }

        // Academic_integrity savepoint reached.
        upgrade_plugin_savepoint(true, 2021042601, 'local', 'academic_integrity');
    }

    if ($oldversion < 2021072900){
        $t = 'local_academic_integrity_inf';
        $DB->delete_records($t);
        $table = new xmldb_table($t);

        $field = new xmldb_field('course');
        if ($dbman->field_exists($table, $field)) {
            $dbman->drop_field($table, $field);
        }

        $field = new xmldb_field('activitycode');
        if ($dbman->field_exists($table, $field)) {
            $dbman->drop_field($table, $field);
        }

        $field = new xmldb_field('courseid', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, null, 'id');
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        $field = new xmldb_field('cmid', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, null, 'courseid');
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        upgrade_plugin_savepoint(true, 2021072900, 'local', 'academic_integrity');
    }

    if ($oldversion < 2021081700){
        $table = new xmldb_table($AI_TABLE);
        $field = new xmldb_field('school');
        if ($dbman->field_exists($table, $field)) {
            $dbman->drop_field($table, $field);
        }

        upgrade_plugin_savepoint(true, 2022091500, 'local', 'academic_integrity');
    }

    if ($oldversion < 2023041900){
        $table = new xmldb_table($AI_TABLE);

        // add new fields
        $new_fields = ['penalty', 'reason', 'state']; // all of these fields have the same settings
        $previous_field = 'infractiondate';
        foreach ($new_fields as $field_name){
            $new_field = new xmldb_field($field_name, XMLDB_TYPE_INTEGER, '2', null, XMLDB_NOTNULL, null, 0, $previous_field);
            if (!$dbman->field_exists($table, $new_field)){
                $dbman->add_field($table, $new_field);
            }
            $previous_field = $field_name;
        }

        // transfer some data
        $old_infraction_field = new xmldb_field('infractiontype');
        if ($dbman->field_exists($table, $old_infraction_field)){
            $upd_values = [
                'minorplagiarism' => INF::PENALTY_MINOR_PLAGIARISM,
                'majorplagiarism' => INF::PENALTY_MAJOR_PLAGIARISM,
                'cheating' => INF::PENALTY_CHEATING,
            ];
            foreach ($upd_values as $old_v => $new_v){
                $DB->set_field($AI_TABLE, 'penalty', $new_v, ['infractiontype' => $old_v]);
            }

            // Remove other infraction types
            $DB->delete_records($AI_TABLE, ['penalty' => 0]);
        }

        // change default reason
        $DB->set_field($AI_TABLE, 'reason', INF::REASON_OTHER, ['reason' => 0]);

        // Change state by approved
        $old_approved_field = new xmldb_field('approved');
        if ($dbman->field_exists($table, $old_approved_field)){
            $DB->set_field($AI_TABLE, 'state', INF::ST_ACTIVE);

            // update state by NGC data
            $NGC = NED::$ned_grade_controller;
            $state_to_update = [
                INF::ST_PAUSE_SHOW => [$NGC::ST_WAIT, $NGC::ST_OBSOLETED],
                INF::ST_PAUSE_HIDE => [$NGC::ST_PAUSED, $NGC::ST_ERROR],
            ];
            foreach ($state_to_update as $state => $ngc_statuses){
                $where = $params =[];
                NED::sql_add_equal('reason', $NGC::REASON_AI, $where, $params);
                NED::sql_add_get_in_or_equal_options('status', $ngc_statuses, $where, $params);
                $ids = $DB->get_records_select_menu($NGC::TABLE, NED::sql_condition($where), $params, '', 'id, relatedid');
                if (empty($ids)) continue;

                $where = $params =[];
                NED::sql_add_get_in_or_equal_options('id', array_values($ids), $where, $params);
                $DB->set_field_select($AI_TABLE, 'state', $state, NED::sql_condition($where), $params);
            }

            $DB->set_field($AI_TABLE, 'state', INF::ST_UNAPPROVED, ['approved' => 0]);

            $records_with_none_penalty = $DB->get_records($AI_TABLE, ['state' => INF::ST_UNAPPROVED], '', 'id');
            $ngc_to_remove = [];
            foreach ($records_with_none_penalty as $record){
                $ngc_record = $NGC::get_record_by_related_id($record->id);
                if ($ngc_record){
                    $ngc_to_remove[] = $ngc_record;
                }
            }
            unset($records_with_none_penalty);

            $NGC::check_and_delete($ngc_to_remove, false);
            unset($ngc_to_remove);
        }

        // update CT field
        $ctid = new xmldb_field('ctid', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, 0, 'student');
        if (!$dbman->field_exists($table, $ctid)){
            $dbman->add_field($table, $ctid);

            // if old field specified, try to transfer some data from it
            $ct_old = new xmldb_field('ct');
            if ($dbman->field_exists($table, $ct_old)){
                // Save old CT to note
                $DB->execute("
                    UPDATE {local_academic_integrity_inf}
                    SET note = CONCAT(TRIM('\n' FROM note), '\n\n[SYSTEM] Initial specified CT: ', TRIM(ct)) 
                    WHERE TRIM(ct) <> ''
                ");

                // Try to get CT id
                $DB->execute("
                    UPDATE {local_academic_integrity_inf} inf 
                    JOIN {user} u
                        ON inf.ct <> '' AND
                        SUBSTRING_INDEX(u.firstname, ' ', 1) LIKE '[__]' AND
                        (CONCAT(u.firstname, ' ', u.lastname) LIKE CONCAT('[__] ', TRIM(inf.ct), '%') OR 
                         CONCAT(u.firstname, ' ', u.lastname) LIKE CONCAT(TRIM(inf.ct), '%')
                        ) 
                    SET inf.ctid = u.id
                ");
            }
        }

        // remove old fields
        $old_fields = ['approved', 'actiontaken', 'status', 'infractiontype', 'ct'];
        foreach ($old_fields as $field_name){
            $old_field = new xmldb_field($field_name);
            if ($dbman->field_exists($table, $old_field)){
                $dbman->drop_field($table, $old_field);
            }
        }

        // remove old ai flag data
        $ai_flag_field = $DB->get_record('user_info_field', ['shortname' => 'ai_status']);
        if ($ai_flag_field){
            $DB->delete_records('user_info_field', ['id' => $ai_flag_field->id]);
            $DB->delete_records('user_info_data', ['fieldid' => $ai_flag_field->id]);
        }

        // remove old indexes
        $old_indexes = ['mdl_lai_cou_ix' => 'student', 'mdl_lai_grad_ix' => 'grader'];
        foreach ($old_indexes as $name => $field){
            $index = new xmldb_index($name, XMLDB_INDEX_NOTUNIQUE, [$field]);
            if ($dbman->index_exists($table, $index)){
                $dbman->drop_index($table, $index);
            }
        }

        // Add new keys
        $replaced_keys = [
            'courseid' => 'course',
            'cmid' => 'course_modules',
            'student' => 'user',
            'ctid' => 'user',
            'grader' => 'user',
        ];
        foreach ($replaced_keys as $name => $key_table){
            $key = new xmldb_key($name, XMLDB_KEY_FOREIGN, [$name], $key_table, ['id']);
            // there no check for existing key in the moodle
            $dbman->add_key($table, $key);
        }

        // Add new indexes
        $new_indexes = ['penalty', 'reason', 'state'];
        foreach ($new_indexes as $name){
            $index = new xmldb_index($name, XMLDB_INDEX_NOTUNIQUE, [$name]);
            if (!$dbman->index_exists($table, $index)){
                $dbman->add_index($table, $index);
            }
        }

        // Savepoint reached
        upgrade_plugin_savepoint(true, 2023041900, 'local', 'academic_integrity');
    }

    if ($oldversion < 2023042200){
        // Change BIG INT fields precision to 10 (from 19), as in core Moodle, and set defaults to 0
        $table = new xmldb_table($AI_TABLE);

        // id is unique field, so do it separately
        $field = new xmldb_field('id', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, XMLDB_SEQUENCE, null);
        if ($dbman->field_exists($table, $field)){
            $dbman->change_field_precision($table, $field);
        }

        $field_names = ['student', 'ctid', 'infractiondate', 'grader', 'timecreated', 'timemodified'];
        // before changing field, we need to drop there indexes
        $replaced_keys = [
            'student' => 'user',
            'ctid' => 'user',
            'grader' => 'user',
        ];
        foreach ($replaced_keys as $name => $key_table){
            $index = new xmldb_index($name, XMLDB_INDEX_NOTUNIQUE, [$name]);
            if ($dbman->index_exists($table, $index)){
                $dbman->drop_index($table, $index);
            }
        }

        foreach ($field_names as $field_name){
            $field = new xmldb_field($field_name, XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, '0');
            if ($dbman->field_exists($table, $field)){
                $dbman->change_field_precision($table, $field);
                $dbman->change_field_default($table, $field);
                $dbman->change_field_notnull($table, $field);
            }
        }

        // return dropped indexes
        foreach ($replaced_keys as $name => $key_table){
            $key = new xmldb_key($name, XMLDB_KEY_FOREIGN, [$name], $key_table, ['id']);
            // there no check for existing key in the moodle
            $dbman->add_key($table, $key);
        }

        upgrade_plugin_savepoint(true, 2023042200, 'local', 'academic_integrity');
    }

    if ($oldversion < 2023052600){
        $table = new xmldb_table($AI_TABLE);
        $field_names = ['authorid', 'editorid'];
        $previous_field = 'note';
        $set_fields = [];
        $update_authors = false;
        foreach ($field_names as $field_name){
            $field = new xmldb_field($field_name, XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, 0, $previous_field);
            $previous_field = $field_name;

            if (!$dbman->field_exists($table, $field)){
                $dbman->add_field($table, $field);

                $key = new xmldb_key($field_name, XMLDB_KEY_FOREIGN, [$field_name], 'user', ['id']);
                $dbman->add_key($table, $key);

                $set_fields[] = "`$field_name` = `grader`";
                $update_authors = $update_authors || $field_name == 'authorid';
            }
        }

        if (!empty($set_fields)){
            $sql_set = join(", ", $set_fields);
            $DB->execute("UPDATE {{$AI_TABLE}} SET $sql_set");
        }
        if ($update_authors){
            // update NGC
            $ngc_table = NED::$ned_grade_controller::TABLE;
            $sql = "UPDATE {{$ngc_table}} ngc
                JOIN {{$AI_TABLE}} ai
                    ON ai.cmid = ngc.cmid 
                    AND ai.student = ngc.userid
                    AND ai.authorid <> ngc.authorid
                    AND ai.authorid > 0
                    AND ai.state <> :ai_unapproved 
                    AND ai.penalty <> :ai_deduction
                    AND ngc.status <> :ngc_error
                SET ngc.authorid = ai.authorid
            ";
            $DB->execute($sql, [
                'ai_unapproved' => INF::ST_UNAPPROVED,
                'ai_deduction' => INF::PENALTY_MINOR_PLAGIARISM,
                'ngc_error' => NED::$ned_grade_controller::ST_ERROR,
            ]);

            // update grades
            $sql = "UPDATE {grade_grades} gg
                JOIN {grade_items} gi
                    ON gi.id = gg.itemid
                    AND gi.itemtype = 'mod'
                    AND gi.itemnumber = 0
                JOIN {modules} m
                    ON m.name = gi.itemmodule 
                JOIN {course_modules} cm
                    ON cm.course = gi.courseid
                    AND cm.instance = gi.iteminstance
                    AND cm.module = m.id
                JOIN {{$AI_TABLE}} ai
                    ON ai.cmid = cm.id
                    AND ai.student = gg.userid
                    AND ai.authorid <> gg.usermodified
                    AND ai.authorid > 0
                    AND ai.state <> :ai_unapproved 
                    AND ai.penalty <> :ai_deduction
                SET gg.usermodified = ai.authorid
            ";
            $DB->execute($sql, [
                'ai_unapproved' => INF::ST_UNAPPROVED,
                'ai_deduction' => INF::PENALTY_MINOR_PLAGIARISM,
            ]);
        }

        upgrade_plugin_savepoint(true, 2023052600, 'local', 'academic_integrity');
    }
}
