<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Contains class local_kica\output\grade_summary
 *
 * @package    local_kica
 * @copyright  2018 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_kica\output;

defined('MOODLE_INTERNAL') || die();

use local_kica\kica_grade;
use local_kica\shared_lib as NED;

/**
 * Class grade_summary
 * @package local_kica\output
 */
class grade_summary implements \renderable, \templatable {
    const COLOR_GREEN = '#23b14c';
    const COLOR_BLUE = '#2f97de';
    const COLOR_LIGHT_ORANGE = '#f59c00';
    const COLOR_ORANGE = '#ea6b14';
    const COLOR_PURPLE = '#893bc3';
    const COLOR_GREY = '#c3c3c3';

    const KICA_COLORS = [
        NED::KICA_K => self::COLOR_GREEN,
        NED::KICA_I => self::COLOR_ORANGE,
        NED::KICA_C => self::COLOR_PURPLE,
        NED::KICA_A => self::COLOR_BLUE,
    ];

    protected $courseid;
    protected $userid;

    /**
     * grade_summary constructor.
     *
     * @param $courseid
     * @param $userid
     */
    public function __construct($courseid, $userid) {
        $this->courseid = $courseid;
        $this->userid = $userid;
    }

    /**
     * Set plot to show kica average course grade
     * Return kica grades
     *
     * @param numeric|object $course_or_id
     * @param array|float[]  $kica_grades - average kica grades, @see \local_kica\kica_grade::get_course_average()
     * @param string         $parent_selector (optional)
     *
     * @return \stdClass|null;
     */
    static public function plot_kica_items($course_or_id, $kica_grades, $parent_selector=''){
        if (empty($kica_grades)){
            return null;
        }

        $kica = NED::get_kica($course_or_id);
        $calc_natual = NED::is_kica_calculation_natural($kica);
        $kica_performance = new \stdClass();
        $kica_performance->stats = [];

        foreach (static::KICA_COLORS as $key => $color){
            $value = $kica_grades[$key] ?? null;
            if (!$calc_natual) {
                $max = $kica->$key ?? 0;
                $value = $max ? ($value/$max * 100) : 0;
            }
            $text = NED::str($key);
            $plot = NED::$plotly::new_plotly("$parent_selector .kica-performance .$key .plot");
            $plot->set_bar();
            $plot->add_value($value, $text, $color);
            $plot->hovertemplate = ''; // hide value by hover

            $kica_performance->stats[] = ['stat' => $key, 'text' => $text];
        }

        return $kica_performance;
    }

    /**
     * Export page contents for template
     *
     * @param \renderer_base $output
     *
     * @return \stdClass|null
     */
    public function export_for_template(\renderer_base $output) {
        if (!$this->userid) return null;

        $average_data = NED::kg_get_kica_average($this->courseid, $this->userid);
        $average_grades = NED::kg_get_course_average($this->courseid, $this->userid);
        $average_calculation = NED::kg_get_course_average_calculation($this->courseid, $average_data, $average_grades);

        if (!empty($average_calculation)){
            $average_calculation = ['kica' => $average_calculation];
        }

        NED::js_call_amd('showcalculation', 'init', [$average_calculation]);

        $data = new \stdClass();
        $data->courseid = $this->courseid;
        $data->userid = $this->userid;
        $data->notavailable = false;
        $data->is_admin = is_siteadmin();
        $data->average = NED::grade_val($average_grades[NED::FINALGRADE] ?? null, true, 2, '-');
        if (has_capability('local/kica:viewgradedisparity', \context_course::instance($this->courseid))) {
            $assign = kica_grade::get_kica_average($this->courseid, $this->userid, null, null, false, null, 'assign');
            $assignment_average_grades = NED::kg_get_course_average($this->courseid, $this->userid, null, false, $assign);
            $quiz = kica_grade::get_kica_average($this->courseid, $this->userid, null, null, false, null, 'quiz');
            $test_average_grades = NED::kg_get_course_average($this->courseid, $this->userid, null, false, $quiz);
            $data->assignments = NED::grade_val($assignment_average_grades[NED::FINALGRADE] ?? null, true, 0, '-');
            $data->tests = NED::grade_val($test_average_grades[NED::FINALGRADE] ?? null, true, 0, '-');
            $data->assigncompleted = $assign['stat']['assigncompleted'] ?? 0;
            $data->assignavailable = $assign['stat']['assignavailable'] ?? 0;
            $data->testcompleted = $quiz['stat']['testcompleted'] ?? 0;
            $data->testavailable = $quiz['stat']['testavailable'] ?? 0;
        }
        $data->kica_performance = static::plot_kica_items($this->courseid, $average_grades, '.kica-grade-summary');
        $data->plotly_add_elements = NED::$plotly::render_all(true, false, true);
        $data->contextid = (\context_course::instance($this->courseid))->id;



        return $data;
    }
}

