<?php

/**
 *
 * @package    local_kica
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_kica\output;
defined('MOODLE_INTERNAL') || die();

use block_ned_teacher_tools\output\menu_bar as MB;
use local_kica\shared_lib as NED;
use local_kica\helper as helper;

/** @var \stdClass $CFG */
require_once($CFG->dirroot.'/local/kica/lib.php');

/**
 * Class menu_bar
 *
 * @package local_kica\output
 *
 * @property-read \local_kica\output\renderer $kica_renderer
 */
class menu_bar extends MB {

    /**
     * @var \local_kica\output\renderer
     */
    protected static $_kica_renderer = null;
    /**
     * @var static
     */
    protected static $_last_this = null;
    /**
     * @var int
     */
    protected $_notification_count = null;
    protected $_incomplete_activities = [];
    protected $_grade_mismatches = [];
    protected $_grade_flagged = [];
    protected $_ungraded_activities = [];
    protected $_zero_graded_users = [];
    protected $_preload_course_grades = false;

    /**
     * Init static functions
     */
    protected function _init_static(){
        if (!static::$_kica_renderer){
            static::$_kica_renderer = NED::get_renderer();
        }

        parent::_init_static();
    }


    /**
     * Return should we show this selector in menu bar
     *
     * @param $selector
     *
     * @return bool
     */
    public function is_show_selector($selector){
        switch ($this->_page){
            case static::PAGE_KICA_SETTINGS:
                return false;
            case static::PAGE_KICA:
            case static::PAGE_KICA_ACTIVITIES:
            case static::PAGE_KICA_GRADE_ACTIVITY:
            case static::PAGE_KICA_NOTIFICATIONS:
                switch ($selector){
                    case static::MENU_STUDENT: return $this->_page == static::PAGE_KICA;
                    case static::MENU_GROUP: return true;
                    case static::MENU_SHOWINACTIVE: return !is_null($this->_show_inactive);
                    case static::MENU_CMSTUDENTS: return $this->is_show_cmstudents_checkbox();
                }
                return false;
        }
        return false;
    }

    /**
     * Render page menu for KICA pages
     *
     * @param bool $return - echo result, if false
     *
     * @return string
     */
    public function render_kica_selector_menu($return=false){
        global $OUTPUT;
        if (!$this->isteacher){
            return '';
        }

        $urls = [
            static::PAGE_KICA => 'gradebookstudentview',
            static::PAGE_KICA_GRADE_ACTIVITY => 'gradebookactivityview',
        ];
        $settings_urls = [
            static::PAGE_KICA_ACTIVITIES => 'kicaactivities',
        ];

        $add2text = [];
        $can_grade = has_capability('moodle/grade:edit', $this->context);
        if ($can_grade){
            $urls[static::PAGE_KICA_NOTIFICATIONS] = 'notifications';
            $settings_urls[static::PAGE_KICA_SETTINGS] = 'kicacoursegradesettings';

            $notifications = $this->get_notification_count();
            if ($notifications > 0){
                $add2text[static::PAGE_KICA_NOTIFICATIONS] = " ($notifications)";
            }
        }

        $links = [];
        foreach ($urls as $url => $name){
            $name = NED::str($name) . ($add2text[$url] ?? '');
            $classes = ['kica-link'];
            if ($url == $this->_page){
                $classes[] = 'active';
            }
            $links[] = NED::link($this->get_url($url), $name, $classes);
        }

        if (NED::has_capability('manage', $this->context)){
            $settings = new \action_menu();
            $settings_add = function($moodle_url, $text='', $classes=[], $link_attr=[]) use (&$settings){
                $classes[] = $text;
                $settings->add(NED::link($moodle_url, $text ?? '', join(' ', $classes), $link_attr));
            };
            $settings->set_menu_trigger(get_string('settings'));
            $settings->triggerextraclasses = 'kica-link';

            foreach ($settings_urls as $url => $name){
                $classes = ['kica-setting-link'];
                if ($url == $this->_page){
                    $classes[] = 'active';
                }
                $settings_add($this->get_url($url), $name, $classes);
            }
            $links[] = $OUTPUT->render($settings);
        }

        $output = NED::div($links, 'kica-selector-menu');
        if (!$return){
            echo $output;
        }
        return $output;
    }

    /**
     * Return count of notification from notification KICA page
     *
     * @return int
     */
    public function get_notification_count(){
        if (!$this->isteacher){
            return 0;
        }

        if (is_null($this->_notification_count)){
            $counts = [];
            list($data, $incompleteactivities) = $this->get_incomplete_activities();
            $counts[] = $incompleteactivities;
            list($data, $numberofuser) = $this->get_grade_mismatches();
            $counts[] = $numberofuser;
            list($data, $numberofuser) = $this->get_grade_flagged();
            $counts[] = $numberofuser;
            list($data, $numberofuser) = $this->get_ungraded_activities();
            $counts[] = $numberofuser;
            $zerogrades = $this->get_zero_graded_users();
            $counts[] = count($zerogrades);

            $this->_notification_count = array_sum($counts);
        }

        return $this->_notification_count;
    }

    /**
     * Set global $PAGE parameters
     *
     * @param string $add_name
     */
    public function set_page_params($add_name=null){
        global $PAGE;
        $url = $this->get_url();
        $PAGE->set_context($this->context);
        $PAGE->set_url($url);
        $title = NED::str('pluginname');
        $title .= $add_name ? (': '.$add_name) : '';
        $PAGE->navbar->add($title, $url);
        $PAGE->set_title($this->course->shortname . ': ' . $title);
        $PAGE->set_heading($this->course->fullname . ': ' . $title);
    }

    /**
     * Renders the provided widget through kica render and returns the HTML to display it.
     *
     * @param \renderable $widget instance with renderable interface
     * @param bool $return - echo rendered object, if false
     *
     * @return string
     */
    public static function render_kica($widget, $return=false){
        $out = static::$_kica_renderer->render($widget);

        if (!$return){
            echo $out;
        }
        return $out;
    }

    /* NOTIFICATIONS FEATURES */

    /**
     * @param string $class
     * @param bool   $return
     *
     * @return string
     */
    public function render_notification_page_content($class='', $return=false){
        if (!$this->isteacher){
            return '';
        }

        $output = \html_writer::start_div($class);
        $output .= $this->render_kica_selector_menu(true);
        if ($this->get_notification_count() > 0){
            $output .= $this->print_activity_errors(true);
            $output .= $this->print_ungraded_activities(true);
            $output .= $this->print_zero_graded(true);
        } else {
            $output .= NED::notification('nonotifications', NED::NOTIFY_SUCCESS);
        }
        $output .= \html_writer::end_div();

        if (!$return) {
            echo $output;
        }
        return $output;
    }

    /**
     * Preloading all users grades by the current course
     */
    protected function _preload_course_grades(){
        if ($this->_preload_course_grades) return;

        NED::kg_get_grades_by_course($this->courseid, array_keys($this->students), false, true);
        $this->_preload_course_grades = true;
    }

    /**
     * @return array - returns activity array that are incomplete
     */
    public function get_incomplete_activities()
    {
        $courseid = $this->courseid;

        if (!isset($this->_incomplete_activities[$courseid])){
            $counter = 0;
            $activities = array();

            if ($kicaitems = NED::ki_get_all_by_course($courseid)) {
                foreach ($kicaitems as $kicaitem) {
                    if ($kicaitem->incomplete) {
                        $activities[$kicaitem->id] = ['item' => $kicaitem];
                        $counter++;
                    }
                }
            }
            $this->_incomplete_activities[$courseid] = [$activities, $counter];
        }

        return $this->_incomplete_activities[$courseid];
    }


    /**
     * @return array - returns activity array and number of users that have grade mismatch
     */
    public function get_grade_mismatches() {
        global $DB;
        $courseid = $this->courseid;
        $show_inactive = (bool)$this->show_inactive;

        if (!isset($this->_grade_mismatches[$courseid][$show_inactive])){
            $kica = NED::get_kica($courseid);
            $pullfromgradebook = $kica->pullfromgradebook ?? false;
            $counter = 0;
            $activities = [];

            if (!$pullfromgradebook) {
                return [$activities, $counter];
            }

            $this->_preload_course_grades();
            $users = $this->students;
            if ($kicaitems = NED::ki_get_all_by_course($courseid)) {
                foreach ($kicaitems as $kicaitem) {
                    $controller = $kicaitem->get_grading_controller();
                    foreach ($users as $user) {
                        $kicagrade = NED::kg_get_by_userid_itemid($user->id, $kicaitem->id);
                        if (empty($kicagrade->id)) continue;

                        if (helper::is_gradingform_needupdate($kicaitem, $user->id, $controller) || $kicagrade->grade_mismatch()) {
                            if (!isset($activities[$kicaitem->id]['item'])) {
                                $activities[$kicaitem->id]['item'] = $kicaitem;
                            }

                            if ($kicaitem->itemmodule === 'quiz') {
                                if ($attempts = $DB->get_records('quiz_attempts', ['quiz' => $kicaitem->iteminstance, 'userid' => $user->id, 'state' => 'finished'], 'attempt DESC')) {
                                    $user->gradingurl = $this->get_url('/local/kicaquizgrading/index.php', [
                                        'status' => '',
                                        'course' => $kicaitem->courseid,
                                        'quiz' => $kicaitem->iteminstance,
                                        'attempt' => (reset($attempts))->id,
                                        'show' => 'student',
                                        'showquestiontext' => 0,
                                        'showmanualquestions' => 1,
                                        'showautoquestions' => 0,
                                        'showquestionswithoutkica' => 0,
                                        'hidegradedresponses' => 0
                                    ]);
                                    $user->gradingstring = get_string('openkicagrader', 'local_kicaquizgrading');
                                }
                            }

                            $activities[$kicaitem->id]['users'][$user->id] = $user;
                            $counter++;
                        }
                    }
                }
            }

            $this->_grade_mismatches[$courseid][$show_inactive] = [$activities, $counter];
        }

        return $this->_grade_mismatches[$courseid][$show_inactive];
    }

    /**
     * @return array - returns activity array and number of users that have grade flagged
     */
    public function get_grade_flagged() {
        $courseid = $this->courseid;
        $show_inactive = (bool)$this->show_inactive;

        if (!isset($this->_grade_flagged[$courseid][$show_inactive])){
            $counter = 0;
            $activities = array();
            $users = $this->students;
            $this->_preload_course_grades();

            if ($kicaitems = NED::ki_get_all_by_course($courseid)) {
                foreach ($kicaitems as $kicaitem) {
                    foreach ($users as $user) {
                        $kicagrade = NED::kg_get_by_userid_itemid($user->id, $kicaitem->id);
                        if (empty($kicagrade->id)) continue;

                        if ($kicagrade->flag) {
                            if (!isset($activities[$kicaitem->id]['item'])) {
                                $activities[$kicaitem->id]['item'] = $kicaitem;
                            }
                            $activities[$kicaitem->id]['users'][$user->id] = $user;
                            $counter++;
                        }
                    }
                }
            }

            $this->_grade_flagged[$courseid][$show_inactive] = [$activities, $counter];
        }

        return $this->_grade_flagged[$courseid][$show_inactive];
    }


    /**
     * @return array - returns activity array and number of users that have grade mismatch
     */
    public function get_ungraded_activities() {
        $courseid = $this->courseid;
        $show_inactive = (bool)$this->show_inactive;

        if (!isset($this->_ungraded_activities[$courseid][$show_inactive])){
            $counter = 0;
            $activities = array();
            $users = $this->students;
            $this->_preload_course_grades();

            if ($kicaitems = NED::ki_get_all_by_course($courseid)) {
                foreach ($kicaitems as $kicaitem) {
                    foreach ($users as $user) {
                        $kicagrade = NED::kg_get_by_userid_itemid($user->id, $kicaitem->id);
                        if (empty($kicagrade->id)) continue;

                        if ($kicagrade->is_gradable() && !$kicagrade->is_graded() && !$kicagrade->grade_mismatch() && !$kicagrade->flag) {
                            if (!isset($activities[$kicaitem->id]['item'])) {
                                $activities[$kicaitem->id]['item'] = $kicaitem;
                            }
                            $activities[$kicaitem->id]['users'][$user->id] = $user;
                            $counter++;
                        }
                    }
                }
            }

            $this->_ungraded_activities[$courseid][$show_inactive] = [$activities, $counter];
        }

        return $this->_ungraded_activities[$courseid][$show_inactive];
    }

    /**
     * @return array - returns activity array and number of users that have zero graded
     * TODO check relevance and maybe rewrite
     */
    public function get_zero_graded_users() {
        global $DB;
        $courseid = $this->courseid;
        $show_inactive = (bool)$this->show_inactive;

        if (!isset($this->_zero_graded_users[$courseid][$show_inactive])){
            $users = $this->students;
            if (empty($users)){
                return [];
            }
            list($insql, $params) = $DB->get_in_or_equal(array_keys($users));
            $params[] = $courseid;

            $sql = "SELECT gg.id,
                       k.courseid,
                       ki.id kicaitemid,
                       ki.itemid,
                       gi.itemname,
                       gi.itemtype,
                       gg.userid,
                       gg.finalgrade,
                       gg.timemodified,
                       (SELECT kg.finalgrade FROM {local_kica_grade_grades} kg WHERE kg.itemid = ki.id AND kg.userid = gg.userid) as kicafinal,
                       (SELECT g.name FROM {groups} g JOIN {groups_members} gm ON g.id = gm.groupid WHERE g.courseid = k.courseid AND gm.userid = u.id LIMIT 1) groupname,
                       u.firstname,
                       u.lastname
                  FROM {local_kica} k
                  JOIN {local_kica_grade_items} ki ON k.courseid = ki.courseid
                  JOIN {grade_items} gi ON ki.itemid = gi.id AND ki.courseid = gi.courseid
                  JOIN {grade_grades} gg ON gi.id = gg.itemid
                  JOIN {user} u ON gg.userid = u.id
                 WHERE k.enabled = 1
                   AND gi.itemtype = 'mod'
                   AND gg.finalgrade = 0
                   AND gg.excluded = 0
                   AND gg.userid {$insql}
                   AND k.courseid = ?
                HAVING kicafinal IS NULL
                ORDER BY gi.itemname, u.lastname";

            $this->_zero_graded_users[$courseid][$show_inactive] = $DB->get_records_sql($sql, $params);
        }

        return $this->_zero_graded_users[$courseid][$show_inactive];
    }

    /**
     * Print notification about activities by notification method
     *
     * @param string $method
     * @param string $title
     * @param string $page
     *
     * @return string
     */
    public function print_notification_activities($method, $title, $page=self::PAGE_KICA_GRADE_ACTIVITY){
        if (!$this->isteacher){
            return '';
        }

        list($data, $numberofuser) = $this->$method();

        if (empty($data)) {
            return '';
        }

        $param_name = $page == self::PAGE_KICA_GRADE_ACTIVITY ? 'itemid' : 'id';
        $output = NED::div(NED::str($title), 'grading-error-title');
        $url = $this->get_url($page);
        foreach ($data as $item) {
            $url->param($param_name, $item['item']->id);
            $pix = NED::img('icon', '', $item['item']->itemmodule, [], $item['item']->itemname);
            $item_name = $item['item']->itemname;
            $users = null;
            if (!empty($item['users'])){
                $users = $item['users'];
                $item_name .= ' ('.count($users).')';
            }

            $content = NED::div($pix.' '.NED::link($url, $item_name));
            $content .= $this->print_users_link_list($users);
            $output .= NED::div($content, 'grading-error-activity');
        }

        return $output;
    }

    /**
     * Print ul with user links to kica page
     *
     * @param array $users
     *
     * @return string
     */
    public function print_users_link_list($users=[]){
        if (empty($users)){
            return '';
        }

        $list = NED::$html_list::get_new_object();
        foreach ($users as $user){
            $gradinglink = '';
            if (!empty($user->gradingurl)) {
                $gradinglink = ' ' . NED::link($user->gradingurl, $user->gradingstring, 'badge badge-info');
            }
            $list->add(NED::link($this->get_url(static::PAGE_KICA, [static::PAR_SETUSER => $user->id]), fullname($user)) . $gradinglink);
        }

        return $list->render(true);
    }

    /**
     * @param bool $return
     *
     * @return string
     */
    public function print_activity_errors($return=false) {
        if (!$this->isteacher){
            return '';
        }

        $output = '';
        $output .= $this->print_notification_activities('get_incomplete_activities', 'graderangemismatch', static::PAGE_KICA_ACTIVITIES_EDIT);
        $output .= $this->print_notification_activities('get_grade_mismatches', 'gradeawardedmismatch');
        $output .= $this->print_notification_activities('get_grade_flagged', 'gradeflagged');

        if (!$return) {
            echo $output;
        }
        return $output;
    }

    /**
     * @param bool $return
     *
     * @return string
     */
    public function print_ungraded_activities($return=false) {
        if (!$this->isteacher){
            return '';
        }

        $output = $this->print_notification_activities('get_ungraded_activities', 'ungradedactivities');

        if (!$return) {
            echo $output;
        }
        return $output;
    }

    /**
     * @param bool $return
     *
     * @return string
     */
    public function print_zero_graded($return=false) {
        if (!$this->isteacher){
            return '';
        }

        $output = '';

        if (!$zerogrades = $this->get_zero_graded_users()) {
            return $output;
        }

        $output .= NED::div(NED::str('zerogradeerror', count($zerogrades)), 'grading-error-title') . ' ' .
            NED::div(NED::link($this->get_url(static::PAGE_KICA_ZEROGRADES), 'viewreport'), 'grading-error-activity');

        if (!$return) {
            echo $output;
        }
        return $output;
    }

    /**
     * @param bool $return
     *
     * @return string
     */
    public function print_grading_errors($return=false) {
        if (!$this->isteacher){
            return '';
        }

        $output = '';
        $warnings = [];
        $error_methods = ['get_incomplete_activities', 'get_grade_mismatches', 'get_grade_flagged'];
        $numberofactivities = 0;
        foreach ($error_methods as $method){
            list($data, $num) = $this->$method();
            if (!empty($data)){
                $numberofactivities += count($data);
            }
        }

        if ($numberofactivities) {
            $warnings[] = NED::link($this->get_url(static::PAGE_KICA_NOTIFICATIONS, ['view' => 'error']),
                NED::str('errorsrequireattention', $numberofactivities), 'btn btn-danger');
        }

        list($ungradedactivities, $numberofungraded) = $this->get_ungraded_activities();
        if ($ungradedactivities) {
            $warnings[] = NED::link($this->get_url(static::PAGE_KICA_NOTIFICATIONS, ['view' => 'ungraded']),
                NED::str('activitiesrequiregrade', $numberofungraded), 'btn btn-warning');
        }

        if (!empty($warnings)) {
            $output .= NED::div($warnings, 'alert alert-error alert-block grading-errors');
        }

        if (!$return) {
            echo $output;
        }
        return $output;
    }
}
