<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Lib.
 *
 * @package     local_kica
 * @category    admin
 * @copyright   2018 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

use block_ned_teacher_tools as TT;
use local_kica as kica;
use local_kica\shared_lib as NED;

defined('MOODLE_INTERNAL') || die();

// Valid istance status.
define('INSTANCE_STATUS_ACTIVE', 1);
// The grade needs to be updated by grader (usually because of changes is grading method).
define('INSTANCE_STATUS_NEEDUPDATE', 2);
// The grader started grading but did clicked neither submit nor cancel.
define('INSTANCE_STATUS_INCOMPLETE', 0);

require_once($CFG->dirroot.'/local/kica/local_lib.php');

/**
 * @param global_navigation $navigation
 * @param $course
 * @param $context
 */
function local_kica_extend_navigation_course($navigation, $course, $context) {
    global $PAGE;

    if (has_capability('moodle/course:update', $context)) {

        $keys = $navigation->get_children_key_list();
        $beforekey = null;
        $i = array_search('gradebooksetup', $keys);
        if ($i === false and array_key_exists(0, $keys)) {
            $beforekey = $keys[0];
        } else if (array_key_exists($i + 1, $keys)) {
            $beforekey = $keys[$i + 1];
        }

        $node = navigation_node::create(get_string('kicagradebooksetup', 'local_kica'),
            new moodle_url('/local/kica/coursesettings.php', array('courseid' => $PAGE->course->id)),
            navigation_node::TYPE_SETTING, null, 'local_kica_store_settings',
            new pix_icon('i/settings', ''));
        $navigation->add_node($node, $beforekey);
    }
}

/**
 * Return users and groups
 * @param      $course
 * @param bool $show_only_active
 *
 * @return array[array, array]
 */
function local_kica_get_users_and_groups($course, $show_only_active=false) {
    global $USER;
    $users = [];
    $groups = [];

    if (!$groupmode = $course->groupmode) {
        return [$users, $groups];
    }

    $context = context_course::instance($course->id);

    $aag = has_capability('moodle/site:accessallgroups', $context);
    $userfields = 'u.*';
    $orderby = null;
    $limitfrom = 0;
    $limitnum = 0;
    $additional_params = [$userfields, $orderby, $limitfrom, $limitnum, $show_only_active];

    if ($groupmode == VISIBLEGROUPS or $aag) {
        $allowedgroups = groups_get_all_groups($course->id, 0, $course->defaultgroupingid);
    } else {
        $allowedgroups = groups_get_all_groups($course->id, $USER->id, $course->defaultgroupingid);
    }

    if ($aag) {
        $users = get_enrolled_users($context, 'mod/assign:submit', 0, ...$additional_params);
    }
    foreach ($allowedgroups as $key => $group){
        $enrolledusers = get_enrolled_users($context, 'mod/assign:submit', $group->id, ...$additional_params);
        if (empty($enrolledusers)) {
            unset($allowedgroups[$key]);
        } else {
            $allowedgroups[$key]->users = $enrolledusers;
            if (!$aag){
                $users += $enrolledusers;
            }
        }
    }

    return [$users, $allowedgroups];
}

/**
 * Return groupid, show_inactive, users, groups by existing parameters and checking TT-block
 * @param      $course
 * @param null $groupid
 * @param null $show_inactive
 * @param bool $only_get_show_inactive
 *
 * @return array [int, int, array, array]
 */
function local_kica_get_group_and_students_from_tt($course, $groupid=null, $show_inactive=null, $only_get_show_inactive=false){
    global $CFG;
    $config = get_config(NED::TT);
    $tt_common_data = $CFG->dirroot . '/blocks/ned_teacher_tools/common_lib.php';
    $use_TT = ($config && (($config->version ?? 0) >= 2020020700) && file_exists($tt_common_data));

    if ($use_TT){
        require_once($tt_common_data);
        $show_inactive = TT\check_show_inactive_with_cache($show_inactive, $course->id);
    }
    $show_inactive = (int)$show_inactive;

    if ($only_get_show_inactive){
        return [$groupid, $show_inactive, [], []];
    }

    list($allusers, $groups) = local_kica_get_users_and_groups($course, !$show_inactive);

    if ($use_TT){
        $groupid = TT\check_group_with_cache($groupid, $course->id, $groups);
    } else {
        $groupid = isset($groups[$groupid]) ? $groupid : 0;
    }
    $groupid = (int)$groupid;

    return [$groupid, $show_inactive, $allusers, $groups];
}
