<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Helper
 *
 * @package    local_kicaquizgrading
 * @copyright  2020 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_kicaquizgrading;

defined('MOODLE_INTERNAL') || die();

/** @var \stdClass $CFG */
require_once($CFG->dirroot.'/lib/questionlib.php');
require_once($CFG->dirroot.'/mod/quiz/locallib.php');
require_once($CFG->dirroot.'/mod/quiz/attemptlib.php');

use local_kicaquizgrading\shared_lib as NED;
use filter_embedquestion\attempt;
use question_usage_by_activity;
use question_attempt;
use question_engine;
use quiz_attempt;
use context_course;
use html_writer;
use moodle_url;

/**
 * Class helper
 * @package local_kicaquizgrading
 */
class helper {

    const AUTO_GRADED_STATES = ['gradedwrong', 'gradedpartial', 'gradedright'];

    /**
     * @return array
     * @throws \coding_exception
     */
    public static function get_courses() {
        global $DB, $USER;

        if (is_siteadmin()) {
            $enrolledcourses = $DB->get_records('course');
        } else {
            $enrolledcourses = enrol_get_my_courses(null, 'fullname');
        }

        foreach ($enrolledcourses as $index => $enrolledcourse) {
            if (!has_capability('mod/quiz:grade', \context_course::instance($enrolledcourse->id))) {
                unset($enrolledcourses[$index]);
            } else if (!$users = self::get_gradebook_users($enrolledcourse->id)) {
                unset($enrolledcourses[$index]);
            } else {
                $enrolledcourse->users = $users;
            }
        }
        return $enrolledcourses;
    }

    /**
     * Quizzes that have KICA tagged questions
     * Returned objects contain next fields:
     * - id = quiz.id
     * - name = quiz.name
     * - cmid = course-module.id
     * - ctxid = quiz context.id
     *
     * @param int $courseid Coursse ID if zero all courses
     * @param string|null $attemptstatus Attampt status graded
     *
     * @return object[]|null
     * @throws \dml_exception
     */
    public static function get_quizzes(int $courseid, $attemptstatus='') {
        global $DB;

        $params = [];
        $quiz_module = $DB->get_field('modules', 'id', ['name' => 'quiz']);
        if (empty($quiz_module)) return null;

        if ($courseid === 0) {
            $coursefilter = '0=0';
        } else {
            $coursefilter = ' q.course = :courseid';
            $params['courseid'] = $courseid;
        }

        $sqlnumofattemtps = self::get_number_of_attempts_sql($attemptstatus);

        $params['module_id'] = $quiz_module;
        $params['component'] = 'core';
        $params['itemtype'] = 'course_modules';
        $params['tagname'] = 'kica';
        $params['pattern_name'] = '%[KICA-%';

        /**
         * @since Moodle 4 there have been a lot of changes related to the quiz tables
         * For references of query changes {@see \mod_quiz\question\bank\qbank_helper::get_question_structure()}
         */
        $quizzes = $DB->get_records_sql("
            SELECT q.id, q.name, cm.id AS cmid, ctx.id as ctxid,  
                    (SELECT COUNT(1) 
                          FROM {tag_instance} ti
                          JOIN {tag} t 
                            ON ti.tagid = t.id 
                         WHERE ti.component = :component
                           AND ti.itemtype = :itemtype
                           AND ti.itemid = cm.id
                           AND t.name = :tagname
                         LIMIT 1
                    ) iskica,
                    (SELECT COUNT(1)
                          FROM {quiz_slots} qs
                          -- Not-random questions
                          LEFT JOIN {question_references} qre
                              ON qre.itemid = qs.id
                             AND qre.usingcontextid = ctx.id AND qre.component = 'mod_quiz' AND qre.questionarea = 'slot'
                          LEFT JOIN {question_bank_entries} qbe ON qbe.id = qre.questionbankentryid
                          -- Random questions
                          LEFT JOIN {question_set_references} qsr
                              ON qsr.itemid = qs.id
                             AND qsr.usingcontextid = ctx.id AND qsr.component = 'mod_quiz' AND qsr.questionarea = 'slot'
                          LEFT JOIN {question_bank_entries} qbe_r
                              ON qbe_r.questioncategoryid = trim(BOTH '\"' FROM JSON_EXTRACT(qsr.filtercondition, '$.questioncategoryid'))
                              
                          JOIN {question_versions} qve 
                              ON qve.questionbankentryid = COALESCE(qbe.id, qbe_r.id)
                          JOIN {question} qu ON qu.id = qve.questionid
                          JOIN {question_categories} qc
                            ON qbe.questioncategoryid = qc.id
                         WHERE qs.quizid = q.id
                           AND qs.maxmark > 0
                           AND qc.name LIKE :pattern_name
                         LIMIT 1
                    ) qs_kica,
                    ({$sqlnumofattemtps} LIMIT 1) numofattempts
            FROM {quiz} q
            JOIN {course_modules} cm ON cm.instance = q.id AND cm.module = :module_id
            JOIN {context} ctx ON ctx.instanceid = cm.id
             
            WHERE {$coursefilter}
            HAVING iskica > 0 AND numofattempts > 0 AND qs_kica > 0", $params
        );

        return $quizzes ?: null;
    }
    /**
     * Quizzes that have KICA tagged questions
     *
     * @param int $courseid Coursse ID if zero all courses
     * @return array|null
     * @throws \dml_exception
     */
    public static function get_quizzes_backup(int $courseid) {
        global $DB;

        if ($courseid === 0) {
            $coursefilter = '0=0';
            $params = null;
        } else {
            $coursefilter = ' q.course = ?';
            $params = [$courseid];
        }
        $sql = "SELECT q.id, q.name,
                       (SELECT COUNT(1) 
                          FROM {quiz_slots} qs 
                          JOIN {tag_instance} ti 
                            ON qs.questionid = ti.itemid
                          JOIN {tag} t ON ti.tagid = t.id
                         WHERE qs.quizid = q.id
                           AND t.name LIKE 'kica-%' 
                           AND ti.component = 'core_question'
                           AND ti.itemtype = 'question') numofq,
                       (SELECT COUNT(1) 
                          FROM {quiz_slots} gs
                          JOIN {quiz_slot_tags} qst
                            ON gs.id = qst.slotid
                         WHERE gs.quizid = q.id
                           AND qst.tagname LIKE 'kica-%') numofslot,
                       (SELECT COUNT(1) 
                          FROM {tag_instance} ti
                          JOIN {tag} t 
                            ON ti.tagid = t.id 
                          JOIN {course_modules} cm
                            ON ti.itemid = cm.id
                          JOIN {modules} m
                            ON cm.module = m.id
                         WHERE ti.component = 'core'
                           AND ti.itemtype = 'course_modules'
                           AND t.name = 'kica'
                           AND m.name = 'quiz'
                           AND cm.instance = q.id) iskica
                  FROM {quiz} q 
                 WHERE {$coursefilter} 
                HAVING (numofq > 0 OR numofslot > 0) AND iskica > 0";
        if ($quizzes = $DB->get_records_sql($sql, $params)) {
            return $quizzes;
        }
        return null;
    }

    /**
     * @param int $quizid
     * @param string $search
     * @return array|null
     * @throws \dml_exception
     */
    public static function get_quiz_questions(int $quizid, int $attemptid, string $search) {
        global $DB, $USER, $PAGE;

        if (empty($quizid)) {
            return null;
        }

        $quiz = $DB->get_record('quiz', ['id' => $quizid], '*', MUST_EXIST);

        if (!$cm = get_coursemodule_from_instance('quiz', $quiz->id, $quiz->course)) {
            NED::print_module_error('invalidcoursemodule');
        }

        //$context = \context_module::instance($cm->id);

        $coursecontext = \context_course::instance($quiz->course);

        $params = [];
        $params['quizid'] = $quizid;
        $params['quizid2'] = $quizid;

        $filter = '';
        $filter2 = '';
        if ($search) {
            $filter = " AND " . $DB->sql_like('qu.idnumber', ':search', false, false);
            $params['search'] = '%'.$DB->sql_like_escape($search).'%';

            $filter2 = " AND " . $DB->sql_like('qu2.idnumber', ':search2', false, false);
            $params['search2'] = '%'.$DB->sql_like_escape($search).'%';
        }

        $sql = "SELECT q.*
                  FROM
                        (SELECT qs.id, qu.name, qu.questiontext, qu.idnumber, eo.graderinfo , qs.slot, qu.id questionid
                          FROM {quiz_slots} qs
                            JOIN {question_references} qre ON qre.itemid = qs.id
                            JOIN {question_bank_entries} qbe ON qbe.id = qre.questionbankentryid
                            JOIN {question_versions} qve ON qve.questionbankentryid = qbe.id
                            JOIN {question} qu ON qu.id = qve.questionid
                          JOIN {qtype_essay_options} eo
                            ON qve.questionid = eo.questionid
                          WHERE qs.quizid = :quizid
                            AND qu.qtype = 'essay'
                            AND qbe.idnumber LIKE 'W%'
                               {$filter}
        
                         UNION
        
                        SELECT qs.id, qu2.name, qu2.questiontext, qu2.idnumber, eo.graderinfo , qs.slot, qu2.id questionid
                          FROM {quiz_slots} qs
                            JOIN {question_set_references} qr ON qr.itemid = qs.id
                            JOIN {question_bank_entries} qbe ON qbe.questioncategoryid = trim(BOTH '\"' FROM JSON_EXTRACT(qr.filtercondition, '$.questioncategoryid'))
                            JOIN {question_versions} qv ON qv.questionbankentryid = qbe.id
                            JOIN {question} qu ON qu.id = qv.questionid
                            JOIN {qtype_essay_options} eo ON qu.id = eo.questionid
                          WHERE qs.quizid = :quizid2 
                            AND qu.qtype = 'essay'
                            AND qbe.idnumber LIKE 'W%'
                               {$filter2}
                        ) q
              ORDER BY q.slot";
        $questions = [];
        if ($rs = $DB->get_recordset_sql($sql, $params)) {
            foreach ($rs as $question) {

                $question->questiontext = self::question_rewrite_question_urls(
                    $question->questiontext, $coursecontext->id, 'question', 'questiontext', [$question->slot], $question->questionid);
                $question->graderinfo = self::question_rewrite_question_urls(
                    $question->graderinfo, $coursecontext->id, 'qtype_essay', 'graderinfo', [$question->slot], $question->questionid);

                $question->questiontext = format_text($question->questiontext);
                $question->graderinfo = format_text($question->graderinfo);

                $questions[$question->questionid] = $question;
            }
            $rs->close();
            return $questions;
        }
        return null;
    }

    /**
     * @param int $quizid
     * @param string|null $attemptstatus
     * @param array $users
     * @return array
     * @throws \dml_exception
     */
    public static function get_quiz_attempts(int $quizid, $attemptstatus, $users) {
        global $DB;

        if (empty($users)) {
            return null;
        }

        [$insql, $params] = $DB->get_in_or_equal($users, SQL_PARAMS_NAMED);
        $params['quizid'] = $quizid;

        $gradefilter = '';
        if (!empty($attemptstatus)) {
            if ($attemptstatus === 'graded') {
                $gradefilter = 'AND qa.sumgrades IS NOT NULL';
            } else if ($attemptstatus === 'requiresgrading') {
                $gradefilter = 'AND qa.sumgrades IS NULL';
            }
        }

        $sql = "SELECT qa.id,
                       qa.userid,
                       qa.attempt,
                       qa.uniqueid,
                       qa.preview,
                       qa.state,
                       qa.timestart,
                       qa.timefinish,
                       qa.sumgrades,
                       u.firstname,
                       u.lastname
                  FROM {quiz_attempts} qa 
                  JOIN {user} u
                    ON qa.userid = u.id
                 WHERE qa.quiz = :quizid
                   AND qa.state = 'finished' 
                   AND qa.userid {$insql}
                       {$gradefilter}
              ORDER BY u.lastname ASC, 
                       qa.attempt ASC";

        return $DB->get_records_sql($sql, $params);
    }

    /**
     * @param int $quizid
     * @return array
     * @throws \dml_exception
     */
    public static function get_quiz_slots(int $quizid) {
        global $DB;

        $sql = "SELECT qs.id,
                       qs.slot,
                       qs.questionid,
                       g.name
                  FROM {quiz_slots} qs 
                  JOIN {question} g
                    ON qs.questionid = g.id
                 WHERE qs.quizid = ?
              ORDER BY qs.slot ASC";

        return $DB->get_records_sql($sql, [$quizid]);
    }

    /**
     * @param $quizid
     * @param $slot
     * @return bool
     */
    public static function get_slot_data($quizid, $slot) {
        global $DB;
        return $DB->get_record('quiz_slots', ['quizid' => $quizid, 'slot' => $slot]);
    }
    /**
     * @param object|int $question
     * @return bool
     */
    public static function is_kica_question($question) {
        if (is_object($question)) {
            $questionid = $question->id;
        } else {
            $questionid = $question;
        }
        if (self::get_kica_tag($questionid)) {
            return true;
        }
        return false;
    }

    /**
     * @param int $questionid
     * @return array
     * @throws \dml_exception
     */
    public static function get_kica_tag($questionid) {
        global $DB;

        $sql = "SELECT qu.id, qc.name 
                  FROM {question} qu 
                  JOIN {question_categories} qc 
                    ON qu.category = qc.id 
                 WHERE qu.id = ?";

        if ($info = $DB->get_record_sql($sql, [$questionid])) {
            $text = strtoupper(trim(str_replace(' ', '', $info->name)));

            $re = '/(\[KICA-.*\])/m';

            preg_match_all($re, $text, $matches, PREG_SET_ORDER, 0);

            if (!empty($matches[0][0])) {
                return str_replace('[', '', str_replace(']', '', $matches[0][0]));
            }
        }
    }


    /**
     * It gets KICA item max grades from question tag;
     * Format KICA-K3:I2:C1:A2
     *
     * @param int $questionid
     * @param int $slotquestionid Random question ID
     * @return false|int[]
     * @throws \dml_exception
     */
    public static function get_kica_max_grades($questionid, $slotquestionid=0) {
        if ($slotquestionid) {
            $questionid = $slotquestionid;
        }
        if (!$tag = self::get_kica_tag($questionid)) {
            return false;
        }
        $tag = trim(strtoupper($tag));
        $tag = str_replace('KICA-', '', $tag);

        $maxgrades = [
            'knowledge' => 0,
            'inquiry' => 0,
            'communication' => 0,
            'application' => 0,
        ];

        $arr = explode(':', $tag);

        foreach ($arr as $item) {
            $identifier = substr($item, 0, 1);
            $maxgrade = substr($item, 1);

            switch ($identifier) {
                case "K":
                    $maxgrades['knowledge'] = $maxgrade;
                    break;
                case "I":
                    $maxgrades['inquiry'] = $maxgrade;
                    break;
                case "C":
                    $maxgrades['communication'] = $maxgrade;
                    break;
                case "A":
                    $maxgrades['application'] = $maxgrade;
                    break;
            }
        }

        return $maxgrades;
    }

    /**
     * @param $usageid
     * @param $slot
     * @return array
     * @throws \coding_exception
     */
    public static function post_process($usageid, $slot) {
        $prefix = self::get_field_prefix($usageid, $slot);
        $kicaitems = NED::KICA_KEYS;

        $errors = [];
        $posted = [];
        $totalkicascore = 0;
        $totalkicamax = 0;

        foreach ($_POST as $index => $item) {
            if (strpos($index, $prefix) !== false) {
                $var = str_replace($prefix, '', $index);
                $posted[$var] = $item;
            }
        }

        foreach ($kicaitems as $kicaitem) {
            if (isset($posted['-'.$kicaitem])) {
                $mark = \question_utils::clean_param_mark($posted['-' . $kicaitem]);
                if ($mark === null || $mark === "") {
                    $errors[$usageid . ':' . $slot][$kicaitem] = get_string('manualgradeinvalidformat', 'question');
                } else if ($posted['-max' . $kicaitem] < $mark || $mark < 0) {
                    $errors[$usageid . ':' . $slot][$kicaitem] = get_string('manualgradeoutofrange', 'question');
                } elseif (!empty($posted['-max' . $kicaitem])) {
                    $totalkicascore += $mark;
                    $totalkicamax += $posted['-max' . $kicaitem];
                }
            }
        }

        if ($totalkicamax > 0) {
            $_POST[$prefix . '-mark'] = ($totalkicascore / $totalkicamax) * $posted['-maxmark'];
        }

        return $errors;
    }
    /**
     * Get the prefix added to variable names to give field names for this
     * question attempt.
     *
     * You should not use this method directly. This is an implementation detail
     * anyway, but if you must access it, use {@link question_usage_by_activity::get_field_prefix()}.
     *
     * @return string The field name to use.
     */
    public static function get_field_prefix($usageid, $slot) {
        return 'q' . $usageid . ':' . $slot . '_';
    }

    /**
     * @param quiz_attempt $attemptobj
     * @param $slot
     * @throws \dml_exception
     */
    public static function save_kica_grades(quiz_attempt $attemptobj, $slot) {
        global $DB;

        $prefix = self::get_field_prefix($attemptobj->get_uniqueid(), $slot);
        $kicaitems =  NED::KICA_KEYS;

        /** @var question_usage_by_activity $quba */
        $quba = question_engine::load_questions_usage_by_activity($attemptobj->get_uniqueid());

        /** @var question_attempt $questionattempt */
        $questionattempt = $quba->get_question_attempt($slot);

        $id = $questionattempt->get_last_step()->get_id();

        $data = new \stdClass();
        $data->id = $id;
        foreach ($kicaitems as $kicaitem) {
            $data->{$kicaitem} = $_POST[$prefix . '-' . $kicaitem] ?? null;
        }

        $DB->update_record('question_attempt_steps', $data);

        self::update_kica_gradebook($attemptobj);
    }

    /**
     * @param quiz_attempt $attemptobj
     *
     * @return bool
     */
    public static function update_kica_gradebook(quiz_attempt $attemptobj) {
        global $DB;

        if (!class_exists('\local_kica\kica_item')) {
            return false;
        }

        $cm = $attemptobj->get_cm();
        $quiz = $attemptobj->get_quiz();

        if (!$kica = NED::get_kica_enabled($cm->course)) {
            return false;
        }

        $kicaitems =  NED::KICA_KEYS;
        $kicascores = [];
        $kicatotal = 0; // It must be equal to quiz sumgrades at the end.

        foreach ($kicaitems as $kicaitem) {
            $kicascores[$kicaitem] = 0;
        }

        /** @var question_usage_by_activity $quba */
        $quba = question_engine::load_questions_usage_by_activity($attemptobj->get_uniqueid());

        $ratio = $quiz->grade / $quiz->sumgrades;
        $slots = $attemptobj->get_slots();

        foreach ($slots as $slot) {
            /** @var question_attempt $questionattempt */
            $questionattempt = $quba->get_question_attempt($slot);

            $slotrec = helper::get_slot_data($quiz->id, $slot);
            // Ungraded question.
            $maxmark = $questionattempt->get_max_mark();
            if (empty($maxmark)) {
                continue;
            }

            $question = $questionattempt->get_question();
            $questionkicagmaxgrades  = self::get_kica_max_grades($question->id, $slotrec->questionid);
            if ($questionkicagmaxgrades === false) {
                continue;
            }
            $questionkicamax = array_sum($questionkicagmaxgrades);
            if (empty($questionkicamax)) {
                continue;
            }
            $perkicavalue = $maxmark / $questionkicamax;
            $manualgraded = self::is_manual_graded($questionattempt);
            $stepdataraw = $DB->get_record('question_attempt_steps', ['id' => $questionattempt->get_last_step()->get_id()]);

            // Auto graded question handling.
            if (!$manualgraded && $maxmark > 0) {
                $fraction = $questionattempt->get_fraction();
                foreach ($kicaitems as $kicaitem) {
                    $stepdataraw->{$kicaitem} = $fraction * $questionkicagmaxgrades[$kicaitem];
                }
                $DB->update_record('question_attempt_steps', $stepdataraw);
            }

            foreach ($kicaitems as $kicaitem) {
                $itemscore = (float) $stepdataraw->{$kicaitem};
                $kicascores[$kicaitem] += $itemscore * $perkicavalue;
            }

            $kicatotal += $maxmark;
        }

        // TODO: Compare $kicatotal and $quiz->sumgrades

        $kicaitemobj = NED::ki_get_by_cm($cm);
        if (!empty($kicaitemobj->id)) {
            $grades = array();
            foreach ($kicascores as $item => $kicascore) {
                $kikaitemmaxgrade = $kicaitemobj->get_section($item);
                if (!is_null($kikaitemmaxgrade)) {
                    $grades[$item] = $kicascore * $ratio;
                }
            }
            \local_kica\helper::save_grade($kicaitemobj->id, $attemptobj->get_userid(), $grades);

            return true;
        }

        return false;
    }

    /**
     * @param \question_attempt $questionattempt
     * @return bool
     */
    public static function is_manual_graded($questionattempt) {
        if ($questionattempt->get_behaviour_name() === 'manualgraded') {
            return true;
        }
        foreach ($questionattempt->get_step_iterator() as $step) {
            if ($step->get_state()->get_state_class(true) == 'requiresgrading') {
                return true;
            }
        }
        return false;
    }

    /**
     * @param $questionattempt
     * @return bool
     */
    public static function is_graded($questionattempt) {
        $gradedstatus = [
            'gradedpartial',
            'gradedright',
            'gradedwrong',
            'mangrright',
            'mangrwrong',
            'mangrpartial',
            'partiallycorrect',
            'correct',
        ];

        foreach ($questionattempt->get_step_iterator() as $step) {
            if ($step->get_state()->is_graded()) {
            //if (in_array($step->get_state()->get_state_class(true), $gradedstatus)) {
                return true;
            }
        }
        return false;
    }

    /**
     * It updates KICA item max values in gradebook
     *
     * @param $quizid
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function update_kica_item($quizid) {
        global $DB;

        $quiz = $DB->get_record('quiz', ['id' => $quizid]);
        $slots = $DB->get_records('quiz_slots', ['quizid' => $quizid]);
        $cm = get_coursemodule_from_instance('quiz', $quizid);

        if (!$quiz || !$slots || !$cm) {
            return;
        }

        $kicaitems =  NED::KICA_KEYS;

        $kicatotalforitem = [];
        foreach ($kicaitems as $kicaitem) {
            $kicatotalforitem[$kicaitem] = 0;
        }

        foreach ($slots as $slot) {
            if ($slot->maxmark == 0) {
                continue;
            }
            if (!$questionkica = self::get_kica_max_grades($slot->questionid, $slot->questionid)) {
                continue;
            }
            $maxkicatotal = ($slot->maxmark/$quiz->sumgrades) * $quiz->grade;
            $maxkicafromlabel = array_sum($questionkica);
            foreach ($kicaitems as $kicaitem) {
                $calculated = $questionkica[$kicaitem] * ($maxkicatotal / $maxkicafromlabel);
                $kicatotalforitem[$kicaitem] += $calculated;
            }
        }

        $kicagradeitem = NED::ki_get_by_cm($cm);
        if (!$kicagradeitem){
            return;
        } elseif (!$kicagradeitem->id){
            $kicagradeitem->kicagroup = 70;
            if (empty($kicagradeitem->itemid)) {
                return;
            }
            foreach ($kicaitems as $kicaitem) {
                $kicagradeitem->{$kicaitem} = $kicatotalforitem[$kicaitem];
            }
            $kicagradeitem->insert();
        } else {
            $updaterequired = false;
            foreach ($kicaitems as $kicaitem) {
                if (!$updaterequired && $kicagradeitem->{$kicaitem} != $kicatotalforitem[$kicaitem]) {
                    $updaterequired = true;
                }
                $kicagradeitem->{$kicaitem} = $kicatotalforitem[$kicaitem];
            }
            if ($updaterequired) {
                $kicagradeitem->update();
            }
        }
    }

    /**
     * @param $cmid
     * @return bool
     * @throws \dml_exception
     */
    public static function quiz_has_kica_tag($cmid) {
        global $DB;
        $sql = "SELECT t.* 
                  FROM {tag_instance} ti 
                  JOIN {tag} t 
                    ON ti.tagid = t.id
                 WHERE ti.component = 'core'
                   AND ti.itemtype = 'course_modules' 
                   AND ti.itemid = ? 
                   AND t.name = 'kica'";
        if ($DB->record_exists_sql($sql, [$cmid])) {
            return true;
        }
        return false;
    }

    /**
     * @param $attemptstatus
     * @return string
     */
    public static function get_number_of_attempts_sql($attemptstatus) {
        if (empty($attemptstatus)) {
            return "SELECT COUNT(1) FROM {quiz_attempts} qa WHERE qa.state = 'finished' AND qa.quiz = q.id";
        } else if ($attemptstatus === 'graded') {
            return "SELECT COUNT(1) FROM {quiz_attempts} qa WHERE qa.state = 'finished' AND qa.quiz = q.id AND qa.sumgrades IS NOT NULL";
        } else if ($attemptstatus === 'requiresgrading') {
            return "SELECT COUNT(1) FROM {quiz_attempts} qa WHERE qa.state = 'finished' AND qa.quiz = q.id AND qa.sumgrades IS NULL";
        }
    }

    /**
     * @param $courseid
     * @return array
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function get_gradebook_users($courseid, $groupid=0) {
        global $CFG, $USER;
        $course = get_course($courseid);
        $gradebookusers = array();
        $context = context_course::instance($courseid);
        foreach (explode(',', $CFG->gradebookroles) as $roleid) {
            $roleid = trim($roleid);
            if ($groupid) {
                $gradebookusers = array_merge($gradebookusers,
                    array_keys(get_role_users($roleid, $context, false, 'u.id', 'u.id ASC', null, $groupid)));
            } else if ((groups_get_course_groupmode($course) == SEPARATEGROUPS) &&
                !has_capability('moodle/site:accessallgroups', $context)) {
                $groups = groups_get_user_groups($courseid, $USER->id);
                foreach ($groups[0] as $_groupid) {
                    $gradebookusers = array_merge($gradebookusers,
                        array_keys(get_role_users($roleid, $context, false, 'u.id', 'u.id ASC', null, $_groupid)));
                }
            } else {
                $gradebookusers = array_merge($gradebookusers,
                    array_keys(get_role_users($roleid, $context, false, 'u.id', 'u.id ASC')));
            }
        }
        return $gradebookusers;
    }
    /**
     * @param $courseid
     * @return array|null
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public static function get_groups($courseid, $quizid, $status) {
        global $DB;
        $course = get_course($courseid);
        $data = null;

        $context = context_course::instance($courseid);
        if ((groups_get_course_groupmode($course) == SEPARATEGROUPS) &&
            !has_capability('moodle/site:accessallgroups', $context)) {
            $groups = groups_get_user_groups($courseid);
            if (empty($groups[0])) {
                return null;
            }
            [$insql, $params] = $DB->get_in_or_equal($groups[0]);
            $data = $DB->get_records_select('groups', "id {$insql}", $params, 'name', 'id,name');
        } else {
            $data = $DB->get_records('groups', ['courseid' => $courseid], 'name', 'id,name');
        }

        if ($data) {
            foreach ($data as $datum) {
                if (!$groupusers = helper::get_gradebook_users($courseid, $datum->id)) {
                    unset($data[$datum->id]);
                    continue;
                }
                if (!$attempts = helper::get_quiz_attempts($quizid, $status, $groupusers)) {
                    unset($data[$datum->id]);
                    continue;
                }
            }
        }

        return $data;
    }

    /**
     * @param int $questionid
     * @param question_attempt $questionattempt
     * @param int $showquestionswithoutkica
     * @param int $showmanualquestions
     * @param int $showautoquestions
     * @param int $hidegradedresponses
     * @return bool
     */
    public static function display_slot($questionid, $questionattempt, $showquestionswithoutkica, $showmanualquestions,
                                        $showautoquestions, $hidegradedresponses) {
        $show = false;

        if (!$showquestionswithoutkica && !self::is_kica_question($questionid)) {
            return false;
        }
        if ($hidegradedresponses && self::is_graded($questionattempt)) {
            return false;
        }
        if ($showmanualquestions && self::is_manual_graded($questionattempt)) {
            return true;
        }
        if ($showautoquestions && !self::is_manual_graded($questionattempt)) {
            return true;
        }
        if ($showquestionswithoutkica) {
            return true;
        }

        return $show;
    }

    /**
     * @param $attemptid
     * @param bool $modal
     * @return string|null
     * @throws \coding_exception
     * @throws \moodle_exception
     */
    public static function render_kica_grade_button($attemptid, $modal=false) {
        global $PAGE;

        [$text, $url] = self::render_kica_grade_link($attemptid);

        if (!empty($url)) {
            if ($modal) {
                $PAGE->requires->js_amd_inline("        
                    require(['jquery', 'core/modal_factory', 'core/modal_events'], function($, ModalFactory, ModalEvents) {
                        ModalFactory.create({
                            type: ModalFactory.types.SAVE_CANCEL,
                            title: '".get_string('pluginname', 'local_kicaquizgrading')."',
                            body: '".get_string('modalbody', 'local_kicaquizgrading')."',
                        }).then(function (modal) {
                            modal.setSaveButtonText('".get_string('openkicagrader', 'local_kicaquizgrading')."');
                            modal.setButtonText('cancel', '".get_string('ignore', 'local_kicaquizgrading')."');
                            var root = modal.getRoot();
                            root.on(ModalEvents.save, function() {
                                window.location.href = '" . $url->out(false) . "';
                            });
                            modal.show();
                        });
                    });          
                ");
                return true;
            }
            return html_writer::link(
                $url, $text, ['class' => 'badge badge-info', 'target' => '_blank']
            );
        }

        return null;
    }

    /**
     * @param $attemptid
     *
     * @return array - [$text, $url]
     */
    public static function render_kica_grade_link($attemptid) {
        global $DB;

        $sql = "SELECT qa.id, qa.quiz, q.course, qa.sumgrades
                  FROM {quiz_attempts} qa
                  JOIN {quiz}  q ON qa.quiz = q.id
                 WHERE qa.id = ?";

        if (!$attempt = $DB->get_record_sql($sql, [$attemptid])) {
            return [null, null];
        }

        if (!$quizzes = helper::get_quizzes($attempt->course)) {
            return [null, null];
        }

        if (!array_key_exists($attempt->quiz, $quizzes)) {
            return [null, null];
        }

        $context = context_course::instance($attempt->course);

        if (has_capability('mod/quiz:grade', $context)) {
            $url = new moodle_url('/local/kicaquizgrading/index.php', [
                'status' => '',
                'course' => $attempt->course,
                'quiz' => $attempt->quiz,
                'attempt' => $attemptid,
                'show' => 'student',
                'showquestiontext' => 0,
                'showmanualquestions' => 1,
                'showautoquestions' => 0,
                'showquestionswithoutkica' => 0,
                'hidegradedresponses' => 0,
            ]);
            $text = get_string('openkicagrader', 'local_kicaquizgrading');
        } else {
            if (is_null($attempt->sumgrades)) {
                return [null, null];
            }
            $url = new moodle_url('/local/kicaquizgrading/view.php', ['attempt' => $attemptid]);
            $text = get_string('viewkicagrades', 'local_kicaquizgrading');

        }

        return [$text, $url];
    }

    /**
     * @param $cm
     * @param $userid
     * @return false|mixed|string
     * @throws \coding_exception
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    public static function get_kica_grader_link_from_cm($cm, $userid) {
        global $DB;

        if ($cm->modname != 'quiz') {
            return false;
        }

        if (!$attepts = $DB->get_records('quiz_attempts', ['quiz' => $cm->instance, 'userid' => $userid, 'state' => 'finished'], 'timefinish DESC')) {
            return false;
        }

        $attept = reset($attepts);

        [$text, $url] = self::render_kica_grade_link($attept->id);

        if (!empty($url)) {
            return $url;
        }
        return false;
    }

    /**
     * @param array $post Posted for data
     * @param bool $single Whether it is single submission or not
     * @return array|null[]
     * @throws \dml_exception
     */
    public static function get_submission_data($post, $single=true) {
        $data = [];
        if ($single) {
            foreach ($post as $index => $item) {
                if (strpos($index, 'submit') !== false) {
                    $data[] = self::parse_submission_data($index, '');
                    break;
                }
            }
        } else {
            foreach ($post as $index => $item) {
                if (strpos($index, 'maxmark') !== false) {
                    $data[] = self::parse_submission_data($index, '-maxmark');
                }
            }
        }
        return $data;
    }

    /**
     * @param $key
     * @return array|null[]
     * @throws \dml_exception
     */
    public static function parse_submission_data($key, $name='') {
        global $DB;

        $default = [null, null, null];

        $arr = explode(':', $key);
        $uniqueid = str_replace('q', '', $arr[0]);
        $slot = str_replace('_' . $name, '', $arr[1]);

        if (!$attempt = $DB->get_record('quiz_attempts', ['uniqueid' => $uniqueid])) {
            return $default;
        }

        return [$attempt->id, $uniqueid, $slot];
    }

}