<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Contains class local_kicaquizgrading\output\renderer
 *
 * @package    local_kicaquizgrading
 * @copyright  2020 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_kicaquizgrading\output;

defined('MOODLE_INTERNAL') || die();

global $CFG;
require_once($CFG->dirroot.'/lib/questionlib.php');

use core_question_renderer;
use html_table;
use html_writer;
use local_kicaquizgrading\shared_lib as NED;
use moodle_url;
use popup_action;
use qbehaviour_manualgraded;
use qbehaviour_manualgraded_renderer;
use qbehaviour_renderer;
use qtype_essay_renderer;
use qtype_renderer;
use question_attempt;
use question_attempt_with_restricted_history;
use question_display_options;
use question_usage_by_activity;
use stdClass;
use question_utils;
use local_kicaquizgrading\helper;

/**
 * Class renderer
 * @package local_kicaquizgrading\output
 */
class renderer extends \plugin_renderer_base {
    /**
     * @param filter $filter
     * @return bool|string
     * @throws \moodle_exception
     */
    public function render_filter(filter $filter) {
        $data = $filter->export_for_template($this);
        return $this->render_from_template('local_kicaquizgrading/filter', $data);
    }

    /**
     * @param kicaquizgrading $graderinfo
     * @return bool|string
     * @throws \moodle_exception
     */
    public function render_kicaquizgrading(kicaquizgrading $graderinfo) {
        $data = $graderinfo->export_for_template($this);
        return $this->render_from_template('local_kicaquizgrading/kicaquizgrading', $data);
    }
    /**
     * @param toprightinfo $toprightinfo
     * @return bool|string
     * @throws \moodle_exception
     */
    public function render_toprightinfo(toprightinfo $toprightinfo) {
        $data = $toprightinfo->export_for_template($this);
        return $this->render_from_template('local_kicaquizgrading/toprightinfo', $data);
    }

    /**
     * Wrapper round print_question from lib/questionlib.php.
     *
     * @param \quiz_attempt $attempt
     * @param int $slot the id of a question in this quiz attempt.
     * @return string HTML of the question.
     */
    public function render_question_for_commenting($attempt, $slot, $context) {
        global $PAGE;

        $options = $attempt->get_display_options(true);
        $options->hide_all_feedback();
        if (has_capability('mod/quiz:grade', $context)) {
            $options->manualcomment = \question_display_options::EDITABLE;
        } else {
            $options->manualcomment = \question_display_options::VISIBLE;
        }
        $options->context = $context;

        $number = $attempt->get_question_number($slot);

        /** @var question_usage_by_activity $quba */
        $quba = \question_engine::load_questions_usage_by_activity($attempt->get_uniqueid());

        /** @var question_attempt $questionattempt */
        $questionattempt = $quba->get_question_attempt($slot);

        /** @var qbehaviour_manualgraded $behaviour */
        $behaviour = $questionattempt->get_behaviour(true);

        /** @var core_question_renderer $qoutput */
        $qoutput = $PAGE->get_renderer('core', 'question');

        /** @var qtype_essay_renderer $qtoutput */
        $qtoutput = $questionattempt->get_question()->get_renderer($PAGE);

        /** @var qbehaviour_manualgraded_renderer $behaviouroutput */
        $behaviouroutput = $behaviour->get_renderer($qoutput->get_page());

        $options = clone($options);
        $behaviour->adjust_display_options($options);

        return $this->question($questionattempt, $behaviouroutput, $qtoutput, $options, $number);
    }

    /**
     * Generate the display of a question in a particular state, and with certain
     * display options. Normally you do not call this method directly. Intsead
     * you call {@link question_usage_by_activity::render_question()} which will
     * call this method with appropriate arguments.
     *
     * @param question_attempt $qa the question attempt to display.
     * @param qbehaviour_renderer $behaviouroutput the renderer to output the behaviour
     *      specific parts.
     * @param qtype_renderer $qtoutput the renderer to output the question type
     *      specific parts.
     * @param question_display_options $options controls what should and should not be displayed.
     * @param string|null $number The question number to display. 'i' is a special
     *      value that gets displayed as Information. Null means no number is displayed.
     * @return string HTML representation of the question.
     */
    public function question(question_attempt $qa, qbehaviour_renderer $behaviouroutput,
                             qtype_renderer $qtoutput, question_display_options $options, $number) {

        $output = '';
        $output .= html_writer::start_tag('div', array(
            'id' => $qa->get_outer_question_div_unique_id(),
            'class' => implode(' ', array(
                'que',
                $qa->get_question(false)->get_type_name(),
                $qa->get_behaviour_name(),
                $qa->get_state_class($options->correctness && $qa->has_marks()),
            ))
        ));

        if (helper::is_graded($qa)) {
            $extracls = 'graded';
        } else {
            $extracls = 'notgraded';
        }

        $output .= html_writer::tag('div',
            $this->info($qa, $behaviouroutput, $qtoutput, $options, $number),
            array('class' => 'info'));

        $output .= html_writer::start_tag('div', array('class' => 'content'));

        $output .= html_writer::tag('div',
            $this->add_part_heading($qtoutput->formulation_heading(),
                $this->formulation($qa, $behaviouroutput, $qtoutput, $options)),
            array('class' => 'formulation clearfix'));
        $output .= html_writer::nonempty_tag('div',
            $this->add_part_heading(get_string('feedback', 'question'),
                $this->outcome($qa, $behaviouroutput, $qtoutput, $options)),
            array('class' => 'outcome clearfix'));
        $output .= html_writer::nonempty_tag('div',
            $this->add_part_heading(get_string('comments', 'question'),
                $this->manual_comment($qa, $behaviouroutput, $qtoutput, $options)),
            array('class' => 'comment clearfix '.$extracls));
        $output .= html_writer::nonempty_tag('div',
            $this->response_history($qa, $behaviouroutput, $qtoutput, $options),
            array('class' => 'history clearfix border p-2'));

        $output .= html_writer::end_tag('div');
        $output .= html_writer::end_tag('div');
        return $output;
    }
    /**
     * Generate the information bit of the question display that contains the
     * metadata like the question number, current state, and mark.
     * @param question_attempt $qa the question attempt to display.
     * @param qbehaviour_renderer $behaviouroutput the renderer to output the behaviour
     *      specific parts.
     * @param qtype_renderer $qtoutput the renderer to output the question type
     *      specific parts.
     * @param question_display_options $options controls what should and should not be displayed.
     * @param string|null $number The question number to display. 'i' is a special
     *      value that gets displayed as Information. Null means no number is displayed.
     * @return string - HTML fragment.
     */
    protected function info(question_attempt $qa, qbehaviour_renderer $behaviouroutput,
                            qtype_renderer $qtoutput, question_display_options $options, $number) {
        $output = '';
        $output .= $this->number($number);
        $output .= $this->status($qa, $behaviouroutput, $options);
        $output .= $this->mark_summary($qa, $behaviouroutput, $options);
        $output .= $this->question_flag($qa, $options->flags);
        $output .= $this->edit_question_link($qa, $options);
        return $output;
    }

    /**
     * Generate the display of the question number.
     * @param string|null $number The question number to display. 'i' is a special
     *      value that gets displayed as Information. Null means no number is displayed.
     * @return string - HTML fragment.
     */
    protected function number($number) {
        if (trim($number) === '') {
            return '';
        }
        $numbertext = '';
        if (trim($number) === 'i') {
            $numbertext = get_string('information', 'question');
        } else {
            $numbertext = get_string('questionx', 'question',
                html_writer::tag('span', $number, array('class' => 'qno')));
        }
        return html_writer::tag('h3', $numbertext, array('class' => 'no'));
    }

    /**
     * Generate the display of the status line that gives the current state of
     * the question.
     * @param question_attempt $qa the question attempt to display.
     * @param qbehaviour_renderer $behaviouroutput the renderer to output the behaviour
     *      specific parts.
     * @param question_display_options $options controls what should and should not be displayed.
     * @return string - HTML fragment.
     */
    protected function status(question_attempt $qa, qbehaviour_renderer $behaviouroutput,
                              question_display_options $options) {
        return html_writer::tag('div', $qa->get_state_string($options->correctness),
            array('class' => 'state'));
    }

    /**
     * Render the question flag, assuming $flagsoption allows it.
     *
     * @param question_attempt $qa the question attempt to display.
     * @param int $flagsoption the option that says whether flags should be displayed.
     */
    protected function question_flag(question_attempt $qa, $flagsoption) {
        global $CFG;

        $divattributes = array('class' => 'questionflag');

        switch ($flagsoption) {
            case question_display_options::VISIBLE:
                $flagcontent = $this->get_flag_html($qa->is_flagged());
                break;

            case question_display_options::EDITABLE:
                $id = $qa->get_flag_field_name();
                // The checkbox id must be different from any element name, because
                // of a stupid IE bug:
                // http://www.456bereastreet.com/archive/200802/beware_of_id_and_name_attribute_mixups_when_using_getelementbyid_in_internet_explorer/
                $checkboxattributes = array(
                    'type' => 'checkbox',
                    'id' => $id . 'checkbox',
                    'name' => $id,
                    'value' => 1,
                );
                if ($qa->is_flagged()) {
                    $checkboxattributes['checked'] = 'checked';
                }
                $postdata = \question_flags::get_postdata($qa);

                $flagcontent = html_writer::empty_tag('input',
                        array('type' => 'hidden', 'name' => $id, 'value' => 0)) .
                    html_writer::empty_tag('input', $checkboxattributes) .
                    html_writer::empty_tag('input',
                        array('type' => 'hidden', 'value' => $postdata, 'class' => 'questionflagpostdata')) .
                    html_writer::tag('label', $this->get_flag_html($qa->is_flagged(), $id . 'img'),
                        array('id' => $id . 'label', 'for' => $id . 'checkbox')) . "\n";

                $divattributes = array(
                    'class' => 'questionflag editable',
                    'aria-atomic' => 'true',
                    'aria-relevant' => 'text',
                    'aria-live' => 'assertive',
                );

                break;

            default:
                $flagcontent = '';
        }

        return html_writer::nonempty_tag('div', $flagcontent, $divattributes);
    }


    /**
     * Generate the display of the marks for this question.
     * @param question_attempt $qa the question attempt to display.
     * @param qbehaviour_renderer $behaviouroutput the behaviour renderer, which can generate a custom display.
     * @param question_display_options $options controls what should and should not be displayed.
     * @return string - HTML fragment.
     */
    protected function mark_summary(question_attempt $qa, qbehaviour_renderer $behaviouroutput, question_display_options $options) {
        global $PAGE;

        $qoutput = $PAGE->get_renderer('core', 'question');

        return html_writer::nonempty_tag('div',
            $behaviouroutput->mark_summary($qa, $qoutput, $options),
            array('class' => 'grade'));
    }

    /**
     * Work out the actual img tag needed for the flag
     *
     * @param bool $flagged whether the question is currently flagged.
     * @param string $id an id to be added as an attribute to the img (optional).
     * @return string the img tag.
     */
    protected function get_flag_html($flagged, $id = '') {
        if ($flagged) {
            $icon = 'i/flagged';
            $alt = get_string('flagged', 'question');
            $label = get_string('clickunflag', 'question');
        } else {
            $icon = 'i/unflagged';
            $alt = get_string('notflagged', 'question');
            $label = get_string('clickflag', 'question');
        }
        $attributes = array(
            'src' => $this->image_url($icon),
            'alt' => $alt,
            'class' => 'questionflagimage',
        );
        if ($id) {
            $attributes['id'] = $id;
        }
        $img = html_writer::empty_tag('img', $attributes);
        $img .= html_writer::span($label);

        return $img;
    }

    /**
     * @param question_attempt $qa
     * @param question_display_options $options
     * @return string
     * @throws \coding_exception
     * @throws \moodle_exception
     */
    protected function edit_question_link(question_attempt $qa,
                                          question_display_options $options) {
        global $CFG;

        if (empty($options->editquestionparams)) {
            return '';
        }

        $params = $options->editquestionparams;
        if ($params['returnurl'] instanceof moodle_url) {
            $params['returnurl'] = $params['returnurl']->out_as_local_url(false);
        }
        $params['id'] = $qa->get_question_id();
        $editurl = new moodle_url('/question/question.php', $params);

        return html_writer::tag('div', html_writer::link(
            $editurl, $this->pix_icon('t/edit', get_string('edit'), '', array('class' => 'iconsmall')) .
            get_string('editquestion', 'question')),
            array('class' => 'editquestion'));
    }


    /**
     * Generate the display of the formulation part of the question. This is the
     * area that contains the quetsion text, and the controls for students to
     * input their answers. Some question types also embed feedback, for
     * example ticks and crosses, in this area.
     *
     * @param question_attempt $qa the question attempt to display.
     * @param qbehaviour_renderer $behaviouroutput the renderer to output the behaviour
     *      specific parts.
     * @param qtype_renderer $qtoutput the renderer to output the question type
     *      specific parts.
     * @param question_display_options $options controls what should and should not be displayed.
     * @return string - HTML fragment.
     */
    protected function formulation(question_attempt $qa, qbehaviour_renderer $behaviouroutput,
                                   qtype_renderer $qtoutput, question_display_options $options) {
        $output = '';
        $output .= html_writer::empty_tag('input', array(
            'type' => 'hidden',
            'name' => $qa->get_control_field_name('sequencecheck'),
            'value' => $qa->get_sequence_check_count()));
        $output .= $qtoutput->formulation_and_controls($qa, $options);
        if ($options->clearwrong) {
            $output .= $qtoutput->clear_wrong($qa);
        }
        $output .= html_writer::nonempty_tag('div',
            $behaviouroutput->controls($qa, $options), array('class' => 'im-controls'));
        return $output;
    }

    /**
     * Add an invisible heading like 'question text', 'feebdack' at the top of
     * a section's contents, but only if the section has some content.
     * @param string $heading the heading to add.
     * @param string $content the content of the section.
     * @return string HTML fragment with the heading added.
     */
    protected function add_part_heading($heading, $content) {
        if ($content) {
            $content = html_writer::tag('h4', $heading, array('class' => 'accesshide')) . $content;
        }
        return $content;
    }

    /**
     * Generate the display of the outcome part of the question. This is the
     * area that contains the various forms of feedback.
     *
     * @param question_attempt $qa the question attempt to display.
     * @param qbehaviour_renderer $behaviouroutput the renderer to output the behaviour
     *      specific parts.
     * @param qtype_renderer $qtoutput the renderer to output the question type
     *      specific parts.
     * @param question_display_options $options controls what should and should not be displayed.
     * @return string - HTML fragment.
     */
    protected function outcome(question_attempt $qa, qbehaviour_renderer $behaviouroutput,
                               qtype_renderer $qtoutput, question_display_options $options) {
        $output = '';
        $output .= html_writer::nonempty_tag('div',
            $qtoutput->feedback($qa, $options), array('class' => 'feedback'));
        $output .= html_writer::nonempty_tag('div',
            $behaviouroutput->feedback($qa, $options), array('class' => 'im-feedback'));
        $output .= html_writer::nonempty_tag('div',
            $options->extrainfocontent, array('class' => 'extra-feedback'));
        return $output;
    }

    /**
     * @param question_attempt $qa
     * @param qbehaviour_renderer $behaviouroutput
     * @param qtype_renderer $qtoutput
     * @param question_display_options $options
     * @return string
     */
    protected function manual_comment(question_attempt $qa, qbehaviour_renderer $behaviouroutput,
                                      qtype_renderer $qtoutput, question_display_options $options) {
        return $qtoutput->manual_comment($qa, $options) .
            $this->behaviour_output_manual_comment($qa, $options);
    }

    /**
     * Display the manual comment, and a link to edit it, if appropriate.
     *
     * @param question_attempt $qa a question attempt.
     * @param question_display_options $options controls what should and should not be displayed.
     * @return string HTML fragment.
     */
    public function behaviour_output_manual_comment(question_attempt $qa, question_display_options $options) {
        if ($options->manualcomment == question_display_options::EDITABLE) {
            return $this->manual_comment_fields($qa, $options);

        } else if ($options->manualcomment == question_display_options::VISIBLE) {
            return $this->manual_comment_view($qa, $options);
        } else {
            return '';
        }
    }

    /**
     * @param question_attempt $qa
     * @param question_display_options $options
     * @return string
     * @throws \coding_exception
     * @throws \dml_exception
     */
    public function manual_comment_fields(question_attempt $qa, question_display_options $options) {
        global $CFG, $OUTPUT, $kicaformerrors;

        $errors = null;
        $key = $qa->get_usage_id() . ':' . $qa->get_slot();
        if (!empty($kicaformerrors[$key])) {
            $errors = $kicaformerrors[$key];
        }
        $cm = get_coursemodule_from_id('quiz', $options->context->instanceid);
        $slotrec = \local_kicaquizgrading\helper::get_slot_data($cm->instance, $qa->get_slot());

        require_once($CFG->dirroot.'/lib/filelib.php');
        require_once($CFG->dirroot.'/repository/lib.php');

        $inputname = $qa->get_behaviour_field_name('comment');
        $id = $inputname . '_id';
        list($commenttext, $commentformat, $commentstep) = $qa->get_current_manual_comment();

        $editor = editors_get_preferred_editor($commentformat);
        $strformats = format_text_menu();
        $formats = $editor->get_supported_formats();
        foreach ($formats as $fid) {
            $formats[$fid] = $strformats[$fid];
        }

        $draftitemareainputname = $qa->get_behaviour_field_name('comment:itemid');
        $draftitemid = optional_param($draftitemareainputname, false, PARAM_INT);

        if (!$draftitemid && $commentstep === null) {
            $commenttext = '';
            $draftitemid = file_get_unused_draft_itemid();
        } else if (!$draftitemid) {
            list($draftitemid, $commenttext) = $commentstep->prepare_response_files_draft_itemid_with_text(
                'bf_comment', $options->context->id, $commenttext);
        }

        $editor->set_text($commenttext);
        $editor->use_editor($id, question_utils::get_editor_options($options->context),
            question_utils::get_filepicker_options($options->context, $draftitemid));

        $commenteditor = html_writer::tag('div', html_writer::tag('textarea', s($commenttext),
            array('id' => $id, 'name' => $inputname, 'rows' => 10, 'cols' => 60)));

        $attributes = ['type'  => 'hidden', 'name'  => $draftitemareainputname, 'value' => $draftitemid];
        $commenteditor .= html_writer::empty_tag('input', $attributes);

        $editorformat = '';
        if (count($formats) == 1) {
            reset($formats);
            $editorformat .= html_writer::empty_tag('input', array('type' => 'hidden',
                'name' => $inputname . 'format', 'value' => key($formats)));
        } else {
            $editorformat = html_writer::start_tag('div', array('class' => 'fitem'));
            $editorformat .= html_writer::start_tag('div', array('class' => 'fitemtitle'));
            $editorformat .= html_writer::tag('label', get_string('format'), array('for'=>'menu'.$inputname.'format'));
            $editorformat .= html_writer::end_tag('div');
            $editorformat .= html_writer::start_tag('div', array('class' => 'felement fhtmleditor'));
            $editorformat .= html_writer::select($formats, $inputname.'format', $commentformat, '');
            $editorformat .= html_writer::end_tag('div');
            $editorformat .= html_writer::end_tag('div');
        }

        $comment = html_writer::tag('div', html_writer::tag('div',
                html_writer::tag('label', get_string('comment', 'question'),
                    array('for' => $id)), array('class' => 'fitemtitle')) .
            html_writer::tag('div', $commenteditor, array('class' => 'felement fhtmleditor', 'data-fieldtype' => "editor")),
            array('class' => 'fitem'));
        $comment .= $editorformat;

        $mark = '';

        if ($qa->get_max_mark()) {
            $manualgraded = helper::is_manual_graded($qa);
            $kicaitems =  NED::KICA_KEYS;
            $kicaitems[] = 'mark';

            $maxgrades = helper::get_kica_max_grades($qa->get_question()->id, $slotrec->questionid);
            if ($maxgrades === false) {
                return '';
            }
            $maxgrades['mark'] = $qa->get_max_mark();

            $range = html_writer::empty_tag('input', array(
                    'type' => 'hidden',
                    'name' => $qa->get_control_field_name('minfraction'),
                    'value' => $qa->get_min_fraction(),
                )) . html_writer::empty_tag('input', array(
                    'type' => 'hidden',
                    'name' => $qa->get_control_field_name('maxfraction'),
                    'value' => $qa->get_max_fraction(),
                ));

            $mark = html_writer::start_div('row mt-3');
            foreach ($kicaitems as $kicaitem) {
                $fieldsize = strlen($qa->format_max_mark($options->markdp)) - 1;
                $markfield = $qa->get_behaviour_field_name($kicaitem);

                $maxmark = $maxgrades[$kicaitem];
                $name = $qa->get_behaviour_field_name('max'.$kicaitem);

                if ($kicaitem == 'mark') {
                    $max = $qa->format_max_mark($options->markdp);
                    $currentmark = $qa->get_current_manual_mark();
                    $error = $qa->validate_manual_mark($currentmark);
                    $attributes = array(
                        'type' => 'text',
                        'size' => $fieldsize,
                        'name' => $markfield,
                        'id'=> $markfield,
                        'disabled' => 'disabled',
                        'mark' => 'mark'
                    );
                } else {
                    $max = format_float($maxmark, $options->markdp);
                    $currentmark = $this->get_current_manual_mark($qa, $kicaitem);
                    $error = $errors[$kicaitem] ?? '';
                    $attributes = array(
                        'type' => 'text',
                        'size' => $fieldsize,
                        'name' => $markfield,
                        'id'=> $markfield
                    );
                    if ($max > 0) {
                        $attributes['max'] = 'max';
                    } else {
                        $attributes['max'] = 'zero';
                        $attributes['disabled'] = 'disabled';
                    }
                }

                $errorclass = '';
                if ($error !== '') {
                    $erroclass = ' error';
                    $error = html_writer::tag('span', $error,
                            array('class' => 'error')) . html_writer::empty_tag('br');
                }

                $markrange = html_writer::empty_tag('input', array(
                        'type' => 'hidden',
                        'name' => $name,
                        'value' => $maxmark,
                    ));

                if (!is_null($currentmark)) {
                    $attributes['value'] = $currentmark;
                }

                $a = new stdClass();
                $a->max = $max;
                $a->mark = html_writer::empty_tag('input', $attributes);

                if ($max > 0) {
                    $wrapper = html_writer::tag('div', $error . get_string('xoutofmax', 'question', $a) .
                        $markrange, array('class' => 'felement ftext' . $errorclass)
                    );
                } else {
                    $wrapper = html_writer::tag('div', $error . $a->mark .
                        $markrange, array('class' => 'felement ftext' . $errorclass)
                    );
                }

                if ($manualgraded) {
                    $mark .= html_writer::tag('div', html_writer::tag('div',
                            html_writer::tag('label', get_string($kicaitem, 'local_kicaquizgrading'),
                                array('for' => $markfield)),
                            array('class' => 'fitemtitle')) .
                        $wrapper, array('class' => ' grade-input max fitemx col-md-2'));
                } else {
                    $attributes = array('class' => 'fitemy col-md-2');
                    if ($kicaitem == 'mark') {
                        $attributes['mark'] = 'mark';
                    } else if ($max > 0) {
                        $attributes['max'] = 'max';
                    } else {
                        $attributes['max'] = 'zero';
                    }
                    if ($max > 0) {
                        $mark .= html_writer::tag('div',
                            html_writer::tag('span', get_string($kicaitem, 'local_kicaquizgrading') . ': ' . $currentmark . ' / ' . $maxmark),
                            $attributes
                        );
                    }
                }
            }
            $mark .= $OUTPUT->render_from_template('local_kicaquizgrading/submitbutton',
                ['submitname' => $qa->get_control_field_name('submit')]);
            $mark .= $range;
            $mark .= html_writer::end_div();
        }

        return html_writer::tag('fieldset', html_writer::tag('div', $comment . $mark,
            array('class' => 'fcontainer clearfix')), array('class' => 'hidden'));
    }

    /**
     * @param question_attempt $qa
     * @param question_display_options $options
     * @return string
     * @throws \coding_exception
     * @throws \moodle_exception
     */
    public function manual_comment_view(question_attempt $qa, question_display_options $options) {
        global $CFG, $OUTPUT;

        $output = '';
        if ($qa->has_manual_comment()) {
            $output .= get_string('commentx', 'question',
                $qa->get_behaviour(false)->format_comment(null, null, $options->context));
        }
        if ($options->manualcommentlink) {
            $url = new moodle_url($options->manualcommentlink, array('slot' => $qa->get_slot()));
            $link = $this->output->action_link($url, get_string('commentormark', 'question'),
                new popup_action('click', $url, 'commentquestion',
                    array('width' => 600, 'height' => 800)));
            $output .= html_writer::tag('div', $link, array('class' => 'commentlink'));
        }

        $cm = get_coursemodule_from_id('quiz', $options->context->instanceid);
        $slotrec = \local_kicaquizgrading\helper::get_slot_data($cm->instance, $qa->get_slot());
        $mark = '';

        if ($qa->get_max_mark()) {
            $kicaitems =  NED::KICA_KEYS;
            $kicaitems[] = 'mark';

            $maxgrades = helper::get_kica_max_grades($qa->get_question()->id, $slotrec->questionid);
            if ($maxgrades === false) {
                return '';
            }
            $maxgrades['mark'] = $qa->get_max_mark();

            $mark = html_writer::start_div('row mt-3');
            foreach ($kicaitems as $kicaitem) {
                $maxmark = $maxgrades[$kicaitem];

                if ($kicaitem == 'mark') {
                    $max = $qa->format_max_mark($options->markdp);
                    $currentmark = $qa->get_current_manual_mark();
                } else {
                    $max = format_float($maxmark, $options->markdp);
                    $currentmark = $this->get_current_manual_mark($qa, $kicaitem);
                }

                $mark .= $OUTPUT->render_from_template('local_kicaquizgrading/kicagradeitemview', [
                    'item' => get_string($kicaitem, 'local_kicaquizgrading'),
                    'currentmark' => $currentmark,
                    'maxmark' => round($max, 2),
                ]);
            }

            $mark .= html_writer::end_div();
        }





        return $output. $mark;
    }

    /**
     * Generate the display of the response history part of the question. This
     * is the table showing all the steps the question has been through.
     *
     * @param question_attempt $qa the question attempt to display.
     * @param qbehaviour_renderer $behaviouroutput the renderer to output the behaviour
     *      specific parts.
     * @param qtype_renderer $qtoutput the renderer to output the question type
     *      specific parts.
     * @param question_display_options $options controls what should and should not be displayed.
     * @return string - HTML fragment.
     */
    protected function response_history(question_attempt $qa, qbehaviour_renderer $behaviouroutput,
                                        qtype_renderer $qtoutput, question_display_options $options) {

        if (!$options->history) {
            return '';
        }

        $table = new html_table();
        $table->head  = array (
            get_string('step', 'question'),
            get_string('time'),
            get_string('action', 'question'),
            get_string('state', 'question'),
        );
        if ($options->marks >= question_display_options::MARK_AND_MAX) {
            $table->head[] = get_string('marks', 'question');
        }

        foreach ($qa->get_full_step_iterator() as $i => $step) {
            $stepno = $i + 1;

            $rowclass = '';
            if ($stepno == $qa->get_num_steps()) {
                $rowclass = 'current';
            } else if (!empty($options->questionreviewlink)) {
                $url = new moodle_url($options->questionreviewlink,
                    array('slot' => $qa->get_slot(), 'step' => $i));
                $stepno = $this->output->action_link($url, $stepno,
                    new popup_action('click', $url, 'reviewquestion',
                        array('width' => 450, 'height' => 650)),
                    array('title' => get_string('reviewresponse', 'question')));
            }

            $restrictedqa = new question_attempt_with_restricted_history($qa, $i, null);
            $row = array(
                $stepno,
                userdate($step->get_timecreated(), get_string('strftimedatetimeshort')),
                s($qa->summarise_action($step)),
                $restrictedqa->get_state_string($options->correctness),
            );

            if ($options->marks >= question_display_options::MARK_AND_MAX) {
                $row[] = $qa->format_fraction_as_mark($step->get_fraction(), $options->markdp);
            }

            $table->rowclasses[] = $rowclass;
            $table->data[] = $row;
        }

        return html_writer::tag('h4', get_string('responsehistory', 'question'),
                array('class' => 'responsehistoryheader')) .
            $options->extrahistorycontent .
            html_writer::tag('div', html_writer::table($table, true),
                array('class' => 'responsehistoryheader'));
    }

    /**
     * This is used by the manual grading code, particularly in association with
     * validation. It gets the current manual mark for a question, in exactly the string
     * form that the teacher entered it, if possible. This may come from the current
     * POST request, if there is one, otherwise from the database.
     *
     * @return string the current manual mark for this question, in the format the teacher typed,
     *     if possible.
     */
    public function get_current_manual_mark(question_attempt $qa, $kicaitem) {
        global $DB;
        // Is there a current value in the current POST data? If so, use that.
        $mark = $qa->get_submitted_var($qa->get_behaviour_field_name($kicaitem), PARAM_RAW_TRIMMED);
        if ($mark !== null) {
            return $mark;
        }

        // Otherwise, use the stored value.
        // If the question max mark has not changed, use the stored value that was input.
        $storedmaxmark = $qa->get_last_behaviour_var('max'.$kicaitem);

        if (!$laststep = $qa->get_last_step()) {
            return null;
        }

        if (!$laststepraw = $DB->get_record('question_attempt_steps', ['id' => $laststep->get_id()])) {
            return null;
        }

        // The max mark for this question has changed so we must re-scale the current mark.
        return format_float($laststepraw->{$kicaitem}, 7, true, true);
    }
}
