<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Quiz grader.
 *
 * @package   local_kicaquizgrading
 * @copyright 2020 Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html gnu gpl v3 or later
 */

use local_kicaquizgrading\helper;
use local_kicaquizgrading\shared_lib as NED;

require_once(dirname(__FILE__, 3) . '/config.php');
require_once($CFG->dirroot . '/mod/quiz/locallib.php');

$status = optional_param('status', 'requiresgrading', PARAM_TEXT);
$courseid = optional_param('course', 0, PARAM_INT);
$quizid = optional_param('quiz', 0, PARAM_INT);
$slot = optional_param('slot', false, PARAM_INT);
$attemptid = optional_param('attempt', 0, PARAM_INT);
$showquestiontext = optional_param('showquestiontext', null, PARAM_INT);
$showmanualquestions = optional_param('showmanualquestions', null, PARAM_INT);
$showautoquestions = optional_param('showautoquestions', null, PARAM_INT);
$showquestionswithoutkica = optional_param('showquestionswithoutkica', null, PARAM_INT);
$hidegradedresponses = optional_param('hidegradedresponses', 1, PARAM_INT);
$group = optional_param('group', null, PARAM_INT);
$show = optional_param('show', 'student', PARAM_TEXT);
$slotnum = optional_param('slotnum', 0, PARAM_INT);

$search = optional_param('search', '', PARAM_TEXT);

require_login(null, false);

$contextsystem = context_system::instance();

if (empty($courseid)) {
    $quizid = 0;
}

if (isset($SESSION->graderinfocourse)) {
    if ($SESSION->graderinfocourse != $courseid) {
        $quizid = 0;
    }
}
$SESSION->graderinfocourse = $courseid;

if (!empty($quizid)) {
    $cm = get_coursemodule_from_instance('quiz', $quizid, $courseid);
    $context = context_module::instance($cm->id);
    require_capability('mod/quiz:grade', $context);
}

$thispageurl = new moodle_url('/local/kicaquizgrading/index.php');

$PAGE->set_url($thispageurl);
$PAGE->set_pagelayout('course');
$PAGE->set_context($contextsystem);

$name = get_string('pluginname', 'local_kicaquizgrading');
$title = get_string('pluginname', 'local_kicaquizgrading');
$heading = $SITE->fullname;

// Breadcrumb.
$PAGE->navbar->add($title, $thispageurl);

$PAGE->set_title($title);
$PAGE->set_heading($heading);

$renderer = $PAGE->get_renderer('local_kicaquizgrading');

echo $OUTPUT->header();
echo html_writer::tag('h1', $title, array('class' => 'page-title'));

if (!get_config('local_kicaquizgrading', 'enable')) {
    echo html_writer::div(get_string('disabledpluginwarning', 'local_kicaquizgrading'), 'alert alert-warning');
    echo $OUTPUT->footer();
    die;
}

if (!empty($cm)) {
    $toprightinfo = new local_kicaquizgrading\output\toprightinfo($cm);
    echo $renderer->render($toprightinfo);
}

$filter = new local_kicaquizgrading\output\filter($courseid, $quizid, $attemptid, $showquestiontext, $showmanualquestions,
    $showautoquestions, $status, $showquestionswithoutkica, $hidegradedresponses, $group, $show, $slotnum);
echo $renderer->render($filter);

if (empty($cm) || (empty($attemptid) && ($show === 'student')) || (empty($slotnum) && ($show === 'question'))) {
    echo $OUTPUT->footer();
    die;
}

$output = $PAGE->get_renderer('mod_quiz');

/** @var \local_kicaquizgrading\output\renderer $renderer */
$renderer = $PAGE->get_renderer('local_kicaquizgrading');

$kicaformerrors = [];

// Process any data that was submitted.
if (data_submitted() && confirm_sesskey()) {
    if (isset($_POST['submitall'])) {
        // All questions
        $data = helper::get_submission_data($_POST, false);
    } else {
        // Single question.
        $data = helper::get_submission_data($_POST, true);
    }

    if ($data) {
        helper::update_kica_item($quizid);
        foreach ($data as $list) {
            list($qattemptid, $quniqueid, $qslot) =  $list;
            $_POST['slots'] = $qslot;
            if (!empty($qattemptid) && !empty($quniqueid) && !empty($qslot)) {
                try {
                    $attemptobj = quiz_create_attempt_handling_errors($qattemptid, $cm->id);
                    $questionattempt = $attemptobj->get_question_attempt($qslot);
                    $slotrec = helper::get_slot_data($quizid, $qslot);
                    if (!helper::is_kica_question($slotrec->questionid) || !helper::is_manual_graded($questionattempt)) {
                        continue;
                    }
                    if ((!$errors = helper::post_process($quniqueid, $qslot))
                        && question_engine::is_manual_grade_in_range($quniqueid, $qslot)) {
                        $transaction = $DB->start_delegated_transaction();
                        $attemptobj->process_submitted_actions(time());
                        $transaction->allow_commit();
                        helper::save_kica_grades($attemptobj, $qslot);
                        // Log this action.
                        $params = array(
                            'objectid' => $attemptobj->get_question_attempt($qslot)->get_question_id(),
                            'courseid' => $attemptobj->get_courseid(),
                            'context' => context_module::instance($attemptobj->get_cmid()),
                            'other' => array(
                                'quizid' => $attemptobj->get_quizid(),
                                'attemptid' => $attemptobj->get_attemptid(),
                                'slot' => $qslot
                            )
                        );
                        $event = \mod_quiz\event\question_manually_graded::create($params);
                        $event->trigger();
                    }
                    $kicaformerrors += $errors;
                } catch (moodle_exception $e) {
                    echo $output->notification(get_string('cannotsavedata', 'error'), 'notifysuccess');
                    echo $OUTPUT->footer();
                    die;
                }
            }
        }
        echo $output->notification(get_string('changessaved'), 'notifysuccess');
    }
}


if ($show === 'question') {
    $attempts = null;
    $users = [];
    if ($courseid) {
        if ($group) {
            $users = helper::get_gradebook_users($courseid, $group);
        } else {
            $users = helper::get_gradebook_users($courseid);
        }
    }

    if ($quizid && !$attempts = helper::get_quiz_attempts($quizid, $status, $users)) {
        echo $OUTPUT->footer();
        die;
    }

    if ($attempts) {
        // Sort Attempts by question id.
        foreach ($attempts as $attempt) {
            /** @var question_usage_by_activity $quba */
            $quba = \question_engine::load_questions_usage_by_activity($attempt->uniqueid);
            /** @var question_attempt $questionattempt */
            $questionattempt = $quba->get_question_attempt($slotnum);

            $question = $questionattempt->get_question();
            $attempt->questionid = $question->id;
        }

        usort($attempts, function($a, $b) {
            return strcmp($a->questionid, $b->questionid);
            }
        );

        // Form start.
        $data = new stdClass();
        $data->formurl = (new moodle_url('/local/kicaquizgrading/index.php'))->out();
        $data->slot = 0;
        $data->course = $courseid;
        $data->quiz = $quizid;
        $data->attemptid = 0;
        $data->sesskey = sesskey();
        $data->showquestiontext = $showquestiontext;
        $data->showmanualquestions = $showmanualquestions;
        $data->showautoquestions = $showautoquestions;
        $data->showquestionswithoutkica = $showquestionswithoutkica;
        $data->hidegradedresponses = $hidegradedresponses;
        $data->group = $group;
        $data->status = $status;
        $data->show = $show;
        $data->slotnum = $slotnum;

        echo $OUTPUT->render_from_template('local_kicaquizgrading/gradingformstart', $data);

        foreach ($attempts as $attempt) {
            try {
                $attemptobj = quiz_create_attempt_handling_errors($attempt->id, $cm->id);
            } catch (moodle_exception $e) {
                echo $OUTPUT->footer();
                die;
            }

            // Can only grade finished attempts.
            if (!$attemptobj->is_finished()) {
                NED::print_module_error('attemptclosed', 'quiz');
            }

            $student = $DB->get_record('user', array('id' => $attemptobj->get_userid()));
            $hidegrades = $hidegradedresponses || NED::grade_is_hidden_now_before_midn($cm, $student->id);

            $slots = $attemptobj->get_slots();

            /** @var question_usage_by_activity $quba */
            $quba = \question_engine::load_questions_usage_by_activity($attemptobj->get_uniqueid());

            if (in_array($slotnum, $slots)) {
                /** @var question_attempt $questionattempt */
                $questionattempt = $quba->get_question_attempt($slotnum);

                $question = $questionattempt->get_question();
                $slotrec = helper::get_slot_data($quizid, $slotnum);

                $display = helper::display_slot($slotrec->questionid, $questionattempt, $showquestionswithoutkica, $showmanualquestions,
                    $showautoquestions, $hidegrades);

                if (!$display) {
                    continue;
                }
                // Prepare summary information about this question attempt.
                $summarydata = array();

                // Student name.
                $userpicture = new user_picture($student);
                $userpicture->courseid = $attemptobj->get_courseid();
                $summarydata['user'] = array(
                    'title' => $userpicture,
                    'content' => new action_link(new moodle_url('/user/view.php', array(
                        'id' => $student->id, 'course' => $attemptobj->get_courseid())),
                        fullname($student, true)),
                );

                // Quiz name.
                $summarydata['quizname'] = array(
                    'title' => get_string('modulename', 'quiz'),
                    'content' => format_string($attemptobj->get_quiz_name()),
                );

                // Add link to Grade controller
                $cmid = $attemptobj->get_cm()->id;
                $studentid = $student->id;
                $records = NED::$ned_grade_controller::get_records_by_params($cmid, $studentid);
                if (!empty($records)){
                    $ngc_record = reset($records);
                    $ngc_text = NED::$ned_grade_controller::get_see_text($ngc_record);
                    if (!empty($ngc_text)){
                        $summarydata['quizname']['content'] .=
                            '('.NED::$C::link([NED::PAGE_GRADE_CONTROLLER, [NED::PAR_ID => $ngc_record->id]], $ngc_text).')';
                    }
                }

                // Question name.
                $summarydata['questionname'] = array(
                    'title' => get_string('question', 'quiz'),
                    'content' => $attemptobj->get_question_name($slotnum),
                );

                // Print quiz information.
                echo $output->review_summary_table($summarydata, 0);
                echo $renderer->render_question_for_commenting($attemptobj, $slotnum, $context);
            }
        }
        echo $OUTPUT->render_from_template('local_kicaquizgrading/gradingformend', []);
    }
} else {

    try {
        $attemptobj = quiz_create_attempt_handling_errors($attemptid, $cm->id);
    } catch (moodle_exception $e) {
        echo $OUTPUT->footer();
        die;
    }

    $student = $DB->get_record('user', array('id' => $attemptobj->get_userid()));

    // Can only grade finished attempts.
    if (!$attemptobj->is_finished()) {
        NED::print_module_error('attemptclosed', 'quiz');
    }

    $slots = $attemptobj->get_slots();

    /** @var question_usage_by_activity $quba */
    $quba = \question_engine::load_questions_usage_by_activity($attemptobj->get_uniqueid());

    $hidegrades = $hidegradedresponses || NED::grade_is_hidden_now_before_midn($cm, $student->id);

    // Form start.
    $data = new stdClass();
    $data->formurl = (new moodle_url('/local/kicaquizgrading/index.php'))->out();
    $data->slot = $slot;
    $data->course = $courseid;
    $data->quiz = $quizid;
    $data->attemptid = $attemptid;
    $data->sesskey = sesskey();
    $data->showquestiontext = $showquestiontext;
    $data->showmanualquestions = $showmanualquestions;
    $data->showautoquestions = $showautoquestions;
    $data->showquestionswithoutkica = $showquestionswithoutkica;
    $data->hidegradedresponses = $hidegrades;
    $data->group = $group;
    $data->status = $status;
    $data->show = $show;
    $data->slotnum = $slotnum;
    echo $OUTPUT->render_from_template('local_kicaquizgrading/gradingformstart', $data);

    foreach ($slots as $slot) {
        /** @var question_attempt $questionattempt */
        $questionattempt = $quba->get_question_attempt($slot);

        $question = $questionattempt->get_question();
        $slotrec = helper::get_slot_data($quizid, $slot);

        $display = helper::display_slot($slotrec->questionid, $questionattempt, $showquestionswithoutkica, $showmanualquestions,
            $showautoquestions, $hidegrades);

        if (!$display) {
            continue;
        }
        // Prepare summary information about this question attempt.
        $summarydata = array();

        // Student name.
        $userpicture = new user_picture($student);
        $userpicture->courseid = $attemptobj->get_courseid();
        $summarydata['user'] = array(
            'title' => $userpicture,
            'content' => new action_link(new moodle_url('/user/view.php', array(
                'id' => $student->id, 'course' => $attemptobj->get_courseid())),
                fullname($student, true)),
        );

        // Quiz name.
        $summarydata['quizname'] = array(
            'title' => get_string('modulename', 'quiz'),
            'content' => format_string($attemptobj->get_quiz_name()),
        );

        // Add link to Grade controller
        $cmid = $attemptobj->get_cm()->id;
        $studentid = $student->id;
        $records = NED::$ned_grade_controller::get_records_by_params($cmid, $studentid);
        if (!empty($records)){
            $ngc_record = reset($records);
            $ngc_text = NED::$ned_grade_controller::get_see_text($ngc_record);
            if (!empty($ngc_text)){
                $summarydata['quizname']['content'] .=
                    '('.NED::$C::link([NED::PAGE_GRADE_CONTROLLER, [NED::PAR_ID => $ngc_record->id]], $ngc_text).')';
            }
        }

        // Question name.
        $summarydata['questionname'] = array(
            'title' => get_string('question', 'quiz'),
            'content' => $attemptobj->get_question_name($slot),
        );

        // Print quiz information.
        echo $output->review_summary_table($summarydata, 0);
        echo $renderer->render_question_for_commenting($attemptobj, $slot, $context);
    }
    echo $OUTPUT->render_from_template('local_kicaquizgrading/gradingformend', []);
}

echo $OUTPUT->footer();