/**
 * Moodle 'core/log' breaks initial methodFactory of the loglevel ('core/loglevel')
 *  and show wrong scope during the logging methods calls.
 * Here we getting clear log object from the loglevel to walk around this issue.
 * Also you can set prefix to your logger, without breaking anything.
 *
 * @module     local_ned_controller/log
 * @package    local_ned_controller
 * @subpackage amd
 * @category   NED
 * @copyright  2022 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 *
 * @example
 * define(['local_ned_controller/log'], function(Log) {
 *     Log.debug('test');                       // debug something through the default logger
 *     let log = Log.getLogger('my_plugin');    // get the new logger object with the name 'my_plugin' (or load existing one with the same name)
 *     log.setPrefix('[MP]');                   // set new prefix to custom logger object;
 *     let dbg = log.debug.bind(log);           // make alias for the debug function of log
 * });
 *
 * About original loglevel see {@link https://github.com/pimterry/loglevel}
 *  or {@link project://lib/amd/src/loglevel.js}
 */
define(['core/loglevel'], function(Log) {
    let defaultLogger = getNewLogger(undefined);
    let _loggersByName = {};

    /**
     * Return new, clear logger from the loglevel
     *
     * @param {String} name
     * @param {String|Number} level - one of the { "TRACE": 0, "DEBUG": 1, "INFO": 2, "WARN": 3, "ERROR": 4, "SILENT": 5}
     *
     * @returns Logger
     */
    function getNewLogger(name, level= Log.levels.TRACE){
        let log = new Log.constructor(name);
        log._originalFactory = log.methodFactory;

        log.setPrefix = setPrefix.bind(log);
        log.getLogger = getLogger.bind(log);
        log.getLoggers = getLoggers.bind(log);

        log.setLevel(level);

        return log;
    }

    /**
     * Set prefix to console methods. If empty - remove prefix
     *
     * @methodOf Logger
     *
     * @param {String} prefix
     */
    function setPrefix(prefix){
        if (prefix){
            this.methodFactory = (methodName, logLevel) => this._originalFactory(methodName, logLevel).bind(this, prefix);
        } else {
            this.methodFactory = (methodName, logLevel) => this._originalFactory(methodName, logLevel).bind(this);
        }
        // Refresh methods
        this.setLevel(this.getLevel());
    }

    /**
     * Get new logger with the custom name and prefix
     *
     * @methodOf Logger
     *
     * @param {String} name - if not provided, set it random
     * @param {String} prefix - if undefined, set default to [${name}], otherwise if empty - set nothing; will set only to the new loggers
     * @param {String|Number} level - one of the { "TRACE": 0, "DEBUG": 1, "INFO": 2, "WARN": 3, "ERROR": 4, "SILENT": 5}
     *
     * @returns Logger
     */
    function getLogger(name, prefix, level= Log.levels.TRACE){
        if ((typeof name !== "symbol" && typeof name !== "string") || name === "") {
            name = (new Date()).getTime() + ':' + Math.random();
            prefix = prefix === undefined ? null : prefix;
        }
        if (prefix === undefined && typeof name !== "symbol"){
            prefix = '['+name+']';
        }

        let logger = _loggersByName[name];
        if (!logger){
            logger = _loggersByName[name] = getNewLogger(name, level);
            if (prefix) logger.setPrefix(prefix);
        }

        return logger;
    }

    /**
     * Return all registered loggers
     *
     * @methodOf Logger
     *
     * @returns {{}}
     */
    function getLoggers(){
        return _loggersByName;
    }

    // ES6 default export, for compatibility
    defaultLogger['default'] = defaultLogger;

    return defaultLogger;
});
