/**
 * JS file required by theme initialisation
 * Used both by NED Clean and NED Boost themes
 *
 * @package    local_ned_controller
 * @subpackage local_ned_controller/theme/init_theme
 * @copyright  2023 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

define(['jquery', 'core/log', 'core/custom_interaction_events'], function($, Log, CustomEvents){

    return {
        init: function(){
            Log.debug('Run NED init_theme');
            FixUndefinedElement();
            FixCustomEventsScrollBottom();
            Log.debug('NED init_theme - done');
        },
    };

    /**
     * Fix error of some methods, which try to find element with undefined id
     * (and error of absence of this element crash all others scripts)
     */
    function FixUndefinedElement(){
        $('body').append($('<br style="display: none; visibility: hidden;" id="undefined" class="ned-fix-element--dont-touch-it">'));
    }

    /**
     * Replace some events of lib/amd/src/custom_interaction_events.js module
     * by modification {CustomEvents.define} method
     */
    function FixCustomEventsScrollBottom(){
        if (!CustomEvents || !CustomEvents.define || !CustomEvents.events || !CustomEvents.events.scrollBottom){
            window.console.error('Something changes in the core CustomEvents, please, update the ned_boost script');
            return;
        }

        /**
         * Static cache of jQuery events that have been handled. This should
         * only be populated by JavaScript generated events (which will keep it
         * fairly small).
         *
         * Required by {@see triggerEvent()}
         * @file lib/amd/src/custom_interaction_events.js
         */
        let triggeredEvents = {};

        /**
         * Which events we would change, all keys should be from the {@see CustomEvents.events}
         */
        let replaced_events = {
            [CustomEvents.events.scrollBottom]: new_addScrollBottomListener,
        };
        CustomEvents._define = CustomEvents.define;

        /**
         * Add all the listeners on the given element for the requested events.
         * Run our replaced events, then run original {@see CustomEvents.define} for all others events
         *
         * @method define
         * @public
         * @param {object} element the DOM element to register event listeners on
         * @param {array} include the array of events to be triggered
         */
        CustomEvents.define = function(element, include){
            let $element = $(element);
            include = include || [];
            if (!$element.length || !include.length) return;

            let new_include = [];
            for (let event of include){
                if (replaced_events[event]){
                    replaced_events[event]($element);
                } else {
                    new_include.push(event);
                }
            }

            if (new_include.length > 0){
                CustomEvents._define(element, new_include);
            }
        };

        /**
         * Trigger the scrollBottom event on the given element if the user scrolls to
         * the bottom of the given element.
         *
         * Replacement of the original {@file lib/amd/src/custom_interaction_events.js::addScrollBottomListener()}
         * The problem with the original method is that it compares rounded integers to fractional numbers,
         * which sometimes makes the event not work when it should.
         * Here, that defect is fixed, but the event may trigger several times if the scroll step is less than one pixel.
         * However, in this case, it is usually better if the event is triggered several times than none at all.
         *
         * @method new_addScrollBottomListener
         * @private
         * @param {object} element jQuery object to add event listeners to
         */
        function new_addScrollBottomListener(element) {
            element.off('scroll.cie.scrollBottom').on('scroll.cie.scrollBottom', function(event){
                let e = element[0];
                if ((e.scrollHeight - e.getBoundingClientRect().height - e.scrollTop) < 1){
                    triggerEvent(CustomEvents.events.scrollBottom, event);
                }
            });
        }

        /**
         * Trigger the custom event for the given jQuery event.
         *
         * Copy of the method from the {@file lib/amd/src/custom_interaction_events.js}
         * without any significant changes
         *
         * This function will only fire the custom event if one hasn't already been
         * fired for the jQuery event.
         *
         * This is to prevent multiple custom event handlers triggering multiple
         * custom events for a single jQuery event as it bubbles up the stack.
         *
         * @param  {string} eventName The name of the custom event
         * @param  {event} e          The jQuery event
         *
         * @return {void}
         */
        function triggerEvent(eventName, e){
            //noinspection JSUnusedAssignment
            let eventTypeKey = "";

            // noinspection DuplicatedCode
            if (!e.hasOwnProperty('originalEvent')) {
                // This is a jQuery event generated from JavaScript not a browser event, so
                // we need to build the cache key for the event.
                eventTypeKey = "" + eventName + e.type + e.timeStamp;

                if (!triggeredEvents.hasOwnProperty(eventTypeKey)) {
                    // If we haven't seen this jQuery event before then fire a custom
                    // event for it and remember the event for later.
                    triggeredEvents[eventTypeKey] = true;
                    $(e.target).trigger(eventName, [{originalEvent: e}]);
                }
                return;
            }

            eventTypeKey = "triggeredCustom_" + eventName;
            if (!e.originalEvent.hasOwnProperty(eventTypeKey)) {
                // If this is a jQuery event generated by the browser then set a
                // property on the original event to track that we've seen it before.
                // The property is set on the original event because it's the only part
                // of the jQuery event that is maintained through multiple event handlers.
                e.originalEvent[eventTypeKey] = true;
                $(e.target).trigger(eventName, [{originalEvent: e}]);
                // noinspection UnnecessaryReturnStatementJS
                return;
            }
        }
    }
});
