<?php
/**
 * @package    local_ned_controller
 * @category   NED
 * @copyright  2023 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller;
use local_ned_controller\shared_lib as NED;

defined('MOODLE_INTERNAL') || die();

/**
 * Trait base_child_class
 * @package local_ned_controller
 *
 * Trait to make child class is more linked to the parent class.
 * Through its methods you can access parent private properties and call parent private methods.
 * Warning: you can call parent private methods, but you can't override parent private methods,
 *  so if you want to change their behavior, you need to override outer parent method,
 *  where calling needed private method (you still can override public and protected methods)
 *
 * For using, you need declare class with "extends" of some parent class,
 *  and get current object through {@see static::bcc_create_from_parent()} with providing declared parent instance
 * Note: technically, you can provide another object that is not a parent to this class,
 *  but their incompatible methods and data will probably cause an error
 *
 *  @see base_child_class_magic for trait with magic methods
 */
trait base_child_class{
    /** @var class_accessor  */
    protected $_bcc_cls_acc = null;

    //region Parent accessor methods
    /**
     * Get property of the object in the parent context
     *
     * @param string $name - name of the property
     *
     * @return mixed
     */
    protected function _bcc_get($name=''){
        if (empty($this->_bcc_cls_acc) || empty($name)) return null;

        return $this->_bcc_cls_acc->get($name);
    }

    /**
     * Get static property of the object in the parent context
     *
     * @param string $name - name of the static property (without "$")
     *
     * @return mixed
     */
    protected function _bcc_get_static($name=''){
        if (empty($this->_bcc_cls_acc) || empty($name)) return null;

        return $this->_bcc_cls_acc->get_static($name);
    }

    /**
     * Get static property of the object in the parent context
     *
     * @param string $name - name of the constant property
     *
     * @return mixed
     */
    protected function _bcc_get_constant($name=''){
        if (empty($this->_bcc_cls_acc) || empty($name)) return null;

        return $this->_bcc_cls_acc->get_constant($name);
    }

    /**
     * Set property of the object in the parent context
     *
     * @param string $name  - name of the property
     * @param mixed  $value - (optional) data to set
     *
     * @return mixed
     */
    protected function _bcc_set($name='', $value=null){
        if (empty($this->_bcc_cls_acc) || empty($name)) return null;

        return $this->_bcc_cls_acc->set($name, $value);
    }

    /**
     * Set static property of the object in the parent context
     *
     * @param string $name  - name of the static property (without "$")
     * @param mixed  $value - (optional) data to set
     *
     * @return mixed
     */
    protected function _bcc_set_static($name='', $value=null){
        if (empty($this->_bcc_cls_acc) || empty($name)) return null;

        return $this->_bcc_cls_acc->set_static($name, $value);
    }

    /**
     * Call method of the object in the parent context
     *
     * @param string      $name - name of the method
     * @param array|mixed $args - (optional) arguments to the method
     *
     * @return mixed
     */
    protected function _bcc_call($name='', ...$args){
        if (empty($this->_bcc_cls_acc) || empty($name)) return null;

        return $this->_bcc_cls_acc->call($name, ...$args);
    }

    /**
     * Call static method of the object in the parent context
     *
     * @param string      $name - name of the static method
     * @param array|mixed $args - (optional) arguments to the method
     *
     * @return mixed
     */
    protected function _bcc_call_static($name='', ...$args){
        if (empty($this->_bcc_cls_acc) || empty($name)) return null;

        return $this->_bcc_cls_acc->call_static($name, ...$args);
    }

    /**
     * Check isset properties of the object in the parent context
     *
     * @param string $name - name of the property
     *
     * @return bool
     */
    protected function _bcc_isset($name=''){
        if (empty($this->_bcc_cls_acc) || empty($name)) return null;

        return $this->_bcc_cls_acc->isset($name);
    }

    /**
     * Unset properties form the object in the parent context
     *
     * @param string $name - name of the property
     *
     * @return null
     */
    protected function _bcc_unset($name=''){
        if (empty($this->_bcc_cls_acc) || empty($name)) return null;

        return $this->_bcc_cls_acc->unset($name);
    }

    /**
     * Call $closure in the context of parent object
     * $closure can contain code with $this, static:: and other using of the object context properties and methods
     *
     * @param \Closure $closure - function with the code to execute
     *
     * @return mixed
     */
    protected function _bcc_exec($closure){
        if (empty($this->_bcc_cls_acc) || empty($closure)) return null;

        return $this->_bcc_cls_acc->exec($closure);
    }
    //endregion

    //region Hooks
    /**
     * Method called before main part of the {@see static::bcc_import_properties()}
     *
     * @param object|static $parent
     *
     * @return void
     */
    protected function _bcc_hook_before_import($parent){

    }

    /**
     * Method called after main part of the {@see static::bcc_import_properties()}
     *
     * @param object|static $parent
     *
     * @return void
     */
    protected function _bcc_hook_after_import($parent){

    }

    /**
     * Method called before main part of the {@see static::bcc_create_from_parent()}
     *
     * @param object|static $parent
     *
     * @return void
     */
    protected function _bcc_hook_before_creation_from_parent($parent){

    }

    /**
     * Method called before main part of the {@see static::bcc_create_from_parent()}
     *
     * @param object|static $parent
     *
     * @return void
     */
    protected function _bcc_hook_after_creation_from_parent($parent){

    }
    //endregion

    //region Public methods
    /**
     * Import parent properties inside object in the parent context
     *
     * @param object|static $parent
     *
     * @return void
     */
    public function bcc_import_properties($parent){
        $this->_bcc_hook_before_import($parent);
        foreach (NED::obj_loop_properties($parent) as $key => $item){
            $this->_bcc_set($key, $item);
        }
        $this->_bcc_hook_after_import($parent);
    }

    /**
     * Create and save class accessor from the parent object
     *
     * @param $parent
     *
     * @return void
     */
    public function bcc_create_accessor($parent){
        $this->_bcc_cls_acc = class_accessor::get_class_accessor($this, $parent);
    }
    //endregion

    //region Static methods
    /**
     * Return new instance of class, without calling __constructor (absolutely clear class object)
     *
     * @return object|static
     * @noinspection PhpReturnDocTypeMismatchInspection
     */
    static public function bcc_get_instance_without_constructor(){
        return NED::obj_get_instance_without_constructor(get_called_class());
    }

    /**
     * Main function to get new class object instead of __construct()
     * Create child class from the parent object,
     *  by importing all its properties in the parent context and skipping __construct calling
     *
     * @param object|static $parent
     *
     * @return object|static
     */
    static public function bcc_create_from_parent($parent){
        $obj = static::bcc_get_instance_without_constructor();
        $obj->_bcc_hook_before_creation_from_parent($parent);
        $obj->bcc_create_accessor($parent);
        $obj->bcc_import_properties($parent);
        $obj->_bcc_hook_after_creation_from_parent($parent);

        return $obj;
    }
    //endregion
}
