<?php
/**
 * @package    local_ned_controller
 * @category   NED
 * @copyright  2023 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller;

defined('MOODLE_INTERNAL') || die();

/**
 * class class_accessor
 * @package local_ned_controller
 *
 * Class allows accessing to the different properties and methods of the object in the private scope.
 *  If both $data_obj and $scope_obj are the same object, you can use it read/write private properties and call private functions.
 *  If them are different objects, then it will use $data_obj in the context of the $scope_obj
 *      - be careful, that it were related objects (for example child and its parent), otherwise it will probably cause an error.
 *
 * If you need just some one-time method, you can use obj_* utils from the shared_lib {@see \local_ned_controller\shared\util },
 *  for example {@see \local_ned_controller\shared\util::obj_get_property()}
 *
 */
class class_accessor {
    const CMD_GET = 'get';
    const CMD_GET_STATIC = 'get_static';
    const CMD_GET_CONST = 'get_const';
    const CMD_SET = 'set';
    const CMD_SET_STATIC = 'set_static';
    const CMD_CALL = 'call';
    const CMD_CALL_STATIC = 'call_static';
    const CMD_ISSET = 'isset';
    const CMD_UNSET = 'unset';
    const CMD_EXEC = 'exec';

    /** @var \Closure */
    protected $_class_closure=null;

    /**
     * @param null|object $data_obj  – The object which contains data, and from which all function will call
     * @param mixed       $scope_obj – (optional) The class scope to which associate the closure is to be associated ($data_obj by default)
     *                               If an object is given, the type of the object will be used instead.
     *                               This determines the visibility of protected and private methods of the bound object.
     *
     * @noinspection PhpVariableIsUsedOnlyInClosureInspection
     */
    protected function __construct($data_obj, $scope_obj=null){
        $th = $this;
        $f = function($cmd, $name='', $value=null) use ($th) {
            if (empty($name) && $cmd != $th::CMD_EXEC) return null;

            switch ($cmd){
                case $th::CMD_GET:
                    return $this->$name ?? null;
                case $th::CMD_GET_STATIC:
                    return $this::${$name};
                case $th::CMD_GET_CONST:
                    return constant('static::'.$name);
                case $th::CMD_SET:
                    return $this->$name = $value;
                case $th::CMD_SET_STATIC:
                    return $this::${$name} = $value;
                case $th::CMD_CALL:
                    return $this->$name(...$value);
                case $th::CMD_CALL_STATIC:
                    return $this::$name(...$value);
                case $th::CMD_ISSET:
                    return isset($this->$name);
                case $th::CMD_UNSET:
                    unset($this->$name);
                    return null;
                case $th::CMD_EXEC:
                    $closure = $value;
                    if (empty($closure) || !is_callable($closure)) return null;
                    return \Closure::bind($closure, $this, $this)();
                default:
                    \debugging("There is no command $cmd in the class_accessor");
                    return null;
            }
        };
        $this->_class_closure = \Closure::bind($f, $data_obj, $scope_obj ?? $data_obj);
    }

    /**
     * Call commands to the class_closure
     *
     * @param string $cmd   - one of the CMD_* constants
     * @param string $name  - most of the command require name of something
     * @param mixed  $value - (optional) some data argument for commands
     *
     * @return mixed
     */
    protected function _do($cmd, $name='', $value=null){
        return ($this->_class_closure)($cmd, $name, $value);
    }

    /**
     * Get property of the object
     *
     * @param string $name - name of the property
     *
     * @return mixed
     */
    public function get($name=''){
        return $this->_do(static::CMD_GET, $name);
    }

    /**
     * Get static property of the object
     *
     * @param string $name - name of the static property (without "$")
     *
     * @return mixed
     */
    public function get_static($name=''){
        return $this->_do(static::CMD_GET_STATIC, $name);
    }

    /**
     * Get static property of the object
     *
     * @param string $name - name of the constant property
     *
     * @return mixed
     */
    public function get_constant($name=''){
        return $this->_do(static::CMD_GET_CONST, $name);
    }

    /**
     * Set property of the object
     *
     * @param string $name  - name of the property
     * @param mixed  $value - (optional) data to set
     *
     * @return mixed
     */
    public function set($name='', $value=null){
        return $this->_do(static::CMD_SET, $name, $value);
    }

    /**
     * Set static property of the object
     *
     * @param string $name  - name of the static property (without "$")
     * @param mixed  $value - (optional) data to set
     *
     * @return mixed
     */
    public function set_static($name='', $value=null){
        return $this->_do(static::CMD_SET_STATIC, $name, $value);
    }

    /**
     * Call method of the object
     *
     * @param string      $name - name of the method
     * @param array|mixed $args - (optional) arguments to the method
     *
     * @return mixed
     */
    public function call($name='', ...$args){
        return $this->_do(static::CMD_CALL, $name, $args);
    }

    /**
     * Call static method of the object
     *
     * @param string      $name - name of the static method
     * @param array|mixed $args - (optional) arguments to the method
     *
     * @return mixed
     */
    public function call_static($name='', ...$args){
        return $this->_do(static::CMD_CALL_STATIC, $name, $args);
    }

    /**
     * Check isset properties of the object
     *
     * @param string $name - name of the property
     *
     * @return bool
     */
    public function isset($name=''){
        return $this->_do(static::CMD_ISSET, $name);
    }

    /**
     * Unset properties form the object
     *
     * @param string $name - name of the property
     *
     * @return null
     */
    public function unset($name=''){
        return $this->_do(static::CMD_UNSET, $name);
    }

    /**
     * Call $closure in the context of object
     * $closure can contain code with $this, static:: and other using of the object context properties and methods
     *
     * @param \Closure $closure - function with the code to execute
     *
     * @return mixed
     */
    public function exec($closure){
        return $this->_do(static::CMD_EXEC, '', $closure);
    }

    /**
     * @param null|object $data_obj  – The object which contains data, and from which all function will call
     * @param mixed       $scope_obj – (optional) The class scope to which associate the closure is to be associated ($data_obj by default)
     *                               If an object is given, the type of the object will be used instead.
     *                               This determines the visibility of protected and private methods of the bound object.
     *
     * @return static
     */
    static public function get_class_accessor($data_obj, $scope_obj=null){
        return new static($data_obj, $scope_obj);
    }
}
