<?php
/**
 * @package    local_ned_controller
 * @category   NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller;
use local_ned_controller\marking_manager\marking_manager as MM;
use local_ned_controller\shared_lib as NED;
use local_ned_controller\support\deadline_notification_record;
use local_ned_controller\support\ned_notify_simple as NNS;
use local_schoolmanager\school_manager as SCM;


/**
 * Class deadline_notification
 *
 * @package local_ned_controller
 */
class deadline_notification {
    use base_empty_class;

    const TABLE = 'local_ned_controller_dlnotif';

    const COND_OVERDUE = 'overdue';
    const COND_DUE_12h = 'due_12h';
    const COND_DUE_24h = 'due_24h';
    const COND_DUE_24h_full = 'due_24h_full';

    const DUE_TABLE_CONDS = [self::COND_OVERDUE, self::COND_DUE_12h, self::COND_DUE_24h];
    const DUE_NOTIF_CONDS = [self::COND_OVERDUE, self::COND_DUE_12h];
    const DUE_ALL_CONDS = [self::COND_OVERDUE, self::COND_DUE_12h, self::COND_DUE_24h, self::COND_DUE_24h_full];

    const STUDENT = 'user';
    const GRADER = 'grader';

    /*
     * Capabilities to see own or other dashboards.
     * All next includes others, so you can check by using ">" or "<"
     */
    const CAP_CANT_SEE_DN = 0;
    const CAP_SEE_OWN_DN = 1;
    const CAP_SEE_OWN_GROUPS_DN = 2;

    protected $_capability = self::CAP_CANT_SEE_DN;
    protected $_ctx;
    protected $_user;
    protected $_userid;

    /**
     * deadline_notification constructor.
     */
    public function __construct(){
        global $USER;

        $this->_ctx = \context_system::instance();
        $this->_capability = static::get_dn_capability($this->_ctx, $USER, true);
        if ($this->_capability < static::CAP_SEE_OWN_DN){
            return;
        }

        $this->_user = $USER;
        $this->_userid = $USER->id;
    }

    /**
     * Set, when user see DN notifications last time
     *
     * @param int|null $t - unix time
     *
     */
    public function set_saw_time($t=null){
        $t = $t ?? time();
        $cache = NED::get_user_cache();
        $cache->set('user_deadline_notes', $t);
    }

    /**
     * Get DN data to show notifications
     *
     * @return array
     */
    public function get_notification_data(){
        if ($this->_capability < static::CAP_SEE_OWN_DN){
            return [];
        }

        $data = [static::STUDENT => null, static::GRADER => null];
        if ($this->_capability >= static::CAP_SEE_OWN_DN){
            $raw_data = static::get_table_useful_data(true, $this->_userid, null, null, null, false);
            $data[static::STUDENT] = static::format_records($raw_data, null, true);
        }

        if ($this->_capability >= static::CAP_SEE_OWN_GROUPS_DN){
            $raw_data = static::get_table_useful_data(true, null, null, null, $this->_userid, false);
            $data[static::GRADER] = static::format_records($raw_data, null, true);
        }

        return $data;
    }

    /**
     *  Check & call deadline_notification if need
     */
    public function check_and_show_notifications(){
        $cache = NED::get_user_cache();
        $dn_time = $cache->get('user_deadline_notes') ?: 0;
        if ($dn_time > (time() - HOURSECS)){
            return;
        }

        $dn = 'deadline_notification';
        $data = $this->get_notification_data();
        $notes = [];
        foreach ($data as $user_type => $datum){
            foreach (static::DUE_NOTIF_CONDS as $cond){
                $check_records = $datum[$cond] ?? null;
                if (empty($check_records)){
                    continue;
                }

                $records = [];
                /** @var deadline_notification_record|object $record */
                foreach ($check_records as $record){
                    if (NED::get_cm_visibility_by_user($record->cmid, $this->_userid, false, false)){
                        $records[] = $record;
                    }
                }
                if (empty($records)){
                    continue;
                }

                $templ_context = new \stdClass();
                $templ_context->username = fullname($this->_user);
                $templ_context->body = NED::str("$dn:$cond:$user_type:text");
                $templ_context->records = $records;
                if ($user_type != static::STUDENT){
                    $show_key = 'show_'.$cond;
                    $templ_context->$show_key = true;
                    $templ_context->show_user_count = true;
                }

                $noty = new NNS();
                $noty->title = NED::str("$dn:$cond:title");
                $noty->notification_type = ($cond == static::COND_OVERDUE) ? NNS::TYPE_NOTICE : NNS::TYPE_REMINDER;
                $noty->template = NED::$PLUGIN_NAME.'/'.$dn;
                $noty->template_context = $templ_context;

                $notes[] = $noty->js_export();
            }
        }

        if (empty($notes)){
            $this->set_saw_time();
            return;
        }

        NNS::show_notifications($notes, 'deadline_notification_data', 'show_deadline_notifications');
    }

    /**
     * Return capability to see own or other dashboards.
     *  All upper capability includes others, so you can check by using ">" or "<"
     *
     * @param null $ctx - context
     * @param integer|\stdClass $user_or_id         (optional) A user id or object. By default (null) checks the permissions of the current user.
     * @param boolean           $ignores_admin_role (optional) If true, ignores effect of admin role assignment
     *
     * @return int
     */
    static public function get_dn_capability($ctx=null, $user_or_id=null, $ignores_admin_role=false){
        $ctx = $ctx ?? \context_system::instance();
        if (NED::has_capability('dn_seeowngroupsnotifications', $ctx, $user_or_id, !$ignores_admin_role)){
            return static::CAP_SEE_OWN_GROUPS_DN;
        }
        if (NED::has_capability('dn_seeownnotifications', $ctx, $user_or_id, !$ignores_admin_role)){
            return static::CAP_SEE_OWN_DN;
        }
        return static::CAP_CANT_SEE_DN;
    }

    /**
     * @param int   $courseid
     * @param int   $cmid
     * @param int   $groupid
     * @param int   $userid
     * @param array $options
     *
     * @return array|mixed
     * @noinspection PhpUnusedParameterInspection
     */
    static public function check_db_params($courseid=0, $cmid=0, $groupid=0, $userid=0, &$options=[]){
        $keys = ['courseid', 'cmid', 'groupid', 'userid'];
        foreach ($keys as $key){
            if ($$key){
                $options[$key] = $$key;
            }
        }
        return $options;
    }

    /**
     * @param array $options
     *
     * @return deadline_notification_record[]|array
     */
    static public function get_records($options=[]){
        global $DB;
        return $DB->get_records(static::TABLE, $options);
    }

    /**
     * @param int   $courseid
     * @param int   $cmid
     * @param int   $groupid
     * @param int   $userid
     * @param array $options
     *
     * @return deadline_notification_record[]|array
     */
    static public function get_records_by_params($courseid=0, $cmid=0, $groupid=0, $userid=0, $options=[]){
        static::check_db_params($courseid, $cmid, $groupid, $userid, $options);
        return static::get_records($options);
    }

    /**
     * Update records in the table
     * @param $records
     *
     * @return bool
     */
    static public function update_records($records){
        global $DB;
        $records = NED::val2arr($records);
        $records = array_values($records);
        $last_i = count($records) - 1;
        foreach ($records as $i => $record){
            $DB->update_record(static::TABLE, $record, $i != $last_i);
        }

        return true;
    }

    /**
     * @param array $options
     *
     * @return bool
     */
    static public function delete_records_by_options($options=[]){
        global $DB;
        if (empty($options)){
            return false;
        }

        return $DB->delete_records(static::TABLE, $options);
    }

    /**
     * @param int[]|int $ids
     *
     * @return bool
     */
    static public function delete_records_by_ids($ids){
        global $DB;
        if (empty($ids)){
            return false;
        }

        $ids = NED::val2arr($ids);
        return $DB->delete_records_list(static::TABLE, 'id', $ids);
    }

    /**
     * @param object[]|deadline_notification_record[]|object|deadline_notification_record $records
     *
     * @return bool
     */
    static public function delete_records_by_records($records){
        $records = NED::val2arr($records);
        $ids = [];
        foreach ($records as $record){
            $ids[] = $record->id;
        }
        return static::delete_records_by_ids($ids);
    }

    /**
     * @param int   $courseid
     * @param int   $cmid
     * @param int   $groupid
     * @param int   $userid
     * @param array $options
     *
     * @return bool
     */
    static public function delete_records_by_params($courseid=0, $cmid=0, $groupid=0, $userid=0, $options=[]){
        static::check_db_params($courseid, $cmid, $groupid, $userid, $options);
        return static::delete_records_by_options($options);
    }

    /**
     * Return true, if all is fine, false otherwise
     *
     * @param object[]|array[] $records
     *
     * @return bool
     */
    static public function save_records($records){
        global $DB;
        $records = NED::val2arr($records);
        $DB->insert_records(static::TABLE, $records);

        return true;
    }

    /**
     * @param deadline_notification_record|object|array $record1
     * @param deadline_notification_record|object|array $record2
     *
     * @return bool
     */
    static public function compare_records($record1, $record2){
        if (empty($record1) || empty($record2)){
            return empty($record1) && empty($record2);
        }
        $record1 = (object)$record1;
        $record2 = (object)$record2;

        foreach (deadline_notification_record::IMPORTANT_KEYS as $key){
            $val1 = $record1->$key ?? null;
            $val2 = $record2->$key ?? null;
            if ($val1 != $val2){
                return false;
            }
        }

        return true;
    }

    /**
     * Copy values by IMPORTANT_KEYS from $record_source to record_base
     * So, from $record_source we copy only important keys, but we have not check $record_base keys
     *
     * @param deadline_notification_record|object|array $record_source
     * @param deadline_notification_record|object|array $record_base
     *
     * @return deadline_notification_record|object
     */
    static public function copy_object($record_source, $record_base=[]){
        $record_base = NED::val2obj($record_base);
        if (empty($record_source)){
            return $record_base;
        }

        $record_source = NED::val2obj($record_source);
        foreach (deadline_notification_record::IMPORTANT_KEYS as $key){
            $record_base->$key = $record_source->$key ?? ($record_base->$key ?? null);
        }

        return $record_base;
    }

    /**
     * Check DN records by courseid with MM help
     *
     * @param int|string    $courseid - if false, check all courses
     * @param null|callable $debug - function for debug text, if you wish
     *
     * @return void
     * @noinspection PhpUnusedParameterInspection
     */
    static public function check_records_by_course($courseid=0, $debug=null){
        if ($debug && is_callable($debug)){
            $d = function(...$args) use (&$debug){
                return $debug(...$args);
            };
        } else {
            $d = function(...$args){
                return null;
            };
        }

        $filter = [MM::BY_ACTIVITY_USER, MM::USE_GROUPS, MM::ST_UNMARKED_DUE_IMPORTANT];
        $OTs = NED::get_online_teachers_ids($courseid);
        $CTs = NED::get_classroom_teachers_ids($courseid);
        if ($courseid){
            $courses = [NED::get_course($courseid)];
            $OTs = [$courseid => $OTs];
            $CTs = [$courseid => $CTs];
        } else {
            $courses = NED::get_all_courses();
        }
        // return $userid, if he can access $cmid, 0 (zero) otherwise
        $check_cm_visibility = function($cmid, $userid){
            if ($cmid && $userid){
                if (NED::get_cm_visibility_by_user($cmid, $userid, false, false)){
                    return $userid;
                }
            }

            return 0;
        };

        $all_courses = count($courses);
        $i = 0;
        foreach ($courses as $course){
            $i++;
            $courseid = $course->id;
            $d("Check $i/$all_courses course with id '$courseid'");
            $context = $context ?? \context_course::instance($courseid);
            $mm_params = ['course' => $course, 'context' => $context, 'groupid' => 0];
            $MM = MM::get_MM_by_params($mm_params);
            $data = $MM->get_raw_data($filter);
            $count_data = count($data);

            $old_records = static::get_records_by_params($courseid);
            $count_old_data = count($old_records);
            $old_records_by_cmid_userid = NED::pack_in_array($old_records, ['cmid', 'userid']);
            if (!$count_old_data && !$count_data){
                $d("\t\tThere are no records in this course, pass");
                continue;
            }
            $d("\t\tGet $count_old_data old record(s) & $count_data new unchecked record(s), check...");

            $new_records = [];
            $upd_records = [];
            foreach ($data as $datum){
                if (!$check_cm_visibility($datum->cmid, $datum->userid)){
                    continue;
                }

                $datum->courseid = $courseid;
                $datum->groupid = $datum->groupid ?? 0;
                $group_ots = $OTs[$courseid][$datum->groupid] ?? [0];
                $datum->ot_id = $check_cm_visibility($datum->cmid, reset($group_ots));
                $group_cts = $CTs[$courseid][$datum->groupid] ?? [0];
                $datum->ct_id = $check_cm_visibility($datum->cmid, reset($group_cts));
                $datum->deadline = $datum->cutoffdate ?? 0;
                $o_record = $old_records_by_cmid_userid[$datum->cmid][$datum->userid] ?? null;
                if ($o_record){
                    unset($old_records[$o_record->id]);
                    if (static::compare_records($o_record, $datum)){
                        continue;
                    } else {
                        $upd_records[] = static::copy_object($datum, $o_record);
                    }
                } else {
                    $new_records[] = static::copy_object($datum);
                }
            }

            if (empty($old_records) && empty($upd_records) && empty($new_records)){
                $d("\t\tAll data in this course still the same.");
            } else {
                if (!empty($old_records)){
                    $c = count($old_records);
                    $d("\t\tRemove $c old record(s).");
                    static::delete_records_by_records($old_records);
                }

                if (!empty($upd_records)){
                    $c = count($upd_records);
                    $d("\t\tUpdate $c old record(s).");
                    static::update_records($upd_records);
                }

                if (!empty($new_records)){
                    $c = count($new_records);
                    $d("\t\tInsert $c new record(s).");
                    static::save_records($new_records);
                }
            }
        }
    }

    /**
     * Return sql condition by its name
     *
     * @param $cond - should be one of the static::DUE_CONDS
     *
     * @return string
     */
    static public function get_sql_condition($cond){
        $sql_now = NED::SQL_NOW;
        $sql_now_12h = NED::SQL_NOW_12h;
        $sql_now_24h = NED::SQL_NOW_24h;
        return match ($cond) {
            static::COND_OVERDUE => "(dln.deadline BETWEEN 1 AND $sql_now)",
            static::COND_DUE_12h => "(dln.deadline BETWEEN ($sql_now+1) AND $sql_now_12h)",
            static::COND_DUE_24h => "(dln.deadline BETWEEN ($sql_now_12h+1) AND $sql_now_24h)",
            static::COND_DUE_24h_full => "(dln.deadline BETWEEN ($sql_now+1) AND $sql_now_24h)",
            default => '1 = 0',
        };
    }

    /**
     * @param bool $sum
     * @param int|string|array $userids
     * @param int|string|array $groupids
     * @param int|string|array $schoolids
     * @param int|string|array $graderids
     * @param bool $table_conds
     *
     * @return array
     */
    static public function get_table_useful_data($sum=false, $userids=null, $groupids=null, $schoolids=null, $graderids=null, $table_conds=true){
        global $DB;
        $t = static::TABLE;
        $params = [];
        $where = [];
        $set_in_or_equal = function($value, $sql_name, $prefix='dn_') use (&$DB, &$where, &$params){
            if (!empty($value)){
                list($where_sql, $where_params) = $DB->get_in_or_equal($value, SQL_PARAMS_NAMED, $prefix);
                $where[] = $sql_name.' '.$where_sql;
                $params = array_merge($params, $where_params);
                return true;
            }
            return false;
        };

        $modtype_join = [];
        $select_name = ["' '"];
        foreach (MM::MOD_TYPES as $type){
            $cls = '\local_ned_controller\marking_manager\marking_manager_'.$type;
            $l = $cls::SQL_TABLE_ALIAS ?? $type[0];
            $modtype_join[] = "
            LEFT JOIN {{$type}} $l
                ON m.name = '$type'
                AND $l.id = cm.instance
            ";
            $select_name[] = "$l.name";
        }
        $modtype_join = join("\n", $modtype_join);
        $select_name = join(", ", $select_name);

        $select = [
                "dln.*", "m.name AS module",
                "CONCAT(ot.firstname, ' ', ot.lastname) AS ot_name",
                "CONCAT(ct.firstname, ' ', ct.lastname) AS ct_name",
                "COALESCE(gr.name, 'None') AS classname",
                "c.shortname AS course_name",
                "c.fullname AS course_fullname",
                "CONCAT_WS($select_name) AS cm_name",
        ];

        $from = ["{{$t}} AS dln
            JOIN {course} c 
                ON c.id = dln.courseid
            JOIN {course_modules} cm
                ON cm.id = dln.cmid
            JOIN {modules} m
                ON m.id = cm.module
            $modtype_join
            LEFT JOIN {user} st 
                ON st.id = dln.userid
            LEFT JOIN {user} ot 
                ON ot.id = dln.ot_id
            LEFT JOIN {user} ct 
                ON ct.id = dln.ct_id
            LEFT JOIN {groups} gr 
                ON gr.id = dln.groupid",
        ];
        $groupby = [];
        $orderby = ['dln.courseid, dln.cmid'];

        $set_in_or_equal($userids, 'dln.userid', 'dn_userid_');
        $set_in_or_equal($groupids, 'dln.groupid', 'dn_groupid_');
        if (!empty($graderids)){
            list($where_sql, $where_params) = $DB->get_in_or_equal($graderids, SQL_PARAMS_NAMED, 'dn_graderid');
            list($where_sql2, $where_params2) = $DB->get_in_or_equal($graderids, SQL_PARAMS_NAMED, 'dn_graderid');
            $where[] = "(dln.ot_id $where_sql OR dln.ct_id $where_sql2)";
            $params = array_merge($params, $where_params, $where_params2);
        }

        $check_conds = $table_conds ? static::DUE_TABLE_CONDS : static::DUE_NOTIF_CONDS;
        if ($sum){
            foreach ($check_conds as $cond){
                $sql_cond = static::get_sql_condition($cond);
                $select[] = "SUM($sql_cond) AS $cond";
            }
            $groupby[] = "cm.id, classname";
        } else {
            foreach ($check_conds as $cond){
                $sql_cond = static::get_sql_condition($cond);
                $select[] = "$sql_cond AS $cond";
            }
            $select[] = "CONCAT(st.firstname, ' ', st.lastname) AS st_name";
        }

        if (NED::is_schm_exists()){
            $from[] = NED::sql_get_school_join('dln.userid', 'school', 'cohort_m', true);
            $select[] = "school.id AS schoolid";
            $select[] = "CONCAT_WS(', ', school.code) AS school_code";
            $set_in_or_equal($schoolids, 'school.id', 'dn_schoolid_');
        }

        $select = "SELECT ". join(', ', $select);
        $from = "\nFROM ".join("\n", $from);
        $where = !empty($where) ? ("\nWHERE (" . join(') AND (', $where) . ')') : '';
        $groupby = !empty($groupby) ? ("\nGROUP BY " . join(',', $groupby)) : '';
        $orderby = !empty($orderby) ? ("\nORDER BY " . join(',', $orderby)) : '';

        $records = $DB->get_records_sql($select.$from.$where.$groupby.$orderby, $params);
        return $records ?: [];
    }

    /**
     * @param int|string|array   $userids
     * @param int|string|array   $groupids
     * @param int|string|array   $schoolids
     *
     * @param string|\moodle_url $base_url
     *
     * @return array
     */
    static public function get_table_summary($userids=null, $groupids=null, $schoolids=null, $base_url=null){
        $raw_data = static::get_table_useful_data(true, $userids, $groupids, $schoolids);
        return static::format_records($raw_data, $base_url);
    }

    /**
     * @param array $raw_data
     * @param null|string|\moodle_url  $school_url
     * @param bool  $sort
     *
     * @return array
     */
    static protected function format_records($raw_data=[], $school_url=null, $sort=false){
        $res = [];

        if (empty($raw_data)){
            return [];
        }

        if ($school_url && is_string($school_url)){
            $school_url = new \moodle_url($school_url);
        }
        $mm_url = new \moodle_url(NED::is_tt_exists() ? NED::MM_PAGE : '#');
        $cp_url = new \moodle_url(NED::is_tt_exists() ? NED::CLASS_PROGRESS_PAGE : '#');
        $mm_params = [
            static::COND_OVERDUE.'_url' => MM::ST_UNMARKED_MISSED_DEADLINE,
            static::COND_DUE_12h.'_url' => MM::ST_UNMARKED_DUE_12h,
            static::COND_DUE_24h.'_url' => MM::ST_UNMARKED_DUE_24h
        ];

        foreach ($raw_data as $record){
            $courseid = $record->courseid;
            $cm_name = $record->cm_name;
            $max_len = 5;
            $pos = min(stripos($cm_name, ' ') ?: $max_len, stripos($cm_name, ':') ?: $max_len, $max_len);
            $record->show_cm_name = substr($cm_name, 0, $pos);

            $record->course_url = (new \moodle_url("/course/view.php", ['id' => $courseid]))->out(false);
            $record->mod_url = (new \moodle_url("/mod/{$record->module}/view.php", ['id' => $record->cmid]))->out(false);
            $record->ot_url = (new \moodle_url('/user/view.php', ['id' => $record->ot_id, 'course' => $courseid]))->out(false);
            $record->ct_url = (new \moodle_url('/user/view.php', ['id' => $record->ct_id, 'course' => $courseid]))->out(false);

            $mm_url_i = clone($mm_url);
            $mm_url_i->params([MM::P_COURSEID => $courseid, MM::P_MID => $record->cmid, MM::P_GROUPID => $record->groupid ?? 0]);
            foreach ($mm_params as $url_name => $param){
                $mm_url_i->param(MM::P_STATUS, $param);
                $record->$url_name = $mm_url_i->out(false);
            }

            $cp_url->params(['courseid' => $courseid, 'group' => $record->groupid ?? 0]);
            $record->class_progress_url = $cp_url->out(false);

            if ($school_url && isset($record->schoolid)){
                $school_url->param('schoolid', $record->schoolid);
                $record->school_url = $school_url->out(false);
            }

            if ($sort){
                foreach (static::DUE_ALL_CONDS as $cond){
                    if ($record->$cond ?? false){
                        if (!isset($res[$cond])){
                            $res[$cond] = [];
                        }
                        $res[$cond][] = $record;
                    }
                }
            } else {
                $res[] = $record;
            }
        }

        return $res;
    }
}
