<?php
/**
 * The mod_assign submission_created abstract event.
 *
 * @package    local_ned_controller
 * @subpackage event
 * @category   NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller\event;

defined('MOODLE_INTERNAL') || die();

/**
 * The mod_assign submission changes abstract event class.
 *
 * @property-read array $other {
 *      Extra information about the event.
 *
 *      - int submissionid: ID number of this submission.
 *      - int submissionattempt: Number of attempts made on this submission.
 *      - string submissionstatus: Status of the submission.
 *      - int groupid: (optional) The group ID if this is a teamsubmission.
 *      - string groupname: (optional) The name of the group if this is a teamsubmission.
 * }
 *
 * @package    local_ned_controller
 */
abstract class submission_changes_base extends \mod_assign\event\base {
    /**
     * Flag for prevention of direct create() call.
     * @var bool
     */
    protected static $preventcreatecall = true;

    /**
     * Init method.
     */
    protected function init() {
        $this->data['crud'] = 'c';
        $this->data['edulevel'] = self::LEVEL_PARTICIPATING;
        $this->data['objecttable'] = 'assign_submission';
    }

    /**
     * Returns localised general event name.
     *
     * @return string
     */
    public static function get_name() {
        return 'submission_changes_base';
    }

    /**
     * Custom validation.
     *
     * @return void
     * @throws \coding_exception
     */
    protected function validate_data() {
        parent::validate_data();
        if (static::$preventcreatecall) {
            throw new \coding_exception('cannot call submission_graded::create() directly, use submission_graded::create_from_grade() instead.');
        }
        if (!isset($this->relateduserid)) {
            throw new \coding_exception('The \'relateduserid\' must be set.');
        }
        if (!isset($this->other['submissionid'])) {
            throw new \coding_exception('The \'submissionid\' value must be set in other.');
        }
        if (!isset($this->other['submissionattempt'])) {
            throw new \coding_exception('The \'submissionattempt\' value must be set in other.');
        }
        if (!isset($this->other['submissionstatus'])) {
            throw new \coding_exception('The \'submissionstatus\' value must be set in other.');
        }
    }

    /**
     * This is used when restoring course logs where it is required that we
     * map the objectid to it's new value in the new course.
     *
     * @return string|array the name of the restore mapping the objectid links to
     */
    public static function get_objectid_mapping() {
        return array('db' => 'assign_submissions', 'restore' => 'submission');
    }

    /**
     * Create instance of event.
     *
     * @param \assign|\local_ned_controller\mod_assign\assign $assign
     * @param \stdClass                                       $submission
     *
     * @return submission_remove_grades
     */
    public static function create_from_submission($assign, $submission) {
        $params = static::get_event_params($assign, $submission);
        return static::create_event_by_params($assign, $submission, $params);
    }

    /**
     * @param \assign|\local_ned_controller\mod_assign\assign $assign
     * @param \stdClass                                       $submission
     *
     * @return array
     */
    public static function get_event_params($assign, $submission){
        $params = array(
            'context' => $assign->get_context(),
            'courseid' => $assign->get_course()->id,
            'objectid' => $submission->id,
            'relateduserid' => $submission->userid,
            'other' => [
                'submissionid' => $submission->id,
                'submissionattempt' => $submission->attemptnumber,
                'submissionstatus' => $submission->status,
                'groupid' => 0,
                'groupname' => null,
            ]
        );

        return $params;
    }

    /**
     * Create instance of event.
     *
     * @param \assign|\local_ned_controller\mod_assign\assign $assign
     * @param \stdClass                                       $submission
     * @param array                                           $params
     *
     * @return submission_remove_grades
     */
    public static function create_event_by_params($assign, $submission, $params){
        static::$preventcreatecall = false;
        /** @var submission_remove_grades $event */
        $event = static::create($params);
        static::$preventcreatecall = true;

        $event->set_assign($assign);
        if ($submission->id ?? 0){
            $event->add_record_snapshot('assign_submission', $submission);
        }

        return $event;
    }
}
