<?php
/**
 * @package    local_ned_controller
 * @subpackage NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @noinspection PhpUnused
 */

namespace local_ned_controller\form;

defined('MOODLE_INTERNAL') || die();

use local_ned_controller\shared_lib as NED;

/**
 * Class form_element for Moodle forms
 *
 * This class is created to help work with HTML_QuickForm & MoodleQuickForm.
 * By this class you can create form-element templates and create/add them as HTML_QuickForm_element to the form.
 * You can also check base_form, which exploit this class and its elements.
 *
 * Note: This class is not fully finished and haven't been tested properly - use with caution, please.
 *       Also, you can change most of the parameters, and they almost are not checked before or during export to form elements,
 *          so if you provided wrong data, it will cause error on the export step.
 *
 * @see \local_ned_controller\form\base_form
 * @see $GLOBALS['HTML_QUICKFORM_ELEMENT_TYPES']:
 * @see \HTML_QuickForm                // and list upper the class
 * @see \MoodleQuickForm_Rule_Required // list below the class
 * @see \MoodleQuickForm
 *
 * @link https://docs.moodle.org/dev/lib/formslib.php_Form_Definition
 *
 * @package local_ned_controller
 */
class form_element {
    const TYPE = 'type';
    const NAME = 'name';
    const ATTRIBUTES = 'attributes';
    const GROUP = 'group';

    const ARGS = 'args';
    const RULES = 'rules';
    const GROUP_RULES = 'group_rules';
    const HELP = 'help';
    const DEFAULT = 'default';
    const OPTIONS = 'options';
    const DISABLEDIFS = 'disabledIfs';
    const HIDEIFS = 'hideIfs';
    const PARAM_TYPE = 'param_type';
    const CUSTOM_DATA = 'custom_data';

    /**
     * @var string - we check to see if a checkbox is not checked
     */
    const COND_NOT_CHECKED = 'notchecked';
    /**
     * @var string - we check to see if a checkbox is checked
     */
    const COND_CHECKED = 'checked';
    /**
     * @var string - we check to see whether nothing is selected in a dropdown list.
     */
    const COND_NO_ITEM_SELECTED = 'noitemselected';
    /**
     * @var string - we check the value of the dependentOn field and check for equality (==)
     * IMPORTANT: don't use 'eq'/'neq' to 1/0 for checkboxes, it works incorrectly - you should use 'checked' instead
     */
    const COND_EQUAL = 'eq';
    /**
     * @var string - we check the value of the dependentOn field and check for nonequality (!=)
     * IMPORTANT: don't use 'eq'/'neq' to 0/1 for checkboxes, it works incorrectly - you should use 'notchecked' instead
     */
    const COND_NOT_EQUAL = 'neq';
    /**
     * @var string - we check to see if a selected item is in the given list or not
     */
    const COND_IN = 'in';

    /** @var string $_type - type of the element */
    protected $_type;
    protected $_rules = [];
    /** @var string|null $_param_type */
    protected $_param_type = null;
    /** @var array|null $_options */
    protected $_options;
    protected $_help = [];
    protected $_disabledIfs = [];
    protected $_hideIfs = [];
    /** @var string|array|null $_attributes */
    protected $_attributes;

    /** @var string|null $name */
    public $name;
    /** @var string|mixed|null $label_or_value */
    public $label_or_value;
    /** @var string|null $text */
    public $text;
    /** @var mixed|null $default */
    public $default;

    public $custom_data = [];

    /**
     * form_element constructor
     *
     * @see \HTML_QuickForm_element
     *
     * @param      $type
     * @param null $name
     * @param null $label_or_value
     * @param null $text
     * @param null $attributes
     */
    protected function __construct($type, $name=null, $label_or_value=null, $text=null, $attributes=null){
        $this->_type = $type;
        $this->name = $name;
        $this->label_or_value = $label_or_value;
        $this->text = $text;
        $this->_attributes = NED::val2arr($attributes);
    }

    /**
     * Raw call of the constructor
     * In normal way, you shouldn't use it, but if you really need it...
     *
     * @param      $type
     * @param null $name
     * @param null $label_or_value
     * @param null $text
     * @param null $attributes
     *
     * @return static
     */
    static public function raw($type, $name=null, $label_or_value=null, $text=null, $attributes=null){
        return new static($type, $name, $label_or_value, $text, $attributes);
    }

    //region Form Elements
    /**
     * html constructor
     *
     * @see \HTML_QuickForm_html
     *
     * @param string $text - raw HTML to add
     *
     * @return static
     */
    public static function html($text=null) {
        return new static('html', null, null, $text);
    }

    /**
     * html DIV constructor
     *
     * @see \HTML_QuickForm_html
     *
     * @param string $text - raw HTML to add inside div
     * @param string|array $class Optional CSS class (or classes as space-separated list)
     * @param array $attributes Optional other attributes as array
     *
     * @return static
     */
    public static function div($text=null, $class=[], $attributes=null) {
        $class = NED::arr2str($class);
        $text = \html_writer::div($text, $class, $attributes);
        return static::html($text);
    }

    /**
     * html SPAN constructor
     *
     * @see \HTML_QuickForm_html
     *
     * @param string $text - raw HTML to add inside span
     * @param string|array $class Optional CSS class (or classes as space-separated list)
     * @param array $attributes Optional other attributes as array
     *
     * @return static
     */
    public static function span($text=null, $class=[], $attributes=null) {
        $class = NED::arr2str($class);
        $text = \html_writer::span($text, $class, $attributes);
        return static::html($text);
    }

    /**
     * static constructor
     *
     * @see \HTML_QuickForm_static
     * @see \MoodleQuickForm_static
     *
     * @param string $elementName  Input field name attribute
     * @param string $elementLabel (optional) Label
     * @param string $text         (optional) Display text
     *
     * @return static
     */
    public static function static($elementName=null, $elementLabel=null, $text=null) {
        return new static('static', $elementName, $elementLabel, $text);
    }

    /**
     * static alias
     *
     * @see \HTML_QuickForm_static
     * @see \MoodleQuickForm_static
     *
     * @param string $elementName  Input field name attribute
     * @param string $elementLabel (optional) Label
     * @param string $text         (optional) Display text
     *
     * @return static
     */
    public static function st($elementName=null, $elementLabel=null, $text=null) {
        return static::static($elementName, $elementLabel, $text);
    }

    /**
     * header constructor
     *
     * @see \HTML_QuickForm_header
     * @see \MoodleQuickForm_header
     *
     * @param string $text        (optional) Display text
     * @param string $elementName Input field name attribute
     *
     * @return static
     */
    public static function header($text=null, $elementName=null) {
        return new static('header', $elementName, null, $text);
    }

    /**
     * hidden constructor
     *
     * @see \HTML_QuickForm_hidden
     * @see \MoodleQuickForm_hidden
     *
     * @param string $elementName Input field name attribute
     * @param string $value       (optional) Input field value
     * @param int    $param_type  (optional) Defines type of data contained in element. Use the constants PARAM_*.
     * @param mixed  $attributes  (optional) HTML attribute string or an associative array
     *
     * @return static
     */
    public static function hidden($elementName=null, $value='', $param_type=PARAM_TEXT, $attributes=null) {
        $fe = new static('hidden', $elementName, $value, null, $attributes);
        $fe->setType($param_type);
        return $fe;
    }

    /**
     * text constructor
     *
     * @see \HTML_QuickForm_text
     * @see \MoodleQuickForm_text
     *
     * @param string $elementName  Input field name attribute
     * @param string $elementLabel (optional) Input field label
     * @param int    $param_type   (optional) Defines type of data contained in element. Use the constants PARAM_*.
     * @param mixed  $attributes   (optional) HTML attribute string or an associative array, has priority on the next parameters
     * @param int    $size
     *
     * @return static
     */
    public static function text($elementName=null, $elementLabel=null, $param_type=PARAM_TEXT, $attributes=null, $size=255) {
        $attributes['size'] = $attributes['size'] ?? $size ?? 255;
        $fe = new static('text', $elementName, $elementLabel, null, $attributes);
        $fe->setType($param_type ?? PARAM_TEXT);
        return $fe;
    }

    /**
     * editor constructor
     *
     * @see  \MoodleQuickForm_editor
     *
     * If you would like to let the user use the filepicker to upload images etc. that are used in the content, then see
     * @link https://docs.moodle.org/dev/Using_the_File_API_in_Moodle_forms#editor
     *
     * To save the data if you don't care about files (if element name is 'fieldname'):
     *      $formdata   = $mform->get_data();
     *      $text       = $formdata->fieldname['text'];
     *      $format     = $formdata->fieldname['format'];
     *
     * @param string $elementName  Input field name attribute
     * @param string $elementLabel (optional) Input field label
     * @param array  $attributes   (optional) HTML attribute string or an associative array
     * @param string $param_type   (optional) Defines type of data contained in element. Use the constants PARAM_*.
     * @param array  $options      (optional) Set of options to initalize filepicker, has priority on the next parameters
     *  Next parameters you can set in the options:
     * @param bool   $autosave
     * @param int    $noclean
     * @param int    $maxbytes
     * @param int    $maxfiles
     * @param null   $context
     * @param int    $subdirs
     * @param int    $changeformat
     * @param int    $areamaxbytes
     * @param int    $return_types          (optional) Default 15 = FILE_INTERNAL | FILE_EXTERNAL | FILE_REFERENCE | FILE_CONTROLLED_LINK.
     * @param bool   $enable_filemanagement (optional) Will display the file management button on true and remove it on false
     * @param bool   $removeorphaneddrafts
     *
     * @return static
     */
    public static function editor($elementName=null, $elementLabel=null, $attributes=null, $options=null, $param_type=PARAM_RAW,
            $autosave=true, $noclean=0, $maxbytes=0, $maxfiles=0, $context=null, $subdirs=0, $changeformat=0,
            $areamaxbytes=FILE_AREA_MAX_BYTES_UNLIMITED, $return_types=15, $enable_filemanagement=true, $removeorphaneddrafts=false
        ) {
        $fe = new static('editor', $elementName, $elementLabel, null, $attributes);
        $fe->setType($param_type);

        $options = NED::val2arr($options);
        $options['autosave']             = $options['autosave']             ?? $autosave;
        $options['noclean']              = $options['noclean']              ?? $noclean;
        $options['maxbytes']             = $options['maxbytes']             ?? $maxbytes;
        $options['maxfiles']             = $options['maxfiles']             ?? $maxfiles;
        $options['context']              = $options['context']              ?? $context;
        $options['subdirs']              = $options['subdirs']              ?? $subdirs;
        $options['changeformat']         = $options['changeformat']         ?? $changeformat;
        $options['areamaxbytes']         = $options['areamaxbytes']         ?? $areamaxbytes;
        $options['return_types']         = $options['return_types']         ?? $return_types;
        $options['enable_filemanagement']= $options['enable_filemanagement']?? $enable_filemanagement;
        $options['removeorphaneddrafts'] = $options['removeorphaneddrafts'] ?? $removeorphaneddrafts;
        $fe->setOptions($options);

        return $fe;
    }

    /**
     * textarea constructor
     *
     * @see \HTML_QuickForm_textarea
     * @see \MoodleQuickForm_textarea
     *
     * @param string $elementName  Input field name attribute
     * @param string $elementLabel (optional) Input field label
     * @param array  $attributes   (optional) HTML attribute string or an associative array, has priority on the next parameters
     * @param string $wrap
     * @param int    $rows
     * @param int    $cols
     * @param string $placeholder
     *
     * @return static
     */
    public static function textarea($elementName=null, $elementLabel=null, $attributes=null, $wrap='virtual',  $rows=5, $cols=50, $placeholder='') {
        $attributes = NED::val2arr($attributes);
        $attributes['wrap'] = $attributes['wrap'] ?? $wrap;
        $attributes['rows'] = $attributes['rows'] ?? $rows;
        $attributes['cols'] = $attributes['cols'] ?? $cols;
        $attributes['placeholder'] = $attributes['placeholder'] ?? $placeholder;

        return new static('textarea', $elementName, $elementLabel, null, $attributes);
    }

    /**
     * checkbox constructor
     *
     * @see \HTML_QuickForm_checkbox
     * @see \MoodleQuickForm_checkbox
     *
     * BEWARE: Unchecked checkboxes return nothing at all (as if they didn't exist). This can surprise the unwary.
     *  You may wish to use advcheckbox instead, which does return a value when not checked. 'Advcheckbox' eliminates this problem.
     *
     * @param string $elementName  Input field name attribute
     * @param string $elementLabel (optional) Input field label - display on the left side of the form
     * @param bool   $value        (optional) Default value to checkbox
     * @param string $text         (optional) Label2 to display on the right side of the form
     * @param mixed  $attributes   (optional) HTML attribute string or an associative array, has priority on the next parameters
     *
     * @return static
     */
    public static function checkbox($elementName=null, $elementLabel=null, $value=false, $text='', $attributes=null) {
        $fe = new static('checkbox', $elementName, $elementLabel, $text, $attributes);
        $fe->default = $value ? 1 : 0;
        return $fe;
    }

    /**
     * advcheckbox constructor
     *
     * @see \HTML_QuickForm_advcheckbox
     * @see \MoodleQuickForm_advcheckbox
     *
     * @param string $elementName  Input field name attribute
     * @param string $elementLabel (optional) Input field label - display on the left side of the form
     * @param bool   $value        (optional) Default value to checkbox
     * @param string $text         (optional) Label2 to display on the right side of the form
     * @param int    $group        (optional) Add a class name to the element, and enable its grouping for a checkbox controller
     *               @see CreateCheckboxController()
     * @param mixed  $attributes   (optional) HTML attribute string or an associative array, has priority on the next parameters
     * @param array  $options      (optional) [$option0, $option1] - where $option0 sent if advcheckbox unchecked, and $option1 - if checked,
     *                                  [0,1] by default (or if you sent null)
     *
     *
     * @return static
     */
    public static function advcheckbox($elementName=null, $elementLabel=null, $value=false, $text='', $group=null, $attributes=null, $options=null) {
        if (!is_null($group) && !isset($attributes['group'])){
            $attributes['group'] = $group;
        }
        $fe = new static('advcheckbox', $elementName, $elementLabel, $text, $attributes);
        $fe->default = $value ? 1 : 0;
        $fe->setOptions($options);
        return $fe;
    }

    /**
     * select constructor
     *
     * @see \HTML_QuickForm_select
     * @see \MoodleQuickForm_select
     *
     * @param string      $elementName  Input field name attribute
     * @param string      $elementLabel (optional) Input field label
     * @param array       $options      (optional) Data to be used to populate options
     * @param int|string  $value        (optional) Default value to selector, sent null to not set it
     * @param bool|string $showchoose   (optional) Add 'choose' option. If it's string, add this string to options
     * @param bool        $multi        (optional) Set select multi-select
     * @param mixed       $attributes   (optional) HTML attribute string or an associative array
     *
     *
     * @return static
     */
    public static function select($elementName=null, $elementLabel=null, $options=null, $value=0,
            $showchoose=false, $multi=false, $attributes=null) {
        $fe = new static('select', $elementName, $elementLabel, null, $attributes);
        $fe->default = $value;

        if ($showchoose){
            $choose = is_string($showchoose) ? $showchoose : (get_string('choose').'...');
            $options = NED::add2list($options, $choose);
        }
        $fe->setOptions($options);

        if ($multi){
            $fe->custom_data['multi'] = true;
        }
        return $fe;
    }

    /**
     * selectyesno constructor
     * No/Yes drop down type form element, where 0 => No and 1 => Yes
     *
     * @see \HTML_QuickForm_select
     * @see \MoodleQuickForm_selectyesno
     *
     * @param string      $elementName  Input field name attribute
     * @param string      $elementLabel (optional) Input field label
     * @param int|string  $value        (optional) Default value to selector, default 0 (No)
     * @param mixed       $attributes   (optional) HTML attribute string or an associative array
     *
     * @return static
     */
    public static function selectyesno($elementName=null, $elementLabel=null, $value=0, $attributes=null) {
        $options = [
            0 => get_string('no'),
            1 => get_string('yes'),
        ];
        $value = $value ? 1 : 0;

        return static::select($elementName, $elementLabel, $options, $value, false, false, $attributes);
    }

    /**
     * autocomplete constructor
     *
     * @see \HTML_QuickForm_autocomplete
     * @see \MoodleQuickForm_autocomplete
     *
     * @param string     $elementName  Input field name attribute
     * @param string     $elementLabel (optional) Input field label
     * @param array      $options      (optional) Data to be used to populate options
     * @param int|string $value        (optional) Default value to autocomplete
     * @param mixed      $attributes   (optional) HTML attribute string or an associative array, has priority on the next parameters + Date format
     *                                      Some attributes value (you can just store them in the $attributes):
     * @param bool     $multiple          (optional) Allow more than one selected item. The data coming from the form will be an array in this case.
     * @param string   $noselectionstring (optional) The text to display when nothing is selected.
     * @param bool     $showsuggestions   (optional) Do not show the list of suggestions when the user starts typing.
     * @param string   $placeholder       (optional) The text to show in the search box when it is empty.
     * @param bool     $casesensitive     (optional) Is the search case-sensitive ?
     * @param bool     $tags              (optional) The user can create new valid entries in the list by typing them and pressing enter.
     * @param string   $ajax              (optional) This string is the name of an AMD module that can fetch and format results.
     * @param callable $valuehtmlcallback (optional) For use with the AJAX option, so that it can format the initial value of the form field.
     *
     * @return static
     */
    public static function autocomplete($elementName=null, $elementLabel=null, $options=null, $value=false, $attributes=null,
            $multiple=false, $noselectionstring='', $showsuggestions=true, $placeholder='', $casesensitive=false, $tags=false,
            $ajax='', $valuehtmlcallback=null
    ) {
        $attributes = NED::val2arr($attributes);
        $attributes['multiple'] =           $attributes['multiple'] ?? $multiple;
        $attributes['noselectionstring'] =  $attributes['noselectionstring'] ?? $noselectionstring;
        $attributes['showsuggestions'] =    $attributes['showsuggestions'] ?? $showsuggestions;
        $attributes['placeholder'] =        $attributes['placeholder'] ?? $placeholder;
        $attributes['casesensitive'] =      $attributes['casesensitive'] ?? $casesensitive;
        $attributes['tags'] =               $attributes['tags'] ?? $tags;
        $attributes['ajax'] =               $attributes['ajax'] ?? $ajax;
        if (!empty($attributes['ajax']) && $valuehtmlcallback){
            $attributes['valuehtmlcallback'] = $attributes['valuehtmlcallback'] ?? $valuehtmlcallback;
        }

        $fe = new static('autocomplete', $elementName, $elementLabel, null, $attributes);
        $fe->default = $value;
        $fe->setOptions($options);

        return $fe;
    }

    /**
     * date_selector constructor
     * This is a date selector. You can select a Day, Month and Year using a group of select boxes.
     * When submitted, submitted data is processed and a timestamp is passed to $form->get_data();
     *
     * @see \MoodleQuickForm_date_selector
     *
     * @param string $elementName  Input field name attribute
     * @param string $elementLabel (optional) Input field label
     * @param array  $options      (optional) Options to control the element's display
     * @param int    $value        (optional) Default value, UNIX timestamp
     * @param array  $attributes   (optional) HTML attribute string or an associative array
     *  Next parameters you can set in the options ($options has priority on next):
     * @param int               $startyear      (optional) start of range of years that can be selected
     * @param int               $stopyear       (optional) last year that can be selected
     * @param int|float|string  $timezone       (optional) timezone modifier used for edge case only
     * @param bool              $optional       (optional) if true, show a checkbox beside the date to turn it on (or off)
     *
     * @return static
     */
    public static function date_selector($elementName=null, $elementLabel=null, $options=null, $value=null, $attributes=null,
        $startyear=null, $stopyear=null, $timezone=99, $optional=false
    ) {
        $fe = new static('date_selector', $elementName, $elementLabel, null, $attributes);
        if (!is_null($value)){
            $fe->default = $value;
        }

        $calendartype = \core_calendar\type_factory::get_calendar_instance();
        $startyear = $startyear ?? $calendartype->get_min_year();
        $stopyear = $stopyear ?? $calendartype->get_max_year();

        $options = NED::val2arr($options);
        $options['startyear']   = $options['startyear']     ?? $startyear;
        $options['stopyear']    = $options['stopyear']      ?? $stopyear;
        $options['timezone']    = $options['timezone']      ?? $timezone;
        $options['optional']    = $options['optional']      ?? $optional;
        $fe->setOptions($options);

        return $fe;
    }

    /**
     * date_time_selector constructor
     * This is a group of select boxes to select a date (Day Month and Year) and time (Hour and Minute).
     * When submitted, submitted data is processed and a timestamp is passed to $form->get_data();
     *
     * @see \MoodleQuickForm_date_time_selector
     *
     * @param string $elementName  Input field name attribute
     * @param string $elementLabel (optional) Input field label
     * @param array  $options      (optional) Options to control the element's display
     * @param int    $value        (optional) Default value, UNIX timestamp
     * @param array  $attributes   (optional) HTML attribute string or an associative array
     *  Next parameters you can set in the options ($options has priority on next):
     * @param int               $startyear      (optional) start of range of years that can be selected
     * @param int               $stopyear       (optional) last year that can be selected
     * @param int|float|string  $timezone       (optional) timezone modifier used for edge case only
     * @param int               $step           (optional) step to increment minutes by
     * @param bool              $optional       (optional) if true, show a checkbox beside the date to turn it on (or off)
     *
     * @return static
     */
    public static function date_time_selector($elementName=null, $elementLabel=null, $options=null, $value=null, $attributes=null,
        $startyear=null, $stopyear=null, $timezone=99, $step=1, $optional=false
    ) {
        $fe = static::date_selector($elementName, $elementLabel, $options, $value, $attributes, $startyear, $stopyear, $timezone, $optional) ;
        $fe->changeElementType('date_time_selector');

        $options = $fe->getOptions();
        $options['step'] = $options['step'] ?? $step;
        $fe->setOptions($options);

        return $fe;
    }

    /**
     * duration constructor
     * This field type lets the user input an interval of time.
     * It comprises a text field, where you can type a number, and a dropdown for selecting a unit (days, hours, minutes or seconds).
     * When submitted the value is converted to a number of seconds.
     *
     * @see \MoodleQuickForm_duration
     *
     * @param string $elementName  Input field name attribute
     * @param string $elementLabel (optional) Input field label
     * @param array  $options      (optional) Options to control the element's display
     * @param int    $value        (optional) Default value, number of seconds
     * @param array  $attributes   (optional) HTML attribute string or an associative array
     *  Next parameters you can set in the options ($options has priority on next):
     * @param bool  $optional    (optional) if true, show a checkbox beside the date to turn it on (or off), false by default
     * @param int   $defaultunit (optional) 1|MINSECS|HOURSECS|DAYSECS|WEEKSECS - the default unit to display, MINSECS by default.
     * @param array $units       (optional) containing some or all of 1, MINSECS, HOURSECS, DAYSECS and WEEKSECS which unit choices to offer.
     *
     * @return static
     */
    public static function duration($elementName=null, $elementLabel=null, $options=null, $value=null, $attributes=null,
        $optional=false, $defaultunit=MINSECS, $units=null
    ) {
        $fe = new static('duration', $elementName, $elementLabel, null, $attributes);
        if (!is_null($value)){
            $fe->default = $value;
        }

        $options = NED::val2arr($options);
        $options['optional']    = $options['optional']    ?? $optional;
        $options['defaultunit'] = $options['defaultunit'] ?? $defaultunit;
        if ($units){
            $options['units']   = $options['units']       ?? $units;
        }

        $fe->setOptions($options);

        return $fe;
    }

    /**
     * group constructor
     * If $appendName is true, elements data will be in different list (named by group name)
     *
     * @see \HTML_QuickForm_group
     * @see \HTML_QuickForm::addGroup()
     * @see \MoodleQuickForm_group
     *
     * @param string       $elementName  (optional) Group name
     * @param string       $elementLabel (optional) Group label
     * @param array        $elements     (optional) Group elements
     * @param string|array $separator    (optional) Use a string for one separator, use an array to alternate the separators.
     * @param bool         $appendName   (optional) whether to change elements' names to the form $groupName[$elementName] or leave them as is.
     *
     *
     * @return static
     */
    public static function group($elementName=null, $elementLabel=null, $elements=null, $separator=null, $appendName=false) {
        $fe = new static('group', $elementName, $elementLabel);
        $fe->setOptions($elements);
        $fe->custom_data['separator'] = $separator;
        $fe->custom_data['appendName'] = $appendName;
        return $fe;
    }

    /**
     * radio constructor
     * It's group element with radio data
     *
     * @see \HTML_QuickForm_group
     * @see \HTML_QuickForm_radio
     * @see \MoodleQuickForm_group
     * @see \MoodleQuickForm_radio
     *
     * @param string     $elementName (optional) Radio name
     * @param array      $elements    (optional) Radio elements as array [value => text]
     * @param int|string $value       (optional) Default value to radio
     * @param mixed      $attributes  (optional) HTML attribute string or an associative array, or array of $attributes, by keys from $elements
     * @param string     $groupName   (optional) Group name
     * @param string     $groupLabel  (optional) Group label
     *
     * @return static
     */
    public static function radio($elementName=null, $elements=null, $value=null, $attributes=null, $groupName=null, $groupLabel='') {
        $radio_elements = [];
        $attributes = NED::val2arr($attributes);
        foreach ($elements as $key => $text){
            $radio_elements[] = static::radio_raw($elementName, $key, $text, $attributes[$key] ?? $attributes);
        }

        $groupName = $groupName ?? $elementName.'_group';
        $fe = static::group($groupName, $groupLabel, $radio_elements);
        if (!is_null($value)){
            $fe->custom_data['custom_defaults'][] = [$elementName, $value];
        }
        return $fe;
    }

    /**
     * Raw radio constructor
     * Create only one radio element. If you need some of them, check static radio() function
     *
     * @see radio()
     * @see \HTML_QuickForm_radio
     * @see \MoodleQuickForm_radio
     *
     * @param string     $elementName (optional) Radio name
     * @param int|string $value       (optional) Default value to radio
     * @param string     $text        (optional) Text to display near the radio
     * @param mixed      $attributes  (optional) HTML attribute string or an associative array, or array of $attributes, by keys from $elements
     *
     * @return static
     */
    public static function radio_raw($elementName=null, $value=null, $text='', $attributes=null) {
        return new static('radio', $elementName, $value, $text, $attributes);
    }

    /**
     * button constructor
     *
     * @see \HTML_QuickForm_button
     * @see \MoodleQuickForm_button
     *
     * @param string $elementName Input field name attribute
     * @param string $value       (optional) Input field value
     * @param mixed  $attributes  (optional) HTML attribute string or an associative array
     *
     * @return static
     */
    public static function button($elementName=null, $value='', $attributes=null) {
        return new static('button', $elementName, $value, null, $attributes);
    }

    /**
     * reset constructor
     *
     * @see \HTML_QuickForm_reset
     *
     * @param string $elementName Input field name attribute
     * @param string $value       (optional) Input field value
     * @param mixed  $attributes  (optional) HTML attribute string or an associative array
     *
     * @return static
     */
    public static function reset($elementName=null, $value='', $attributes=null) {
        return new static('reset', $elementName, $value, null, $attributes);
    }

    /**
     * submit constructor
     * You can leave all parameters empty, if it's one such element
     *
     * @see \HTML_QuickForm_submit
     * @see \MoodleQuickForm_submit
     *
     * @param string        $elementName Input field name attribute
     * @param string        $value      (optional) Input field value
     * @param bool|null     $primary Is this button a primary button?
     * @param string|array  $class      (optional) HTML class of the button
     * @param mixed         $attributes (optional) HTML attribute string or an associative array
     *
     * @return static
     */
    public static function submit($elementName=null, $value='', $primary=null, $class='', $attributes=null) {
        $elementName = $elementName ?? 'submit';
        $attributes = NED::val2arr($attributes);
        $class = NED::val2arr($class);
        $attributes['customclassoverride'] = NED::arr2str($class, $attributes['customclassoverride'] ?? '');
        $attributes['primary'] = $primary;

        return new static('submit', $elementName, $value, null, $attributes);
    }

    /**
     * cancel constructor
     * You can leave all parameters empty, if it's one such element
     *
     * @see \MoodleQuickForm_cancel
     *
     * @param string $elementName Input field name attribute
     * @param string $value       (optional) Input field value
     * @param mixed  $attributes  (optional) HTML attribute string or an associative array
     *
     * @return static
     */
    public static function cancel($elementName=null, $value='', $attributes=null) {
        $elementName = $elementName ?? 'cancel';
        return new static('cancel', $elementName, $value, null, $attributes);
    }
    //endregion

    //region Create not-elements
    /**
     * Adds a link/button that controls the checked state of a group of checkboxes.
     *
     * @see moodleform::add_checkbox_controller()
     * @see advcheckbox()
     *
     * @param int    $groupid       The id of the group of advcheckboxes this element controls ($group param from the advcheckbox() function)
     * @param string $text          The text of the link. Defaults to selectallornone ("select all/none")
     * @param array  $attributes    associative array of HTML attributes
     * @param int    $originalValue The original general state of the checkboxes before the user first clicks this element
     *
     * @return array - arguments for the add_checkbox_controller function
     */
    public static function CreateCheckboxController($groupid, $text=null, $attributes=null, $originalValue=0){
        return [$groupid, $text, $attributes, $originalValue];
    }

    /**
     * Adds a validation rule for the given field
     *
     * If the element is in fact a group, it will be considered as a whole.
     * To validate grouped elements as separated entities,
     * use addGroupRule instead of addRule.
     *
     * The usual Rule types are: 'required', 'maxlength', 'minlength', 'rangelength', 'email',
     *      'regex', 'lettersonly', 'alphanumeric', 'numeric',
     *      'nopunctuation', 'nonzero', 'callback', 'compare'
     *
     * @see \MoodleQuickForm::addRule()
     * @see \HTML_QuickForm::addRule()
     * @see \HTML_QuickForm::getRegisteredRules()
     *
     * @param    string     $element       Form element name
     * @param    string     $type          Rule type, use getRegisteredRules() to get types
     * @param    string     $message       (optional) Message to display for invalid data
     * @param    string     $format        (optional) Required for extra rule data
     * @param    string     $validation    (optional) Where to perform validation: "server", "client"
     * @param    boolean    $reset         (optional) Client-side validation: reset the form element to its original value if there is an error?
     * @param    boolean    $force         (optional) Force the rule to be applied, even if the target form element does not exist
     * @param    bool       $without_name  (optional) Return without element name
     *
     * @return array - arguments for the form addRule() function, list($element, $message, $type, $format, $validation, $reset, $force)
     */
    public static function CreateRule($element, $type, $message='', $format=null, $validation='client', $reset=false, $force=false, $without_name=0){
        if ($without_name){
            return [$message, $type, $format, $validation, $reset, $force];
        }
        return [$element, $message, $type, $format, $validation, $reset, $force];
    }

    /**
     * Adds a validation rule 'required' for the given field
     *
     * @see CreateRule()
     *
     * @param    string     $element       Form element name
     * @param    string     $message       (optional) Message to display for invalid data, get_string('required') by default
     * @param    string     $validation    (optional) Where to perform validation: "server", "client"
     * @param    boolean    $reset         (optional) Client-side validation: reset the form element to its original value if there is an error?
     * @param    boolean    $force         (optional) Force the rule to be applied, even if the target form element does not exist
     * @param    bool       $without_name  (optional) Return without group name
     *
     * @return array - arguments for the form addRule() function, list($element, $message, $type, $format, $validation, $reset, $force)
     */
    public static function CreateRequired($element, $message=null, $validation='client', $reset=false, $force=false, $without_name=false){
        $message = $message ?? get_string('required');
        return static::CreateRule($element, 'required', $message, null, $validation, $reset, $force, $without_name);
    }

    /**
     * Adds a validation rule for the given group of elements
     *
     * Only groups with a name can be assigned a validation rule
     * Use addGroupRule when you need to validate elements inside the group.
     * Use addRule if you need to validate the group as a whole. In this case,
     * the same rule will be applied to all elements in the group.
     * Use addRule if you need to validate the group against a function.
     *
     * @see \HTML_QuickForm::addGroupRule()
     * @see CreateRule()
     *
     * @param    string     $group         Form group name
     * @param string|array  $arg1          (optional) Array for multiple elements or error message string for one element
     * @param    string     $type          (optional) Rule type use getRegisteredRules() to get types
     * @param    string     $format        (optional) Required for extra rule data
     * @param    int        $howmany       (optional) How many valid elements should be in the group
     * @param    string     $validation    (optional) Where to perform validation: "server", "client"
     * @param    bool       $reset         (optional) Client-side: whether to reset the element's value to its original state if validation failed.
     * @param    bool       $without_name  (optional) Return without group name
     *
     * @return array - arguments for the form addGroupRule() function - list($group, $arg1, $type, $format, $howmany, $validation, $reset)
     */
    public static function CreateGroupRule($group, $type='', $arg1='', $format=null, $howmany=0, $validation='client', $reset=false, $without_name=0){
        if ($without_name){
            return [$arg1, $type, $format, $howmany, $validation, $reset];
        }
        return [$group, $arg1, $type, $format, $howmany, $validation, $reset];
    }
    //endregion

    //region Form static
    /**
     * Try to create moodle form element by $mform
     * $element be both current class or manual array
     *
     * @see createByForm()
     * @see createElementFromArray()
     * @see \MoodleQuickForm::createElement()
     *
     * @param \MoodleQuickForm                      $mform
     * @param array|static|\HTML_QuickForm_element  $element
     * @param bool                                  $and_add - add to form after creating
     * @param string                                $group_name - group name if elements for group
     *
     * @return \HTML_QuickForm_element|null
     */
    static public function &createElement($mform, $element, $and_add=false, $group_name=''){
        $res = null;
        if (empty($element)){
            return $res;
        }

        if (is_object($element)){
            if (is_a($element, 'HTML_QuickForm_element')){
                $res = &$element;
                if ($and_add){
                    $res = &$mform->addElement($res);
                }
            } elseif (is_a($element, get_called_class())){
                $res = &$element->createByForm($mform, $and_add, $group_name);
            }
        } elseif (is_array($element)){
            $res = &static::createElementFromArray($mform, $element, $and_add);
        }

        return $res;
    }

    /**
     * Create moodle form elements by $mform
     *
     * @see createElement()
     *
     * @param \MoodleQuickForm                                  $mform
     * @param array|array[]|static[]|\HTML_QuickForm_element[]  $elements
     * @param string                                            $group_name - group name if elements for group
     *
     * @return \HTML_QuickForm_element[]
     */
    static public function createElements($mform, $elements, $group_name=''){
        $res = [];
        if (empty($elements)){
            return $res;
        }
        foreach ($elements as $element){
            $el = &static::createElement($mform, $element, false, $group_name);
            if ($el){
                $res[] = $el;
            }
        }
        return $res;
    }

    /**
     * Create and add element to the $mform
     *
     * @see createElement()
     * @see \MoodleQuickForm::addElement()
     *
     * @param \MoodleQuickForm                      $mform
     * @param array|static|\HTML_QuickForm_element  $element
     * @param string                                $group_name - group name if elements for group
     *
     * @return \HTML_QuickForm_element|null
     */
    static public function &addElement($mform, $element, $group_name=''){
        return static::createElement($mform, $element, true, $group_name);
    }

    /**
     * Create and add list of elements to the $mform
     *
     * @see addElement()
     * @see \MoodleQuickForm::addElement()
     *
     * @param \MoodleQuickForm                                  $mform
     * @param array|array[]|static[]|\HTML_QuickForm_element[]  $elements
     * @param string                                            $group_name - group name if elements for group
     *
     * @return \HTML_QuickForm_element[]
     */
    static public function addElements($mform, $elements, $group_name=''){
        $res = [];
        if (empty($elements)){
            return $res;
        }
        foreach ($elements as $element){
            $el = &static::addElement($mform, $element, $group_name);
            if ($el){
                $res[] = $el;
            }
        }
        return $res;
    }

    /**
     * Create element from the manual array
     * Function works with the same logic as createByForm
     * You can get needed array from export() function, but it's better to use createByForm function, if you can
     * Possible keys for the array you can find here or in export() function
     *
     * @see createByForm()
     * @see export()
     *
     * @param \MoodleQuickForm  $mform
     * @param array             $element
     * @param bool              $and_add - add to form after creating
     *
     * @return \HTML_QuickForm_element
     */
    static public function &createElementFromArray($mform, $element, $and_add=false){
        if ($element[static::GROUP] ?? false){
            $element[static::OPTIONS] = static::createElements($mform, $element[static::OPTIONS] ?? []);
        }

        /** @var \HTML_QuickForm_element|\HTML_QuickForm_select|object $qf_element */
        $qf_element =& $mform->createElement($element[static::ARGS]);
        if ($and_add){
            $qf_element =& $mform->addElement($qf_element);
        }

        if ($element[static::CUSTOM_DATA]['multi'] ?? false){
            $qf_element->setMultiple(true);
        }
        if (!empty($element[static::CUSTOM_DATA]['custom_defaults'])){
            foreach ($element[static::CUSTOM_DATA]['custom_defaults'] as $custom_default){
                $mform->setDefault(...$custom_default);
            }
        }

        if (empty($element[static::NAME])){
            return $qf_element;
        }

        $name = $element[static::NAME];
        if (!empty($element[static::PARAM_TYPE])){
            $mform->setType($name, $element[static::PARAM_TYPE]);
        }
        if (!empty($element[static::DEFAULT])){
            $mform->setDefault($name, $element[static::DEFAULT]);
        }
        if (!empty($element[static::HELP])){
            $mform->addHelpButton(...$element[static::HELP]);
        }
        if (!empty($element[static::GROUP_RULES])){
            $rules = $element[static::GROUP_RULES];
            foreach ($rules as $rule){
                $mform->addGroupRule(...$rule);
            }
        }
        if (!empty($element[static::RULES])){
            $rules = $element[static::RULES];
            foreach ($rules as $rule){
                $mform->addRule(...$rule);
            }
        }
        if (!empty($element[static::DISABLEDIFS])){
            $disabledIfs = $element[static::DISABLEDIFS];
            foreach ($disabledIfs as $disabledIf){
                $mform->disabledIf(...$disabledIf);
            }
        }
        if (!empty($element[static::HIDEIFS])){
            $hideIfs = $element[static::HIDEIFS];
            foreach ($hideIfs as $hideIf){
                $mform->hideIf(...$hideIf);
            }
        }

        return $qf_element;
    }
    //endregion

    //region Simple methods
    /**
     * Get args for addElement function
     *
     * @see \MoodleQuickForm::addElement
     *
     * @return array
     */
    public function getArgs(){
        switch ($this->_type){
            case 'html':
                return [$this->_type, $this->text];
            case 'static':
                return [$this->_type, $this->name, $this->label_or_value, $this->text];
            case 'header':
                return [$this->_type, $this->name, $this->text];
            case 'hidden':
            case 'text':
            case 'textarea':
            case 'button':
            case 'reset':
            case 'submit':
                $attr = $this->getAttributes();
                $options = ['customclassoverride' => ($attr['customclassoverride'] ?? false)];
                return [$this->_type, $this->name, $this->label_or_value, $attr, $attr['primary'] ?? null, $options];
            case 'cancel':
                return [$this->_type, $this->name, $this->label_or_value, $this->getAttributes()];
            case 'editor':
                return [$this->_type, $this->name, $this->label_or_value, $this->getAttributes(), $this->getOptions()];
            case 'checkbox':
                return [$this->_type, $this->name, $this->label_or_value, $this->text, $this->getAttributes()];
            case 'advcheckbox':
                return [$this->_type, $this->name, $this->label_or_value, $this->text, $this->getAttributes(), $this->getOptions()];
            case 'select':
            case 'autocomplete':
            case 'date_selector':
            case 'date_time_selector':
            case 'duration':
                return [$this->_type, $this->name, $this->label_or_value, $this->getOptions(), $this->getAttributes()];
            case 'radio':
                return [$this->_type, $this->name, '', $this->text, $this->label_or_value, $this->getAttributes()];
            case 'group':
                // you also need to check
                return [$this->_type, $this->name, $this->label_or_value, $this->getOptions(),
                    $this->custom_data['separator'], $this->custom_data['appendName']];
        }

        return [$this->_type];
    }

    /**
     * Return base element type (kind/form)
     *
     * @return string
     */
    public function getElementType(){
        return $this->_type;
    }

    /**
     * Changes base element type (kind/form)
     * It possibly can break many things
     * Do it only if you really know, what do you do
     *
     * @param string $type
     */
    public function changeElementType($type){
        $this->_type = $type;
    }

    /**
     * @param string $name
     */
    public function setName($name){
        $this->name = $name;
    }

    /**
     * @return bool
     */
    public function hasName(){
        return !empty($this->name);
    }

    /**
     * @return string
     */
    public function getName(){
        return $this->name;
    }

    /**
     * Return name with group name like "GroupName[ElementName]
     * If group name is empty, return just name
     *
     * @param string $group_name - group name if element for group
     *
     * @return string
     */
    public function getNameWithGroup($group_name=''){
        $name = $this->getName();
        if (!empty($name) && !empty($group_name)){
            $name = $group_name.'['.$name.']';
        }

        return $name;
    }

    /**
     * Get array of the HTML attributes
     *
     * @return array
     */
    public function getAttributes(){
        return NED::val2arr($this->_attributes);
    }

    /**
     * Replace all HTML attributes
     *
     * @param array $_attributes
     */
    public function setAttributes($_attributes){
        $this->_attributes = NED::val2arr($_attributes);
    }

    /**
     * Get HTML attribute by key
     *
     * @param string $key
     *
     * @return mixed|null
     */
    public function getAttribute($key){
        $attributes = $this->getAttributes();
        return $attributes[$key] ?? null;
    }

    /**
     * Set HTML attribute by key
     *
     * @param string $key
     * @param mixed  $value
     */
    public function setAttribute($key, $value){
        $attributes = $this->getAttributes();
        $attributes[$key] = $value;
        $this->setAttributes($attributes);
    }

    /**
     * Set HTML attribute by key
     * Alias of the {@see setAttribute()}
     *
     * @param string $key
     * @param mixed  $value
     */
    public function addAttribute($key, $value){
        $this->setAttribute($key, $value);
    }

    /**
     * Merge new HTML attributes with the existing one
     *
     * @param array $new_attributes
     */
    public function mergeAttributes($new_attributes){
        $attributes = $this->getAttributes();
        $attributes = array_merge($attributes, $new_attributes);
        $this->setAttributes($attributes);
    }

    /**
     * Return HTML class of the element
     *
     * @return string
     */
    public function getHtmlClass(){
        return $this->getAttribute('class') ?? '';
    }

    /**
     * Add new HTML class to the element
     *
     * @param string|array $add_class
     *
     * @return void
     */
    public function addHtmlClass($add_class=''){
        $this->setAttribute('class', NED::arr2str($this->getHtmlClass(), $add_class));
    }

    /**
     * Replace HTML class of the element
     *
     * @param string|array $new_class
     *
     * @return void
     */
    public function setHtmlClass($new_class=''){
        $this->setAttribute('class', NED::arr2str($new_class));
    }

    /**
     * @return bool
     */
    public function isGroup(){
        return $this->_type == 'group';
    }

    /**
     * @return array
     */
    public function getOptions(){
        return $this->_options;
    }

    /**
     * @param $options
     */
    public function setOptions($options){
        $this->_options = $options;
    }

    /**
     * Set default value to the element
     *
     * @param mixed|null $val - if it will be null, it will unset default
     */
    public function setDefault($val=null){
        $this->default = $val;
    }

    /**
     * @return bool
     */
    public function hasDefault(){
        return $this->hasName() && !is_null($this->default);
    }

    /**
     * Add rule to the element
     *
     * @see CreateRule()
     * @see \MoodleQuickForm::addRule()
     * @see \HTML_QuickForm::addRule()
     *
     * @param string|array  $rule_or_type  Rule argument (array) or Rule type(string), use getRegisteredRules() to get types
     * @param    string     $message       (optional) Message to display for invalid data
     * @param    string     $format        (optional) Required for extra rule data
     * @param    string     $validation    (optional) Where to perform validation: "server", "client"
     * @param    boolean    $reset         (optional) Client-side validation: reset the form element to its original value if there is an error?
     * @param    boolean    $force         (optional) Force the rule to be applied, even if the target form element does not exist
     */
    public function addRule($rule_or_type, $message='', $format=null, $validation='client', $reset=false, $force=false){
        if (!is_array($rule_or_type)){
            if (empty($message)){
                $message = get_string('err_'.$rule_or_type, 'form', $format);
            }
            $rule = static::CreateRule($this->name, $rule_or_type, $message, $format, $validation, $reset, $force, true);
        } else {
            $rule = $rule_or_type;
        }

        $this->_rules[] = $rule;
    }

    /**
     * Adds a validation rule 'required' for the given element
     *
     * @see CreateRule()
     *
     * @param    string     $message       (optional) Message to display for invalid data, get_string('required') by default
     * @param    string     $validation    (optional) Where to perform validation: "server", "client"
     * @param    boolean    $reset         (optional) Client-side validation: reset the form element to its original value if there is an error?
     * @param    boolean    $force         (optional) Force the rule to be applied, even if the target form element does not exist
     */
    public function setRequired($message=null, $validation='client', $reset=false, $force=false){
        if ($this->_type == 'text'){
            $message = $message ?? get_string('required');
            $this->_rules[] = static::CreateRule($this->name, 'minlength', $message, 1, $validation, $reset, $force, true);
        }

        $this->_rules[] = static::CreateRequired($this->name, $message, $validation, $reset, $force, true);
    }

    /**
     * Adds a simple validation rule 'required' for the given element
     * If you wish to use some settings - use setRequired() function instead
     *
     * @see setRequired()
     *
     * @param boolean $required (optional) If true - set element required
     */
    public function setSimpleRequired($required=true){
        if (!$required){
            return;
        }
        $this->setRequired();
    }

    /**
     * Adds a validation rule for the given group of elements
     *
     * Only groups with a name can be assigned a validation rule
     * Use addGroupRule when you need to validate elements inside the group.
     * Use addRule if you need to validate the group as a whole. In this case,
     * the same rule will be applied to all elements in the group.
     * Use addRule if you need to validate the group against a function.
     *
     * @see CreateGroupRule()
     *
     * @param    string     $type          (optional) Rule type use getRegisteredRules() to get types
     * @param string|array  $arg1          (optional) Array for multiple elements or error message string for one element
     * @param    string     $format        (optional) Required for extra rule data
     * @param    int        $howmany       (optional) How many valid elements should be in the group
     * @param    string     $validation    (optional) Where to perform validation: "server", "client"
     * @param    bool       $reset         Client-side: whether to reset the element's value to its original state if validation failed.
     */
    public function addGroupRule($type='', $arg1='', $format=null, $howmany=0, $validation='client', $reset=false){
        if (!$this->isGroup()){
            return;
        }

        $this->custom_data['group_rules'] = $this->custom_data['group_rules'] ?? [];
        $this->custom_data['group_rules'][] = static::CreateGroupRule($this->name, $type, $arg1, $format, $howmany, $validation, $reset, true);
    }

    /**
     * @return bool
     */
    public function hasRules(){
        return $this->hasName() && !empty($this->_rules);
    }

    /**
     * @see addRule()
     * @see \MoodleQuickForm::addRule()
     * @see \HTML_QuickForm::addRule()
     *
     * @param string $group_name - group name if element for group
     *
     * @return array[] - list with args for the \MoodleQuickForm::addRule() function
     */
    public function getRules($group_name=''){
        if (!$this->hasName()){
            return [];
        }

        $rules = [];
        foreach ($this->_rules as $rule){
            [$message, $type, $format, $validation, $reset, $force] = $rule;
            $rules[] = [$this->getNameWithGroup($group_name), $message, $type, $format, $validation, $reset, $force];
        }

        return $rules;
    }

    /**
     * @see addRule()
     * @see \MoodleQuickForm::addRule()
     * @see \HTML_QuickForm::addRule()
     *
     * @param $rules - list with args for the \MoodleQuickForm::addRule() function, but without element name
     */
    public function setRules($rules){
        $this->_rules = $rules;
    }

    /**
     * @see addGroupRule()
     *
     * @return bool
     */
    public function hasGroupRules(){
        return $this->hasName() && $this->isGroup() && !empty($this->custom_data['group_rules']);
    }

    /**
     * @see addGroupRule()
     * @see \MoodleQuickForm::addGroupRule()
     * @see \HTML_QuickForm::addGroupRule()
     *
     * @param string $group_name - group name if element for group
     *
     * @return array[] - list with args for the \MoodleQuickForm::addGroupRule() function
     */
    public function getGroupRules($group_name=''){
        if (!$this->hasGroupRules()){
            return [];
        }

        $rules = [];
        foreach ($this->custom_data['group_rules'] as $rule){
            [$type, $arg1, $format, $howmany, $validation, $reset] = $rule;
            $rules[] = [$this->getNameWithGroup($group_name), $type, $arg1, $format, $howmany, $validation, $reset];
        }

        return $rules;
    }

    /**
     * @see addGroupRule()
     * @see \MoodleQuickForm::addGroupRule()
     * @see \HTML_QuickForm::addGroupRule()
     *
     * @param $rules - list with args for the \MoodleQuickForm::addGroupRule() function, but without element name
     */
    public function setGroupRules($rules){
        $this->custom_data['group_rules'] = $rules;
    }

    /**
     * @see setHelp()
     *
     * @return bool
     */
    public function hasHelp(){
        return $this->hasName() && !empty($this->_help);
    }

    /**
     * Add a help button to element, only one button per element is allowed.
     *
     * @see \MoodleQuickForm::addHelpButton()
     *
     * Typically, you will provide the same identifier and the component as you have used for the
     * label of the element. The string identifier with the _help suffix added is then used
     * as the help string.
     *
     * There has to be two strings defined:
     *   1/ get_string($identifier, $component) - the title of the help page
     *   2/ get_string($identifier.'_help', $component) - the actual help page text
     *
     * You can also add the language string like
     *   get_string($identifier.'_link', $component) - to add a link to more help on Moodle docs
     * 1/ By default, these link string should consist of a short relative path like 'component/thing' and link to the 'https://docs.moodle.org
     * 2/ Use <nowiki>'YOUR_LINK';</nowiki> to set other external link.
     * 3/ Or start the link with %%WWWROOT%%. That is replaced by $CFG->wwwroot, so, it creates link to page on your site
     *
     * @param string $identifier    Help string identifier without _help suffix
     * @param string $component     (optional) Component name to look the help string in
     * @param string $linktext      (optional) Text to display next to the icon
     * @param bool   $suppresscheck (optional) Set to true if the element may not exist
     */
    public function setHelp($identifier, $component=NED::CTRL, $linktext='', $suppresscheck=false){
        $this->_help = [$identifier, $component, $linktext, $suppresscheck];
    }

    /**
     * Return args for the MoodleQuickForm::addHelpButton() function
     *
     * @see setHelp()
     * @see \MoodleQuickForm::addHelpButton()
     *
     * @param string $group_name - group name if element for group
     *
     * @return array - $elementname, $identifier, $component, $linktext, $suppresscheck
     */
    public function getHelp($group_name=''){
        if (!$this->hasHelp()){
            return [];
        }

        [$identifier, $component, $linktext, $suppresscheck] = $this->_help;
        return [$this->getNameWithGroup($group_name), $identifier, $component, $linktext, $suppresscheck];
    }

    /**
     * @see \MoodleQuickForm::disabledIf()
     *
     * Adds a dependency for element which will be disabled if $condition is met.
     * If $condition = 'notchecked' (default) then the condition is that the $dependentOn element
     * is not checked. If $condition = 'checked' then the condition is that the $dependentOn element
     * is checked. If $condition is something else (like "eq" for equals) then it is checked to see if the value
     * of the $dependentOn element is $condition (such as equal) to $value.
     *
     * When working with multiple selects, the dependentOn has to be the real name of the select, meaning that
     * it will most likely end up with '[]'. Also, the value should be an array of required values, or a string
     * containing the values separated by pipes: array('red', 'blue') or 'red|blue'.
     *
     * $condition will be 'notchecked', 'checked', 'noitemselected', 'eq', 'in' or, if it is anything else, we test for 'neq'.
     *      • If $condition is 'eq' or 'neq' then we check the value of the dependentOn field and check for equality (==) or nonequality (!=) in js
     *      • If $condition is 'checked' or 'notchecked' then we check to see if a checkbox is checked or not.
     *      • If $condition is 'in' then we check to see if a selected item is in the given list or not.
     *      • If $condition is 'noitemselected' then we check to see whether nothing is selected in a dropdown list.
     *
     * Important: don't use 'eq'/'neq' to 1/0 for checkboxes, it works incorrectly - you should use 'checked' or 'notchecked' instead
     *
     * @param string $dependentOn - the name of the element whose state will be checked for condition
     * @param string $condition   - the condition to check
     * @param mixed  $value       - used in conjunction with condition.
     */
    public function addDisabledIf($dependentOn, $condition='notchecked', $value='1'){
        $this->_disabledIfs[] = [$dependentOn, $condition, $value];
    }

    /**
     * @see addDisabledIf()
     *
     * @param array[] $disabledIfs - it should be listed with args for the disabledIf() function, but without element name
     */
    public function setDisabledIfs($disabledIfs){
        $this->_disabledIfs = $disabledIfs;
    }

    /**
     * @see addDisabledIf()
     *
     * @return bool
     */
    public function hasDisabledIf(){
        return $this->hasName() && !empty($this->_disabledIfs);
    }

    /**
     * @see \MoodleQuickForm::disabledIf()
     * @see addDisabledIf()
     *
     * @param string $group_name - group name if element for group
     *
     * @return array[] - list with args for the disabledIf() function
     */
    public function getDisabledIfs($group_name=''){
        if (!$this->hasDisabledIf()){
            return [];
        }

        $disabledIfs = [];
        foreach ($this->_disabledIfs as $disabledIf){
            [$dependentOn, $condition, $value] = $disabledIf;
            $disabledIfs[] = [$this->getNameWithGroup($group_name), $dependentOn, $condition, $value];
        }

        return $disabledIfs;
    }

    /**
     * @see \MoodleQuickForm::hideIf()
     *
     * Adds a dependency for element which will be hidden if $condition is met.
     * If $condition = 'notchecked' (default) then the condition is that the $dependentOn element
     * is not checked. If $condition = 'checked' then the condition is that the $dependentOn element
     * is checked. If $condition is something else (like "eq" for equals) then it is checked to see if the value
     * of the $dependentOn element is $condition (such as equal) to $value.
     *
     * When working with multiple selects, the dependentOn has to be the real name of the select, meaning that
     * it will most likely end up with '[]'. Also, the value should be an array of required values, or a string
     * containing the values separated by pipes: array('red', 'blue') or 'red|blue'.
     *
     * $condition will be 'notchecked', 'checked', 'noitemselected', 'eq', 'in' or, if it is anything else, we test for 'neq'.
     *      • If $condition is 'eq' or 'neq' then we check the value of the dependentOn field and check for equality (==) or nonequality (!=) in js
     *      • If $condition is 'checked' or 'notchecked' then we check to see if a checkbox is checked or not.
     *      • If $condition is 'in' then we check to see if a selected item is in the given list or not.
     *      • If $condition is 'noitemselected' then we check to see whether nothing is selected in a dropdown list.
     *
     * Important: don't use 'eq'/'neq' to 1/0 for checkboxes, it works incorrectly - you should use 'checked' or 'notchecked' instead
     *
     * @param string $dependentOn - the name of the element whose state will be checked for condition
     * @param string $condition   - the condition to check
     * @param mixed  $value       - used in conjunction with condition.
     */
    public function addHideIf($dependentOn, $condition='notchecked', $value='1'){
        $this->_hideIfs[] = [$dependentOn, $condition, $value];
    }

    /**
     * @see addHideIf()
     *
     * @param array[] $hideIfs - it should be listed with args for the hideIf() function, but without element name
     */
    public function setHideIfs($hideIfs){
        $this->_hideIfs = $hideIfs;
    }

    /**
     * @see addHideIf()
     *
     * @return bool
     */
    public function hasHideIf(){
        return $this->hasName() && !empty($this->_hideIfs);
    }

    /**
     * @see \MoodleQuickForm::hideIf()
     * @see addHideIfs()
     *
     * @param string $group_name - group name if element for group
     *
     * @return array[] - list with args for the disabledIf() function
     */
    public function getHideIfs($group_name=''){
        if (!$this->hasHideIf()){
            return [];
        }

        $hideIfs = [];
        foreach ($this->_hideIfs as $hideIf){
            [$dependentOn, $condition, $value] = $hideIf;
            $hideIfs[] = [$this->getNameWithGroup($group_name), $dependentOn, $condition, $value];
        }

        return $hideIfs;
    }

    /**
     * @see \MoodleQuickForm::setType()
     *
     * @param string $param_type defines type of data contained in element. Use the constants PARAM_*.
     */
    public function setType($param_type=PARAM_RAW){
        $this->_param_type = $param_type;
    }

    /**
     * @see \MoodleQuickForm::setType()
     *
     * @return string - type of data contained in element (see constants PARAM_*)
     */
    public function getType(){
        return $this->_param_type;
    }

    /**
     * @return bool
     */
    public function hasType(){
        return $this->hasName() && !empty($this->_param_type);
    }
    //endregion

    //region Form export methods
    /**
     * Return element data as array
     * Result array (or similar) you can transform to the form element by createElementFromArray()
     *
     * @see createElementFromArray()
     *
     * @return array[]
     */
    public function export(){
        $res = [];
        $res[static::TYPE] = $this->getElementType();
        $res[static::NAME] = $this->getName();
        $res[static::PARAM_TYPE] = $this->getType();
        $res[static::ATTRIBUTES] = $this->getAttributes();
        $res[static::GROUP] = $this->isGroup();

        $res[static::ARGS] = $this->getArgs();
        $res[static::RULES] = $this->getRules();
        $res[static::GROUP_RULES] = $this->getGroupRules();
        $res[static::HELP] = $this->getHelp();
        $res[static::OPTIONS] = $this->getOptions();
        $res[static::DISABLEDIFS] = $this->getDisabledIfs();
        $res[static::HIDEIFS] = $this->getHideIfs();
        $res[static::CUSTOM_DATA] = $this->custom_data;

        return $res;
    }

    /**
     * Get form element by the objects data
     *
     * @see \MoodleQuickForm::createElement
     *
     * @param \MoodleQuickForm $mform
     * @param bool             $and_add - add to form after creating
     * @param string           $group_name - group name if elements for group
     *
     * @return \HTML_QuickForm_element|object
     */
    public function &createByForm($mform, $and_add=false, $group_name=''){
        $name = $this->getName();
        $rule_name = $this->getNameWithGroup($group_name);
        if ($this->isGroup()){
            $next_group_name = $group_name;
            if (!empty($this->custom_data['appendName'])){
                $next_group_name = $rule_name;
            }
            $options = static::createElements($mform, $this->getOptions(), $next_group_name);
            $this->setOptions($options);
        }

        /** @var \HTML_QuickForm_element|\HTML_QuickForm_select|object $element */
        $element =& $mform->createElement(...$this->getArgs());
        if ($and_add){
            $element =& $mform->addElement($element);
        }

        if ($this->custom_data['multi'] ?? false){
            $element->setMultiple(true);
        }
        if (!empty($this->custom_data['custom_defaults'])){
            foreach ($this->custom_data['custom_defaults'] as $custom_default){
                $mform->setDefault(...$custom_default);
            }
        }

        $this_attr = $this->getAttributes();
        if (!empty($this_attr)){
            $element->_attributes = array_merge($element->_attributes ?: [], $this_attr);
        }

        if (!$this->hasName()) return $element;
        // some static elements require force set of the name
        $element->setName($name);

        if ($this->hasType()){
            $mform->setType($rule_name, $this->getType());
        }
        if ($this->hasDefault()){
            $mform->setDefault($rule_name, $this->default);
        }
        if ($this->hasHelp()){
            $mform->addHelpButton(...$this->getHelp($group_name));
        }
        if ($this->hasGroupRules()){
            $rules = $this->getGroupRules($group_name);
            foreach ($rules as $rule){
                $mform->addGroupRule(...$rule);
            }
        }
        if ($this->hasRules()){
            $rules = $this->getRules($group_name);
            foreach ($rules as $rule){
                $mform->addRule(...$rule);
            }
        }
        if ($this->hasDisabledIf()){
            $disabledIfs = $this->getDisabledIfs($group_name);
            foreach ($disabledIfs as $disabledIf){
                $mform->disabledIf(...$disabledIf);
            }
        }
        if ($this->hasHideIf()){
            $hideIfs = $this->getHideIfs($group_name);
            foreach ($hideIfs as $hideIf){
                $mform->hideIf(...$hideIf);
            }
        }

        return $element;
    }

    /**
     * Create and add form element by the objects data to the $mform
     *
     * @see createByForm()
     * @see \MoodleQuickForm::addElement()
     *
     * @param \MoodleQuickForm $mform
     *
     * @return \HTML_QuickForm_element|object
     */
    public function &addByForm($mform){
        return $this->createByForm($mform, true);
    }
    //endregion
}
