<?php
/**
 * Override standard base class for mod_assign (assignment types).
 *
 * @package    local_ned_controller
 * @subpackage mod_assign
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller\mod_assign;
use local_ned_controller\ned_grade_grade;
use local_ned_controller\shared_lib as NED;

defined('MOODLE_INTERNAL') || die();
/** @var \stdClass $CFG */
require_once($CFG->dirroot . '/mod/assign/locallib.php');


/**
 * Class ned_assign
 *
 * @package local_ned_controller
 */
class assign extends \assign{
    const URL =  '/local/ned_controller/ned_assign_view.php';
    const MOODLE_URL = '/mod/assign/view.php';

    const TABLE_SUBMISSION = 'assign_submission';
    const TABLE_GRADES = 'assign_grades';
    const TABLE_GRADE_HISTORY = 'grade_grades_history';

    protected static $is_need_substitute_page_list = [];
    protected static $update_effective_access_list = [];
    protected static $_assigns = [];
    protected static $_ngc_records = [];

    protected $_submissions = [];

    /**
     * Constructor for the base assign class.
     *
     * Note: For $coursemodule you can supply a stdclass if you like, but it
     * will be more efficient to supply a cm_info object.
     *
     * @param mixed $coursemodulecontext context|null the course module context
     *                                   (or the course context if the coursemodule has not been
     *                                   created yet).
     * @param mixed $coursemodule the current course module if it was already loaded,
     *                            otherwise this class will load one from the context as required.
     * @param mixed $course the current course  if it was already loaded,
     *                      otherwise this class will load one from the context as required.
     */
    public function __construct($coursemodulecontext, $coursemodule, $course){
        parent::__construct($coursemodulecontext, $coursemodule, $course);
        static::$_assigns[$this->get_course_module()->id] = $this;
    }

    /**
     * @param \assign|static $assign
     *
     * @return static
     */
    public static function get_ned_assign_from_assign($assign){
        if ($assign instanceof static){
            return $assign;
        } else {
            return new static($assign->get_context(), $assign->get_course_module(), $assign->get_course());
        }
    }

    /**
     * Get the settings for the current instance of this assignment.
     *
     * @return \stdClass|null The settings
     */
    public function get_default_instance() {
        try {
            return parent::get_default_instance();
        } catch (\Throwable){
            return null;
        }
    }

    /**
     * Get the settings for the current instance of this assignment
     *
     * @param int|null $userid the id of the user to load the assign instance for.
     *
     * @return \stdClass The settings
     */
    public function get_instance(int $userid = null) : \stdClass{
        try {
            return parent::get_instance($userid);
        } catch (\Throwable){
            return new \stdClass();
        }
    }

    /**
     * Get the current course module.
     *
     * @return \cm_info|null The course module or null if not known
     */
    public function get_course_module() {
        try {
            return parent::get_course_module();
        } catch (\Throwable){
            return null;
        }
    }

    /**
     * Load the submission object for a particular user, optionally creating it if required.
     *
     * For team assignments there are 2 submissions - the student submission and the team submission
     * All files are associated with the team submission but the status of the students contribution is
     * recorded separately.
     *
     * @param int  $userid The id of the user whose submission we want or 0 in which case USER->id is used
     * @param bool $create If set to true a new submission object will be created in the database with the status set to "new".
     * @param int  $attemptnumber - -1 means the latest attempt
     *
     * @return \stdClass|false - The submission
     */
    public function get_user_submission($userid, $create, $attemptnumber=-1) {
        if (!isset($this->_submissions[$userid][$attemptnumber]) || $create){
            $submission = parent::get_user_submission($userid, $create, $attemptnumber);
            $this->_submissions[$userid][$attemptnumber] = $submission;
            if ($create && $attemptnumber != -1){
                unset($this->_submissions[$userid][-1]);
            } elseif ($submission && $attemptnumber == -1){
                $this->_submissions[$userid][$submission->attemptnumber] = $submission;
            }
        }

        return $this->_submissions[$userid][$attemptnumber];
    }

    /**
     * Get NGC record for the current assign by userid
     *
     * @param $userid
     *
     * @return \local_ned_controller\support\ned_grade_controller_record|object|null
     */
    public function get_ngc_record($userid){
        $cmid = $this->get_course_module()->id;
        if (!isset(static::$_ngc_records[$cmid][$userid])){
            $records = NED::$ned_grade_controller::get_records_by_params($cmid, $userid);
            static::$_ngc_records[$cmid][$userid] = empty($records) ? false : reset($records);
        }

        return static::$_ngc_records[$cmid][$userid] ?: null;
    }

    /**
     * Load the group submission object for a particular user, optionally creating it if required.
     *
     * @param int  $userid The id of the user whose submission we want
     * @param int  $groupid The id of the group for this user - may be 0 in which
     *                     case it is determined from the userid.
     * @param bool $create If set to true a new submission object will be created in the database
     *                     with the status set to "new".
     * @param int  $attemptnumber - -1 means the latest attempt
     *
     * @return \stdClass|false - The submission
     */
    public function get_group_submission($userid, $groupid, $create, $attemptnumber=-1) {
        if (!isset($this->_submissions[0][$userid][$groupid][$attemptnumber]) || $create){
            $submission = parent::get_group_submission($userid, $groupid, $create, $attemptnumber);
            $this->_submissions[0][$userid][$groupid][$attemptnumber] = $submission;
            if ($create && $attemptnumber != -1){
                unset($this->_submissions[0][$userid][$groupid][-1]);
            } elseif ($submission && $attemptnumber == -1){
                $this->_submissions[0][$userid][$groupid][$submission->attemptnumber] = $submission;
            }
        }

        return $this->_submissions[0][$userid][$groupid][$attemptnumber];
    }

    /**
     * Update grades in the gradebook based on submission time.
     *
     * @param \stdClass $submission
     * @param int       $userid
     * @param bool      $updatetime
     * @param bool      $teamsubmission
     *
     * @return bool
     */
    protected function update_submission(\stdClass $submission, $userid, $updatetime, $teamsubmission){
        if ($teamsubmission) {
            return $this->update_team_submission($submission, $userid, $updatetime);
        }

        $res = parent::update_submission($submission, $userid, $updatetime, $teamsubmission);
        $this->_submissions[$userid][$submission->attemptnumber] = $submission;
        if (($this->_submissions[$userid][-1]->id ?? 0) == $submission->id){
            $this->_submissions[$userid][-1] = $submission;
        }

        return $res;
    }

    /**
     * Update team submission.
     *
     * @param \stdClass $submission
     * @param int       $userid
     * @param bool      $updatetime
     *
     * @return bool
     */
    protected function update_team_submission(\stdClass $submission, $userid, $updatetime){
        $res = parent::update_team_submission($submission, $userid, $updatetime);
        $this->_submissions[0][$userid][$submission->attemptnumber] = $submission;
        if (($this->_submissions[0][$userid][-1]->id ?? 0) == $submission->id){
            $this->_submissions[0][$userid][-1] = $submission;
        }

        return $res;
    }

    /**
     * Unlock the student submission.
     *
     * @param int   $userid
     * @param bool  $force - skip requiring capability
     *
     * @return bool
     */
    public function unlock_submission($userid, $force=false) {
        global $DB;

        if (!$force){
            // Need grade permission.
            require_capability('mod/assign:grade', $this->get_context());
        }

        // Give each submission plugin a chance to process the unlocking.
        $plugins = $this->get_submission_plugins();
        $submission = $this->get_user_submission($userid, false);

        $flags = $this->get_user_flags($userid, true);
        $flags->locked = 0;
        $this->update_user_flags($flags);

        foreach ($plugins as $plugin) {
            if ($plugin->is_enabled() && $plugin->is_visible()) {
                $plugin->unlock($submission, $flags);
            }
        }

        $user = $DB->get_record('user', array('id' => $userid), '*', MUST_EXIST);
        \mod_assign\event\submission_unlocked::create_from_user($this, $user)->trigger();
        return true;
    }

    /**
     * Add a new attempt for a user.
     *
     * @param int|string      $userid      - The user to add the attempt for
     * @param int|string|null $set_author
     * @param int|string|null $set_duedate - change user due dae through the TT plugin
     *
     * @return bool - true if successful.
     */
    public function force_add_attempt($userid, $set_author=null, $set_duedate=null){
        $graderid = NED::get_userid_or_global();
        $set_author = $set_author ?: $graderid;
        $cm = $this->get_course_module();

        $oldsubmission = $this->get_needed_user_submission($userid, -1, true);
        $grade_grade = $this->get_grade_grade($userid);

        // Create the new submission record for the group/user.
        if ($this->get_instance()->teamsubmission) {
            $mostrecentteamsubmission = $this->get_group_submission($userid, 0, false, -1);
            if (isset($mostrecentteamsubmission)) {
                // Team submissions can end up in this function for each user (via save_grade). We don't want to create
                // more than one attempt for the whole team.
                if ($mostrecentteamsubmission->attemptnumber == $oldsubmission->attemptnumber) {
                    $newsubmission = $this->get_group_submission($userid, 0, true, $oldsubmission->attemptnumber + 1);
                } else {
                    $newsubmission = $this->get_group_submission($userid, 0, false, $oldsubmission->attemptnumber);
                }
            } else {
                debugging('Please use set_most_recent_team_submission() before calling add_attempt', DEBUG_DEVELOPER);
                $newsubmission = $this->get_group_submission($userid, 0, true, $oldsubmission->attemptnumber + 1);
            }
        } else {
            $newsubmission = $this->get_user_submission($userid, true, $oldsubmission->attemptnumber + 1);
        }

        // Set the status of the new attempt to reopened.
        $newsubmission->status = ASSIGN_SUBMISSION_STATUS_REOPENED;

        // Give each submission plugin a chance to process the add_attempt.
        $plugins = $this->get_submission_plugins();
        /** @var \assign_submission_plugin $plugin */
        foreach ($plugins as $plugin) {
            if ($plugin->is_enabled() && $plugin->is_visible()) {
                $plugin->add_attempt($oldsubmission, $newsubmission);
            }
        }

        $this->update_submission($newsubmission, $userid, false, $this->get_instance()->teamsubmission);
        $flags = $this->get_user_flags($userid, false);
        if (!empty($flags) && property_exists($flags, 'locked') && $flags->locked) { // May not exist.
            $this->unlock_submission($userid, true);
        }

        // change grade_grades_history
        $gh = null;
        $timecreated = $newsubmission->timecreated ?? time();
        do {
            $grades_history = $this->get_grade_history($userid);
            if (empty($grades_history)){
                break;
            }

            $gh = reset($grades_history);
            // sometime timemodified is different from assign_submission timecreated, but it shouldn't
            $gh->timemodified = $timecreated;

            // set other author in ggh table
            if ($graderid == $set_author){
                break;
            }

            $gh->loggeduser = $set_author;
        } while(false);

        if ($gh){
            NED::db()->update_record(static::TABLE_GRADE_HISTORY, $gh);
        }

        // Check grade
        if ($grade_grade){
            /**
             * If grade was overridden, update previous assign attempt,
             *  but only after adding new attempt (as changed attempt shouldn't be last) to avoid some gradebook updates
             */
            if ($grade_grade->overridden){
                $assign_grade = $this->get_user_grade($userid, true, $oldsubmission->attemptnumber);
                $assign_grade->timecreated = $grade_grade->timecreated ?? $assign_grade->timecreated;
                $assign_grade->timemodified = $timecreated;
                $assign_grade->grader = $graderid;
                $assign_grade->grade = $grade_grade->finalgrade ?? ($grade_grade->rawgrade ?? -1);
                $this->update_grade($assign_grade, false);
            }

            // Reset and convert grade to regular if necessary
            $grade_grade->overridden = 0;
            $grade_grade->excluded = 0;
            $grade_grade->finalgrade = null;
            $grade_grade->rawgrade = null;
            $grade_grade->usermodified = $graderid;
            $grade_grade->timemodified = $timecreated;
            $this->grade_grade_update($grade_grade, true, false, false);
        }

        // change due date
        if (!is_null($set_duedate) && NED::is_tt_exists()){
            $dma = new \block_ned_teacher_tools\mod\deadline_manager_assign($cm);
            $set_unlimited = !$set_duedate;
            $dma->set_user_override($userid, $set_duedate, null, $set_unlimited);
        }

        // Change NGC after grade and deadline updates
        $NGC = NED::$ned_grade_controller;
        $ngc_records = $NGC::get_records_by_params_done($cm->id, $userid);
        $NGC::check_and_change_obsolete_state($ngc_records, true, false, $set_author ?: null);

        \local_ned_controller\event\submission_force_add_attempt::create_from_submission($this, $newsubmission, $set_author)->trigger();

        return true;
    }

    /**
     * Set the action and parameters that can be used to return to the current page.
     *
     * @param string $action The action for the current page
     * @param array  $params An array of name value pairs which form the parameters
     *                       to return to the current page.
     *
     * @return void
     */
    public function register_return_link($action, $params) {
        if ($this->is_need_substitute_page()){
            global $PAGE;
            $params['action'] = $action;
            $cm = $this->get_course_module();
            if ($cm) {
                $currenturl = new \moodle_url(static::URL, array('id' => $cm->id));
            } else {
                $currenturl = new \moodle_url('/mod/assign/index.php', array('id' => $this->get_course()->id));
            }

            $currenturl->params($params);
            $PAGE->set_url($currenturl);
        } else {
            parent::register_return_link($action, $params);
        }
    }

    /**
     * Display the assignment, used by view.php
     *
     * The assignment is displayed differently depending on your role,
     * the settings for the assignment and the status of the assignment.
     *
     * @param string $action The current action if any.
     * @param array $args Optional arguments to pass to the view (instead of getting them from GET and POST).
     * @return string - The page output.
     */
    public function view($action='', $args = array()) {
        $o = parent::view($action, $args);
        if ($this->is_need_substitute_page()){
            return static::change_url($o);
        }
        else {
            return $o;
        }
    }

    /**
     * Updates the assign properties with override information for a user.
     *
     * Algorithm:  For each assign setting, if there is a matching user-specific override,
     *   then use that otherwise, if there are group-specific overrides, return the most
     *   lenient combination of them.  If neither applies, leave the assign setting unchanged.
     *
     * @param int $userid The userid.
     */
    public function update_effective_access($userid){
        if (isset(static::$update_effective_access_list[$userid])){
            return;
        }

        parent::update_effective_access($userid);
        static::$update_effective_access_list[$userid] = true;
    }

    /**
     * Is this assignment open for submissions?
     * NED: if submission is draft, it always open
     *
     * Check the due date,
     * prevent late submissions,
     * has this person already submitted,
     * is the assignment locked?
     *
     * @param int       $userid       - Optional userid so we can see if a different user can submit
     * @param bool      $skipenrolled - Skip enrollment checks (because they have been done already)
     * @param \stdClass $submission   - Pre-fetched submission record (or false to fetch it)
     * @param \stdClass $flags        - Pre-fetched user flags record (or false to fetch it)
     * @param \stdClass $gradinginfo  - Pre-fetched user gradinginfo record (or false to fetch it)
     *
     * @return bool
     */
    public function submissions_open($userid=0, $skipenrolled=false, $submission=false, $flags=false, $gradinginfo=false) {

        global $USER;

        if (!$userid) {
            $userid = $USER->id;
        }

        $finaldate = $this->get_instance()->cutoffdate;
        // Note you can pass null for submission and it will not be fetched.
        if ($submission === false){
            $submission = $this->get_needed_user_submission($userid);
        }
        /* resubmission check is deprecated
        $submission_info = new assign_info($this->get_course_module(), $userid);
        $is_resubmission = $submission_info->exist && $submission_info->submissions_count > 1;
        */
        $is_draft = $submission && ($submission->status == ASSIGN_SUBMISSION_STATUS_DRAFT);

        if ($finaldate && $is_draft){
            // set ignore to time limit
            if (!$flags) {
                $flags = $this->get_user_flags($userid, false);
                if (!$flags){
                    // default object from get_user_flags
                    $flags = (object) ['assignment' => $this->get_instance()->id, 'userid' => $userid,
                        'locked' => 0, 'extensionduedate' => 0, 'workflowstate' => '', 'allocatedmarker' => 0];

                }
            }

            $flags->extensionduedate = time() + (24 * 3600); // extensionduedate will be always tomorrow
        }

        return parent::submissions_open($userid, $skipenrolled, $submission, $flags, $gradinginfo);
    }

    /**
     * Update a grade in the grade table for the assignment and in the gradebook.
     *
     * @param \stdClass $grade a grade record keyed on id
     * @param bool $reopenattempt If the attempt reopen method is manual, allow another attempt at this assignment.
     * @return bool true for success
     */
    public function update_grade($grade, $reopenattempt=false) {
        $res = parent::update_grade($grade, $reopenattempt);
        if ($res){
            $submission = $this->get_needed_user_submission($grade->userid);
            $grade->maxattempt = $submission->attemptnumber ?? $grade->attemptnumber;
            \local_ned_controller\event\submission_graded::create_from_grade($this, $grade)->trigger();
        }

        return $res;
    }

    /**
     * Apply a grade from a grading form to a user (may be called multiple times for a group submission).
     *
     * @param \stdClass $formdata      - the data from the form
     * @param int       $userid        - the user to apply the grade to
     * @param int       $attemptnumber - The attempt number to apply the grade to.
     *
     * @return void
     */
    protected function apply_grade_to_user($formdata, $userid, $attemptnumber) {
        parent::apply_grade_to_user($formdata, $userid, $attemptnumber);
    }

    /**
     * Save outcomes submitted from grading form.
     *
     * @param int       $userid
     * @param \stdClass $formdata
     * @param int       $sourceuserid The user ID under which the outcome data is accessible. This is relevant
     *                          for an outcome set to a user but applied to an entire group.
     */
    protected function process_outcomes($userid, $formdata, $sourceuserid = null) {
        parent::process_outcomes($userid, $formdata, $sourceuserid);
    }

    /**
     * Utility function to get the userid for every row in the grading table
     * so the order can be frozen while we iterate it.
     *
     * @param boolean $cached       If true, the cached list from the session could be returned.
     * @param string  $useridlistid String value used for caching the participant list.
     *
     * @return array An array of userids
     */
    protected function get_grading_userid_list($cached = false, $useridlistid = '') {
        return parent::get_grading_userid_list($cached, $useridlistid);
    }

    /**
     * Utility function to get the userid for every row in the grading table
     * so the order can be frozen while we iterate it.
     * Public alias for {@see get_grading_userid_list()}
     *
     * @param boolean $cached       If true, the cached list from the session could be returned.
     * @param string  $useridlistid String value used for caching the participant list.
     *
     * @return array An array of userids
     */
    public function get_grading_userid_list2($cached = false, $useridlistid = '') {
        return $this->get_grading_userid_list($cached, $useridlistid);
    }

    /**
     * Remove totally the current submission.
     *
     * @param int $userid
     *
     * @return boolean
     */
    protected function process_remove_submission($userid=0) {
        require_sesskey();

        if (!$userid) {
            $userid = required_param('userid', PARAM_INT);
        }

        if ($this->remove_submission($userid)){
            return $this->remove_last_assign_submission($userid);
        }

        return false;
    }

    // ADDITIONAL FUNCTIONS (there are no in moodle assign class)

    /**
     * Debug messages for the assign
     *
     * @param string     $txt
     * @param int        $userid
     * @param int|string $level
     */
    public function debug($txt, $userid=0, $level=DEBUG_DEVELOPER){
        $course = $this->get_course();
        $cm = $this->get_course_module();

        $postfix = "{$course->fullname}[id: {$course->id}] -> {$cm->name}[id: {$cm->id}]";
        if ($userid){
            $postfix .= " for userid '$userid'";
        }

        debugging($txt . ' in ' . $postfix, $level);
    }

    /**
     * @param     $userid
     * @param int $attemptnumber = -1
     * @param bool $create If set to true a new submission object will be created in the database with the status set to "new".
     *
     * @return false|\stdClass|null
     */
    public function get_needed_user_submission($userid, $attemptnumber=-1, $create=false){
        $submission = null;
        $attemptnumber = $attemptnumber ?? -1;
        if ($this->get_instance()->teamsubmission) {
            $submission = $this->get_group_submission($userid, 0, $create, $attemptnumber);
        } else {
            $submission = $this->get_user_submission($userid, $create, $attemptnumber);
        }

        return $submission;
    }

    /**
     * Get the primary grade item for this assign instance.
     *
     * @return null|\grade_item The grade_item record
     */
    public function get_grade_item(){
        try {
            return parent::get_grade_item();
        } catch (\Throwable){
            return null;
        }
    }

    /**
     * Return grade history from the grade_grades_history table
     * By default sorted from the last - to the old ones
     *
     * @param int|string $userid
     * @param string     $sort
     *
     * @return array
     */
    public function get_grade_history($userid, $sort='timemodified DESC'){
        global $DB;
        $gi = $this->get_grade_item();
        if (!$gi){
            return [];
        }

        return $DB->get_records(static::TABLE_GRADE_HISTORY, ['itemid' => $gi->id, 'userid' => $userid, 'source' => 'mod/assign'], $sort);
    }

    /**
     * Get grade_grade object
     *
     * @param      $userid
     * @param bool $clone - to clone object for independence, highly recommended, if it will be using for writing,
     *                    use 'false' for read-only
     *
     * @return ned_grade_grade|\grade_grade|false - return false, if find nothing
     */
    public function get_grade_grade($userid, $clone=true){
        return NED::get_grade_grade($this->get_course_module(), $userid, $clone);
    }

    /**
     * Get all assignment grades, from last to first by default
     * If you need only one last grade, @see get_user_grade()
     *
     * @param numeric $userid
     * @param string  $sort - sql sort, "attemptnumber DESC" by default
     *
     * @return array|object[]
     */
    public function get_all_user_grades($userid, $sort='attemptnumber DESC'){
        $params = ['assignment' => $this->get_instance($userid)->id, 'userid' => $userid];
        return NED::db()->get_records(static::TABLE_GRADES, $params, $sort);
    }

    /**
     * Updates final grade value for given user
     * Skipped many checks and sets - if you don't sure, what you do, use \grade_item::update_final_grade() better
     *
     * @see \grade_item::update_final_grade()
     *
     * @param ned_grade_grade  $grade
     * @param bool             $check_changes - if true, try to check changes from the old grade
     * @param bool             $call_events - if true, triggers grade events
     * @param bool             $update_grade_history - if false, uses other insert/update methods to not change grade history table
     * @param bool             $force_regrading - if true, call $force_regrading without checks
     *
     * @return bool - success
     */
    public function grade_grade_update($grade, $check_changes=true, $call_events=true, $update_grade_history=true, $force_regrading=false){
        return NED::grade_grade_update($this->get_course_module(), $grade, $check_changes, $call_events, $update_grade_history, $force_regrading);
    }

    /**
     * @param int $userid
     *
     * @return mixed
     */
    public function is_need_substitute_page($userid=0){
        global $USER;

        if (!$userid) {
            $userid = $USER->id;
        }

        if (!isset(static::$is_need_substitute_page_list[$userid])){
            do {
                /* Turn off "always open during resubmission" by PIPE-106
                $this->update_effective_access($userid);
                $res = $this->submissions_open($userid) != parent::submissions_open($userid);
                if ($res) break;
                */
                // change page, if we have draft
                $submission = $this->get_needed_user_submission($userid);
                $res = (($submission->status ?? '') == ASSIGN_SUBMISSION_STATUS_DRAFT);
                if ($res) break;

                // change page, if we need to remove submission
                $action = optional_param('action', '', PARAM_ALPHA);
                $res = $action == 'removesubmissionconfirm' || $action == 'removesubmission';
                if ($res) break;
            } while(false);

            static::$is_need_substitute_page_list[$userid] = $res;
        }

        return static::$is_need_substitute_page_list[$userid];

    }

    /**
     * @param $output
     *
     * @return mixed
     */
    protected static function change_url($output){
        return str_replace(static::MOODLE_URL, static::URL, $output);
    }

    /**
     * Check capabilities for teacher
     *
     * @param numeric|object|null $user_or_id - if null $USER is used, otherwise user object or id expected
     *
     * @return bool
     */
    public function is_teacher($user_or_id=null){
        $user_or_id = $user_or_id ?: null;
        $ctx = $this->get_context();
        if (!is_enrolled($ctx, $user_or_id, '', true) && !is_viewing($ctx, $user_or_id) && !is_siteadmin($user_or_id)){
            return false;
        }
        return has_capability('moodle/grade:viewall', $ctx, $user_or_id);
    }

    /**
     * Submit submission, if it's not submitted
     * Return true if all is OK, false otherwise
     *
     * @param      $submission
     * @param      $userid
     * @param bool $updatetime
     * @param null $instance
     *
     * @return bool
     */
    protected function submit_submission($submission, $userid, $updatetime=true, $instance=null){
        if ($submission->status != ASSIGN_SUBMISSION_STATUS_SUBMITTED) {
            $instance = $instance ?: $this->get_instance($userid);

            // Give each submission plugin a chance to process the submission.
            $plugins = $this->get_submission_plugins();
            /** @var \assign_submission_plugin $plugin */
            foreach ($plugins as $plugin) {
                if ($plugin->is_enabled() && $plugin->is_visible()) {
                    $plugin->submit_for_grading($submission);
                }
            }

            $submission->status = ASSIGN_SUBMISSION_STATUS_SUBMITTED;
            $this->update_submission($submission, $userid, $updatetime, $instance->teamsubmission);
            $completion = new \completion_info($this->get_course());
            if ($completion->is_enabled($this->get_course_module()) && $instance->completionsubmit) {
                $this->update_activity_completion_records($instance->teamsubmission,
                    $instance->requireallteammemberssubmit,
                    $submission,
                    $userid,
                    COMPLETION_COMPLETE,
                    $completion);
            }

            $this->notify_graders($submission);
            $this->notify_student_submission_receipt($submission);

            \mod_assign\event\assessable_submitted::create_from_submission($this, $submission, false)->trigger();

            return true;
        }
        return false;
    }

    /**
     * Check to make sure this user can edit this submission by NED capabilities
     *
     * @param numeric|object|null $user_or_id   - (optional) The user whose submission is to be edited (default to $USER).
     * @param numeric|object|null $grader_or_id - (optional) The user who will do the editing (default to $USER).
     *
     * @return bool
     */
    public function can_user_edit_submission_ned($user_or_id, $grader_or_id=null){
        $userid = NED::get_userid_or_global($user_or_id);
        $graderid = NED::get_userid_or_global($grader_or_id);
        if ($userid == $graderid){
            return $this->can_edit_submission($userid, $graderid);
        } else {
            return NED::has_capability('editothersubmission', $this->get_context(), $graderid);
        }
    }

    /**
     * Submit draft submission.
     * Return true, if all is OK, string error otherwise
     *
     * @param numeric|object    $user_or_id
     * @param bool              $check_permissions
     *
     * @return bool|string
     */
    public function draft2submit($user_or_id, $check_permissions=true){
        $userid = NED::get_id($user_or_id);
        if ($check_permissions && !$this->can_user_edit_submission_ned($userid)){
            return NED::str('ned_assign:nopermission');
        }

        $submission = $this->get_needed_user_submission($userid, -1, true);
        if ($submission->status != ASSIGN_SUBMISSION_STATUS_DRAFT){
            return NED::str('ned_assign:notdraft');
        }

        if ($result = $this->submit_submission($submission, $userid, false)){
            return $result;
        }

        return NED::str('ned_assign:unexpectederror');
    }

    /**
     * Return to draft submitted submission.
     * Return true, if all is OK, string error otherwise
     *
     * @param numeric|object    $user_or_id
     * @param bool              $check_permissions
     *
     * @return bool|string
     */
    public function submitted2draft($user_or_id, $check_permissions=true){
        $userid = NED::get_id($user_or_id);
        if ($check_permissions && !$this->can_user_edit_submission_ned($userid)){
            return NED::str('ned_assign:nopermission');
        }

        $submission = $this->get_needed_user_submission($userid);
        if (!$submission || $submission->status != ASSIGN_SUBMISSION_STATUS_SUBMITTED){
            return NED::str('ned_assign:notsubmitted');
        }

        if ($result = $this->_to_draft($userid, $submission)){
            return $result;
        }

        return NED::str('ned_assign:unexpectederror');
    }

    /**
     * Revert to draft.
     * It's based on parent revert_to_draft method
     * @see revert_to_draft()
     *
     * @param numeric|object        $user_or_id
     * @param false|object|null     $submission
     * @param numeric|object|null   $grader_or_id
     *
     * @return bool
     */
    protected function _to_draft($user_or_id, $submission=null, $grader_or_id=null) {
        $userid = NED::get_id($user_or_id);
        $graderid = NED::get_userid_or_global($grader_or_id);
        $submission = $submission ?? $this->get_needed_user_submission($userid);
        $cm = $this->get_course_module();
        if (!$submission || !$cm) {
            return false;
        }

        $submission->status = ASSIGN_SUBMISSION_STATUS_DRAFT;
        $this->update_submission($submission, $userid, false, $this->get_instance()->teamsubmission);

        // Give each submission plugin a chance to process the reverting to draft.
        $plugins = $this->get_submission_plugins();
        foreach ($plugins as $plugin) {
            if ($plugin->is_enabled() && $plugin->is_visible()) {
                $plugin->revert_to_draft($submission);
            }
        }

        // Update the modified time on the grade (grader modified).
        $grade = $this->get_user_grade($userid, true);
        $grade->grader = $graderid;
        $this->update_grade($grade);

        $completion = new \completion_info($this->get_course());
        if ($completion->is_enabled($cm) &&
            $this->get_instance()->completionsubmit) {
            $completion->update_state($cm, COMPLETION_INCOMPLETE, $userid);
        }

        $NGC = NED::$ned_grade_controller;
        $NGC::check_and_delete_records_by_params($cm->id, $userid, ['grade_type' => $NGC::GT_DEDUCTION, 'reason' => $NGC::REASON_SUBMISSION]);

        \mod_assign\event\submission_status_updated::create_from_submission($this, $submission)->trigger();
        return true;
    }

    /**
     * Check, does this assign is valid and have necessary data in the BD
     *
     * @param int  $courseid - check course id too, if it sent
     * @param bool $through_error - can through error, if true
     *
     * @return bool
     *
     * @throws \moodle_exception
     */
    public function is_valid($courseid=0, $through_error=false){
        $error = true;

        do{
            if (!$this->get_course_module()){
                break;
            }
            if (!$this->get_default_instance()){
                break;
            }
            if ($courseid){
                if ($this->get_course()->id != $courseid){
                    break;
                }
            }

            $error = false;
        } while(false);


        if ($error && $through_error) {
            throw new \moodle_exception('invalidmodule', 'error');
        }
        return !$error;
    }

    /**
     * Remove any data from the current submission.
     * WARNING: There are none security check
     *
     * @param int  $userid
     * @param bool $status_updated_event - call submission_status_updated if true and user id is not 0
     *
     * @return boolean
     */
    public function remove_submission_file($userid, $status_updated_event=false) {
        $submission = $this->get_needed_user_submission($userid);

        if (!$submission) {
            return false;
        }

        // Tell each submission plugin we were saved with no data.
        $plugins = $this->get_submission_plugins();
        foreach ($plugins as $plugin) {
            if ($plugin->is_enabled() && $plugin->is_visible()) {
                $plugin->remove($submission);
            }
        }

        if ($status_updated_event && $submission->userid != 0) {
            \mod_assign\event\submission_status_updated::create_from_submission($this, $submission)->trigger();
        }

        return true;
    }

    /**
     * Remove last submission and try to revert grade to previous state
     *  return false if can do nothing, true if can at least change assign_submission
     *
     * Changes tables: assign_submission +
     * @see remove_last_assign_grade
     *
     * WARNING: There are none security check
     *
     * @param int|string        $userid
     * @param null|string       $check_status - string status, if you wish check submission
     * @param bool              $reset - when true, if there is resubmission, it will be reset, but not deleted
     *
     * @return bool
     */
    public function remove_last_assign_submission($userid, $check_status=null, $reset=false){
        global $DB;

        $cm = $this->get_course_module();
        $cmid = $cm->id;
        $assign_id = $cm->instance;
        $is_grader = $this->can_grade();

        $clear_grade_attempt = null;
        $event_submission = null;
        // check submission, which we have what to delete
        $submissions = $DB->get_records(static::TABLE_SUBMISSION, ['assignment' => $assign_id, 'userid' => $userid], 'attemptnumber');
        if (!empty($submissions)){
            $reset = $reset || \count($submissions) == 1;
            $resubmission = \count($submissions) > 1;

            $last_submission = end($submissions);
            $clear_grade_attempt = $last_submission->attemptnumber;
            if ($check_status && $last_submission->status != $check_status){
                $this->debug("Submission to removing has status '{$last_submission->status}' instead of '$check_status'", $userid);
                return false;
            }
            $event_submission = clone($last_submission);

            if ($reset){
                $last_submission->status = $resubmission ? ASSIGN_SUBMISSION_STATUS_REOPENED : ASSIGN_SUBMISSION_STATUS_NEW;
                $this->update_submission($last_submission, $userid, false, $this->get_instance()->teamsubmission);
            } else {
                $new_last_submission = prev($submissions);
                // change assign_submission
                $DB->delete_records(static::TABLE_SUBMISSION, ['id' => $last_submission->id]);
                unset($this->_submissions[$userid][-1]);

                $new_last_submission->latest = 1;
                $this->update_submission($new_last_submission, $userid, false, $this->get_instance()->teamsubmission);
            }
        } else {
            if ($check_status && $check_status != ASSIGN_SUBMISSION_STATUS_NEW){
                $this->debug("There are no submissions which we can remove", $userid);
                return false;
            }

            $clear_grade_attempt = 0;
            $reset = true;
            $event_submission = $this->get_needed_user_submission($userid, $clear_grade_attempt, true);
        }

        if ($is_grader){
            // remove NGC deductions and zeros
            $ngc_records = NED::$ned_grade_controller::get_records_by_params($cmid, $userid);
            NED::$ned_grade_controller::check_and_delete($ngc_records, false, $reset);

            // remove AI infractions
            if (NED::is_ai_exists()){
                \local_academic_integrity\infraction::delete_records_by_userid_cmid($cmid, $userid);
            }

            // remove user override & extension
            if (NED::is_tt_exists()){
                $dma = new \block_ned_teacher_tools\mod\deadline_manager_assign($cm);
                $dma->delete_user_override($userid);
                $dma->delete_extension($userid);
            }
        }

        // remove file submission
        $this->remove_submission_file($userid, false);

        if ($is_grader){
            // remove last grade manually
            $this->remove_last_assign_grade($userid, $clear_grade_attempt, $event_submission);
        }

        // try to get the original grade, check regrading, update completion state
        NED::grade_grade_refresh($cm, $userid);

        \local_ned_controller\event\submission_removed::create_from_submission($this, $event_submission, $reset)->trigger();

        return true;
    }

    /**
     * Remove last grades and try to revert grade to previous state
     *  return false if can do nothing, true if can at least delete or change something
     *
     * Changes tables: assign_grades, grade_grades_history, local_kica_grade_grades, grade_grades
     *
     * WARNING: If you clear resubmission, it can remove also grade_grades_history record about resubmission moment
     *
     * @param int|string     $userid
     * @param int|string     $attempt - if specified, check all attempts after, uses last otherwise
     * @param null|\stdClass $event_submission
     * @param bool           $restore_prev_grade
     *
     * @return bool
     */
    public function remove_last_assign_grade($userid, $attempt=null, $event_submission=null, $restore_prev_grade=false){
        global $DB;
        $result = false;
        $course = $this->get_course();
        $courseid = $course->id;
        // delete all grades for the first attempt
        $reset = !is_null($attempt) && !$attempt;
        // we need grade_item if we want to change grades
        $gi = $this->get_grade_item();
        if (!$gi){
            $this->debug("Can't find grade_item");
            return $result;
        }

        // change assign_grades
        $last_assign_grade = null;
        $time_x = 0;
        $assign_grades = $this->get_all_user_grades($userid);
        if (!empty($assign_grades)){
            $ag_id_remove = [];
            if (is_null($attempt)){
                if ($event_submission){
                    $attempt = $event_submission->attemptnumber;
                } else {
                    $real_last_ag = reset($assign_grades);
                    $attempt = $real_last_ag->attemptnumber;
                }
            }

            foreach ($assign_grades as $ag){
                if ($reset || $ag->attemptnumber >= $attempt){
                    $ag_id_remove[] = $ag->id;
                } else {
                    $last_assign_grade = $ag;
                    break;
                }
            }

            if (!empty($ag_id_remove)){
                $result = $DB->delete_records_list(static::TABLE_GRADES, 'id', $ag_id_remove) || $result;
            }
        }

        if (!$reset){
            if ($event_submission){
                $time_x = $event_submission->timecreated;
            } elseif ($last_assign_grade){
                $time_x = $last_assign_grade->timemodified;
            }
        }

        // change grade_grades_history
        $last_grades_history = null;
        $grades_history = $this->get_grade_history($userid);
        if (!empty($grades_history)){
            $gh_id_remove = [];
            foreach ($grades_history as $gh){
                if ($gh->timemodified > $time_x){
                    $gh_id_remove[] = $gh->id;
                } else {
                    $last_grades_history = $gh;
                    break;
                }
            }

            if (!empty($gh_id_remove)){
                $result = $DB->delete_records_list(static::TABLE_GRADE_HISTORY, 'id', $gh_id_remove) || $result;
            }
        }

        if (NED::is_kica_exists()){
            $kgi = NED::ki_get_by_grade_item($gi);
            if ($kgi->id ?? false){
                NED::kg_delete_records(['itemid' => $kgi->id, 'courseid' => $courseid, 'userid' => $userid]);
            }
        }

        // change grade_grades, if need
        do {
            if (!$restore_prev_grade && !$reset) break;

            $gg = $this->get_grade_grade($userid);
            if (empty($gg)) break;

            $have_to_reset = $restore_prev_grade && !$reset && empty($last_assign_grade) && empty($last_grades_history);

            if ($reset || $have_to_reset){

                if ($reset){
                    $gg->timemodified = null;
                } else {
                    // if grade is overridden, we clear it from any attempt (time)
                    if ($gg->timemodified < $time_x && !$gg->overridden && !$gg->excluded){
                        break;
                    } elseif ($attempt > 0){
                        $this->debug('There are no data to restore current grade', $userid);
                    }

                    $gg->timemodified = time();
                }

                $gg->rawgrade = null;
                $gg->finalgrade = null;
                $gg->usermodified = -1;
            } else {
                // if !$have_to_reset && !$reset && $restore_prev_grade
                if ($last_assign_grade){
                    $gg->timecreated = $last_assign_grade->timecreated ?? time();
                    $gg->timemodified = $last_assign_grade->timemodified ?? time();
                    $rawgrade = $last_assign_grade->grade > -1 ? $last_assign_grade->grade : null;
                    $gg->rawgrade = $rawgrade;
                    $gg->finalgrade = $rawgrade;
                    $gg->usermodified = $last_assign_grade->grader;
                } elseif ($last_grades_history){
                    $gg_data = $gg->get_record_data();
                    foreach ($gg_data as $key => $value){
                        if ($key == 'id' || !isset($last_grades_history->$key)){
                            continue;
                        }
                        $gg->$key = $last_grades_history->$key;
                    }
                    $gg->timecreated = $last_grades_history->timemodified; // it's strange, but records in the table follow this logic
                    if ($event_submission){
                        $gg->timemodified = $event_submission->timemodified;
                    }
                }
            }

            /**
             * MDL-31713 rawgramemin and max must be up to date so conditional access %'s works properly.
             * @see \grade_item::update_final_grade()
             */
            $gg->rawgrademin = $gi->grademin;
            $gg->rawgrademax = $gi->grademax;
            $gg->rawscaleid  = $gi->scaleid;

            $gg->overridden = 0;
            $gg->excluded = 0;

            $result = $this->grade_grade_update($gg, !$reset, false, false, false) || $result;
        } while(false);

        $event_submission = $event_submission ?? $this->get_needed_user_submission($userid, $attempt ?? -1);
        \local_ned_controller\event\submission_remove_grades::create_from_submission($this, $event_submission)->trigger();

        return $result;
    }

    /**
     * Copy grade from the previous attempt to the current
     *
     * Note: it checks only assign attempt grades,
     *  if you need more complicated search, @see remove_last_assign_grade()
     *
     * @param numeric       $userid
     * @param null|string   $check_status - string status, if you wish check submission, usually ASSIGN_SUBMISSION_STATUS_REOPENED
     * @param bool          $check_ungraded - if true, will ot change grade, if attempt already graded
     * @param bool          $set_zero_if_no_grades - if true, and there are no previous grades, set 0 (zero)
     *
     * @return bool
     */
    public function copy_last_assign_grade($userid, $check_status=null, $check_ungraded=false, $set_zero_if_no_grades=false){
        do {
            $last_submission = $this->get_needed_user_submission($userid);

            if (empty($last_submission)) break;
            if ($check_status && $last_submission->status != $check_status) break;
            if (!$set_zero_if_no_grades && $last_submission->attemptnumber < 1)  break; // there is only one attempt

            $last_attempt = $last_submission->attemptnumber;
            $assign_grade = $this->get_user_grade($userid, true, $last_attempt);
            if ($check_ungraded){
                if ($assign_grade->grade != -1) break;
            }

            $previous_grade = null;
            $assign_grades = $this->get_all_user_grades($userid);
            foreach ($assign_grades as $a_grade){
                if ($a_grade->attemptnumber >= $last_attempt){
                    continue;
                }

                if ($a_grade->grade != -1){
                    $previous_grade = $a_grade;
                    break;
                }
            }

            if ($previous_grade){
                $assign_grade->grade = $previous_grade->grade;
            } else {
                if ($set_zero_if_no_grades){
                    $assign_grade->grade = 0;
                } else {
                    break;
                }
            }

            return $this->update_grade($assign_grade, false);
        } while (false);

        return false;
    }

    /**
     * Try to remove user course module completion status
     * It can possibly fail, if a user should have completion in the current situation and the USER haven't ability to override it
     *
     * @param $userid
     *
     * @return void
     */
    public function remove_course_module_completion_status($userid){
        global $USER;
        $cm = $this->get_course_module();
        $course = $this->get_course();
        $completion = new \completion_info($course);
        $override = $completion->user_can_override_completion($USER);

        $completion->update_state($cm, COMPLETION_INCOMPLETE, $userid, $override);
    }

    /**
     * Return, have this assign "Wrong File" warning for this user or not
     *
     * @param numeric|object|null $user_or_id - user, for whom check warning (global $USER by default)
     *
     * @return bool
     */
    public function have_wrong_file_warning($user_or_id=null){
        $userid = NED::get_userid_or_global($user_or_id);
        do {
            if (!$userid) break;

            // there is no sense in the warning, if user can't edit submission
            $can_edit = $this->is_any_submission_plugin_enabled() && $this->can_edit_submission($userid);
            if (!$can_edit) break;

            // check "Wrong File" NGC record
            $NGC = NED::$ned_grade_controller;
            $ngc_record = $this->get_ngc_record($userid);
            if (!$ngc_record) break;
            if ($ngc_record->status != $NGC::ST_DONE || $ngc_record->reason != $NGC::REASON_FILE) break;

            // Check submission: it should be draft, and last edited before NGC record update
            $submission = $this->get_needed_user_submission($userid);
            if (!$submission || $submission->status != ASSIGN_SUBMISSION_STATUS_DRAFT) break;

            if ($ngc_record->timemodified >= $submission->timemodified){
                return true;
            }
        } while(false);

        return false;
    }

    /**
     * @param \cm_info|int|string $cm_or_id
     * @param object|numeric      $courseorid - Optional course object (or its id) if loaded, improves optimization if $cm_or_id is represented as ID
     *
     * @return static
     * @throws \moodle_exception
     */
    public static function get_assign_by_cm($cm_or_id, $courseorid=null){
        $cmid = $cm_or_id->id ?? $cm_or_id;
        if (!isset(static::$_assigns[$cmid])){
            $cm = NED::get_cm_by_cmorid($cm_or_id, $courseorid, null, NED::ASSIGN);
            if (!$cm){
                NED::print_error("There are no assign course_module with id $cmid");
            }

            new static($cm->context, $cm, $cm->course);
            if (!isset(static::$_assigns[$cmid])){
                NED::print_error("Can't get assign by course_module with id $cmid");
            }
        }

        return static::$_assigns[$cmid];
    }

    /**
     * Return global assign, if exists, otherwise null
     *
     * @param bool $return_only_ned_assign - if true (default), return $assign only if it's ned_assign
     *
     * @return \assign|assign|null
     */
    public static function get_global_assign($return_only_ned_assign=true){
        global $assign;
        if (isset($assign)){
            if ($return_only_ned_assign){
                if ($assign instanceof assign){
                    return $assign;
                }
            } else {
                if ($assign instanceof \assign){
                    return $assign;
                }
            }
        }

        return null;
    }

    /**
     * Return, should this assign page shows "Wrong File" warning or not
     *
     * @return bool
     */
    public static function do_show_wrong_file_warning_on_page(){
        static $_res = null;
        if (is_null($_res)){
            $_res = false;
            // we need check, that we on the NED assign page
            $ned_assign = static::get_global_assign();
            if ($ned_assign){
                $_res = $ned_assign->have_wrong_file_warning();
            }
        }

        return $_res;
    }

    /**
     * Return first time of last period of grading by cmid and userid
     * @see \local_ned_controller\shared\db_util::sql_get_first_last_grade_join() for additional info
     *
     * @param \cm_info|numeric $cm_or_id   - course-module or its id, for assign activity
     * @param object|numeric   $user_or_id - user id, uses global $USER by default
     *
     * @return int|null - $first_last_grade_time
     */
    public static function get_first_last_grade_time($cm_or_id, $user_or_id=null){
        $gi = NED::get_grade_item($cm_or_id, null, null, 'assign');
        if (empty($gi)){
            return null;
        }

        return static::get_first_last_grade_time_by_gi_id($gi->id, NED::get_userid_or_global($user_or_id));
    }

    /**
     * Return first time of last period of grading, by grade item id and userid
     * @see \local_ned_controller\shared\db_util::sql_get_first_last_grade_join() for additional info
     *
     * @param numeric $gi_id - grade item id, should be for assign activity
     * @param numeric $userid - user id
     *
     * @return int - $first_last_grade_time
     */
    public static function get_first_last_grade_time_by_gi_id($gi_id, $userid){
        $select = "LEAST(COALESCE(ggh_first_graded_time.timemodified, gg.timemodified), gg.timemodified) AS first_grade_time";
        $joins = " 
        JOIN {grade_items} gi 
            ON gi.id = :gi_id
            AND gi.itemmodule = 'assign'
            AND gi.itemtype = 'mod'
            AND gi.itemnumber = ".NED::GRADE_ITEMNUMBER_USUAL."
        JOIN {grade_grades} gg
            ON gg.itemid = gi.id
            AND gg.userid = u.id
        LEFT JOIN {assign_submission} a_s
          ON a_s.userid = u.id
          AND a_s.assignment = gi.iteminstance
          AND a_s.latest = 1
        ".NED::sql_get_first_last_grade_join(\local_ned_controller\marking_manager\marking_manager_assign::SQL_USER_TIMEMODIFIED);
        $where = "u.id = :uid AND ggh_last_ungraded_time_fake.id IS NULL AND ggh_first_graded_time_fake.id IS NULL";
        $sql = NED::sql_generate($select, $joins, 'user', 'u', $where);

        return NED::db()->get_field_sql($sql, ['uid' => $userid, 'gi_id' => $gi_id]) ?: 0;
    }
}
