<?php
/**
 * @package    local_ned_controller
 * @subpackage mod_assign
 * @category   NED
 * @copyright  2023 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller\mod_assign;
use local_ned_controller\shared_lib as NED;

/**
 * class assign_grading_table
 * @package local_ned_controller
 *
 * Changes from the {@see \assign_grading_table}:
 * - hide "edit" menu, if user hasn't "edit other submission" permission
 * - hide grades and graded status, if grades are before midnight and user hasn't capability to view them
 *      {@see \local_ned_controller\shared\grade_util::grade_is_hidden_now_before_midn()}
 *
 * @property assign         $assignment
 * @property int            $perpage
 * @property int            $rownum                       (global index of current row in table)
 * @property \renderer_base $output                       for getting output
 * @property \stdClass      $gradinginfo
 * @property int            $tablemaxrows
 * @property boolean        $quickgrading
 * @property boolean        $hasgrantextension            - Only do the capability check once for the entire table
 * @property boolean        $hasgrade                     - Only do the capability check once for the entire table
 * @property array          $groupsubmissions             - A static cache of group submissions
 * @property array          $submissiongroups             - A static cache of submission groups
 * @property array          $plugincache                  - A cache of plugin lookups to match a column name to a plugin efficiently
 * @property array          $scale                        - A list of the keys and descriptions for the custom scale
 */
class assign_grading_table extends \assign_grading_table{
    use \local_ned_controller\base_child_class_magic;

    /** @var bool $_can_edit_other_submission - use access through the {@see static::can_edit_other_submission()} */
    protected $_can_edit_other_submission = null;
    /** @var bool $_can_view_grades_before_midn - use access through the {@see static::can_view_grades_before_midn()} */
    protected $_can_view_grades_before_midn = null;

    /**
     * @return bool
     */
    public function can_edit_other_submission(){
        if (is_null($this->_can_edit_other_submission)){
            $this->_can_edit_other_submission = NED::$C::has_capability('editothersubmission', $this->get_context());
        }
        return $this->_can_edit_other_submission;
    }

    /**
     * @return bool
     */
    public function can_view_grades_before_midn(){
        if (is_null($this->_can_view_grades_before_midn)){
            $this->_can_view_grades_before_midn = NED::$C::cap_can_view_grades_before_midn($this->get_context());
        }
        return $this->_can_view_grades_before_midn;
    }

    /**
     * Call appropriate methods on this table class to perform any processing on values before displaying in table.
     * Takes raw data from the database and process it into human-readable format, perhaps also adding html linking when
     * displaying table as html, adding a div wrap, etc.
     *
     * See for example col_fullname below which will be called for a column whose name is 'fullname'.
     *
     * @param array|object $row row of data from db used to make one row of the table.
     * @return array one row for the table, added using add_data_keyed method.
     */
    function format_row($row){
        if (is_array($row)){
            $row = (object)$row;
        }

        if (!empty($row->userid) && !$this->can_view_grades_before_midn()){
            // don't use $row->timemarked as grade time here, as it shows only submission grade, but not gradebook's grade (overridden)
            $grade_time = NED::grade_get_shown_midn_grade_time($this->assignment->get_course_module(), $row->userid, null, false);
            $row->timemarked = $grade_time;

            if (time() < $grade_time){
                // Hide grade status
                $row->gradeid = null;
                $row->grade = null;
                $row->timemarked = null;
            }
        }

        return parent::format_row($row);
    }

    /**
     * Create edit menu for the user record
     *
     * @param \stdClass $row
     * @return string
     */
    public function col_userid(\stdClass $row){
        if ($this->can_edit_other_submission()){
            return parent::col_userid($row);
        } else {
            return '';
        }
    }

    /**
     * Display a grade with scales etc.
     *
     * @param string  $grade
     * @param boolean $editable
     * @param int     $userid   The user id of the user this grade belongs to
     * @param int     $modified Timestamp showing when the grade was last modified
     *
     * @return string The formatted grade
     */
    public function display_grade($grade, $editable, $userid, $modified){
        if (!$this->can_view_grades_before_midn()){
            $grade_time = NED::grade_get_shown_midn_grade_time($this->assignment->get_course_module(), $userid, $modified, false);
            $modified = $grade_time;

            if (time() < $modified){
                $grade = null;
                $modified = null;
            }
        }

        return parent::display_grade($grade, $editable, $userid, $modified);
    }

    /**
     * @param \assign_grading_table $parent
     *
     * @return static|object
     */
    static function get_ned_assign_grading_table($parent){
        return static::bcc_create_from_parent($parent);
    }
}
