<?php
/**
 * Override standard base class definition to represent an individual user's grade
 *
 * @package    local_ned_controller
 * @subpackage NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller;

defined('MOODLE_INTERNAL') || die();

use local_ned_controller\shared_lib as NED;

NED::require_lib('/grade/grade_grade.php');

/**
 * ned_grade_grade is an object mapped to DB table {prefix}grade_grades
 *
 * @package    local_ned_controller
 * @copyright  2021 NED {@link http://ned.ca}
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class ned_grade_grade extends \grade_grade {
    const TABLE = 'grade_grades';
    const CLASSNAME = '\local_ned_controller\ned_grade_grade';

    //region STATIC NATIVE
    /**
     * Finds and returns a grade_grade instance based on params.
     *
     * @param array $params associative arrays varname => value
     *
     * @return static Returns a grade_grade instance or false if none found
     */
    public static function fetch($params) {
        return \grade_object::fetch_helper(static::TABLE, static::CLASSNAME, $params);
    }

    /**
     * Finds and returns all grade_grade instances based on params.
     *
     * @param array $params associative arrays varname=>value
     * @return static[] array of static instances or false if none found.
     */
    public static function fetch_all($params) {
        return \grade_object::fetch_all_helper(static::TABLE, static::CLASSNAME, $params);
    }
    //endregion

    //region STATIC NEW
    /**
     * @param \grade_item|false|null $grade_item
     * @param int|string             $userid
     * @param bool                   $create_if_none
     *
     * @return static|null
     */
    public static function get_by_grade_item(&$grade_item, $userid, $create_if_none=false){
        if (!$grade_item || !$userid){
            return null;
        }
        $gg = static::fetch(['itemid' => $grade_item->id, 'userid' => $userid]);
        if (!$gg && $create_if_none){
            $gg = new static(['itemid' => $grade_item->id, 'userid' => $userid]);
        }

        if ($gg){
            $gg->grade_item = &$grade_item;
        }
        return $gg ?: null;
    }
    //endregion

    //region NON-STATIC NEW
    /**
     * Insert the grade_grade instance into the database.
     * It ignores simple checks and grade history update - use grade_grade insert method in common cases
     * @see \grade_grade::insert()
     *
     * @param false $update - set to true, if you wish to update $grade after insert
     * @param bool $notify_changed - to notify the completion system that a user's grade has changed, clear up a possible score cache.
     *
     * @return int|false - The new grade_grade ID if successful, false otherwise
     */
    public function raw_insert($update=false, $notify_changed=true){
        global $DB;

        $data = $this->get_record_data();
        unset($data->id);
        $this->id = $DB->insert_record($this->table, $data);

        if ($update){
            // set all object properties from real db data
            $this->update_from_db();
        }

        if ($notify_changed){
            $this->notify_changed(false);
        }

        return $this->id;
    }

    /**
     * Updates this object in the Database, based on its object variables. ID must be set.
     * It ignores simple checks and grade history update - use grade_grade update method in common cases
     * @see \grade_grade::update()
     *
     * @param bool $notify_changed - to notify the completion system that a user's grade has changed, clear up a possible score cache.
     *
     * @return bool - success
     */
    public function raw_update($notify_changed=true){
        global $DB;

        if (empty($this->id)) {
            debugging('Can not update grade object, no id!');
            return false;
        }

        $data = $this->get_record_data();
        $DB->update_record($this->table, $data);

        if ($notify_changed){
            $this->notify_changed(false);
        }

        return true;
    }

    /**
     * Get all grades by itemid(s) and/or userids
     * You can left something empty, but you need set at least something
     *
     * Made by \grade_object::fetch_all_helper example
     * @see \grade_object::fetch_all_helper()
     *
     * @param numeric|array $itemids - id(s) of grade items; if empty - load for all items
     * @param numeric|array $userids - user id(s); if empty - load for all users
     *
     * @return array|static[] id => $object
     */
    public static function fetch_all_itemid_userid($itemids=[], $userids=[]){
        $result = [];
        if (empty($itemids) && empty($userids)){
            // do not return whole table
            return $result;
        }

        $itemids = NED::val2arr($itemids);
        $userids = NED::val2arr($userids);
        list($where, $params) = NED::sql_get_in_or_equal_options(['itemid' => $itemids, 'userid' => $userids]);
        $rs = NED::db()->get_recordset_select(static::TABLE, $where, $params);

        if ($rs->valid()) {
            foreach($rs as $data) {
                $instance = new static();
                /** @var static|\stdClass $instance */
                static::set_properties($instance, $data);
                $result[$instance->id] = $instance;
            }
        }
        $rs->close();

        return $result;
    }
    //endregion
}