<?php
/**
 * Plugins overview
 *
 * @package     local_ned_controller
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 *
 * Here 3 classes, ned_notifications (main), ned_notify & ned_user_notify
 *  it's part of local_ned_controller notification system
 */

namespace local_ned_controller;
use local_ned_controller\shared_lib as NED;

defined('MOODLE_INTERNAL') || die();
/** @var \stdClass $CFG */
require_once($CFG->dirroot . '/local/ned_controller/local_lib.php');
require_once($CFG->dirroot . '/user/profile/lib.php');
require_once($CFG->dirroot . '/user/externallib.php');
require_once($CFG->dirroot . '/lib/classes/user.php');
require_once($CFG->libdir . '/badgeslib.php');
require_once($CFG->dirroot . '/cohort/lib.php');

/**
 * @property-read ned_notify[] $active = [];
 * @property-read $userid = 0;
 * @property-read ned_user_notify[] $user_notes = [];
 * @property-read $profile;
 * @property-read $cohorts;
 * @property-read $badges;
 * @property-read $dm_complete;
 * @property-read $kica_gradebook_complete;
 */
class ned_notifications
{
    const MIN_WEIGHT = -10;
    const MAX_WEIGHT = 10;
    const TYPE_MESSAGE = 0;
    const TYPE_INFO = ned_notify::TYPE_INFO;
    const TYPE_REMINDER = ned_notify::TYPE_REMINDER;
    const TYPE_NOTICE = ned_notify::TYPE_NOTICE;
    const RULE_OR = 0;          // rule "any"
    const RULE_AND = 1;         // rule "all"
    const MATCH = 0;
    const NOT_MATCH = 1;
    const REPEAT_ONCE = 0;
    const REPEAT_THRICE = 1;
    const REPEAT_UNLIMITED = 2;
    const MORE = 0;
    const LESS = 1;
    const BEFORE = 0;
    const AFTER = 1;
    const CLASS_START = 0;
    const CLASS_END = 1;

    const TIMEOUT_NO = 0;
    const TIMEOUT_15MIN = 900;
    const TIMEOUT_HOUR = 3600;
    const TIMEOUT_DAY = 86400;
    const TIMEOUT_BY_LOGIN = 1;

    const COND_TYPE_PROFILE = 'profile';
    const COND_TYPE_COHORT = 'cohort';
    const COND_TYPE_BADGE = 'badge';
    const COND_TYPE_DM = 'deadline_manager';
    const COND_TYPE_KICA_GRADEBOOK = 'kica_gradebook';
    const COND_TYPE_ACTIVITY= 'activity';
    const COND_TYPE_CLASS_TIME = 'class_time';
    const COND_TYPES = [
        self::COND_TYPE_PROFILE,
        self::COND_TYPE_COHORT,
        self::COND_TYPE_BADGE,
        self::COND_TYPE_DM,
        self::COND_TYPE_KICA_GRADEBOOK,
        self::COND_TYPE_ACTIVITY,
        self::COND_TYPE_CLASS_TIME,
    ];

    const REDIRECT_NONE = 0;
    const REDIRECT_DM = 1;
    const REDIRECT_ACTIVITY_COND = 2;
    const REDIRECT_ACTIVITY_OTHER = 3;
    const REDIRECT_PAGE = 9;

    const REDIRECTS_GENERAL = [
        self::REDIRECT_NONE => 'notifications:redirect:none',
        self::REDIRECT_PAGE => 'notifications:redirect:webpage',
    ];
    const REDIRECTS_COURSE = [
        self::REDIRECT_NONE           => 'notifications:redirect:none',
        self::REDIRECT_DM             => 'notifications:redirect:dm',
        self::REDIRECT_ACTIVITY_COND  => 'notifications:redirect:condition_activity',
        self::REDIRECT_ACTIVITY_OTHER => 'notifications:redirect:other_activity',
        self::REDIRECT_PAGE           => 'notifications:redirect:webpage',
    ];

    const CONDITIONS_NAME = 'conditions';
    const COURSE_LEVEL_NAME = 'courselevel';

    const TABLE_NOTIFY = 'local_ned_controller_notify';
    const TABLE_NOTIFY_USER = 'local_ned_controller_unotes';
    const COL_USERIDS_REMOVE = 'userids_remove';
    const COL_USERIDS_ADD = 'userids_add';

    /** @var \cache_application[]|\cache_session[]|\cache_store[] $_cache */
    static protected $_cache = null;
    /** @var ned_notify[] $_active */
    static protected $_active = null;

    protected $_userid = 0;
    /** @var ned_user_notify[] | ned_user_notify[][] $_user_notes */
    protected $_user_notes = [];
    /** @var \cache_application[]|\cache_session[]|\cache_store[] $_usercache */
    protected $_usercache = null;
    protected $_profile = null;
    protected $_cohorts = null;
    protected $_badges = null;
    protected $_dm_complete = null;
    protected $_kica_gradebook_complete = null;

    public $courseid = 0;
    public $now = 0;

    /**
     * ned_notifications constructor.
     *
     * @param object|numeric $user_or_id
     * @param object|numeric $course_or_id
     */
    public function __construct($user_or_id=null, $course_or_id=null){
        $this->now = time();
        $this->_userid = NED::get_userid_or_global($user_or_id);
        $this->courseid = NED::get_courseid_or_global($course_or_id);

        $this->_usercache = NED::get_user_cache();
        $this->_load_user_cache();
    }

    /**
     * @param $name
     *
     * @return \mixed
     */
    public function __get($name){
        switch($name){
            case 'cache':
            case 'usercache':
                return null;
            case 'active':
                return static::get_active();
            case 'dm_complete':
                return $this->get_dm_complete();
            case 'kica_gradebook_complete':
                return $this->get_kica_gradebook_complete();
        }

        $name = '_' . $name;
        return property_exists($this, $name) ? $this->$name : null;
    }

    /**
     * load user_notes cache and check it for actuality
     */
    protected function _load_user_cache(){
        if (!$this->_userid){
            $this->_user_notes = [];
            return;
        }
        /** @var ned_user_notify[] $user_notes */
        $user_notes = $this->_usercache->get('user_notes');
        $actives = $this->active;
        $update = false;
        $load_form_db = false;
        if (!$user_notes){
            $user_notes = [];
            $load_form_db = true;
            $records = static::_db()->get_records(static::TABLE_NOTIFY_USER, ['userid' => $this->_userid]);
            foreach ($records as $record){
                if (isset($actives[$record->notifyid])){
                    if ($actives[$record->notifyid]->foreachcourse){
                        $user_notes[$record->notifyid][$record->courseid] = new ned_user_notify($record);
                    } else {
                        $user_notes[$record->notifyid] =  new ned_user_notify($record);
                    }
                }
            }
        }

        foreach ($actives as $act){
            if (!isset($user_notes[$act->id])){
                $note_exist = false;
                $un = null;
            } elseif ($act->foreachcourse) {
                if (is_array($user_notes[$act->id])){
                    $note_exist = isset($user_notes[$act->id][$this->courseid]);
                } else {
                    $note_exist = $user_notes[$act->id]->courseid == $this->courseid;
                    $user_notes[$act->id] = [$user_notes[$act->id]->courseid => $user_notes[$act->id]];
                }
                $un = $note_exist ? $user_notes[$act->id][$this->courseid] : 0;
            } else { // !$act->foreachcourse
                if (is_array($user_notes[$act->id])){
                    if (isset($user_notes[$act->id][0])){
                        $un = $user_notes[$act->id][0];
                    } else {
                        $un = new ned_user_notify();
                        $un->userid = $this->_userid;
                        $un->notifyid = $act->id;
                        foreach ($user_notes[$act->id] as $unote){
                            $un->iteration += $unote->iteration;
                            $un->dontshow = $un->dontshow || $unote->dontshow;
                            if ($unote->timemodified > $un->timemodified){
                                $un->timemodified = $unote->timemodified;
                            }
                        }
                    }
                    $un->id = 0;
                    $un->courseid = 0;
                    $user_notes[$act->id] = $un;
                    $update = true;
                    $note_exist = true;
                    static::reset_active($act->id, $this->_userid,false);
                    $this->update_user_db_notification($user_notes[$act->id]);
                } else {
                    $un = $user_notes[$act->id];
                    $note_exist = true;
                }
            }


            if ( !$note_exist || (!$load_form_db && $un && $this->is_active_reset($un)) ){
                $update = true;
                $un = new ned_user_notify();
                $un->userid = $this->_userid;
                $un->notifyid = $act->id;
                if ($act->foreachcourse){
                    $un->courseid = $this->courseid;
                    $user_notes[$act->id][$this->courseid] = $un;
                } else {
                    $user_notes[$act->id] = $un;
                }
            }
        }

        // check actual of status
        foreach ($user_notes as $nid => $unote){
            if (!isset($actives[$nid])){
                $update = true;
                unset($user_notes[$nid]);
            } else {
                $un = $actives[$nid]->foreachcourse ? $unote[$this->courseid] : $unote;
                if ($actives[$nid]->timemodified >= $un->lastcheck){
                    $update = $this->_check_and_set_actual($un) || $update;
                }
            }
        }

        if ($load_form_db || $update){
            $this->_usercache->set('user_notes', $user_notes);
        }

        $this->_user_notes = $user_notes;
    }

    /**
     * Check and set actuality of current user-notify
     *      Return needing of update
     * @param ned_user_notify $unote
     *
     * @return bool
     */
    protected function _check_and_set_actual($unote){
        $active = static::get_active($unote->notifyid);
        $res = false;
        $update_time = true;

        do {
            if ($active->disabled){
                break;
            }

            if ($active->courselevel && !$this->is_course_page()){
                $update_time = false;
                break;
            }

            if ($unote->id != 0){
                // user press "don't show again" before
                if ($active->dontshow && $unote->dontshow){
                    break;
                }

                switch($active->iteration){
                    case static::REPEAT_ONCE:
                        if ($unote->iteration > 0){
                            break 2;
                        }
                        break;
                    case static::REPEAT_THRICE:
                        if ($unote->iteration > 2){
                            break 2;
                        }
                        break;
                }
            }

            if (in_array($unote->userid, $active->userids_add)){
                $res = true;
                break;
            } elseif (in_array($unote->userid, $active->userids_remove)){
                break;
            }

            $update_time = !$active->impermanent;
            foreach ($active->conditions as $condition){
                $check = $this->_check_condition($condition);
                if (is_null($check)){
                    continue;
                } elseif ($check && $active->notification_rule == static::RULE_OR){
                    $res = true;
                    break 2;
                } elseif (!$check && $active->notification_rule == static::RULE_AND){
                    break 2;
                }
            }

            $res = $active->notification_rule == static::RULE_AND;
            break;

        } while(false);

        $unote->actual = $res;
        if ($update_time){
            $unote->update_time();
        }
        return $update_time;
    }

    /**
     * @param array $condition
     *
     * @return null|bool
     */
    protected function _check_condition($condition){
        /**
         * For the data in the $condition @see \local_ned_controller\form\notifications_edit_form::get_condition_element()
         */
        $type = $condition['type'] ?? '';
        if (static::is_course_level_type($type) && !$this->is_course_page()) return false;

        $is_match = null;
        if (isset($condition['equal'])){
            $is_match = $condition['equal'] == static::MATCH;
        }

        switch ($type){
            default:
                trigger_error ("Unknown condition type \"$type\".", E_USER_NOTICE);
                return null;
            case static::COND_TYPE_PROFILE:
                $profile = $this->get_profile();
                $field = $condition['select_value'];
                if (isset($profile->$field) && !empty($profile->$field)){
                    if ($is_match){
                        return $profile->$field == $condition['text_value'];
                    } else {
                        return !str_contains($profile->$field, $condition['text_value']);
                    }
                } else {
                    return empty($condition['text_value']) == $is_match;
                }
            case static::COND_TYPE_COHORT:
                $cohorts = $this->get_cohorts();
                return isset($cohorts[$condition['select_value']]) == $is_match;
            case static::COND_TYPE_BADGE:
                $badges = $this->get_badges();
                return isset($badges[$condition['select_value']]) == $is_match;
            case static::COND_TYPE_DM:
                return $this->dm_complete === $is_match;
            case static::COND_TYPE_KICA_GRADEBOOK:
                return $this->kica_gradebook_complete === $is_match;
            case static::COND_TYPE_ACTIVITY:
                return $this->check_activity_completion($condition['name_value'] ?? '', $is_match);
            case static::COND_TYPE_CLASS_TIME:
                return $this->check_class_time($condition['select_type'] ?? null, $condition['more_less'] ?? null,
                $condition['before_after'] ?? null, $condition['time_value'] ?? null);
        }

        /** @noinspection PhpUnreachableStatementInspection */
        trigger_error ("Unexpected error.", E_USER_NOTICE);
        return null;
    }

    /**
     * @return bool
     */
    public function is_course_page(){
        return $this->courseid && $this->courseid != SITEID;
    }

    /**
     * Get dm_completion state
     *
     * @see ned_notifications::_calculate_dm_completed()
     *
     * @return bool
     */
    public function get_dm_complete(){
        if (is_null($this->_dm_complete)){
            $this->_dm_complete = $this->_calculate_dm_completed();
        }
        return $this->_dm_complete;
    }

    /**
     * Calculate DM condition completion
     *
     * @return bool
     */
    protected function _calculate_dm_completed(){
        global $PAGE;
        $is_dm_check = NED::is_tt_exists() && $this->is_course_page();
        if (!$is_dm_check) return 0;

        $block_check = ($this->courseid != $PAGE->course->id) || ($PAGE->blocks->is_block_present(NED::TT_NAME));
        if (!$block_check) return 0;

        $dm = new \block_ned_teacher_tools\deadline_manager($this->courseid);
        return !$dm->has_missed_schedule();
    }

    /**
     * @return bool|int
     */
    public function get_kica_gradebook_complete(){
        do {
            if (!is_null($this->_kica_gradebook_complete)) break;

            $this->_kica_gradebook_complete = 0;

            if (!$this->is_course_page() || !NED::get_kica_enabled($this->courseid)) break;

            $KMB = \local_kica\output\menu_bar::get_menu_bar_by_courseid($this->courseid);
            [$incompleteactivities, $numberofincompleted] = $KMB->get_incomplete_activities();
            [$mismatchactivities, $numberofmismatch] = $KMB->get_grade_mismatches();
            [$flaggedactivities, $numberofflagged] = $KMB->get_grade_flagged();
            $this->_kica_gradebook_complete = empty($incompleteactivities) && empty($mismatchactivities) && empty($flaggedactivities);
        } while (false);

        return $this->_kica_gradebook_complete;
    }

    /**
     * Check activity completion condition
     *
     * @param string $activity_name - activity name to check
     * @param bool   $is_completed - should activity be completed (true) or incompleted (false)
     *
     * @return bool
     */
    public function check_activity_completion($activity_name='', $is_completed=false){
        if (empty($activity_name)) return false;

        $cm = NED::cm_find_activity_by_name($this->courseid, $activity_name);
        if (empty($cm)) return false;

        $has_incomplete_cm = NED::check_cms_completion_by_course($this->courseid, $this->_userid, [$cm], COMPLETION_INCOMPLETE);
        return $has_incomplete_cm != $is_completed;
    }

    /**
     * Check class time condition
     *
     * @param $class_start_or_end - check class start or end time
     * @param $more_or_less - it should more or less than X from the NOW
     * @param $before_or_after - checked X before (-) or after (+) NOW
     * @param $time_delay - time delay (X) from NOW
     *
     * @return bool
     */
    public function check_class_time($class_start_or_end=self::CLASS_START, $more_or_less=self::MORE, $before_or_after=self::BEFORE, $time_delay=0){
        $class_start_or_end = $class_start_or_end ?? static::CLASS_START;
        $more_or_less = $more_or_less ?? static::MORE;
        $before_or_after = $before_or_after ?? static::BEFORE;
        $time_delay = $time_delay ?? 0;

        if ($class_start_or_end == static::CLASS_START){
            $class_time = NED::get_user_start_date($this->courseid, $this->_userid);
        } else {
            $class_time = NED::get_user_end_date($this->courseid, $this->_userid);
        }
        if (is_null($class_time)) return false;

        $checked_time = time();
        $invert_more_less = false;
        if (!empty($time_delay)){
            $time_delay = abs($time_delay);
            if ($before_or_after == static::BEFORE){
                /**
                 * We suppose, that for time value checked "before" now, "less" means "closer to the now",
                 * which for int numbers mean ">", not "<".
                 */
                $invert_more_less = true;
                $time_delay = -$time_delay;
            }
            $checked_time += $time_delay;
        }

        if (($more_or_less == static::MORE) == !$invert_more_less){
            return $class_time >= $checked_time;
        } else {
            return $class_time <= $checked_time;
        }
    }

    /**
     * Return true, if user has at least one actual for him notification
     * @return bool
     */
    public function have_actual(){
        foreach ($this->_user_notes as $user_note){
            if ($user_note->actual){
                return true;
            }
        }
        return false;
    }

    /**
     * Get user_note by id or actual user_note with the biggest weight
     * @param int $id
     *
     * @return \local_ned_controller\ned_user_notify|\stdClass|null
     */
    public function get_actual_note($id=0){
        $active_list = $this->active;

        if ($id){
            if (isset($this->_user_notes[$id])){
                if ($active_list[$id]->courselevel && !$this->is_course_page()){
                    return null;
                } elseif ($active_list[$id]->foreachcourse){
                    return $this->_user_notes[$id][$this->courseid] ?? null;
                } else {
                    return $this->_user_notes[$id];
                }
            }
            return null;
        }

        $weight = static::MIN_WEIGHT - 1;
        $note = null;
        foreach ($active_list as $active){
            $user_note = $this->get_actual_note($active->id);
            if (!$user_note || !$user_note->actual || !static::check_note_timeout($user_note)){
                continue;
            }
            if ($active->weight > $weight){
                $weight = $active->weight;
                $note = $active;
            }
        }
        return $note ? $note->js_export() : null;
    }

    /**
     * Return array of actual user_notify in weight order
     * @return ned_user_notify[]
     */
    public function get_actual_notes_by_weight(){
        $active_list = $this->active;
        $weight_list = [];
        $note_list = [];
        foreach ($active_list as $active){
            $user_note = $this->get_actual_note($active->id);
            if (!$user_note || !$user_note->actual || !static::check_note_timeout($user_note)){
                continue;
            }
            $weight_list[$active->id] = $active->weight;
        }

        arsort($weight_list);
        foreach ($weight_list as $notifyid => $weight){
            $note_list[] = $active_list[$notifyid]->js_export();
        }

        return $note_list;
    }

    /**
     * Get actual notes and show it
     * You can call it from outside
     */
    public function check_and_show_notifications(){
       $notes = $this->get_actual_notes_by_weight();
       static::show_notifications($notes);
    }

    /**
     * Call it, when user press "close" button
     *  add count of view and set "don't show again" if it needs
     *
     * @param      $notifyid
     * @param bool $dontshow
     *
     * @return bool
     */
    public function update_user_notifications($notifyid, $dontshow=false){
        $actual_note = $this->get_actual_note($notifyid);
        if (!$this->_userid || is_null($actual_note) || !$actual_note->actual){
            return false;
        }

        $actual_note->iteration++;
        $actual_note->timemodified = time();
        $active = static::get_active($notifyid);
        if ($active->dontshow && $dontshow){
            $actual_note->dontshow = true;
        }

        $this->update_user_db_notification($actual_note);

        // check actual AFTER you get id for it
        $this->_check_and_set_actual($actual_note);

        if ($active->foreachcourse){
            $this->_user_notes[$notifyid][$actual_note->courseid] = $actual_note;
        } else {
            $this->_user_notes[$notifyid] = $actual_note;
        }
        $this->clear_user_cache(true);

        return true;
    }

    /**
     * @param ned_user_notify $actual_note
     *
     * @return int
     */
    public function update_user_db_notification($actual_note){
        if ($actual_note->id > 0){
            static::_db()->update_record(static::TABLE_NOTIFY_USER, $actual_note->export());
        } else {
            $actual_note->id = static::_db()->insert_record(static::TABLE_NOTIFY_USER, $actual_note->export());
        }
        return $actual_note->id;
    }

    /**
     * Update user_cache by local value or remove it
     * @param bool $update
     */
    public function clear_user_cache($update = true){
        if ($update){
            $this->_usercache->set('user_notes', $this->_user_notes);
        } else {
            $this->_usercache->delete('user_notes');
            $this->_load_user_cache();
        }
    }

    /**
     * Check, has this active notification reset value,
     *        which newer than user_notification data update time
     *
     * @param ned_user_notify | int | string $note
     *
     * @return bool
     */
    public function is_active_reset($note){
        if (is_object($note)){
            $id = $note->notifyid;
        } else {
            $id = $note;
            $note = $this->get_actual_note($id);
        }

        if (!$note){
            return false;
        }

        $reset_time = static::get_active_reset($id);
        if (!$reset_time){
            return false;
        }

        if ($note->lastcheck > $reset_time){
            return false;
        }

        return true;
    }

    /**
     *
     * @return object|null
     * @throws \dml_exception
     */
    public function get_profile(){
        if (is_null($this->_profile)){
            $profile = (array)\core_user::get_user($this->_userid, '*');
            $profile += (array)profile_user_record($this->_userid, false);
            $this->_profile = (object)$profile;
        }
        return $this->_profile;
    }

    /**
     * @return array|null
     */
    public function get_cohorts(){
        if (is_null($this->_cohorts)){
            $this->_cohorts = cohort_get_user_cohorts($this->_userid);
        }
        return $this->_cohorts;
    }

    /**
     * @return array|null
     */
    public function get_badges(){
        if (is_null($this->_badges)){
            $this->_badges = [];
            $badges = badges_get_user_badges($this->_userid);
            foreach ($badges as $badge){
                $this->_badges[$badge->name] = $badge->name;
            }
        }
        return $this->_badges;
    }

    /**
     * @return \cache_application|\cache_application[]|\cache_session|\cache_session[]|\cache_store|\cache_store[]
     * @noinspection PhpReturnDocTypeMismatchInspection
     */
    static protected function _get_cache(){
        if (is_null(static::$_cache)){
            static::$_cache = \cache::make(NED::$PLUGIN_NAME, 'notification');
        }
        return static::$_cache;
    }

    /**
     * Usually you can just use ->active
     * @param int $id
     *
     * @return ned_notify[] | ned_notify
     */
    static public function get_active($id=0){
        if (is_null(static::$_active)){
            $data = static::_get_cache()->get('active');
            if (!$data){
                $data = [];
                $records = static::_db()->get_records(static::TABLE_NOTIFY, ['disabled' => 0]);
                foreach ($records as $record){
                    $data[$record->id] = new ned_notify(static::unpack_data($record));
                }
                static::_get_cache()->set('active', $data);
            }
            static::$_active = $data;
        }

        if ($id){
            return static::$_active[$id] ?? null;
        }
        return static::$_active;
    }

    /**
     * @param int $id
     *
     * @return null | \local_ned_controller\ned_notify
     */
    static public function get_notify_from_db($id=0){
        $record = static::_db()->get_record(static::TABLE_NOTIFY, ['id' => $id]);
        if (empty($record)){
            return null;
        }
        return new ned_notify(static::unpack_data($record));
    }

    /**
     * @return bool
     */
    static public function check_db_tables(){
        $res = static::_get_cache()->get('tables_exists');
        if ($res){
            return true;
        }

        $tables = static::_db()->get_tables();
        $res = isset($tables[static::TABLE_NOTIFY]) && isset($tables[static::TABLE_NOTIFY_USER]);
        if ($res){
            static::_get_cache()->set('tables_exists', true);
        }

        return $res;
    }

    /**
     *  Clear application cache of notifications
     */
    static public function clear_active_cache(){
        static::_get_cache()->delete('active');
        static::$_active = null;
    }

    /**
     * Reset views or "don't show again" of notification
     *
     * @param      $id
     * @param null $userid
     * @param bool $reset_cache
     *
     */
    static public function reset_active($id, $userid=null, $reset_cache=true){
        $active = static::get_active($id);
        if (!$active){
            return;
        }

        $params = ['notifyid' => $id];
        if (!is_null($userid)){
            $params['userid'] = $userid;
        }

        static::_db()->delete_records(static::TABLE_NOTIFY_USER, $params);
        if ($reset_cache){
            static::_get_cache()->set('reset' . $id, time());
        }
    }

    /**
     * Get reset time for notification, if it exists
     * @param $id
     *
     * @return false|int|null
     */
    static public function get_active_reset($id){
        $active = static::get_active($id);
        if ($active){
            return static::_get_cache()->get('reset' . $id);
        }
        return null;
    }

    /**
     * @param ned_user_notify $user_note
     *
     * @return bool
     */
    static public function check_note_timeout($user_note){
        global $USER;
        $now = time();
        $active = static::get_active($user_note->notifyid);
        if (!$active){
            return false;
        }
        switch ($active->timeout){
            /** @noinspection PhpMissingBreakStatementInspection */
            default:
                trigger_error ("Unknown timeout \"$active->timeout\".", E_USER_NOTICE);
                $active->timeout = static::TIMEOUT_NO;
            case static::TIMEOUT_NO:
            case static::TIMEOUT_15MIN:
            case static::TIMEOUT_HOUR:
            case static::TIMEOUT_DAY:
                return ($now - $user_note->timemodified) >= $active->timeout;
            case static::TIMEOUT_BY_LOGIN:
                return $USER->currentlogin > $user_note->timemodified;
        }
    }

    /**
     * @return array
     */
    static public function get_condition_types(){
        return static::COND_TYPES;
    }

    /**
     * @param string $type - one of the {@see COND_TYPES}}
     *
     * @return bool
     */
    static public function is_course_level_type($type){
        return match ($type) {
            static::COND_TYPE_DM,
            static::COND_TYPE_KICA_GRADEBOOK,
            static::COND_TYPE_ACTIVITY,
            static::COND_TYPE_CLASS_TIME => true,

            default => false,
        };
    }

    /**
     * @param string $type - one of the {@see COND_TYPES}}
     *
     * @return bool
     */
    static public function is_impermanent_type($type){
        return static::is_course_level_type($type);
    }

    /**
     * Used by form for selector options
     * @return array
     */
    static public function get_profile_fields(){
        $choices = [];
        $standard_fields = [
            'firstname', 'lastname', 'email', 'city', 'country', 'url', 'icq', 'skype', 'aim', 'yahoo',
            'msn', 'idnumber', 'institution', 'department', 'phone1', 'phone2', 'address'
        ];
        $social_fields = \profilefield_social\helper::get_networks();

        foreach ($standard_fields as $field){
            $choices[$field] = $social_fields[$field] ?? \core_user\fields::get_display_name($field);
        }

        $custom_fields =  profile_get_custom_fields();
        foreach ($custom_fields as $field){
            $choices[$field->shortname] = $field->name;
        }

        return $choices;
    }

    /**
     * Used by form for selector options
     * @return array
     */
    static public function get_cohort_list(){
        $cohorts = cohort_get_all_cohorts(0, 0);
        $cohort_list = [];
        foreach ($cohorts['cohorts'] as $cohort){
            $cohort_list[$cohort->id] = $cohort->name;
        }
        return $cohort_list;
    }

    /**
     * Used by form for selector options
     * @return array
     */
    static public function get_badge_list(){
        global $DB;

        $sql = "SELECT DISTINCT b.name, b.name AS name2 FROM {badge} AS b";
        $params = ['deleted' => \BADGE_STATUS_ARCHIVED];
        $where = ['b.status <> :deleted'];

        $types = [\BADGE_TYPE_SITE, \BADGE_TYPE_COURSE];
        if (!empty($types)){
            if (count($types) == 1){
                $where[] = 'b.type = :type';
                $params['type'] = reset($types);
            } else {
                $conds = [];
                foreach ($types as $i => $type){
                    $key = 'type_'.$i;
                    $conds[] = ':'.$key;
                    $params[$key] = $type;
                }
                $where[] = 'b.type in ('.join(', ', $conds).')';
            }
        }
        if (!empty($where)){
            $sql .= ' WHERE ' . join(' AND ', $where);
        }

        return $DB->get_records_sql_menu($sql, $params);
    }

    /**
     * Used by form for class-time selector options
     *
     * @return array
     */
    static public function get_class_time_types(){
        return [
            static::CLASS_START => NED::str('notifications:class_time:class_start'),
            static::CLASS_END => NED::str('notifications:class_time:class_end'),
        ];
    }

    /**
     * More/less list
     *
     * @return array
     */
    static public function get_more_less_list(){
        return [
            static::MORE => NED::str('notifications:common:more'),
            static::LESS => NED::str('notifications:common:less'),
        ];
    }

    /**
     * Before/after list
     *
     * @return array
     */
    static public function get_before_after_list(){
        return [
            static::BEFORE => NED::str('notifications:common:before'),
            static::AFTER => NED::str('notifications:common:after'),
        ];
    }

    /**
     * Get table with notification list
     *
     * @return \html_table
     * @throws \coding_exception
     * @throws \dml_exception
     * @throws \moodle_exception
     */
    static public function get_table_view(){
        global $PAGE;
        $table = new \html_table();
        $table->attributes['class'] = 'generaltable ned_notifications';
        $table->head = [get_string('name'), NED::str('preview'), NED::str('notifications:description'),
            NED::str('type'), get_string('active')];
        $records = static::_db()->get_records(static::TABLE_NOTIFY, null, 'id DESC', 'id, name, description, weight, notification_type, disabled');
        foreach ($records as $note){
            $table->data[] = NED::row([
                NED::link(['~/notifications_edit.php', ['id' => $note->id]], $note->name),
                NED::fa('fa-external-link btn_notification_preview', '', '', ['data-notifyid' => $note->id]),
                $note->description,
                NED::fa(ned_notify::get_type_icon($note->notification_type)),
                $note->disabled ? NED::fa('fa-times') : NED::fa('fa-check')
            ]);
        }
        $PAGE->requires->js_call_amd(NED::$PLUGIN_NAME . '/show_preview_ned_notifications', 'init',
            [NED::str('loading')]);

        return $table;
    }

    /**
     * Save data from form - to DB
     * @param $data
     *
     * @return bool|int
     * @throws \dml_exception
     */
    static public function save_form_data($data){
        $data = (array)$data;
        $t_cols = static::_db()->get_columns(static::TABLE_NOTIFY);
        $t_data = [];
        $c = static::CONDITIONS_NAME;

        foreach ($t_cols as $col => $col_info){
            if ($col == 'id' || $col == $c) continue;

            if ($col == static::COL_USERIDS_ADD || $col == static::COL_USERIDS_REMOVE){
                $t_data[$col] = isset($data[$col]) ? join(';', $data[$col]) : '';
            } elseif (isset($data[$col]['text']) && isset($data[$col]['format'])) {
                $t_data[$col] = $data[$col]['text'];
            } else {
                $def = !empty($col_info->not_null) ? 0 : null;
                $t_data[$col] = $data[$col] ?? $def;
            }
        }

        $t_data['timemodified'] = time();
        $c_data = [];

        if (isset($data[$c]) && isset($data[$c]['count']) && $data[$c]['count'] > 0){
            for ($i=0; $i < $data[$c]['count']; $i++){
                if (static::is_course_level_type($data[$c][$i]['type'])){
                    $t_data[static::COURSE_LEVEL_NAME] = true;
                }
                $c_data[$i] = $data[$c][$i];
            }
            $c_data['count'] = $data[$c]['count'];
        }
        $t_data[$c] = base64_encode(serialize($c_data));

        if (!isset($data['id']) || $data['id'] < 1){
            $data['id'] = static::_db()->insert_record(static::TABLE_NOTIFY, (object)$t_data);
        } else {
            $t_data['id'] = $data['id'];
            static::_db()->update_record(static::TABLE_NOTIFY, (object)$t_data);
            if (isset($data['resetviews']) && $data['resetviews']){
                static::reset_active($data['id']);
            }
        }

        static::clear_active_cache();
        return $data['id'];
    }

    /**
     * Load data from BD - for updating form
     * @param $id
     *
     * @return array
     * @throws \dml_exception
     */
    static public function load_data_for_form($id){
        $t_data = static::_db()->get_record(static::TABLE_NOTIFY, ['id' => $id]);
        return static::unpack_data($t_data);
    }

    /**
     * Transform raw data from BD to normal structure
     * @param $t_data
     *
     * @return array|bool
     */
    static public function unpack_data($t_data){
        if (!$t_data){
            return false;
        }
        $t_data = (array)$t_data;
        $c = static::CONDITIONS_NAME;
        $data = [];

        foreach ($t_data as $col => $val){
            if ($col == $c){
                continue;
            }

            if ($col == static::COL_USERIDS_ADD || $col == static::COL_USERIDS_REMOVE){
                if (!empty($val)){
                    $data[$col] = explode(';', $val);
                } else {
                    $data[$col] = [];
                }
            } else {
                $data[$col] = $val;
            }
        }

        $data[$c] = unserialize(base64_decode($t_data[$c]));
        return $data;
    }

    /**
     * Remove notification from BD
     * @param $id
     *
     * @throws \dml_exception
     */
    static public function delete_notification($id){
        if (!$id){
            return;
        }
        static::_db()->delete_records(static::TABLE_NOTIFY, ['id' => $id]);
        static::_db()->delete_records(static::TABLE_NOTIFY_USER, ['notifyid' => $id]);
        static::_get_cache()->delete('reset' . $id);
        static::clear_active_cache();
    }

    /**
     * Show notification on page by notification list
     *
     * @param $notes
     */
    static public function show_notifications($notes){
        global $PAGE;
        ned_notify::show_notifications($notes, ['ned_notification_data', $PAGE->course->id]);
    }

    /**
     * @param array $record
     *
     * @return \local_ned_controller\ned_notify
     */
    static public function get_new_ned_notify($record=[]){
        return new ned_notify($record);
    }

    /**
     * You can see, how many times this class use $DB, if set breakpoint here
     * @return \moodle_database
     */
    static protected function _db(){
        global $DB;
        return $DB;
    }
}

/**
 * Class ned_notify
 * Structure for data from local_ned_controller_notify table
 *
 * @package local_ned_controller
 */
class ned_notify extends support\ned_notify_simple
{
    public $id = 0;
    public $name = '';
    public $weight = 0;
    public $disabled = false;
    public $notification_rule = 0;
    /** For the data in the $condition @see \local_ned_controller\form\notifications_edit_form::get_condition_element() */
    public $conditions = [];
    public $condition_count = 0;
    public $impermanent = false;
    public $iteration = 0;
    public $dontshow = true;
    public $userids_remove = [];
    public $userids_add = [];
    public $timemodified = 0;
    public $timeout = 0;
    public $courselevel = false;
    public $foreachcourse = false;
    public $redirect_option = ned_notifications::REDIRECT_NONE;
    public $redirect_data = '';

    /**
     * ned_notify constructor.
     *
     * @param array|\stdClass $record - record from DB
     */
    public function __construct($record=[]){
        parent::__construct($record);

        if (isset($this->conditions['count'])){
            unset($this->conditions['count']);
        }
        $this->condition_count = count($this->conditions);

        foreach ($this->conditions as $condition){
            if (ned_notifications::is_impermanent_type($condition['type'])){
                $this->impermanent = true;
                break;
            }
        }
        $this->dontshow = $this->iteration == ned_notifications::REPEAT_ONCE ? false : $this->dontshow;

        $this->foreachcourse = $this->courselevel ? $this->foreachcourse : false;
    }

    /**
     * Return moodle url to redirect after closing notification, or null if there is no need redirection
     *
     * @param numeric $courseid - course id of the current page
     *
     * @return \moodle_url|null
     */
    public function get_redirect_url($courseid=null){
        switch ($this->redirect_option){
            default:
            case ned_notifications::REDIRECT_NONE:
                break;
            case ned_notifications::REDIRECT_DM:
                if (NED::is_tt_exists()){
                    return new \moodle_url(NED::PAGE_DM, [NED::PAR_COURSE => $courseid, NED::PAR_DM_TO_MISSED => 1]);
                }
                break;
            case ned_notifications::REDIRECT_ACTIVITY_COND:
                foreach ($this->conditions as $condition){
                    $type = $condition['type'];
                    if (empty($type) || $type != ned_notifications::COND_TYPE_ACTIVITY || empty($condition['name_value'])) continue;

                    $cm = NED::cm_find_activity_by_name($courseid, $condition['name_value']);
                    if (!empty($cm->url)){
                        return new \moodle_url($cm->url);
                    }
                }
                break;
            case ned_notifications::REDIRECT_ACTIVITY_OTHER:
                if (!empty($this->redirect_data)){
                    $cm = NED::cm_find_activity_by_name($courseid, $this->redirect_data);
                    if (!empty($cm->url)){
                        return new \moodle_url($cm->url);
                    }
                }
                break;
            case ned_notifications::REDIRECT_PAGE:
                return new \moodle_url($this->redirect_data);
        }

        return null;
    }

    /**
     * Get only data, which need to show message on js
     *
     * @param null|object $obj
     *
     * @return \stdClass
     */
    public function js_export($obj=null){
        $keys = ['id', 'dontshow'];
        $obj = $obj ?? new \stdClass();
        foreach ($keys as $key){
            $obj->$key = $this->$key;
        }

        return parent::js_export($obj);
    }
}

/**
 * Class ned_user_notify
 *  Structure for data from local_ned_controller_unotes table
 * @package local_ned_controller
 */
class ned_user_notify
{
    public $id = 0;
    public $notifyid = 0;
    public $userid = 0;
    public $iteration = 0;
    public $dontshow = false;
    public $courseid = 0;
    public $timemodified = 0;

    public $actual = false;
    public $lastcheck = 0; // time of $actual status

    /**
     * ned_user_notify constructor.
     *
     * @param array|\stdClass $record - record from DB
     */
    public function __construct($record=[]){
        NED::import_array_to_object($this, $record);
        $this->lastcheck = 0;
    }

    /**
     * Update lastcheck
     * @param null $time
     */
    public function update_time($time=null){
        $this->lastcheck = is_null($time) ? time() : $time;
    }

    /**
     * Return only data, which stored in BD
     * @return object
     */
    public function export(){
        $keys = ['id', 'notifyid', 'userid', 'iteration', 'dontshow', 'courseid', 'timemodified'];
        $data = [];
        foreach ($keys as $key){
            $data[$key] = $this->$key;
        }
        return (object)$data;
    }
}

