<?php
/**
 * @package    local_ned_controller
 * @category   NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller\output\core;
use local_ned_controller\shared_lib as NED;

/**
 * Class admin_renderer
 * Standard HTML output renderer for core_admin subsystem
 *
 * @package local_ned_controller\output\core
 */
class admin_renderer extends \core_admin_renderer {

    /**
     * Displays the info about available Moodle core and plugin updates
     *
     * The structure of the $updates param has changed since 2.4. It contains not only updates
     * for the core itself, but also for all other installed plugins.
     *
     * * NED change: add NED plugin versions check
     *
     * @param array|null $updates array of (string)component => array of \core\update\info objects or null
     * @param int|null $fetch timestamp of the most recent updates fetch or null (unknown)
     *
     * @return string
     */
    protected function available_updates($updates, $fetch){
        $output = [];
        $output[] = parent::available_updates($updates, $fetch);
        $output[] = $this->available_updates_ned();
        $output[] = $this->available_updates_ctrl_admin_alerts();

        return join('', $output);
    }

    /**
     * Displays the info about NED plugin versions check

     * @return string
     */
    public function available_updates_ned(){
        $output = [];
        $add_warning = function($plugin, $version, $str_key, $type='warning') use (&$output) {
            $plugin_name = NED::str_check('pluginname', null, $plugin, $plugin);
            $text = NED::str($str_key, ['plugin' => $plugin_name, 'version' => $version]);
            $output[] = $this->warning($text, $type.' font-weight-bolder');
        };

        foreach (NED::PLUGIN_DIRS as $plugin => $plugin_dir){
            $d_version = NED::PLUGIN_DEPENDENCIES[$plugin] ?? 0;
            $r_version = NED::PLUGIN_REQUIREMENTS[$plugin] ?? 0;
            if (!($d_version || $r_version)){
                continue;
            }

            if (NED::is_plugin_exists($plugin)){
                $version = (int)get_config($plugin, 'version');
                if ($d_version && $d_version > $version){
                    $add_warning($plugin, $d_version, 'mustupdateplugin', 'danger');
                } elseif ($r_version && $r_version > $version){
                    $add_warning($plugin, $r_version, 'shouldupdateplugin');
                }
            } else {
                if ($d_version){
                    $add_warning($plugin, $d_version, 'mustinstallplugin', 'danger');
                }
            }
        }

        return empty($output) ? '' : join('', $output);
    }

    /**
     * Displays admin alerts
     * @see \local_ned_controller\shared\controller_util::ctrl_admin_alert_add()

     * @return string
     */
    public function available_updates_ctrl_admin_alerts(){
        $output = [];
        $add_warning = function($plugin, $key, $text, $type=NED::NOTIFY_INFO) use (&$output) {
            if (empty($text)){
                $text = NED::str_check($key, null, $key, $plugin);
            }
            $classes = [$type, $key, $plugin];
            $output[] = $this->warning($text, NED::arr2str($classes));
        };

        $admin_alerts = NED::ctrl_admin_alert_get_all();
        foreach ($admin_alerts as $plugin => $plugin_admin_alert){
            if (empty($plugin_admin_alert)) continue;

            foreach ($plugin_admin_alert as $key => $admin_alert){
                $text = $admin_alert[0] ?? '';
                $type = $admin_alert[1] ?? NED::NOTIFY_INFO;
                $add_warning($plugin, $key, $text, $type);
            }
        }

        return empty($output) ? '' : join('', $output);
    }
}
