<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * core_renderer for NED themes
 *
 * @package    local_ned_controller
 * @subpackage output
 * @category   NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 *
 * @noinspection PhpMultipleClassDeclarationsInspection
 * @noinspection PhpUndefinedClassInspection
 */

namespace local_ned_controller\output;
use local_ned_controller\shared_lib as NED;
use local_schoolmanager\school_manager as SM;

defined('MOODLE_INTERNAL') || die;
NED::require_lib('badgeslib.php');
NED::require_file('~lib.php');

/**
 * Trait core_renderer
 *
 * @package local_ned_controller\output
 */
trait core_renderer {
    protected $compactlogourl = null;
    protected $logourl = null;

    protected $additionalhtmlhead = null;
    protected $additionalhtmltopofbody = null;
    protected $additionalhtmlfooter = null;
    protected $customcss = null;

    protected $islaunched = null;

    /**
     * Constructor
     *
     * @param \moodle_page $page the page we are doing output for.
     * @param string $target one of rendering target constants
     */
    public function __construct(\moodle_page $page, $target) {
        parent::__construct($page, $target);

        if (empty($this->islaunched)) {
            $this->_init_school_logo();
            [$this->islaunched, $this->logourl, $this->compactlogourl, $this->customcss, $this->additionalhtmlhead,
                    $this->additionalhtmltopofbody,
                    $this->additionalhtmlfooter] =
                    local_ned_controller_get_active_rules_data(false, $this->logourl, $this->compactlogourl);
        }
    }

    /**
     * Set school logo (if exists) as site logo
     *
     * @return void
     */
    protected function _init_school_logo(){
        if (!NED::is_school_manager_exists()) return;

        $schoolid = NED::get_user_school(NED::get_userid_or_global(), true);
        if(!empty($schoolid)){
            $this->logourl = SM::get_logo_url($schoolid) ?: '';
            $this->compactlogourl = SM::get_compact_logo_url($schoolid) ?: '';
        }
    }

    /**
     * @param $name
     *
     * @return null
     */
    public function __get($name){
        return $this->$name ?? null;
    }

    /**
     * Returns HTML attributes to use within the body tag. This includes an ID and classes.
     *
     * @param string|array $additionalclasses Any additional classes to give the body tag,
     * @return string
     */
    public function body_attributes($additionalclasses = array()) {
        if (!is_array($additionalclasses)) {
            $additionalclasses = explode(' ', $additionalclasses);
        }
        $additionalclasses[] = 'theme-' . str_replace('_', '-', $this->page->theme->name);

        return parent::body_attributes($additionalclasses);
    }

    /**
     * Returns standard navigation between activities in a course.
     *
     * M3.4 onwards but should be harmless in less than as long as not called.
     *
     * @return string the navigation HTML.
     */
    public function activity_navigation() {
        // First we should check if we want to add navigation.
        $context = $this->page->context;
        if (($this->page->pagelayout !== 'incourse' && $this->page->pagelayout !== 'frametop')
            || $context->contextlevel != CONTEXT_MODULE) {
            return '';
        }

        // If the activity is in stealth mode, show no links.
        if ($this->page->cm->is_stealth()) {
            return '';
        }

        // Get a list of all the activities in the course.
        $course = $this->page->cm->get_course();
        $modules = get_fast_modinfo($course->id)->get_cms();

        // Put the modules into an array in order by the position they are shown in the course.
        $mods = [];
        $activitylist = [];
        foreach ($modules as $module) {
            // Only add activities the user can access, aren't in stealth mode and have a url (eg. mod_label does not).
            if (!$module->uservisible || $module->is_stealth() || empty($module->url)) {
                continue;
            }
            $mods[$module->id] = $module;

            // No need to add the current module to the list for the activity dropdown menu.
            if ($module->id == $this->page->cm->id) {
                continue;
            }
            // Module name.
            //$modname = $module->get_formatted_name();
            // Display the hidden text if necessary.
            //if (!$module->visible) {
            //    $modname .= ' ' . get_string('hiddenwithbrackets');
            //}
            // Remove jumptomenu setting
            /*
            if ((!empty($this->page->theme->settings->jumptomenu)) && ($this->page->theme->settings->jumptomenu == 2)) {
                // Module URL.
                $linkurl = new \moodle_url($module->url, array('forceview' => 1));
                // Add module URL (as key) and name (as value) to the activity list array.
                $activitylist[$linkurl->out(false)] = $modname;
            }
            */
        }

        $nummods = count($mods);

        // If there is only one mod then do nothing.
        if ($nummods == 1) {
            return '';
        }

        // Get an array of just the course module ids used to get the cmid value based on their position in the course.
        //$modids = array_keys($mods);

        // Get the position in the array of the course module we are viewing.
        //$position = array_search($this->page->cm->id, $modids);

        $prevmod = null;
        $nextmod = null;
        // Remove forwardbacklinks setting
        /*
        if ((!empty($this->page->theme->settings->forwardbacklinks)) && ($this->page->theme->settings->forwardbacklinks == 2)) {
            // Check if we have a previous mod to show.
            if ($position > 0) {
                $prevmod = $mods[$modids[$position - 1]];
            }

            // Check if we have a next mod to show.
            if ($position < ($nummods - 1)) {
                $nextmod = $mods[$modids[$position + 1]];
            }
        }
        */

        $activitynav = new \core_course\output\activity_navigation($prevmod, $nextmod, $activitylist);
        $renderer = $this->page->get_renderer('core', 'course');
        return $renderer->render($activitynav);
    }

    /**
     * Return user badges to navbar
     * Rendered through {@see static::render_custom_menu()}
     *  If you wish to change it, you may find useful these functions:
     *      /badges/lib.php: core_badges_myprofile_navigation
     *      /badges/renderer.php: print_badges_list
     *
     * @return string
     */
    protected function _get_badge_navbar_list() {
        global $USER, $OUTPUT;
        if (!$USER->id) return '';

        $badges = badges_get_user_badges($USER->id);
        if (empty($badges)) return '';

        $res = '';
        $now = time();
        $ctx = NED::ctx();

        foreach ($badges as $badge){
            $context = ($badge->type == BADGE_TYPE_SITE) ? $ctx : NED::ctx($badge->courseid);
            $name = $badge->name;
            $imageurl = \moodle_url::make_pluginfile_url($context->id, 'badges', 'badgeimage',
                $badge->id, '/', 'f2', false);

            $add_to_image = '';
            if (!empty($badge->dateexpire) && $badge->dateexpire < $now) {
                $add_to_image = $OUTPUT->pix_icon('i/expired',
                    get_string('expireddate', 'badges', userdate($badge->dateexpire)),
                    'moodle',
                    ['class' => 'expireimage']);
                $name .= '(' . get_string('expired', 'badges') . ')';
            }

            $image = \html_writer::img($imageurl, $name, ['class' => 'badge-image']) . $add_to_image;

            $url = new \moodle_url('/badges/badge.php', ['hash' => $badge->uniquehash]);
            $res .= NED::link($url, $image, ['title' => $name, 'class' => 'badge-link']);
        }

        return NED::div($res, 'badge-navbar-list');
    }

    /**
     * Renders a custom menu object (located in outputcomponents.php)
     *
     * The custom menu this method produces makes use of the YUI3 menunav widget
     * and requires very specific html elements and classes.
     *
     * @staticvar int $menucount
     *
     * @param \custom_menu $menu
     *
     * @return string
     */
    protected function render_custom_menu(\custom_menu $menu){
        $content = parent::render_custom_menu($menu);
        //$content .= $this->_get_badge_navbar_list();
        return $content;
    }

    /**
     * Returns the page heading menu.
     *
     * @return string HTML.
     */
    public function page_heading_menu() {
        $content = parent::page_heading_menu();
        $content .= $this->_get_badge_navbar_list();
        return $content;
    }

    /**
     * The standard tags (typically script tags that are not needed earlier) that
     * should be output after everything else. Designed to be called in theme layout.php files.
     *
     * @return string HTML fragment.
     */
    public function standard_end_of_body_html() {
        global $CFG;

        // This function is normally called from a layout.php file in {@link core_renderer::header()}
        // but some of the content won't be known until later, so we return a placeholder
        // for now. This will be replaced with the real content in {@link core_renderer::footer()}.
        $output = '';
        if ($this->page->pagelayout !== 'embedded') {
            if (!empty($CFG->additionalhtmlfooter)) {
                $output .= "\n" . $CFG->additionalhtmlfooter;
            }
            if (!empty($this->additionalhtmlfooter)) {
                $output .= $this->additionalhtmlfooter;
            }
        }

        $output .= $this->unique_end_html_token;
        return $output;
    }

    /**
     * Returns information about an activity.
     * NED: adding waring about wrong file submission, if need
     *
     * @param \cm_info $cminfo The course module information.
     * @param \core_completion\cm_completion_details $completiondetails The completion details for this activity module.
     * @param array $activitydates The dates for this activity module.
     * @return string the activity information HTML.
     * @throws coding_exception
     */
    public function activity_information(\cm_info $cminfo, \core_completion\cm_completion_details $completiondetails, array $activitydates): string {
        $output = parent::activity_information($cminfo, $completiondetails, $activitydates);

        if (NED::$ned_assign::do_show_wrong_file_warning_on_page()){
            $text = NED::div(NED::$C::str('wrongfiles_draft_warning'));
            /** @noinspection PhpUndefinedMethodInspection */
            $output .= $this->box($text, 'generalbox p-3 ned-wrong-submission');
        }

        return $output;
    }
}
