<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * myprofile rendrer
 *
 * @package    local_ned_controller
 * @category   core
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */


namespace local_ned_controller\output\core_user;
use local_ned_controller\shared_lib as NED;

defined('MOODLE_INTERNAL') || die;

/** @var \stdClass $CFG */
require_once($CFG->dirroot.'/blocks/moodleblock.class.php');
require_once($CFG->dirroot.'/blocks/badges/block_badges.php');

/**
 * Class myprofile_renderer
 *
 * @package local_ned_controller\output\core_user
 */
class myprofile_renderer extends \core_user\output\myprofile\renderer {
    const PLUGIN = NED::CTRL;

    /**
     * Render the whole tree.
     *
     * @param \core_user\output\myprofile\tree $tree
     *
     * @return string
     */
    public function render_tree(\core_user\output\myprofile\tree $tree) {
        $this->change_profile($tree);
        $odd = '';
        $even = '';
        $categories = $this->filter($tree->categories);
        $categories = $this->change_order($categories);
        if (empty($categories)){
            return '';
        }

        $i = 1;
        foreach ($categories as $category) {
            $category->order = $i;
            $res = $this->render($category);
            if ($i++ % 2){
                $odd .= $res;
            } else {
                $even .=  $res;
            }
        }
        $odd = \html_writer::div($odd,'profile_tree_column sections_odd');
        $even = \html_writer::div($even,'profile_tree_column sections_even');
        $return = \html_writer::div($odd . $even,'profile_tree');
        return $return;
    }
    /**
     * Render a category.
     *
     * @param \core_user\output\myprofile\category $category
     *
     * @return string
     */
    public function render_category(\core_user\output\myprofile\category $category) {
        $classes = $category->classes;
        $classes = empty($classes) ? '' : $category->classes;
        $classes .= 'class="node_category ' . str_replace(' ', '-', strtolower($category->name)) . ' ';
        if (isset($category->order)){
            $classes.= ' ' . 'category_' . $category->order;
        }
        $o = parent::render_category($category);
        $o = str_replace('class="node_category', $classes, $o);
        return $o;
    }

    /**
     * Render a node.
     *
     * @param \core_user\output\myprofile\node $node
     *
     * @return string
     */
    public function render_node(\core_user\output\myprofile\node $node){
        if (isset($node->replaced_content) || isset($node->replaced_title)){
            $content = $node->replaced_content ?? $node->content;
            $title = $node->replaced_title ?? $node->title;
            $node2 = new \core_user\output\myprofile\node($node->parentcat, $node->name, $title, $node->after, $node->url,
                $content, $node->icon, $node->classes);
            return parent::render_node($node2);
        }
        return parent::render_node($node);
    }

    // New functions

    /**
     * Filter $categories by theme options and empty nodes
     * @param array $categories
     *
     * @return array
     */
    public function filter(array $categories) {
        global $USER;

        $userid = optional_param('id', 0, PARAM_INT);

        $show_categories = [];
        $hiddenprofilecategories = explode(',', get_config(static::PLUGIN, 'hiddenprofilecategories'));

        if ($courseid = optional_param('course', 0, PARAM_INT)) {
            $context = \context_course::instance($courseid);
        } else {
            $context = \context_system::instance();
        }

        $viewfulluserprofile = (empty($userid)) || ($userid == $USER->id) ||
            has_capability('local/ned_controller:viewfulluserprofile', $context);

        if (!$viewfulluserprofile) {
            $hiddenprofilecategories[] = 'coursedetails';
            $hiddenprofilecategories[] = 'miscellaneous';
            $hiddenprofilecategories[] = 'loginactivity';
            $hiddenprofilecategories[] = 'administration';
        }

        if(empty($hiddenprofilecategories)){
            return $categories;
        }

        foreach ($categories as $category) {
            $nodes = $category->nodes;
            if (in_array($category->name, $hiddenprofilecategories) || empty($nodes)){
                continue;
            }
            $show_categories[] = $category;
        }

        return $show_categories;
    }

    /**
     * Sort $categories by additional keys "force_order" and "order"
     * @param array $categories
     *
     * @return array
     */
    public function change_order(array $categories){
        if (empty($categories)){
            return [];
        }

        $force_order = [];
        $order_list = [];
        $other_categories = [];
        $categories2 = [];

        foreach ($categories as $category) {
            $without_order = true;
            if (isset($category->force_order) && $category->force_order > 0){
                if (!isset($force_order[$category->force_order])){
                    $force_order[$category->force_order] = $category;
                    $category->order = 0;
                } elseif(!isset($category->order) || $category->order < 1){
                    $category->order = $category->force_order;
                }
                $without_order = false;
            }

            if (isset($category->order) && $category->order > 0){
                if (!isset($order_list[$category->order])){
                    $order_list[$category->order] = [];
                }
                $order_list[$category->order][] = $category;
                $without_order = false;
            }

            if ($without_order){
                $other_categories[] = $category;
            }
        }

        $i = 1;
        ksort($order_list);
        reset($order_list);
        reset($other_categories);
        $order_keys = array_merge(array_keys($force_order), array_keys($order_list));

        if (empty($order_keys)){
            $limit = 0;
        } elseif (count($order_keys) == 1){
            $limit = $order_keys[0] + 1;
        } else {
            $limit = max(...$order_keys) + 1;
        }

        $order = empty($order_list) ? $limit+1 : key($order_list);
        $other_categories_not_end = !empty($other_categories);

        while($i < $limit || $order < $limit){
            $cat = null;
            if (isset($force_order[$i])){
                $cat = $force_order[$i];
            } elseif ($order <= $i && isset($order_list[$order])){
                $cat = current($order_list[$order]);
                if(next($order_list[$order]) === false){
                    if (next($order_list) === false){
                        $order = $limit + 1;
                    } else {
                        $order = key($order_list);
                        reset($order_list[$order]);
                    }
                }
            } elseif($other_categories_not_end) {
                $cat = current($other_categories);
                $other_categories_not_end = (bool)next($other_categories);
            }

            if ($cat){
                $categories2[] = $cat;
            }
            $i++;
        }

        if ($other_categories_not_end){
            $categories2 = array_merge($categories2, array_slice($other_categories, key($other_categories)));
        }

        return $categories2;
    }

    /**
     * @param \core_user\output\myprofile\tree $tree
     *
     */
    public function change_profile(\core_user\output\myprofile\tree $tree){
        /** @var []\core_user\output\myprofile\node $nodes */
        $nodes = $tree->nodes;
        if (isset($nodes['localbadges'])){
            $nodes['localbadges']->replaced_content = $this->change_badge_node();
            $nodes['localbadges']->replaced_title = '';
        }
    }

    /**
     *
     * @return string
     */
    public function change_badge_node(){
        $userid = optional_param('id', null, PARAM_INT);
        /** @var \local_ned_controller\output\core\badges_renderer $output */
        $output = $this->page->get_renderer('core', 'badges');
        $res = $output->print_badges_list(null, $userid, true);

        return $res;
    }

    /**
     * Profile context menu
     *
     * @param numeric                      $userid   User ID
     * @param numeric                      $courseid Course ID
     * @param numeric|object|\cm_info|null $cm_or_id Course Module object or ID
     *
     * @return string
     */
    public static function get_profile_context_menu($userid, $courseid=null, $cm_or_id=null) {
        global $DB;
        $menuhtml = '';
        if (empty($userid)) return $menuhtml;

        $has_course = !empty($courseid) && $courseid != SITEID;
        $issiteadmin = is_siteadmin();
        $context = NED::ctx($courseid);

        if (NED::is_tt_exists()){
            $canseetracker = $issiteadmin || \block_ned_teacher_tools\grading_tracker::has_any_capabilities($context);
        } else {
            $canseetracker = false;
        }

        if ($has_course){
            $is_watcher_teacher = has_capability('moodle/grade:viewall', $context, null);
            $is_user_student = has_capability('mod/assignment:submit', $context, $userid, false);
        } else {
            $is_user_student = NED::role_is_user_default_student($userid);

            if ($issiteadmin){
                $is_watcher_teacher = true;
            } else {
                $def_viewer_role = NED::role_get_user_def_role(null);
                $is_watcher_teacher = NED::str_has($def_viewer_role, 'teacher', 0, false);
            }
        }

        if (!$is_watcher_teacher || !$is_user_student) {
            return $menuhtml;
        }

        $user_groupids = NED::get_user_groupings($courseid, $userid);
        $groupid = !empty($user_groupids[0]) ? reset($user_groupids[0]) : 0;
        $menu = new \local_ned_controller\output\ned_action_menu();
        $triggerextraclasses = [];
        $triggerextraclasses[] = 'dimmed';
        $menu->attributessecondary['class'] .= ' ned-user-profile-context-menu user-profile-context-menu';
        $action_attributes = ['class' => 'context-menu-link', 'target' => '_blank'];

        $menu_add = function($url_page, $url_params=[], $text='', $add_class=[]) use (&$menu, &$action_attributes){
            $url = new \moodle_url($url_page, $url_params);
            if (!empty($add_class)){
                $action_attributes['class'] = ($action_attributes['class'] ?? '').' '.NED::arr2str($add_class);
            }
            $menu->add_secondary_action(new \action_link($url, NED::fa('custom').$text, null, $action_attributes));
        };

        if ($has_course){
            // KICA Gradebook.
            if (NED::is_kica_exists()){
                $url_params = [
                    'courseid' => $courseid,
                    'setuser' => $userid,
                    'group' => $groupid,
                ];
                $menu_add(NED::PLUGIN_DIRS[NED::KICA].'/grade_user.php', $url_params, NED::str('kicagradebook'));
            }

            if (NED::is_tt_exists()){
                // Class Progress.
                $url_params = [
                    'courseid' => $courseid,
                    'prshow' => 'allstatustypes',
                    'setuser' => $userid,
                ];
                $menu_add(NED::PLUGIN_DIRS[NED::TT].'/progress_report.php', $url_params, NED::str('simplegradebook'));

                $base_url_params = [
                    'courseid' => $courseid,
                    'setuser' => $userid,
                    'group' => $groupid,
                ];

                // Student Progress.
                $menu_add(NED::PLUGIN_DIRS[NED::TT].'/student_progress.php', $base_url_params, NED::str('studentprogress'));

                // Marking Manager.
                $menu_add(NED::PLUGIN_DIRS[NED::TT].'/marking_manager.php', $base_url_params, NED::str('markingmanager'));

                // Deadline Manager.
                $menu_add(NED::PLUGIN_DIRS[NED::TT].'/deadline_manager.php', $base_url_params, NED::str('deadlinemanager'));
            }

            // Grade Override.
            if ($issiteadmin) {
                $url_params = [
                    'id' => $courseid,
                    'item' => 'user',
                    'group' => $groupid,
                    'itemid' => $userid
                ];
                $menu_add('/grade/report/singleview/index.php', $url_params, NED::str('overrideactivity_title'));
            }

            // Today’s Logs
            $url_params = [
                'id' => $userid,
                'course' => $courseid,
                'mode' => 'today'
            ];
            if ($has_course){
                $url_params['course'] = $courseid;
            }
            $menu_add('/report/log/user.php', $url_params, get_string('todaylogs'));

            // All logs.
            $url_params = [
                'id' => $userid,
                'course' => $courseid,
                'mode' => 'all'
            ];
            $menu_add('/report/log/user.php', $url_params, get_string('alllogs'));
        }

        if ($canseetracker){
            // Grading Monitor.
            /** @var \block_ned_teacher_tools\output\grading_tracker_render $GTR */
            $GTR = '\block_ned_teacher_tools\output\grading_tracker_render';
            $url_params = [
                NED::PAR_USER => $userid,
                $GTR::PAR_VIEW => $GTR::VIEW_EXPANDED,
                NED::PAR_STATUS => $GTR::STATUS_ALL,
            ];
            if ($has_course){
                $url_params[NED::PAR_COURSE] = $courseid;
            }
            $menu_add($GTR::get_url($url_params), [], NED::str('gradingtracker'));
        }

        $NGC = NED::$ned_grade_controller;
        if ($NGC::has_any_see_capability($context, $issiteadmin)){
            // Grade Controller
            $url_params = [
                NED::PAR_USER => $userid,
                NED::PAR_GROUP => $groupid,
            ];
            if ($has_course){
                $url_params[NED::PAR_COURSE] = $courseid;
                $url_params[$NGC::$NGC_RENDER::PAR_COURSE_VIEW] = true;
            }
            $menu_add($NGC::$NGC_RENDER::get_url($url_params), [], NED::str('gradecontroller'));
        }

        // Grading Page.
        if (!empty($cm_or_id) && $cm = NED::get_cm_by_cmorid($cm_or_id, $courseid)){
            if ($cm->modname == 'quiz' && class_exists('\\local_kicaquizgrading\\helper')){
                if ($url = \local_kicaquizgrading\helper::get_kica_grader_link_from_cm($cm, $userid)){
                    $url_params = [
                        NED::PAR_USER => $userid,
                        NED::PAR_GROUP => $groupid,
                    ];
                    $menu_add($url, $url_params, NED::str('gradingpage'));
                }
            } else {
                $url_params = [
                    'id' => $cm->id,
                    'userid' => $userid,
                    'action' => 'grader',
                    'group' => 0
                ];
                $menu_add("/mod/$cm->modname/view.php", $url_params, NED::str('gradingpage'));
            }
        }

        // AI page
        if (NED::is_ai_exists()){
            if (\local_academic_integrity\shared_lib::can_view_user_ai($userid)){
                $url_params = [
                    NED::PAR_USER => $userid,
                    NED::PAR_GROUP => $groupid,
                ];
                if ($has_course){
                    $url_params[NED::PAR_COURSE] = $courseid;
                }
                $menu_add(NED::PAGE_AI_INFRACTIONS, $url_params, NED::str('ai_records'));
            }
        }

        // Deadline extensions.
        $contextsystem = \context_system::instance();
        if (class_exists('\report_ghs\helper') &&
            (has_capability('report/ghs:viewactivityextensions', $contextsystem)
            || \report_ghs\helper::has_capability_in_any_course('report/ghs:viewactivityextensions'))) {

            $cls = '';
            if (!$DB->record_exists('block_ned_teacher_tools_exte', ['userid' => $userid])) {
                $cls = 'dimmed_text';
            }
            $url_params = [
                'filterstudent' => $userid
            ];
            $menu_add('/report/ghs/ghs_activity_extension.php', $url_params, NED::str('deadlineextentions', null, 'local_schoolmanager'), $cls);
        }

        // Admin links
        if ($issiteadmin){
            if (isset($GTR) && $has_course){
                $url_params = [
                    NED::PAR_USER => $userid,
                    NED::PAR_COURSE => $courseid,
                    NED::PAR_ACTION => $GTR::ACTION_USER_UPDATE,
                ];

                $menu_add($GTR::get_url($url_params), [], NED::str('gt_user_update'), 'ned-user-profile-admin-link');
            }
        }

        $menu->attributes['class'] = 'float-right ml-1';
        $menu->set_menu_trigger(' ', implode(' ', $triggerextraclasses));

        return NED::render($menu);
    }
}
