<?php
/**
 * @package    local_ned_controller
 * @subpackage custom_ned_menu
 * @category   NED
 * @copyright  2022 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller\output\custom_ned_menu;

use local_ned_controller\shared_lib as NED;

/**
 * Class custom_ned_menu
 *
 * Custom menu with links, submenus, icons and role restrictions
 * Used by the {@see \format_ned} and {@see \block_ned_custom_menu}
 *
 * You can get its description by {@see custom_ned_menu::get_menu_format_description()}
 * When you changes this menu logic, don't forget to change its description too
 *
 * @package local_ned_controller\output\custom_ned_menu
 *
 * @property-read int id
 * @property-read string label
 * @property-read string role
 * @property-read bool active
 * @property-read bool current
 *
 * @property-read int level - {@see custom_ned_menu::get_level()}
 * @property-read \moodle_url|null url - {@see custom_ned_menu::get_url()}
 * @property-read string icon - {@see custom_ned_menu::get_icon()}
 * @property-read string href - {@see custom_ned_menu::get_href()}
 * @property-read string color_icon - {@see custom_ned_menu::get_color_icon()}
 * @property-read string color_bg - {@see custom_ned_menu::get_color_bg()}
 */
class custom_ned_menu implements \renderable, \templatable
{
    const LEVEL_MAIN = -2;
    const LEVEL_TITLE = self::LEVEL_MAIN+1; // -1
    const LEVEL_ITEM = self::LEVEL_TITLE+1; // 0
    const LEVEL_SUBITEM = self::LEVEL_ITEM+1; // 1, it multiplies on count of parent items, so all subitems have level > 0

    const ICON_DEFAULT_TITLE = 'fa-circle';
    const ICON_DEFAULT_ITEM = 'fa-square';

    protected const READ_FIELDS = [
        'id' => '_id',
        'label' => '_label',
        'role' => '_role',
        'current' => '_current',
        'active' => '_active',

        'level' => 'get_level',
        'url' => 'get_url',
        'href' => 'get_href',
        'icon' => 'get_icon',
        'color_icon' => 'get_color_icon',
        'color_bg' => 'get_color_bg',
    ];

    static protected $_ids = 0;

    protected $_id = 0;
    /** @var static */
    protected $_parent = null;
    /** @var settings_parser - only for top-parent element */
    protected $_stg_parser = null;
    /** @var bool does the user has capabilities to view this element */
    protected $_can_view = null;
    /** @var bool true if this element is current. Current element highlight yellow color */
    protected $_current = false;
    /** @var bool true if this element is current or this element contain current element */
    protected $_active = false;

    protected $_label = '';
    protected $_level = 0;

    /** @var string */
    protected $_raw_url = '';
    /** @var \moodle_url */
    protected $_url;

    protected $_role = '';
    protected $_icon = '';
    protected $_color_icon = '';
    protected $_color_bg = '';
    /** @var static[] */
    protected $_submenu = [];

    /**
     * Usually, you do not need call it directly,
     *  you can use {@see custom_ned_menu::new_menu()} (or alias {@see NED::new_custom_menu_item()}) instead of it
     *
     * @constructor custom_ned_menu
     *
     * @param static $parent - optional, parent element
     * @param string $init_input - optional, if provided - init element from this input
     */
    public function __construct($parent=null, $init_input=''){
        $this->_id = static::$_ids++;
        $this->_parent = $parent ?: null;
        $this->_level = static::LEVEL_MAIN;
        if (isset($init_input[0])){
            $this->init($init_input);
        }
    }

    /**
     * Function to export the renderer data in a format that is suitable for a
     * mustache template. This means:
     * 1. No complex types - only stdClass, array, int, string, float, bool
     * 2. Any additional info that is required for the template is pre-calculated (e.g. capability checks).
     *
     * @param \renderer_base $output Used to do a final render of any components that need to be rendered for export.
     *
     * @return object|array
     */
    public function export_for_template(\renderer_base $output){
        $this->init_current_active_item();
        return $this->get_header_menu_item();
    }

    //region Magic
    /**
     * @param $name
     *
     * @return mixed|null
     */
    public function __get($name){
        $real_name = static::READ_FIELDS[$name] ?? null;
        if ($real_name){
            if ($real_name[0] === '_'){
                return $this->$real_name ?? null;
            } else {
                return $this->$real_name();
            }
        }

        return null;
    }

    /**
     * Basic method doesn't check __get() method, so we do it
     *
     * @param $name
     *
     * @return bool
     */
    public function __isset($name){
        $val = $this->$name;
        return isset($val);
    }

    public function __destruct(){
        $this->clear_submenu();
    }
    //endregion

    //region Inits
    /**
     * Init element
     *
     * @param string $input - input to parse item
     * @param static $set_parent - optional, change parent element
     *
     * @return bool - true, if init was finished, false if it was aborted
     */
    public function init($input='', $set_parent=null){
        if ($set_parent){
            $this->_parent = $set_parent;
        }

        if (!isset($input[0])) return false;

        $is_title = static::_is_title($input);
        if ($is_title){
            $input = static::_format_title_input($input);
            if (!isset($input[0])) return false;
        }

        // Parse all params
        [$params, $url, $role] = array_pad(explode(settings_parser::SYMBOL_SECTION_DELIMITER, $input), 3, '');
        [$label, $icon, $color_icon, $color_bg] = array_pad(explode(settings_parser::SYMBOL_GROUP_DELIMITER, $params), 4, '');

        $label = trim($label);
        $icon = trim($icon);
        if (static::_is_empty($label, $icon)){
            return false;
        }

        // Count level, parse label
        if (empty($this->_parent)){
            $this->_level = static::LEVEL_MAIN;
            if ($this->_stg_parser){
                $this->_stg_parser->init();
            }
        } else {
            if ($is_title){
                $level = static::LEVEL_TITLE;
                $parent = $this->_parent->get_top_parent();
            } else {
                // it's level from input, but it can be wrong
                $level = static::_get_option_level($label);
                $label = static::_format_subitem_label($label, $level);
                $parent = $this->_parent->find_parent_for_level($level);
            }
            $this->set_parent($parent, $level);
        }
        $this->_label = clean_text($label);

        $this->_raw_url = trim($url);
        $this->_role = trim($role);
        $this->_icon = $icon;
        $this->_color_icon = trim($color_icon);
        $this->_color_bg = trim($color_bg);

        return true;
    }

    /**
     * Init current active item (item with the same url as current page)
     * It also inits active subtree from parent to active child
     * @see custom_ned_menu::get_current_item() if you need the item itself
     *
     * @param bool $reset - if true, reset current item if it have been found before
     *
     * @return void
     */
    public function init_current_active_item($reset=false){
        if ($reset){
            $this->reset_current_item();
        }
        $this->get_current_item();
    }

    /**
     * Filled submenu with options from parsed $input
     *
     * @param string $input - string with options for submenu to parse
     */
    public function parse_submenu($input=''){
        if (!isset($input[0])) return;

        $options = array_map('trim', explode("\n", $input));
        if (empty($options)) return;

        $prev_item = $this;
        foreach ($options as $input_option){
            if (!isset($input_option[0]) || $input_option[0] == settings_parser::SYMBOL_COMMENT) continue;

            $mi = new static($prev_item, $input_option);
            if ($mi->is_empty()) continue;

            $prev_item = $mi;
        }
    }
    //endregion
    //region Tree/submenu
    /**
     * Clear item submenu (remove all elements in it)
     */
    public function clear_submenu(){
        $ids = array_keys($this->_submenu);
        foreach ($ids as $id){
            $this->rem_child($id, true);
        }
        $this->_submenu = [];
    }

    /**
     * Set parent of the item, change item level and process adding child for the parent
     *
     * @param static    $parent - new parent
     * @param int|null  $level - option level, but my be corrected by parent level, if null - uses level from this
     */
    public function set_parent($parent, $level=null){
        if ($this->_parent){
            $this->_parent->rem_child($this->_id);
        }
        $this->_parent = $parent;
        $this->_level = max($parent->get_level() + 1, $level ?? $this->_level);
        $parent->add_child($this);
    }

    /**
     * Clear item parent
     */
    public function rem_parent(){
        if ($this->_parent){
            $this->_parent->rem_child($this->_id);
        }
        $this->_parent = null;
    }

    /**
     * Save child item into the submenu, change its parent if necessary
     *
     * @param static $child
     */
    public function add_child($child){
        if ($child->get_parent()->id !== $this->id) $child->set_parent($this);
        else $this->_submenu[$child->id] = $child;
    }

    /**
     * Remove child element from the item by id
     *
     * @param int $id
     * @param bool $clear_submenu - if true, also cleared child submenu
     */
    public function rem_child($id, $clear_submenu=false){
        $child = $this->_submenu[$id] ?? null;
        unset($this->_submenu[$id]);
        if ($child){
            if ($clear_submenu){
                $child->clear_submenu();
            }
            $child->rem_parent();
        }

        unset($child);
    }

    /**
     * Check item level from current - to top-parent, return the fist suitable element with lower level
     *
     * @param int $level - level of the child, which need parent
     *
     * @return static
     */
    public function find_parent_for_level($level=0){
        if (empty($this->_parent)) return $this;
        if ($this->get_level() < $level) return $this;

        return $this->_parent->find_parent_for_level($level);
    }

    /**
     * Return first direct active child (or null if nothing find)
     *
     * @param bool $only_visible
     *
     * @return static|null
     */
    public function find_active_child($only_visible=true){
        foreach ($this->children($only_visible) as $child){
            if ($child->active) return $child;
        }

        return null;
    }
    //endregion

    //region Is_/Has_
    /**
     * @return bool
     */
    public function has_submenu(){
        return !empty($this->_submenu);
    }

    /**
     * Return true, if item has nothing
     *
     * @return bool
     */
    public function is_empty(){
        return !$this->has_submenu() && static::_is_empty($this->_label, $this->_icon);
    }

    /**
     * Check, can user see this item
     *
     * @return bool
     */
    public function is_can_view(){
        if (is_null($this->_can_view)){
            if ($this->is_top_parent()) return true;
            if ($this->is_empty()) return false;
            if (!$this->get_parent()->is_can_view()) return false;

            $this->_can_view = (bool)$this->get_stg_parser()->check_unparsed_roles($this->_role);
        }

        return $this->_can_view;
    }

    /**
     * Return true, if it's top-parent item
     * For circle or recursive moments it will be better to use raw checking by empty
     *
     * @return bool
     */
    public function is_top_parent(){
        return empty($this->_parent);
    }

    /**
     * Return true, if item should have default icon-bullet
     *
     * @return bool
     */
    public function has_icon_bullet(){
        return $this->level <= static::LEVEL_ITEM || $this->has_submenu();
    }

    /**
     * @return bool
     */
    public function is_use_units(){
        return $this->get_stg_parser()->is_use_units();
    }
    //endregion
    //region Getters
    /**
     * Return current (parsed & corrected) item level
     * @return int
     */
    public function get_level(){
        if ($this->is_top_parent()) return static::LEVEL_MAIN;
        return $this->_level;
    }

    /**
     * Return parent of the item, or itself if it's top-parent element
     *
     * @return static
     */
    public function get_parent(){
        if (empty($this->_parent)) return $this;

        return $this->_parent;
    }

    /**
     * Return top-parent
     *
     * @return static
     */
    public function get_top_parent(){
        if (empty($this->_parent)) return $this;

        return $this->_parent->get_top_parent();
    }

    /**
     * @return \moodle_url|null
     */
    public function get_url(){
        if (is_null($this->_url)){
            if ($this->is_top_parent()) return null;
            if (!$this->is_can_view()) return null;

            $this->_url = $this->get_stg_parser()->parse_url($this->_raw_url, $this->_label);
        }

        return $this->_url ?: null;
    }

    /**
     * @return string
     */
    public function get_icon(){
        if (empty($this->_icon)){
            if ($this->has_icon_bullet()){
                return static::ICON_DEFAULT_TITLE;
            } else {
                return static::ICON_DEFAULT_ITEM;
            }
        }
        return $this->_icon ?: '';
    }

    /**
     * @return string
     */
    public function get_href(){
        $url = $this->get_url();
        return $url ? $url->out(false) : '';
    }

    /**
     * @return string
     */
    public function get_color_icon(){
        return static::_format_color($this->_color_icon);
    }

    /**
     * @return string
     */
    public function get_color_bg(){
        return static::_format_color($this->_color_bg);
    }

    /**
     * Return format html title of current item
     *
     * @return string
     */
    public function get_format_title(){
        $txt = format_string($this->_label);
        if (!isset($txt[0])){
            if (isset($this->_icon[0])) $txt = NED::fa($this->_icon);
        }
        if (!isset($txt[0])) return '';

        $title_attr = [];
        $span_attr = [];
        if ($this->_color_icon){
            $span_attr['style'] = 'color: '.$this->get_color_icon().';';
        }
        $span = NED::span($txt, '', $span_attr);

        if ($this->_color_bg){
            $title_attr['style']  = 'background-color: '.$this->get_color_bg().';';
        }

        return NED::div($span, 'format-title', $title_attr);
    }

    /**
     * Return block html attributes for menu
     * @return array
     */
    public function get_block_html_attributes(){
        return static::get_standard_block_html_attributes();
    }
    //endregion
    //region Setters
    /**
     * Change "active" state for current item and all its parents
     *
     * @param bool $val
     */
    public function set_active($val=true){
        $this->_active = $val;
        if ($this->is_top_parent()){
            $this->reset_header_menu_item();
        } else {
            $this->get_parent()->set_active($val);
        }
    }

    /**
     * Change "current" state for current item
     *
     * @param bool $val
     */
    public function set_current($val=true){
        $this->_current = $val;
        $this->set_active($val);
    }
    //endregion

    //region Settings parser
    /**
     * Set new settings parser, if it's top-parent item
     *
     * @param settings_parser $settings_parser
     */
    public function set_stg_parser($settings_parser){
        if (!$this->is_top_parent()){
            debugging('Only top parent can have settings!');
            return;
        }

        $this->_stg_parser = $settings_parser;
    }

    /**
     * Get top-parent settings_parser
     *
     * @return settings_parser
     */
    public function get_stg_parser(){
        if ($this->is_top_parent()){
            return $this->_stg_parser;
        }

        return $this->get_top_parent()->get_stg_parser();
    }

    /**
     * Get current active menu item (item with the same url as current page)
     *
     * @return custom_ned_menu|null
     */
    public function get_current_item(){
        return $this->get_stg_parser()->get_current_menu_item($this);
    }

    /**
     * Reset current item, so you will need to search the new one
     */
    public function reset_current_item(){
        $this->get_stg_parser()->reset_current_menu_item();
    }

    /**
     * Get current active menu item
     *
     * @return custom_ned_menu
     */
    public function get_header_menu_item(){
        return $this->get_stg_parser()->get_header_menu_item($this);
    }

    /**
     * Reset header item, so you will need to search the new one
     */
    public function reset_header_menu_item(){
        $this->get_stg_parser()->reset_header_menu_item();
    }

    /**
     * Get custom_ned_menu_header object
     *
     * @return custom_ned_menu_header
     */
    public function get_menu_course_header(){
        return $this->get_stg_parser()->get_menu_course_header($this);
    }
    //endregion
    //region Iterators
    /**
     * Base iterate method to realize other methods
     *
     * @param bool $only_visible - if true, return item only if it can be viewed
     * @param bool $it_this - if true, iterate $this first
     * @param bool $it_children - if true, call iterate method on children
     * @param bool $it_this_next - passed as $it_this to children iterate, if $it_children is true
     * @param bool $it_children_next - passed as $it_children to children iterate, if $it_children is true
     *
     * @return \Generator|static[]
     */
    protected function _iterate($only_visible=true, $it_this=false, $it_children=false, $it_this_next=false, $it_children_next=false){
        if ($only_visible && !$this->is_can_view()) return;

        if ($it_this){
            yield $this->_id => $this;
        }

        if ($it_children && $this->has_submenu()){
            foreach ($this->_submenu as $menu_item){
                yield from $menu_item->_iterate($only_visible, $it_this_next, $it_children_next, $it_this_next, $it_children_next);
            }
        }
    }

    /**
     * Iterate all: first $this, after children, after tha there children etc.
     *
     * @param bool $only_visible
     *
     * @return \Generator|static[]
     */
    public function iterate($only_visible=true){
        yield from $this->_iterate($only_visible, true, true, true, true);
    }

    /**
     * Iterate only direct children
     *
     * @param bool $only_visible
     *
     * @return \Generator|static[]
     */
    public function children($only_visible=true){
        yield from $this->_iterate($only_visible, false, true, true, false);
    }

    /**
     * Iterate all children recursively
     *
     * @param bool $only_visible
     *
     * @return \Generator|static[]
     */
    public function all_children($only_visible=true){
        yield from $this->_iterate($only_visible, false, true, true, true);
    }
    //endregion

    //region Static Protected
    /**
     * Return true, if there are empty label and icon (so item will be invisible)
     *
     * @param string $label
     * @param string $icon
     *
     * @return bool
     */
    static protected function _is_empty($label='', $icon=''){
        return !(isset($label[0]) || isset($icon[0]));
    }

    /**
     * Return, if string is title (eg "[some title]")
     *
     * @param $str - raw input for menu
     *
     * @return bool
     */
    static protected function _is_title($str){
        return (isset($str[1]) && $str[0] == settings_parser::SYMBOL_TITLE_START && $str[-1] == settings_parser::SYMBOL_TITLE_END);
    }

    /**
     * Remove [] from title ("[some title]" -> "some title")
     *
     * @param $str - raw input for menu
     *
     * @return string - updated input
     */
    static protected function _format_title_input($str){
        return trim(substr($str, 1, -1));
    }

    /**
     * Count '-' at the beginning of the string $option and return it
     *
     * @param string $option
     *
     * @return int option level
     */
    static protected function _get_option_level($option){
        /** @noinspection PhpStatementHasEmptyBodyInspection */
        for ($i=0; isset($option[$i]) && $option[$i] == settings_parser::SYMBOL_SUBITEM; $i++);

        return $i ? ($i * static::LEVEL_SUBITEM) : static::LEVEL_ITEM;
    }

    /**
     * Parse subitem labels, by removing level symbols
     *
     * @param string $label - original label
     * @param int    $level - item level
     *
     * @return string
     */
    static protected function _format_subitem_label($label, $level=0){
        if ($level > 0){
            $label = trim(substr($label, $level));
        }
        return $label;
    }

    /**
     * Fixed color for CSS if needed
     *
     * @param string $color
     *
     * @return string
     */
    static protected function _format_color($color){
        if (!isset($color[0])) return '';

        if (ctype_xdigit($color)){
            return '#'.$color;
        }
        return clean_text($color);
    }
    //endregion
    //region Static Public
    /**
     * Return standard html attributes for menu block
     * @return array
     */
    static public function get_standard_block_html_attributes(){
        return ['class' => 'block local_ned_controller-custom_ned_menu'];
    }

    /**
     * Get new top-parent custom_menu_
     *
     * @param string         $raw_settings - initial raw text settings
     * @param bool           $use_units - set option of using units
     * @param numeric|object $course_or_id - course or its id (global by default)
     * @param numeric|object $user_or_id - user or its id (global by default)
     * @param bool   $init - if true (default), init all right now
     *
     * @return static - return new parent menu
     */
    static public function new_menu($raw_settings='', $use_units=false, $course_or_id=null, $user_or_id=null, $init=true){
        $m = new static();
        $raw_settings = trim($raw_settings);
        $stg_parser = new settings_parser($raw_settings, $use_units, $course_or_id, $user_or_id);
        $m->set_stg_parser($stg_parser);
        if ($init){
            $stg_parser->init();
            $m->parse_submenu($raw_settings);
        }

        return $m;
    }

    /**
     * Shows description of format settings
     *
     * @return string
     */
    static public function get_menu_format_description(){
        return NED::render_from_template('~/output/custom_ned_menu/custom_ned_menu_description');
    }
    //endregion
}
