<?php
/**
 * @package    local_ned_controller
 * @subpackage NED
 * @copyright  2022 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @noinspection PhpUnused
 * @noinspection PhpUnnecessaryStopStatementInspection
 */

namespace local_ned_controller\output;
use local_ned_controller\shared_lib as NED;

/**
 * Abstract class base_table_page_render
 * Use as class-template for page-table render classes
 *
 * Useful methods:
 *  Render class for page: {@see ned_base_table_page_render::render_full_page()}
 *
 *  Main render content method: {@see ned_base_table_page_render::_render_page_content()}
 *  Main get_data method: {@see ned_base_table_page_render::get_table_data()}
 *  Main get_data SQL method: {@see ned_base_table_page_render::get_db_table_data()}
 *
 *  Start init point: {@see ned_base_table_page_render::init()}
 *  Main check params: {@see ned_base_table_page_render::_check_params()}
 *  Export content for template: {@see ned_base_table_page_render::export_for_template()}
 *
 * @package local_ned_controller\output
 *
 * @property-read \core_renderer $o              - $OUTPUT
 * @property-read \context       $ctx
 * @property-read object         $viewer         - current $USER
 * @property-read int            $viewerid
 * @property-read bool           $is_admin
 * @property-read array          $params         - page url params
 *
 * @property-read array          $ids
 * @property-read array          $filter_ids
 * @property-read int            $id
 * @property-read int            $filter_id
 * @property-read int            $courseid
 * @property-read int            $cmid
 * @property-read int            $userid
 * @property-read int            $groupid
 * @property-read int            $schoolid
 * @property-read int            $graderid
 * @property-read int            $action
 *
 * @property-read bool           $course_view    if true, use course context and page type
 *
 * @property-read int            $page
 * @property-read int            $perpage
 * @property-read int            $total_count
 * @property-read bool           $header_printed - true, if class also print headers
 *
 * @property-read int            $cap_view
 * @property-read int            $cap_edit
 * @property-read bool           $params_init    - true, after loaded and checked params
 */
abstract class ned_base_table_page_render implements \renderable, \templatable {
    use \local_ned_controller\base_empty_class;

    //region Config to set up base class work
    /** @var string - current plugin for some relative things, as string translation or templates paths */
    protected const _PLUGIN = NED::CTRL;
    /** @var string - default type for params  */
    protected const _PARAM_TYPE_DEFAULT = PARAM_INT;
    /** @var mixed - default value for param */
    protected const _PARAM_VALUE_DEFAULT = 0;
    /** @var bool - does page use course view */
    protected const _USE_COURSE_VIEW = true;
    /** @var bool - what value we should use if we have PAR_COURSE, but not PAR_COURSE_VIEW */
    protected const _DEFAULT_COURSE_VIEW = true;
    /** @var bool - if true, and checked param not in PARAM_DATA, also check {@see \local_ned_controller\shared_lib::PARAM_DATA} */
    protected const _USE_NED_PARAM_DATA = true;
    /** @var bool - if true, check some base params with default check {@see _check_base_params()} */
    protected const _USE_BASE_PARAM_CHECK = true;
    /** @var bool if true, add course name to navbar by default */
    protected const _NAVBAR_ADD_COURSE_NAME = true;
    /** @var bool if true, add plugin name to navbar by default */
    protected const _NAVBAR_ADD_PLUGIN_NAME = true;
    /** @var string|null - if not null, add this value as plugin url in navbar */
    protected const _NAVBAR_PLUGIN_URL = null;
    protected const _USE_NED_SCHOOL_YEAR_FILTER = false;
    //endregion

    //region SQL data
    /**
     * You can set some sql_* consts to empty - it will turn off some functionality with this consts
     */
    /** @var string @abstract */
    protected const _SQL_TABLE = 'table';
    /** @var string @abstract */
    protected const _SQL_ALIAS = 't';

    protected const _SQL_SCHOOL_ALIAS = 'sch';
    protected const _SQL_GROUP_ALIAS = 'grp';

    protected const _SQL_COURSEID = 'courseid';
    protected const _SQL_CMID = 'cmid';
    protected const _SQL_USERID = 'userid';
    protected const _SQL_GRADERID = 'graderid';

    protected const _SQL_TIMECREATED = 'timecreated';
    protected const _SQL_TIMEMODIFIED = 'timemodified';
    //endregion

    //region School Year
    const SCHOOL_YEAR_CURRENT = 0;
    const SCHOOL_YEAR_PAST = 1;
    const SCHOOL_YEAR_ALL = 2;
    const SCHOOL_YEAR_OPTIONS = [
        self::SCHOOL_YEAR_CURRENT => 'ned_school_year_current',
        self::SCHOOL_YEAR_PAST => 'ned_school_year_past',
        self::SCHOOL_YEAR_ALL => 'ned_school_year_all',
    ];
    //endregion

    //region Params
    const PARAMS = [
        NED::PAR_ID, NED::PAR_IDS, NED::PAR_FILTER_ID, NED::PAR_FILTER_IDS,
        NED::PAR_COURSE,
        NED::PAR_COURSE_VIEW,
        NED::PAR_CM,
        NED::PAR_USER,
        NED::PAR_GRADER,
        NED::PAR_GROUP,
        NED::PAR_SCHOOL,
        NED::PAR_SCHOOL_YEAR,
        NED::PAR_ACTION,
        NED::PAR_PAGE,
        NED::PAR_PERPAGE,
    ];

    /**
     * By default, params uses _PARAM_TYPE_DEFAULT and _PARAM_VALUE_DEFAULT, but you can set defaults to others here
     * Possible keys for rewrite array:
     *  • type - change default type of param
     *  • default - change default value of param
     *  • property - load raw param value for $this as $this->{$property}
     * Also {@see \local_ned_controller\shared\C::PARAM_DATA}
     *
     * @var array[] - keys from PARAMS, value is array
     */
    const PARAM_DATA = [
        NED::PAR_GRADER => ['type' => PARAM_INT, 'default' => self::GRADER_ALL],
    ];
    //endregion

    //region Other Consts
    /** @var string @abstract */
    const URL = '';
    const GRADER_ALL = -1; // as there is grader with id 0 (None)
    const COURSE_NONE = -1;
    const PERPAGES = NED::PERPAGES;
    //endregion

    //region Properties
    // Protected
    /** @var \core_renderer $_o */
    protected $_o;
    /** @var \context_system|\context_course|\context */
    protected $_ctx;

    protected $_viewer;
    protected $_viewerid;
    protected $_is_admin;
    protected $_params = [];

    // values for default params
    protected $_ids = [];
    protected $_id = 0;
    protected $_filter_ids = [];
    protected $_filter_id = null;
    protected $_courseid = 0;
    protected $_course_view = false;
    protected $_cmid = 0;
    protected $_userid = 0;
    protected $_graderid = 0;
    protected $_groupid = 0;
    protected $_schoolid = 0;
    protected $_school_year = null;
    protected $_action = 0;
    protected $_page = 0;
    protected $_perpage = 0;

    protected $_total_count = 0;
    protected $_header_printed = false;
    protected $_params_init = false;

    protected $_cap_view = false;
    protected $_cap_edit = false;

    /**
     * @var bool|null capability to view all schools
     *  if empty, require $_limit_schoolids list for accessible schools
     */
    protected $_view_all_schools = null;
    /**
     * @var array - list of accessible schools
     *  if empty and $_view_all_schools is empty, can deny access to all users
     */
    protected $_limit_schoolids = [];

    /** @var array - store "static" function data, but for $this context */
    protected $_static_data = [];
    /** @var array - notifications to show when page will be rendered */
    protected $_notifications = [];

    // Public
    /** @var object content for the {@see export_for_template()}  */
    public $content;

    //endregion

    //region Init methods
    /**
     * @constructor
     */
    public function __construct(){
        global $USER;

        if (empty($USER->id)){
            require_login();
        }

        $this->_o = NED::output();
        $this->_viewer = $USER;
        $this->_viewerid = $USER->id;
        $this->_is_admin = is_siteadmin($this->_viewer);
        $this->_ctx = NED::ctx();
        $this->_cap_view = false;
        $this->_cap_edit = false;
        $this->_params_init = false;
        $this->_view_all_schools = $this->_view_all_schools ?? true;
        $this->content = new \stdClass();
    }

    /**
     * Init class, send here additional params if you need
     * Start chains of checks and rendering
     *
     * @param bool  $setup_page      - if true, setup page for view
     * @param array $params          - default page params
     * @param bool  $load_url_params - if true, also load param from url options
     */
    public function init($setup_page=true, $params=[], $load_url_params=true){
        $this->set_and_check_params(true, $params, $load_url_params);
        if ($setup_page){
            $this->setup_page();
        }
    }

    /**
     * Call load_params & check_params
     *
     * @param false $clear_params_first
     * @param array $params
     * @param bool  $load_page_params
     */
    public function set_and_check_params($clear_params_first=false, $params=[], $load_page_params=true){
        if ($clear_params_first){
            $this->_params = [];
        }

        $this->_load_params($params, $load_page_params);
        $this->_check_params();
        $this->_check_course_view();
    }

    /**
     * @return array
     */
    static public function get_all_params(){
        return static::PARAMS;
    }

    /**
     * Params to load for page
     *
     * @return array
     */
    protected static function _get_params_list_to_load(){
        return static::get_all_params();
    }

    /**
     * Load $params and url params if they exist in static::PARAMS
     *
     * @param array $params
     * @param bool  $load_url_params
     */
    protected function _load_params($params=[], $load_url_params=true){
        $params = NED::val2arr($params, true, true);
        $params2load = static::_get_params_list_to_load();

        foreach ($params2load as $param){
            if (isset($params[$param])){
                $this->_params[$param] = $params[$param];
            } else {
                $def = static::_param_default($param);
                $this->_params[$param] = $def;
                if ($load_url_params){
                    $type = static::_param_type($param);
                    $this->_params[$param] = optional_param($param, $def, $type);
                }
            }

            $property = static::_param_property($param);
            if ($property){
                $this->$property = $this->_params[$param];
            }
        }
    }

    /**
     * Check loaded params
     * Normally called by {@see set_and_check_params()} after param loading
     */
    protected function _check_params(){
        if (!$this->can_see()) return;

        if (static::_USE_BASE_PARAM_CHECK){
            $this->_check_base_params();
        }

        $this->_params_init = true;
    }

    /**
     * Check "course_view" for ability to use
     */
    protected function _check_course_view(){
        if (!$this->_course_view) return;

        $cant_see_course = !static::_USE_COURSE_VIEW || !$this->can_see() ||
            empty($this->_courseid) || $this->_courseid == static::COURSE_NONE ||
            !NED::course_can_view_course_info($this->_courseid, $this->_viewerid);
        if ($cant_see_course){
            $this->_course_view = false;
        }
    }

    /**
     * Base check of the base params
     * You may not call it at all, if you make all params check by yourself
     * Normally calling from the {@see _check_params()}
     */
    protected function _check_base_params(){
        $this->_course_view = false;
        if (static::_USE_COURSE_VIEW){
            if (isset($this->_params[NED::PAR_COURSE_VIEW])){
                $this->_course_view = (bool)$this->_params[NED::PAR_COURSE_VIEW];
            } else {
                $this->_course_view = static::_DEFAULT_COURSE_VIEW;
            }
        }

        if (array_key_exists(NED::PAR_COURSE, $this->_params)){
            $courseid = $this->_params[NED::PAR_COURSE] ?? NED::ALL;
            if ($courseid != static::COURSE_NONE && $courseid != NED::ALL){
                $course = NED::get_course($courseid);
                $courseid = $course->id ?? NED::ALL;
            }

            $options = $this->get_courses_options();
            $this->_courseid = NED::isset_key($options, $courseid, static::COURSE_NONE);
        }

        if ($this->_courseid != static::COURSE_NONE && $this->_courseid != NED::ALL){
            if (array_key_exists(NED::PAR_GROUP, $this->_params)){
                $options = $this->get_group_options();
                $this->_groupid = NED::isset_key($options, $this->_params[NED::PAR_GROUP], NED::ALL);
            }

            if (array_key_exists(NED::PAR_CM, $this->_params)){
                $options = $this->get_cm_options();
                $this->_cmid = NED::isset_key($options, $this->_params[NED::PAR_CM], NED::ALL);
            }
        } else {
            $this->_groupid = NED::ALL;
            $this->_cmid = NED::ALL;
        }

        if (array_key_exists(NED::PAR_SCHOOL, $this->_params)){
            $options = $this->get_schools_options();
            $this->_schoolid = NED::isset_key($options, $this->_params[NED::PAR_SCHOOL], NED::ALL);
        }

        if (array_key_exists(NED::PAR_USER, $this->_params)){
            $options = $this->get_users_options();
            $this->_userid = NED::isset_key($options, $this->_params[NED::PAR_USER], NED::ALL);
        }

        if (array_key_exists(NED::PAR_GRADER, $this->_params)){
            $options = $this->get_graders_options();
            $this->_graderid = NED::isset_key($options, $this->_params[NED::PAR_GRADER], static::GRADER_ALL);
        }

        if (array_key_exists(NED::PAR_PERPAGE, $this->_params)){
            $options = $this->get_perpage_options();
            $this->_params[NED::PAR_PERPAGE] = NED::isset_key($options, $this->_params[NED::PAR_PERPAGE]);
            $this->_perpage = $options[$this->_params[NED::PAR_PERPAGE]];
            if ($this->_perpage === NED::PERPAGE_ALL){
                $this->_perpage = 0;
                $this->_page = 0;
            } else {
                $this->_page = (int)abs($this->_params[NED::PAR_PAGE]);
            }
        }

        if (static::_USE_NED_SCHOOL_YEAR_FILTER){
            $this->_school_year = static::SCHOOL_YEAR_CURRENT;
            $options = $this->get_school_year_options();
            $this->_school_year = NED::isset_key($options, $this->_params[NED::PAR_SCHOOL_YEAR], $this->_school_year);
        } else {
            $this->_school_year = null;
        }

        $this->_action = $this->_params[NED::PAR_ACTION] ?? null;
    }
    //endregion

    //region Setters
    /**
     * @param int|string|array $ids
     */
    public function set_filter_ids($ids){
        $this->_filter_ids = NED::val2arr($ids);
    }
    //endregion

    //region Capabilities methods
    /**
     * Return true, if current user is able to see this page
     *
     * @return bool
     */
    public function can_see(){
        return $this->_cap_view;
    }

    /**
     * Return true, if current user is able to edit data on this page
     *
     * @return bool
     */
    public function can_edit(){
        return $this->_cap_edit;
    }
    //endregion

    //region Params methods
    /**
     * Check PARAM_DATA for default param properties
     *
     * @param string $param - param name
     * @param string $key - string property for search def value
     * @param mixed $def - return, if there is no data in the PARAM_DATA
     *
     * @return mixed
     */
    static protected function _param_get_def_param_data($param, $key, $def){
        if (isset(static::PARAM_DATA[$param]) && array_key_exists($key, static::PARAM_DATA[$param])){
            return static::PARAM_DATA[$param][$key];
        } elseif (static::_USE_NED_PARAM_DATA && isset(NED::PARAM_DATA[$param]) && array_key_exists($key, NED::PARAM_DATA[$param])){
            return NED::PARAM_DATA[$param][$key];
        }

        return $def;
    }

    /**
     * Get default param type
     *
     * @param string $param
     *
     * @return string
     */
    static protected function _param_type($param){
        return static::_param_get_def_param_data($param, 'type', static::_PARAM_TYPE_DEFAULT);
    }

    /**
     * Get default param value
     *
     * @param string $param
     *
     * @return mixed|null
     */
    static protected function _param_default($param){
        return static::_param_get_def_param_data($param, 'default', static::_PARAM_VALUE_DEFAULT);
    }

    /**
     * Return name of property or null
     *
     * @param string $param
     *
     * @return string|null
     */
    static protected function _param_property($param){
        return static::_param_get_def_param_data($param, 'property', null);
    }
    //endregion

    //region Setup Page methods
    /**
     * Call this method before header output, then you can redirect here
     */
    public function before_header_output(){
        return;
    }

    /**
     * Call this method after header output, you can add something between header and main page content
     */
    public function after_header_output(){
        return;
    }

    /**
     * Call this method before footer output
     */
    public function before_footer_output(){
        return;
    }

    /**
     * Setup PAGE data
     *
     * @param bool $set_header
     */
    public function setup_page($set_header=true){
        global $PAGE;

        require_login();

        $title = $this->get_page_title();
        if ($this->_course_view){
            $this->_ctx = NED::ctx($this->_courseid);
            $course = NED::get_course($this->_courseid);
            require_login($course);
            $PAGE->set_pagelayout('course');
        }

        $PAGE->set_context($this->_ctx);
        $this->setup_page_navbar();
        NED::page_set_title($title, $this->get_main_url());

        if ($set_header){
            $this->before_header_output();
            echo $this->_o->header();
            $this->_header_printed = true;
            $this->after_header_output();
        }
    }

    /**
     * @abstract
     * @return string
     */
    public function get_page_title(){
        return '';
    }

    /**
     * Add items to navbar during page setup
     */
    public function setup_page_navbar(){
        /**
         * You can set here additional breadcrumbs
         * Page title will be automatically added after it
         */
        NED::page()->navbar->ignore_active();
        if (static::_NAVBAR_ADD_COURSE_NAME && (!static::_USE_COURSE_VIEW || $this->_course_view) && $this->_courseid > NED::ALL){
            $course = NED::get_course($this->_courseid);
            NED::page()->navbar->add($course->shortname, new \moodle_url(NED::PAGE_COURSE, ['id' => $this->_courseid]));
        }
        if (static::_NAVBAR_ADD_PLUGIN_NAME){
            NED::page()->navbar->add(NED::str('pluginname', null, static::_PLUGIN), static::_NAVBAR_PLUGIN_URL);
        }
    }
    //endregion

    //region URL methods
    /**
     * Params for URL export
     *
     * @return array
     */
    protected static function _get_params_url(){
        return static::get_all_params();
    }

    /**
     * Return checked params for URL
     *
     * @param array $params
     * @param array $not_usual_params - url params, which are not from static::PARAMS
     *
     * @return array
     */
    static protected function _check_url_param($params=[], $not_usual_params=[]){
        $url_params = [];
        $export_params = static::_get_params_url();
        foreach ($export_params as $param){
            $def = static::_param_default($param);
            $val = $params[$param] ?? null;
            if (!is_null($val) && (is_null($def) || $val != $def || (bool)$val != (bool)$def)){
                $url_params[$param] = $val;
            }
        }

        return array_merge($url_params, $not_usual_params);
    }

    /**
     * Get params for URL
     *
     * @param array $add_params
     *
     * @return array
     */
    protected function _get_url_params($add_params=[]){
        return array_merge($this->_params, $add_params);
    }

    /**
     * Return page url
     *
     * @param array $params
     * @param array $not_usual_params - url params, which are not from static::PARAMS
     *
     * @return \moodle_url
     */
    static public function get_url($params=[], $not_usual_params=[]){
        return new \moodle_url(static::URL, static::_check_url_param($params, $not_usual_params));
    }

    /**
     * Return page url for $this context:
     *  if it has site context, it will be the same as {@see get_url()},
     *  otherwise it added current course id and course view
     *
     * @param array $params
     *
     * @return \moodle_url
     */
    public function get_main_url($params=[]){
        $url = null;
        if (empty($params)){
            $url = $this->_static_data[__FUNCTION__] ?? null;
        }

        if (is_null($url)){
            $base_params = [];
            if ($this->_course_view){
                $base_params[NED::PAR_COURSE] = $this->_courseid;
                $base_params[NED::PAR_COURSE_VIEW] = $this->_course_view;
            }
            $url = static::get_url(array_merge($base_params, $params));

            if (empty($params)){
                $this->_static_data[__FUNCTION__] = $url;
            }
        }

        return new \moodle_url($url);
    }

    /**
     * Return page url for $this
     *
     * @param array $params
     * @param array $not_usual_params - url params, which are not from static::PARAMS
     *
     * @return \moodle_url
     */
    public function get_my_url($params=[], $not_usual_params=[]){
        $url = null;
        if (empty($params)){
            $url = $this->_static_data[__FUNCTION__] ?? null;
        }

        if (is_null($url)){
            $url_params = $this->_get_url_params($params);
            $url = static::get_url($url_params);
            if (empty($params)){
                $this->_static_data[__FUNCTION__] = $url;
            }
        }

        return new \moodle_url($url, $not_usual_params);
    }
    //endregion

    //region SQL static utils
    /**
     * Return calling sql name with table alias
     *
     * @param $name
     *
     * @return string
     */
    static protected function _sql_a($name){
        return static::_SQL_ALIAS.'.'.$name;
    }

    /**
     * Generate SQL query for current TABLE
     *
     * @param string|array $select
     * @param string|array $joins
     * @param string|array $where
     * @param string|array $groupby
     * @param string|array $orderby
     *
     * @return string - $sql query
     */
    static protected function _sql_generate_sql($select=[], $joins=[], $where=[], $groupby=[], $orderby=[]){
        return NED::sql_generate($select, $joins, static::_SQL_TABLE, static::_SQL_ALIAS, $where, $groupby, $orderby);
    }

    /**
     * Add construct with some condition SQL fragment for $where & $params
     *
     * Warning: use only simple value. For arrays - {@see sql_add_get_in_or_equal_options()}
     * Also, function doesn't check param array for having such key already
     *
     * @param string $sql_field  - name of where condition
     * @param mixed  $val        - value for where condition
     * @param array  $where      - where list, if you already have it
     * @param array  $params     - params array, if you already have it
     * @param string $condition - one of the {@see C::SQL_CONDITIONS} values
     * @param string $param_name - (optional) param name for $params, if empty - uses field name
     * @param string $param_postfix - (optional) - if $param_name is empty, add it to the $sql_field to create $param_name
     *
     * @return void - result saves in the $where and $params
     */
    static protected function _sql_add_condition($sql_field, $val=null, &$where=[], &$params=[], $condition=NED::SQL_COND_EQ,
        $param_name='', $param_postfix=''){
        if (empty($sql_field)) return;

        NED::sql_add_condition(static::_sql_a($sql_field), $val, $where, $params, $condition, $param_name, $param_postfix);
    }

    /**
     * Add constructs '=' sql fragment for $where & $params,
     *  with transforming $sql_field with table alias
     *
     * Warning: use only simple value. For arrays - {@see _sql_add_simple_limit()}
     * Also, function doesn't check param array for having such key already
     *
     * @param string $sql_field  - name of where condition
     * @param mixed  $val        - value for where condition
     * @param array  $where      - where list, if you already have it
     * @param array  $params     - params array, if you already have it
     * @param string $param_name - (optional) param name for $params, if empty - uses field name
     * @param string $param_postfix - (optional) - if $param_name is empty, add it to the $sql_field to create $param_name
     *
     * @return void - result saves in the $where and $params
     */
    static protected function _sql_add_equal($sql_field, $val=null, &$where=[], &$params=[], $param_name='', $param_postfix=''){
        static::_sql_add_condition($sql_field, $val, $where, $params, NED::SQL_COND_EQ, $param_name, $param_postfix);
    }

    /**
     * Add simple sql limitation by table field and its value
     * Note: for empty arrays as value, it always returns NONE condition
     *
     * @param string      $sql_field     - name of where condition
     * @param mixed       $val           - value for where condition
     * @param array       $where         - where list, if you already have it
     * @param array       $params        - params array, if you already have it
     * @param bool        $pass_if_empty - if true, and value is empty, do not set limitation
     * @param string|null $prefix        - prefix for SQL parameters, null means default "param" value
     * @param bool        $equal         - true means we want to equate to the constructed expression, false means we don't want to equate to it.
     *
     * @return void - result saves in the $where and $params
     */
    static protected function _sql_add_simple_limit($sql_field, $val=null, &$where=[], &$params=[], $pass_if_empty=false, $prefix='param', $equal=true){
        if (empty($sql_field)) return;
        if (empty($val)){
            if (is_array($val) || is_object($val)){
                // for empty array return None result
                $where[] = NED::SQL_NONE_COND;
                return;
            } elseif ($pass_if_empty){
                // none limitation
                return;
            }
        }

        NED::sql_add_get_in_or_equal_options(static::_sql_a($sql_field), $val, $where, $params, $prefix, $equal);
    }

    /**
     * Add some simple sql filter by your items, save result in $where and $params
     * If you wish skip adding of some items, send NULL for it
     *
     * @param array        $where  - list to save new conditions
     * @param array        $params - list to save new params
     * @param numeric|null $courseid
     * @param numeric|null $cmid
     * @param numeric|null $userid
     * @param numeric|null $groupid
     * @param numeric|null $graderid
     * @param numeric|null $schoolid
     * @param array|null   $filter_ids
     * @param numeric|null $filter_id
     * @param numeric|null $school_year - {@see static::SCHOOL_YEAR_OPTIONS}
     */
    static protected function _sql_set_simple_filters(&$where=[], &$params=[],
        $courseid=null, $cmid=null, $userid=null, $groupid=null, $graderid=null, $schoolid=null, $filter_ids=null, $filter_id=null,
        $school_year=null){
        $school_t = static::_SQL_SCHOOL_ALIAS;
        if ($schoolid && !empty($school_t) && NED::is_school_manager_exists()){
            NED::sql_add_find_in_set("$school_t.school_ids", $schoolid, $where, $params);
        }

        $group_t = static::_SQL_GROUP_ALIAS;
        if ($groupid && !empty($group_t)){
            NED::sql_add_find_in_set("$group_t.group_ids", $groupid, $where, $params);
        }

        if (isset($graderid) && $graderid > static::GRADER_ALL){
            static::_sql_add_equal(static::_SQL_GRADERID, $graderid, $where, $params);
        }
        if (isset($courseid) && $courseid > NED::ALL){
            static::_sql_add_equal(static::_SQL_COURSEID, $courseid, $where, $params);
        }
        if ($cmid){
            static::_sql_add_equal(static::_SQL_CMID, $cmid, $where, $params);
        }
        if ($userid){
            static::_sql_add_equal(static::_SQL_USERID, $userid, $where, $params);
        }

        if (!empty($filter_ids)){
            static::_sql_add_simple_limit('id', $filter_ids, $where, $params, true, 'filter_ids_');
        }
        if ($filter_id){
            static::_sql_add_equal('id', $filter_id, $where, $params, 'filter_id');
        }

        if (isset($school_year)){
            $school_year_condition = null;
            switch ($school_year){
                default:
                case static::SCHOOL_YEAR_ALL:
                    break;
                case static::SCHOOL_YEAR_CURRENT:
                    $school_year_condition = NED::SQL_COND_GTE;
                    break;
                case static::SCHOOL_YEAR_PAST:
                    $school_year_condition = NED::SQL_COND_LT;
                    break;
            }
            if ($school_year_condition){
                $school_year_start = NED::$C::get_config('ned_school_year_start') ?: 0;
                static::_sql_add_condition(static::_SQL_TIMECREATED, $school_year_start, $where, $params, $school_year_condition);
            }
        }
    }
    //endregion

    //region SQL menu filter methods
    /**
     * Return menu-array for get_*_options() methods
     *
     * @param string          $sql_key           - SQL code for keys of the menu, don't use "DISTINCT" word
     * @param string          $sql_value         - SQL code for values of the menu
     * @param array|string    $joins             - additional tables to join
     * @param array|string    $where             - SQL conditions, don't use "WHERE" word
     * @param array           $params            - params for query
     * @param bool|string|int $add_all           - add option "All" to the menu, if it's string or int, uses this value as key
     * @param bool            $ignore_view_limit - if true, do not use different view limitation (so get all possible options)
     * @param bool            $sort              - sort menu result; if $add_all is used, it will be first anyway
     * @param array|string    $orderby           - SQL order by, not used when $sort is true
     *
     * @return array
     */
    protected function _sql_menu_filter($sql_key, $sql_value, $joins=[], $where=[], $params=[], $add_all=true,
        $ignore_view_limit=false, $sort=true, $orderby=[]){
        $records = [];
        if ($this->can_see()){
            [$where, $joins] = NED::val2arr_multi(true, $where, $joins);

            if (!$ignore_view_limit){
                $this->_sql_add_view_limitation($joins, $where, $params);
            }

            $select = "DISTINCT $sql_key, $sql_value";
            $sql = static::_sql_generate_sql($select, $joins, $where, [], $sort ? [] : $orderby);

            $records = NED::db()->get_records_sql_menu($sql, $params) ?: [];
        }

        if ($sort){
            asort($records, SORT_STRING);
        }

        if ($add_all || is_string($add_all) || is_int($add_all)){
            $key = (is_string($add_all) || is_int($add_all)) ? $add_all : NED::ALL;
            $records = [$key => get_string('all')] + $records;
        }

        return $records;
    }

    /**
     * Add limitation by course, changing $where and $params for sql query
     *
     * @param array              $joins
     * @param array              $where
     * @param array              $params
     * @param array|numeric|null $checked_ids   - if not null, rewrite current data to check
     * @param bool               $pass_if_empty - if true, and value is empty, do not set limitation
     *
     * @return void
     */
    protected function _sql_add_limit_course(&$joins=[], &$where=[], &$params=[], $checked_ids=null, $pass_if_empty=true){
        static::_sql_add_simple_limit(static::_SQL_COURSEID, $checked_ids ?? $this->_courseid, $where, $params, $pass_if_empty);
    }

    /**
     * Add limitation by group, changing $where and $params for sql query
     *
     * @param array              $joins
     * @param array              $where
     * @param array              $params
     * @param array|numeric|null $checked_ids   - if not null, rewrite current data to check
     * @param bool               $pass_if_empty - if true, and value is empty, do not set limitation
     *
     * @return void
     */
    protected function _sql_add_limit_group(&$joins=[], &$where=[], &$params=[], $checked_ids=null, $pass_if_empty=true){
        $val = $checked_ids ?? $this->_groupid;
        if (empty($val)){
            if ($pass_if_empty){
                // none limitation
                return;
            }
            if (is_array($val) || is_object($val)){
                // for empty array return None result
                $where[] = NED::SQL_NONE_COND;
                return;
            }
        }

        $group_alias = static::_SQL_GROUP_ALIAS;
        $course_alias = static::_sql_a(static::_SQL_COURSEID);
        $user_alias = static::_sql_a(static::_SQL_USERID);
        NED::sql_add_get_in_or_equal_options($group_alias.'.id', $val, $where, $params);

        $group_join = NED::sql_get_group_join($course_alias, $user_alias, $group_alias);
        if (empty($joins) || !in_array($group_join, $joins)){
            $joins[] = $group_join;
        }
    }

    /**
     * Add limitation by school, changing $where and $params for sql query
     *
     * @param array              $joins
     * @param array              $where
     * @param array              $params
     * @param array|numeric|null $checked_ids   - if not null, rewrite current data to check
     * @param bool               $pass_if_empty - if true, and value is empty, do not set limitation
     *
     * @return void
     */
    protected function _sql_add_limit_school(&$joins=[], &$where=[], &$params=[], $checked_ids=null, $pass_if_empty=true){
        if (!NED::is_schm_exists()) return;

        $val = $checked_ids ?? $this->_schoolid;
        if (empty($val)){
            if ($pass_if_empty){
                // none limitation
                return;
            }

            if (is_array($val) || is_object($val)){
                // for empty array return None result
                $where[] = NED::SQL_NONE_COND;
                return;
            }
        }

        $school_alias = static::_SQL_SCHOOL_ALIAS;
        $user_alias = static::_sql_a(static::_SQL_USERID);
        NED::sql_add_get_in_or_equal_options($school_alias.'.id', $val, $where, $params);

        $school_join = NED::sql_get_school_join($user_alias, $school_alias);
        if (empty($joins) || !in_array($school_join, $joins)){
            $joins[] = $school_join;
        }
    }

    /**
     * Add limitation by grader, changing $where and $params for sql query
     *
     * @param array              $joins
     * @param array              $where
     * @param array              $params
     * @param array|numeric|null $checked_ids   - if not null, rewrite current data to check
     * @param bool               $pass_if_empty - if true, and value is empty, do not set limitation
     *
     * @return void
     */
    protected function _sql_add_limit_grader(&$joins=[], &$where=[], &$params=[], $checked_ids=null, $pass_if_empty=true){
        static::_sql_add_simple_limit(static::_SQL_GRADERID, $checked_ids ?? $this->_graderid, $where, $params, $pass_if_empty);
    }

    /**
     * Add limitation by user, changing $where and $params for sql query
     *
     * @param array              $joins
     * @param array              $where
     * @param array              $params
     * @param array|numeric|null $checked_ids - if not null, rewrite current data to check
     * @param bool               $pass_if_empty - if true, and value is empty, do not set limitation
     *
     * @return void
     */
    protected function _sql_add_limit_user(&$joins=[], &$where=[], &$params=[], $checked_ids=null, $pass_if_empty=true){
        static::_sql_add_simple_limit(static::_SQL_USERID, $checked_ids ?? $this->_userid, $where, $params, $pass_if_empty);
    }

    /**
     * Add additional view limitation for {@see ned_base_table_page_render::_sql_menu_filter()}
     *  changing $where and $params for sql query
     *
     * @param array $joins
     * @param array $where
     * @param array $params
     *
     * @return void
     */
    protected function _sql_add_view_limitation(&$joins=[], &$where=[], &$params=[]){
        /**
         * You can use _sql_add_limit_* methods here, or add something new
         */
    }
    //endregion

    //region SQL main utils methods
    /**
     * Check single school id filter and list of limited school ids
     * - if we need only filter by single id, return it
     * - otherwise add sql filter to use only users from the limited school ids (and return null)
     * Can changing $joins, $where, $params and $groupby for sql query
     *
     * @param array      $joins            - (modified) for later SQL query
     * @param array      $where            - (modified) for later SQL query
     * @param array      $params           - (modified) for later SQL query
     * @param array      $groupby          - (modified) for later SQL query
     * @param null|int   $schoolid         - (optional) other school id to check, by default (NULL) uses {@see _schoolid}
     * @param null|array $limit_schoolids  - (optional) other limit school ids to check, by default (NULL) uses {@see _limit_schoolids}
     * @param null|bool  $view_all_schools - (optional) other capability to view all schools, by default (NULL) uses {@see _view_all_schools}
     *
     * @return int|null - return $simple_school_id_filter (school id) if we need only simple filter by the single id, otherwise return null
     */
    protected function _sql_provide_school_limitation(&$joins=[], &$where=[], &$params=[], &$groupby=[],
        $schoolid=null, $limit_schoolids=null, $view_all_schools=null){
        $simple_school_id_filter = null;
        $school_t = static::_SQL_SCHOOL_ALIAS;

        if (NED::is_school_manager_exists() && !empty($school_t)){
            $schoolid = $schoolid ?? $this->_schoolid;
            $simple_school_id_filter = $schoolid ?: null;
            $limit_schoolids = $limit_schoolids ?? $this->_limit_schoolids;
            $view_all_schools = $view_all_schools ?? $this->_view_all_schools;
            if (!$view_all_schools && empty($limit_schoolids)){
                $where[] = NED::SQL_NONE_COND;
            } elseif (!$simple_school_id_filter && !empty($limit_schoolids)){
                if (count($limit_schoolids) === 1){
                    $simple_school_id_filter = reset($limit_schoolids);
                } else {
                    $school_t2 = $school_t.'2';
                    $joins[] = NED::sql_get_school_join(static::_sql_a(static::_SQL_USERID), $school_t2);
                    NED::sql_add_get_in_or_equal_options("$school_t2.id", $limit_schoolids, $where, $params);
                    $where[] = "FIND_IN_SET($school_t2.id, $school_t.school_ids)";
                    $groupby[] = static::_sql_a('id');
                }
            }
        }

        return $simple_school_id_filter;
    }
    //endregion

    //region Get main table data methods
    /**
     * Get and render data for main page table
     * @see get_db_table_data() for DB information
     * @see _process_table_raw_records() for process/render DB records
     *
     * @param bool $only_keys - if true, return only records key (normally it will be listed of record ids)
     *
     * @return array - list of records or keys
     */
    public function get_table_data($only_keys=false){
        if (isset($this->_static_data[__FUNCTION__])){
            $records = $this->_static_data[__FUNCTION__];
            if ($only_keys){
                return array_keys($records);
            }

            return $records;
        }

        if (!$this->can_see()){
            return [];
        }

        $select = [];
        $joins = [];
        $where = [];
        $params = [];
        $groupby = [];
        $orderby = [];
        $set_limit = (bool)$this->_perpage;

        $raw_records = $this->get_db_table_data($select, $joins, $where, $params, $groupby, $orderby, $set_limit);
        if ($only_keys){
            return array_keys($raw_records);
        }

        $records = $this->_process_table_raw_records($raw_records);
        $this->_static_data[__FUNCTION__] = $records;
        return $records;
    }

    /**
     * Process raw records from the DB and return rendered result
     * Normally called from the {@see get_table_data()}
     * @abstract
     *
     * @param array $raw_records
     *
     * @return array
     */
    protected function _process_table_raw_records(&$raw_records=[]){
        /**
         * This realisation is more like example, then real working method,
         * as base class doesn't have real DB table
         */

        $records = [];
        /** @var object $record */
        foreach ($raw_records as $record){
            $record->course_name = NED::q_course_link($record->courseid, false, true);
            $record->course = NED::q_course_link($record->courseid);
            $record->cm = NED::q_cm_grade_link($record->cmid, $record->userid, $record->courseid);
            $record->student = NED::q_user_link($record->userid, $record->courseid);
            $record->gradername = NED::q_user_link($record->graderid, $record->courseid);

            $records[$record->id] = $record;
        }

        return $records;
    }

    /**
     * Get data from the DB for main table
     * Also can update some pager data, {@see _db_calc_count_and_get_limitfrom()}
     *
     * @param array|string $select
     * @param array|string $joins
     * @param array|string $where
     * @param array        $params
     * @param array|string $groupby
     * @param array|string $orderby
     * @param bool         $set_limit - if true, uses page limitations to query
     *
     * @return array
     */
    public function get_db_table_data($select='', $joins=[], $where=[], $params=[], $groupby=[], $orderby=[], $set_limit=false){
        if (!$this->_db_process_sql_query($select, $joins, $where, $params, $groupby, $orderby)){
            return [];
        }
        return $this->_db_calc_count_and_get_table_data($select, $joins, $where, $params, $groupby, $orderby, $set_limit);
    }

    /**
     * Get base joins for {@see _db_process_sql_query()}
     *
     * @param array $add_joins  - add additional joins to it
     * @param array $params     - array to save additional params
     * @param array $base_joins - joins for adding to the beginning of the join list
     *
     * @return array
     */
    protected function _db_get_base_join($add_joins=[], &$params=[], $base_joins=[]){
        $base_joins = $base_joins ?: [];
        $group_t = static::_SQL_GROUP_ALIAS;
        $a_userid = static::_sql_a(static::_SQL_USERID);
        $a_courseid = static::_sql_a(static::_SQL_COURSEID);
        if (!empty($group_t)){
            $base_joins[] = NED::sql_join_groups_as_subquery_table($group_t, $a_userid, $a_courseid);
        }

        $school_t = static::_SQL_SCHOOL_ALIAS;
        if (NED::is_school_manager_exists() && !empty($school_t)){
            $base_joins[] = NED::sql_join_school_as_subquery_table($school_t, $a_userid);
        }

        if (!empty($add_joins)){
            $base_joins = array_merge($base_joins, $add_joins);
        }

        return $base_joins;
    }

    /**
     * Process SQL query for the getting data from the DB for main table
     * @abstract
     *
     * @param array|string $select
     * @param array|string $joins
     * @param array|string $where
     * @param array        $params
     * @param array|string $groupby
     * @param array|string $orderby
     *
     * @return bool - saves result in the params, return true if you can continue query
     */
    protected function _db_process_sql_query(&$select=[], &$joins=[], &$where=[], &$params=[], &$groupby=[], &$orderby=[]){
        /**
         * This realisation is more like example, then real working method,
         * as base class doesn't have real DB table
         */

        [$select, $joins, $where, $groupby, $orderby] = NED::val2arr_multi(true, $select, $joins, $where, $groupby, $orderby);
        if (empty($select)){
            $select[] = static::_sql_a('*');
        }
        $joins = $this->_db_get_base_join($joins, $params);

        if ($this->_is_admin && !empty($this->_ids)){
            $ids = $this->_ids;
            // if we have filter, we should filter even admin call
            if (!empty($this->_filter_ids)){
                $ids = array_intersect($ids, $this->_filter_ids);
                if (empty($ids)){
                    return false;
                }
            }
            NED::sql_add_get_in_or_equal_options(static::_sql_a('id'), $ids, $where, $params);
        } else {
            static::_sql_set_simple_filters($where, $params,
                $this->_courseid, $this->_cmid, $this->_userid, $this->_groupid, $this->_graderid, $this->_schoolid,
                $this->_filter_ids, $this->_filter_id
            );
        }

        return true;
    }

    /**
     * Count all records by SQL conditions and return data from the DB for main table
     * Also can update some pager data, {@see _db_calc_count_and_get_limitfrom()}
     * Normally called from the {@see get_db_table_data()}
     *
     * @param array|string $select
     * @param array|string $joins
     * @param array|string $where
     * @param array        $params
     * @param array|string $groupby
     * @param array|string $orderby
     * @param bool         $set_limit - if true, uses page limitations to query
     *
     * @return array
     */
    protected function _db_calc_count_and_get_table_data($select=[], $joins=[], $where=[], $params=[], $groupby=[], $orderby=[], $set_limit=false){
        $limitfrom = $this->_db_calc_count_and_get_limitfrom($joins, $where, $params, $select);
        if ($set_limit){
            $limitnum = $this->_perpage;
        } else {
            $limitfrom = 0;
            $limitnum = 0;
        }

        $sql = static::_sql_generate_sql($select, $joins, $where, $groupby, $orderby);
        $records = NED::db()->get_records_sql($sql, $params, $limitfrom, $limitnum);
        return $records ?: [];
    }

    /**
     * Count all records by SQL conditions
     * Also update pager data, as _total_count and _page
     * Normally called from the {@see get_db_table_data()}
     *
     * @param array $joins
     * @param array $where
     * @param array $params
     * @param array $select - optional, if you want to get result not by main id
     *
     * @return int
     */
    protected function _db_calc_count_and_get_limitfrom($joins=[], $where=[], $params=[], $select=[]){
        if (empty($select)){
            $select = "COUNT(DISTINCT ".static::_sql_a('id').')';
        } else {
            $select_for_count = str_replace('*', 'id', reset($select));
            $select = "COUNT(DISTINCT $select_for_count)";
        }
        $sql_count = static::_sql_generate_sql($select, $joins, $where);
        $this->_total_count = NED::db()->count_records_sql($sql_count, $params);
        $limitfrom = 0;
        $last = $this->_total_count - 1;
        if ($this->_perpage && $this->_total_count > 1){
            $limitfrom = $this->_page * $this->_perpage;
            if ($limitfrom > $last){
                $this->_page = floor($last/$this->_perpage);
                $limitfrom = $this->_page * $this->_perpage;
            }
            $limitfrom = max(min($limitfrom, $last), 0);
        }
        return $limitfrom;
    }
    //endregion

    //region Get filter options [get_*_options()]
    /**
     * @return array
     */
    public function get_graders_options(){
        $data = $this->_static_data[__FUNCTION__] ?? null;
        if (is_null($data)){
            $key = static::_sql_a(static::_SQL_GRADERID);
            $val = "IF(u.id IS NOT NULL, CONCAT(u.firstname, ' ', u.lastname), 'None')";
            $joins = ["LEFT JOIN {user} u ON u.id = $key"];
            $data = $this->_sql_menu_filter($key, $val, $joins, [], [], static::GRADER_ALL, false);
            $this->_static_data[__FUNCTION__] = $data;
        }
        return $data;
    }

    /**
     * @return array
     */
    public function get_schools_options(){
        if (!NED::is_school_manager_exists()) return [];

        $data = $this->_static_data[__FUNCTION__] ?? null;
        if (is_null($data)){
            $school_alias = static::_SQL_SCHOOL_ALIAS;
            $key = $school_alias.".id";
            $val = $school_alias.".code";
            $joins = NED::sql_get_school_join(static::_sql_a(static::_SQL_USERID), $school_alias);
            $data = $this->_sql_menu_filter($key, $val, $joins);
            $this->_static_data[__FUNCTION__] = $data;
        }
        return $data;
    }

    /**
     * @return array
     */
    public function get_courses_options(){
        $data = $this->_static_data[__FUNCTION__] ?? null;
        if (is_null($data)){
            $key = static::_sql_a(static::_SQL_COURSEID);
            $val = "c.shortname";
            $joins = ["JOIN {course} c ON c.id = $key"];
            $data = $this->_sql_menu_filter($key, $val, $joins);
            $data[static::COURSE_NONE] = NED::str('none');
            $this->_static_data[__FUNCTION__] = $data;
        }
        return $data;
    }

    /**
     * @return array
     */
    public function get_cm_options(){
        if ($this->_courseid <= 0){
            return null;
        }

        $data = $this->_static_data[__FUNCTION__] ?? null;
        if (is_null($data)){
            $key = static::_sql_a(static::_SQL_CMID);
            $val = $key." AS val";
            $joins = ["JOIN {course_modules} cm ON cm.id = $key"];
            $where = [];
            $params = [];

            $this->_sql_add_limit_course($joins, $where, $params);
            $raw_data = $this->_sql_menu_filter($key, $val, $joins, $where, $params, false);

            $data = [NED::ALL => get_string('all')];
            foreach ($raw_data as $cmid){
                $cm = NED::get_cm_by_cmid($cmid, $this->_courseid);
                if (!$cm) continue;

                $data[$cm->id] = strip_tags($cm->get_formatted_name());
            }
            $this->_static_data[__FUNCTION__] = $data;
        }
        return $data;
    }

    /**
     * Get student class options
     *
     * @return array
     */
    public function get_group_options(){
        if ($this->_courseid <= 0){
            return null;
        }

        $data = $this->_static_data[__FUNCTION__] ?? null;
        if (is_null($data)){
            $group_alias = static::_SQL_GROUP_ALIAS;
            $key = $group_alias.".id";
            $val = $group_alias.".name";

            $joins = NED::sql_get_group_join(static::_sql_a(static::_SQL_COURSEID), static::_sql_a(static::_SQL_USERID), $group_alias);
            $where = [];
            $params = [];

            $this->_sql_add_limit_course($joins, $where, $params);
            $data = $this->_sql_menu_filter($key, $val, $joins, $where, $params);
            $this->_static_data[__FUNCTION__] = $data;
        }
        return $data;
    }

    /**
     * @return array
     */
    public function get_users_options(){
        $data = $this->_static_data[__FUNCTION__] ?? null;
        if (is_null($data)){
            $key = static::_sql_a(static::_SQL_USERID);
            $val = "IF(u.id IS NOT NULL, CONCAT(u.firstname, ' ', u.lastname), 'None')";
            $joins = ["LEFT JOIN {user} u ON u.id = $key"];
            $where = [];
            $params = [];

            $this->_sql_add_limit_course($joins, $where, $params);
            $this->_sql_add_limit_group($joins, $where, $params);

            $data = $this->_sql_menu_filter($key, $val, $joins, $where, $params);
            $this->_static_data[__FUNCTION__] = $data;
        }
        return $data;
    }

    /**
     * @return array
     */
    public function get_school_year_options(){
        return $this->_is_admin ? static::SCHOOL_YEAR_OPTIONS : [];
    }

    /**
     * @return array
     */
    public function get_perpage_options(){
        return static::PERPAGES;
    }
    //endregion

    //region Render utils methods, including r_*_selector() functions (filter selectors)
    /**
     * Here you can add some info to ->content or call the error, if user can't see this page
     * Called from the {@see export_for_template()} if user can't see the page
     */
    public function r_can_not_see(){
        NED::print_error('nopermissions', 'error', '', get_string('checkpermissions', 'core_role'));
    }

    /**
     * @return string
     */
    public function r_pager(){
        if ($this->_total_count > 0 && $this->_perpage && ($this->_total_count > $this->_perpage)){
            return $this->_o->paging_bar($this->_total_count, $this->_page, $this->_perpage, $this->get_my_url(), NED::PAR_PAGE);
        }
        return '';
    }

    /**
     * @return string
     */
    public function r_grader_selector(){
        $options = $this->get_graders_options();
        if (!empty($options)){
            return NED::single_select($this->get_my_url(), NED::PAR_GRADER, $options, $this->_graderid, NED::str('grader'));
        }
        return '';
    }

    /**
     * @return string
     */
    public function r_school_selector(){
        if ($this->_view_all_schools || count($this->_limit_schoolids ?? []) > 1){
            $options = $this->get_schools_options();
            if (!empty($options)){
                return NED::single_autocomplete($this->get_my_url(), NED::PAR_SCHOOL, $options, $this->_schoolid, NED::str('school'));
            }
        }

        return '';
    }

    /**
     * @return string
     */
    public function r_course_selector(){
        $options = $this->get_courses_options();
        return NED::single_autocomplete($this->get_my_url(), NED::PAR_COURSE, $options, $this->_courseid, NED::str('course'));
    }

    /**
     * @return string
     */
    public function r_cm_selector(){
        $options = $this->get_cm_options();
        return NED::single_autocomplete($this->get_my_url(), NED::PAR_CM, $options, $this->_cmid, NED::str('activity'));
    }

    /**
     * @return string
     */
    public function r_group_selector(){
        $options = $this->get_group_options();
        return NED::single_autocomplete($this->get_my_url(), NED::PAR_GROUP, $options, $this->_groupid, NED::str('class'));
    }

    /**
     * @return string
     */
    public function r_users_selector(){
        $options = $this->get_users_options();
        return NED::single_autocomplete($this->get_my_url(), NED::PAR_USER, $options, $this->_userid, NED::str('student'));
    }

    /**
     * @return string
     */
    public function r_school_year_selector(){
        $options = $this->get_school_year_options();
        if (!empty($options)){
            return NED::single_select($this->get_my_url(), NED::PAR_SCHOOL_YEAR, static::get_list_as_options($options),
                $this->_school_year, NED::str('ned_school_year_title'));
        }
        return '';
    }

    /**
     * @return string
     */
    public function r_perpage_selector(){
        $options = static::get_list_as_options($this->get_perpage_options());
        return NED::single_select($this->get_my_url(), NED::PAR_PERPAGE, $options, $this->_params[NED::PAR_PERPAGE], NED::str('perpage'));
    }

    /**
     * @return string
     */
    public function r_course_view_button(){
        return NED::button_link(
            $this->get_my_url([NED::PAR_COURSE_VIEW => !$this->_course_view]),
            $this->_course_view ? 'sitepage' : 'coursepage'
        );
    }

    /**
     * @return string
     */
    public function r_reset_button(){
        return NED::button_link($this->get_main_url(), 'reset');
    }
    //endregion

    //region Notifications methods
    /**
     * Add notification, which should be rendered later
     * Method was divided, because notifications may be created before the page is set up.
     *
     * @param string $message
     * @param string $type - any of the NED::NOTIFY_*
     */
    public function add_notification($message, $type=NED::NOTIFY_INFO){
        $this->_notifications[] = [$message, $type];
    }

    /**
     * Render the notifications.
     *
     * @param array $output - array to store result
     *
     * @return array - return modified $output
     */
    protected function _render_notifications(&$output=[]){
        foreach ($this->_notifications as $n_data){
            $output[] = $this->_o->notification(...$n_data);
        }
        $this->_notifications = [];

        return $output;
    }
    //endregion

    //region Main render page content methods
    /**
     * Function to export the renderer data in a format that is suitable for a
     * mustache template. This means:
     * 1. No complex types - only stdClass, array, int, string, float, bool
     * 2. Any additional info that is required for the template is pre-calculated (e.g. capability checks).
     * @noinspection PhpReturnDocTypeMismatchInspection
     *
     * @param \renderer_base $output Used to do a final render of any components that need to be rendered for export.
     *
     * @return \stdClass|array
     */
    public function export_for_template(\renderer_base $output){
        $this->content = $this->content ?? new \stdClass();
        if (!$this->can_see()){
            $this->r_can_not_see();
            return $this->content;
        }

        $this->_before_start_export();
        $this->_render_page_content();
        $this->_before_finish_export();

        return $this->content;
    }

    /**
     * Render page content
     * Normally calling from the {@see export_for_template()}
     * You can rewrite this method and not changing original {@see export_for_template()}
     * Please, do not use redirect methods here, normally they should be called from {@see before_header_output()}
     * @abstract
     * @noinspection DuplicatedCode
     */
    protected function _render_page_content(){
        /**
         * This realisation is more like example, then real working method,
         * as base class doesn't have real page or template
         */

        $this->content->data = array_values($this->get_table_data());

        $this->content->pager = $this->r_pager();

        $grader_selector = $this->r_grader_selector();
        if (!empty($grader_selector)){
            $this->content->control_panel1[] = $grader_selector;
        }

        if (static::_USE_NED_SCHOOL_YEAR_FILTER){
            $this->content->control_panel1[] = $this->r_school_year_selector();
        }

        $school_selector = $this->r_school_selector();
        if (!empty($school_selector)){
            $this->content->control_panel2[] = $school_selector;
        }


        if (!$this->_course_view){
            $this->content->control_panel2[] = $this->r_course_selector();
        }

        if ($this->_courseid > 0){
            $this->content->control_panel2[] = $this->r_cm_selector();
            $this->content->control_panel2[] = $this->r_group_selector();
        }

        $this->content->control_panel2[] = $this->r_users_selector();
        $this->content->control_panel3[] = $this->r_perpage_selector();
    }

    /**
     * Init some content values before {@see export_for_template()}
     * Normally calling from the {@see export_for_template()}
     * @abstract
     */
    protected function _before_start_export(){
        /**
         * This realisation is more like example, then real working method,
         * as base class doesn't have real page or template
         */

        $this->content = $this->content ?? new \stdClass();
        $this->content->contextid = $this->_ctx->id;
        $this->content->table_class = [];

        $this->content->control_panel1 = [];
        $this->content->control_panel2 = [];
        $this->content->control_panel3 = [];
        $this->content->buttons = [];

        $this->content->is_admin = $this->_is_admin;
        $this->content->data = [];
    }

    /**
     * Some last content changes at final {@see export_for_template()}
     * Normally calling from the {@see export_for_template()}
     * @abstract
     */
    protected function _before_finish_export(){
        /**
         * This realisation is more like example, then real working method,
         * as base class doesn't have real page or template
         */

        $this->content = $this->content ?? new \stdClass();
        $this->content->table_class = join(' ', $this->content->table_class);
        $this->content->has_data = !empty($this->content->data);
        $this->content->notifications = $this->_render_notifications();
    }
    //endregion

    //region Control methods
    /**
     * Clear stored cache data
     *
     * @param bool $clear_content - if true, also clear content and notifications
     *
     * @return void
     */
    public function clear_cache($clear_content=false){
        foreach ($this->_static_data as $key => $data){
            unset($data);
            unset($this->_static_data[$key]);
        }
        $this->_static_data = [];

        if ($clear_content){
            unset($this->content);
            $this->content = new \stdClass();

            foreach ($this->_notifications as $key => $data){
                unset($data);
                unset($this->_notifications[$key]);
            }
            $this->_notifications = [];
        }
    }
    //endregion

    //region Static Utils methods
    /**
     * @param array $list
     * @param bool  $add_all   - add "All" option
     * @param bool  $translate - try to translate list values
     *
     * @return array
     */
    static public function get_list_as_options($list=[], $add_all=false, $translate=true){
        $res = [];
        if ($add_all){
            $res[isset($list[NED::ALL]) ? -1 : NED::ALL] = get_string('all');
        }
        foreach ($list as $key => $item){
            $r = $item;
            if ($translate && !is_numeric($item) && is_string($item)){
                $r = NED::str_check2($item, null, $item, static::_PLUGIN, NED::CTRL);
            }
            $res[$key] = $r;
        }

        return $res;
    }

    /**
     * Get records from main table with some simple filters. Don't support filters, which requires join of other tables
     * If you wish skip filter by some param, send NULL for it
     *
     * @param numeric|null $courseid
     * @param numeric|null $cmid
     * @param numeric|null $userid
     * @param numeric|null $graderid
     * @param numeric|null $school_year - {@see static::SCHOOL_YEAR_OPTIONS}
     * @param array|null   $filter_ids
     * @param numeric|null $filter_id
     *
     * @return array|object[] - list of records from main table
     */
    static public function get_records_by_simple_filters($courseid=null, $cmid=null, $userid=null, $graderid=null, $school_year=null,
        $filter_ids=null, $filter_id=null){
        $where = $params = [];
        // don't call params, which requires join other tables
        static::_sql_set_simple_filters($where, $params, $courseid, $cmid, $userid, null, $graderid, null,
            $filter_ids, $filter_id, $school_year);

        $sql = static::_sql_generate_sql([], [], $where);
        return NED::db()->get_records_sql($sql, $params);
    }
    //endregion

    //region Main rendering class/page methods
    /**
     * Render this class element
     *
     * @param bool $return
     *
     * @return string
     */
    public function rendered($return=false){
        $res = NED::render($this);
        if (!$return){
            echo $res;
        }
        return $res;
    }

    /**
     * Render full page, including header and footer
     */
    static public function render_full_page(){
        $elem = new static();
        $elem->init();

        if (!$elem->header_printed){
            $elem->before_header_output();
            echo $elem->o->header();
            $elem->after_header_output();
        }
        $elem->rendered();

        $elem->before_footer_output();
        echo $elem->o->footer();
    }
    //endregion
}
