<?php
/**
 * @package    local_ned_controller
 * @subpackage shared
 * @category   NED
 * @copyright  2022 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 *
 * @noinspection DuplicatedCode
 */

namespace local_ned_controller\shared;

defined('MOODLE_INTERNAL') || die();

/**
 * Trait cm_util (util for working with course modules)
 *
 * @package local_ned_controller\shared
 * @mixin base_trait
 */
trait cm_util {
    use data_util, db_util, util;

    //region Getters (cm, course etc)
    /**
     * Efficiently retrieves the $course (stdclass) and $cm (cm_info) objects, given
     * a cmid. If module name is also provided, it will ensure the cm is of that type.
     *
     * Usage:
     * list($course, $cm) = get_course_and_cm_from_cmid($cmid, 'forum');
     *
     * Using this method has a performance advantage because it works by loading
     * modinfo for the course - which will then be cached, and it is needed later
     * in most requests. It also guarantees that the $cm object is a cm_info and
     * not a stdclass.
     *
     * The $course object can be supplied if already known and will speed
     * up this function - although it is more efficient to use this function to
     * get the course if you are starting from a cmid.
     *
     * WARNING: Multiple uses of this method (e.g. in a loop) without $courseorid, may cause speed problems or memory leaks.
     *
     * @param object|int $cmorid - Id of course-module, or database object
     * @param string     $modulename Optional modulename (improves security)
     * @param object|int $courseorid Optional course object if already loaded
     * @param object|int $userorid Optional userid (default = current)
     *
     * @return (object|\course_modinfo|\cm_info|null)[] Array with 2 elements $course and $cm (or null and null)
     */
    static public function get_course_and_cm_from_cmid($cmorid, $modulename='', $courseorid=null, $userorid=null){
        $cmid = static::get_id($cmorid);
        if (!$cmid){
            return [null, null];
        }
        $courseid = (int)($cmorid->course ?? 0);
        $userid = static::get_userid_or_global($userorid);

        $res = static::g_get(__FUNCTION__, [$cmid, $userid]);
        if (is_null($res)){
            /**
             * Core {@see get_course_and_cm_from_cmid()} function is not very memory sensitive,
             * so it is better to use our methods to load \cm_info and course object separately.
             */
            //$res = get_course_and_cm_from_cmid($cmorid, '', $courseorid ?: $courseid, $userid);
            $cm = static::get_cm_by_cmorid($cmorid, $courseorid ?: $courseid, $userorid);
            $course = null;
            if ($cm){
                $course = (!empty($courseorid) && is_object($courseorid)) ? $courseorid : null;
                if (empty($course->id) || $course->id != $cm->course){
                    $course = static::get_course($cm->course);
                }
            }
            $res = [$course, $cm];

            static::g_set(__FUNCTION__, [$cmid, $userid], $res);
        }

        [$course, $cm] = $res ?: [null, null];
        if ($cm && !empty($modulename) && $cm->modname !== $modulename){
            [$course, $cm] = [null, null];
        }

        return [$course, $cm];
    }

    /**
     * @param \stdClass|int     $courseorid - Course object (or its id) if already loaded
     * @param \stdClass|int     $userorid   - Optional userid (default = current)
     * @param string[]|string   $filter_modnames - Optional array or single value to check cm modname
     *
     * @return \cm_info[] Array from course-module instance to cm_info object within this course, in
     *   order of appearance
     */
    static public function get_course_cms($courseorid, $userorid=null, $filter_modnames=[]){
        $userid = static::get_userid_or_global($userorid);
        $courseid = static::get_id($courseorid);
        $course_info = static::get_fast_modinfo($courseid, $userid);
        if (empty($course_info)) return [];

        $filter_modnames = static::val2arr($filter_modnames);
        if (count($filter_modnames) == 1){
            $cms = [];
            $cms_by_inst = $course_info->get_instances_of(reset($filter_modnames));
            foreach ($cms_by_inst as $cm){
                $cms[$cm->id] = $cm;
            }
        } else {
            $cms = static::filter_cms_by_modnames($course_info->get_cms(), $filter_modnames);
        }

        return $cms;
    }

    /**
     * Return course module by kica item, or grade item,
     *  or any other object with 'courseid', 'itemmodule', 'iteminstance' keys
     *
     * @param \local_kica\kica_item|\grade_item|object $item
     *
     * @return null|object|\cm_info
     */
    static public function get_cm_by_kica_or_grade_item($item){
        if (!$item){
            return null;
        }

        return static::get_cm_by_params($item->courseid, $item->itemmodule, $item->iteminstance);
    }

    /**
     * Return course module by courseid, itemmodule, iteminstance
     *
     * @param object|numeric    $course_or_id
     * @param string            $itemmodule - modname
     * @param string|numeric    $iteminstance - instance
     *
     * @return null|object|\cm_info
     */
    static public function get_cm_by_params($course_or_id, $itemmodule, $iteminstance){
        $modinfo = static::get_fast_modinfo($course_or_id);

        return $modinfo ? ($modinfo->instances[$itemmodule][$iteminstance] ?? null) : null;
    }

    /**
     * Get activities for NED plugins
     * If you need all course course-modules,
     *  @see \local_ned_controller\shared\data_util::get_course_cms()
     *
     * @param \stdClass|int $courseorid - Course object (or its id) if already loaded
     * @param \stdClass|int $userorid   - Optional userid (default = current)
     *
     * @return \cm_info[] Array from course-module instance to cm_info object within this course,
     *                      only which user can see and  which has view
     */
    static public function get_course_activities($courseorid, $userorid=null){
        $userid = static::get_userid_or_global($userorid);
        $courseid = static::get_id($courseorid);

        $activities = static::g_get(__FUNCTION__, [$courseid, $userid]);
        if (is_null($activities)){
            $cms = static::get_course_cms($courseid, $userid);
            $activities = [];
            foreach ($cms as $cm) {
                // Exclude activities that aren't visible or have no view link (e.g. label)
                if (!$cm->uservisible || !$cm->has_view() || !isset(C::MOD_VALS[$cm->modname])) {
                    continue;
                }

                $activities[$cm->id] = $cm;
            }

            static::g_set(__FUNCTION__, [$courseid, $userid], $activities);
        }

        return $activities;
    }

    /**
     * Return $cm (cm_info) objects, by a cmid. If module name is also provided,
     *  it will ensure the cm is of that type.
     *
     * The $course object (or its id) can be supplied if already known and will speed
     * up this function.
     *
     * Return null if it finds nothing (or $modulename is wrong)
     *
     * @see get_course_and_cm_from_cmid
     *
     * @param int           $cmid       - Id of course-module, or database object
     * @param \stdClass|int $courseorid - Optional course object (or its id) if already loaded
     * @param \stdClass|int $userorid   - Optional user object (or its id; default = current)
     * @param string        $modulename - Optional modulename (improves security)
     *
     * @return \cm_info|null
     */
    static public function get_cm_by_cmid($cmid, $courseorid=null, $userorid=null, $modulename=''){
        static $_last_used_courseid = null;

        if ($courseorid){
            $cms = static::get_course_cms($courseorid, $userorid);
            $cm = $cms[$cmid] ?? null;
            if ($cm && !empty($modulename) && $cm->modname !== $modulename){
                $cm = null;
            }
        } else {
            /**
             * Multiple CM requests (e.g. in a loop) without a course can lead to memory leaks,
             *  so check for CM in the global or last used course first - this is much more optimized than requesting CM without a course
             */
            $_last_used_courseid = $_last_used_courseid ?: static::get_courseid_or_global();
            if ($_last_used_courseid && $_last_used_courseid != SITEID){
                $cm = static::get_cm_by_cmid($cmid, $_last_used_courseid, $userorid, $modulename);
            }

            if (empty($cm)){
                try {
                    [, $cm] = \get_course_and_cm_from_cmid($cmid, $modulename, $courseorid, static::get_id($userorid));
                } catch (\Throwable){
                    $cm = null;
                }
            }
        }

        $_last_used_courseid = $cm ? $cm->course : null;

        return $cm;
    }

    /**
     * @param \cm_info|object|numeric $cm_or_id
     * @param object|numeric          $courseorid - Optional course obj (or its id) if loaded, improves optimization if $cm_or_id is represented as ID
     * @param object|numeric          $userorid   - Optional user object (or its id; default = current)
     * @param string                  $modulename - Optional modulename (improves security)
     *
     * @return \cm_info|object|null
     */
    static public function get_cm_by_cmorid($cm_or_id, $courseorid=null, $userorid=null, $modulename=''){
        if ($cm_or_id instanceof \cm_info){
            $cm = $cm_or_id;
            if ($courseorid && $cm->course != static::get_id($courseorid)) return null;

            if (!empty($modulename) && $cm->modname != $modulename) return null;

            if ($userorid && $cm->get_modinfo()->userid != static::get_id($userorid)){
                return static::get_cm_by_cmid($cm->id, $courseorid, $userorid, $cm->modname);
            }

            return $cm;
        } else {
            if (is_object($cm_or_id)){
                $courseorid = $courseorid ?? ($cm_or_id->course ?? null);
                $modulename = $modulename ?? ($cm_or_id->modname ?? null);
            }
            return static::get_cm_by_cmid(static::get_id($cm_or_id), $courseorid, $userorid, $modulename);
        }
    }

    /**
     * Get course id from cm
     *
     * @param \cm_info|int|string $cm_or_id
     *
     * @return int
     */
    static public function get_courseid_by_cmorid($cm_or_id){
        $cm = static::get_cm_by_cmorid($cm_or_id);
        return $cm ? $cm->course : 0;
    }

    /**
     * Return course fast modinfo, but if there is no course, return null instead of error.
     * Usually it will be better to use {@see static::get_fast_modinfo()}
     * @see get_fast_modinfo()
     *
     * @param object|numeric $course_or_id
     * @param object|numeric $user_or_id - Current (global) user by default (if null|0)
     *
     * @return \course_modinfo|null
     */
    static protected function _get_fast_modinfo($course_or_id, $user_or_id=null){
        $userid = static::get_userid_or_global($user_or_id);
        try {
            $res = get_fast_modinfo($course_or_id, $userid);
        } catch (\Exception){
            $res = null;
        }
        return $res;
    }

    /**
     * Returns reference to full info about modules in course (including visibility).
     * @see get_fast_modinfo()
     *
     * @param object|numeric $course_or_id
     * @param object|numeric $user_or_id - Current (global) user by default (if null|0)
     *
     * @return \course_modinfo|null
     */
    static public function get_fast_modinfo($course_or_id, $user_or_id=null){
        $courseid = static::get_id($course_or_id);
        $g_userid = static::get_userid_or_global();
        $userid = static::get_userid_or_global($user_or_id);
        if ($userid == $g_userid){
            $res = static::g_get(__FUNCTION__, [$courseid, $userid]);
            if (is_null($res)){
                $res = static::_get_fast_modinfo($course_or_id, $userid);
                static::g_set(__FUNCTION__, [$courseid, $userid], $res);
            }
        } else {
            // do not save fast_modinfo for other users: it takes some time to create, but it takes too much memory to store
            $res = static::_get_fast_modinfo($course_or_id, $userid);
        }

        return $res ?: null;
    }

    /**
     * @param \cm_info|numeric $cm_or_id
     *
     * @return \core_availability\info_module
     */
    static public function get_availability_info_module($cm_or_id){
        $cmid = static::get_id($cm_or_id);
        $res = static::g_get(__FUNCTION__, [$cmid]);
        if (is_null($res)){
            $res = new \core_availability\info_module(static::get_cm_by_cmorid($cm_or_id));
            static::g_set(__FUNCTION__, [$cmid], $res);
        }
        return $res;
    }

    /**
     * Get instance object
     *
     * @param numeric $instance_id id of activity in its table
     * @param string  $modname Name of module (not full frankenstyle) e.g. 'label'
     * @param numeric $courseid (optional) only for additional check
     *
     * @return object|null instance record from the activity table, or null if nothing found
     */
    static public function get_module_instance($instance_id, $modname, $courseid=null){
        if (empty($instance_id) || empty($modname)){
            return null;
        }

        $instance_id = (int)$instance_id;
        $keys = [$instance_id, $modname];
        $res = static::g_get(__FUNCTION__, $keys);
        if (is_null($res)){
            $res = false;
            $modnames = \get_module_types_names(true);
            if (isset($modnames[$modname])){
                $res = static::db()->get_record($modname, ['id' => $instance_id]);
            }

            static::g_set(__FUNCTION__, $keys, $res ?: false);
        }

        if ($courseid){
            $courseid = (int)$courseid;
            $res_courseid = (int)($res->courseid ?? ($res->course ?? 0));
            if ($res_courseid != $courseid){
                return null;
            }
        }

        return $res ?: null;
    }

    /**
     * @param int|\cm_info|object   $cm_or_id  - Id of course-module, or database object
     *
     * @return object|null
     */
    static public function get_module_instance_by_cm($cm_or_id){
        $cm = static::get_cm_by_cmorid($cm_or_id);
        if ($cm){
            return static::get_module_instance($cm->instance, $cm->modname, $cm->course);
        }

        return null;
    }
    //endregion

    //region Visibility and Access
    /**
     * NED calculation logic of activity visibility
     * Normally you can get all this data (except $unavailable_as_invisible) from course-module(cm, \cm_info)
     *
     * If you wish to get data for other user by the same cm, you can use _get_visibility_data_by_cm_user()
     * @see _get_visibility_data_by_cm_user()
     *
     * @param bool   $unavailable_as_invisible - if true, then with false uservisible - return false, despite available info
     * @param bool   $uservisible
     * @param string $availableinfo
     * @param bool   $has_view
     *
     * @return bool
     */
    static protected function _calc_activity_visibility($unavailable_as_invisible=false, $uservisible=false, $availableinfo='',
        $has_view=false){
        if (!$uservisible){
            // this is a student who is not allowed to see the module but might be allowed
            // to see availability info (i.e. "Available from ...")
            if ($unavailable_as_invisible || empty($availableinfo)){
                return false;
            }
        }

        if (!$has_view){
            return false;
        }

        return true;
    }

    /**
     * Get cm-user data necessary for calculate activity visibility
     * @see _calc_activity_visibility()
     *
     * If cm was loaded for not called user, it tries to emulate standard cm checks for other user without loading cms for this user
     * #core_moodle - you have to update this function to moodle core functionality
     *
     * Note: Function is supposed to be used for check many users - for one activity,
     *      if you need checks only one user, you may load already checked activities with get_important_activities()
     * @see get_important_activities()
     *
     * @param \cm_info|numeric $cm_or_id
     * @param object|numeric   $user_or_id
     *
     * @return array($uservisible, $availableinfo, $has_view) = list($uservisible, $availableinfo, $has_view)
     */
    static protected function _get_visibility_data_by_cm_user($cm_or_id, $user_or_id=null){
        $cmid = static::get_id($cm_or_id);
        $userid = static::get_userid_or_global($user_or_id);
        $res = static::g_get(__FUNCTION__, [$cmid, $userid]);
        if (is_null($res)){
            $uservisible = false;
            $availableinfo = '';
            $has_view = false;
            /**
             * For optimization purposes, we will NOT load cm for checked userid:
             *      if we will check many users, it will cause loading course cms for all of them
             */
            $cm = static::get_cm_by_cmorid($cm_or_id);
            do {
                if (!$cm || !$userid){
                    break;
                } elseif ($cm->get_modinfo()->get_user_id() == $userid){
                    // cm was loaded for the same user
                    [$uservisible, $availableinfo, $has_view] =
                        [$cm->uservisible, $cm->availableinfo, $cm->has_view()];
                    break;
                }

                /**
                 * Next we try to emulate course module checks, without loading cm for this user
                 * You may have to update it, after moodle core update
                 * #core_moodle
                 */
                $modinfo = static::get_fast_modinfo($cm->course, $userid);
                if (!$modinfo) break;

                $uservisible = true;
                $available = true;
                $has_view = $cm->has_view();

                /**
                 * @see \cm_info::obtain_dynamic_data()
                 */
                if (!empty(static::cfg('enableavailability'))){
                    // Get availability information.
                    $ci = static::get_availability_info_module($cm);

                    // Note that the modinfo currently available only includes minimal details (basic data)
                    // but we know that this function does not need anything more than basic data.
                    $available = $ci->is_available($availableinfo, true, $userid, $modinfo);
                }

                // Check parent section.
                if ($available) {
                    $parentsection = $modinfo->get_section_info($cm->sectionnum);
                    if (!$parentsection->available) {
                        // Do not store info from section here, as that is already
                        // presented from the section (if appropriate) - just change
                        // the flag
                        $available = false;
                    }
                }

                /**
                 * Update visible state for current user
                 * @see \cm_info::update_user_visible()
                 */
                // If the module is being deleted, set the uservisible state to false and return.
                if ($cm->deletioninprogress) {
                    $uservisible = false;
                    $availableinfo = '';
                    break;
                }

                // If the user cannot access the activity set the uservisible flag to false.
                // Additional checks are required to determine whether the activity is entirely hidden or just greyed out.
                $ctx = $cm->context;
                if ((!$cm->visible && !has_capability('moodle/course:viewhiddenactivities', $ctx, $userid)) ||
                    (!$available && !has_capability('moodle/course:ignoreavailabilityrestrictions', $ctx, $userid))){
                    $uservisible = false;
                }

                /**
                 * Check group membership
                 * @see \cm_info::is_user_access_restricted_by_capability()
                 */
                if (static::cm_is_user_access_restricted_by_capability($cm, $userid)){
                    $uservisible = false;
                    // Ensure activity is completely hidden from the user.
                    $availableinfo = '';
                }

                /*
                $uservisibleoncoursepage = $uservisible &&
                    ($cm->visibleoncoursepage ||
                        has_capability('moodle/course:manageactivities', $ctx, $userid) ||
                        has_capability('moodle/course:activityvisibility', $ctx, $userid));
                // Activity that is not available, not hidden from course page and has availability
                // info is actually visible on the course page (with availability info and without a link).
                if (!$uservisible && $cm->visibleoncoursepage && $availableinfo) {
                    $uservisibleoncoursepage = true;
                }
                */

                /**
                 * "Let module make dynamic changes at this point"
                 *      - we can't really check it here, so, let's hope, that there are nothing relevant
                 */
                //$cm->call_mod_function('cm_info_dynamic');
            } while(false);

            $res = [$uservisible, $availableinfo, $has_view];
            static::g_set(__FUNCTION__, [$cmid, $userid], $res);
        }

        return $res;
    }

    /**
     * Return visibility data by course modules (cms) and users
     * Activity visibility checks much faster (around 10 times) when users - in outer circle, so,
     *  if in your case users should be in the inner circle, there will be more quickly to get visibility data in separate loop,
     *  for example by this function.
     *
     * @param array|object[]|\cm_info[]|numeric[] $cm_or_ids                - list of kica items
     * @param array|object[]|numeric[]            $users_or_ids             - list of users or users ids
     * @param bool                                $by_userid_cmid           - (optional) if true, result array will be by userid and cmid
     * @param bool                                $unavailable_as_invisible - (optional) if true, then with false uservisible - return false, despite available info
     * @param bool                                $check_global_visibility  - (optional) if true, return false, if loaded cm (or $USER) can't see this activity, despite $userid
     *
     *
     * @return array [$cmid => [$userid => true]] or [$userid => [$cmid => true]]
     */
    static public function get_cm_visibility_data($cm_or_ids, $users_or_ids, $by_userid_cmid=false,
        $unavailable_as_invisible=false, $check_global_visibility=false){
        $visibility_data = [];
        foreach ($users_or_ids as $user_or_id){
            $userid = static::get_id($user_or_id);
            foreach ($cm_or_ids as $cm_or_id){
                if (static::get_cm_visibility_by_user($cm_or_id, $user_or_id, $unavailable_as_invisible, $check_global_visibility)){
                    $cmid = static::get_id($cm_or_id);
                    if ($by_userid_cmid){
                        $visibility_data[$userid][$cmid] = true;
                    } else {
                        $visibility_data[$cmid][$userid] = true;
                    }
                }
            }
        }

        return $visibility_data;
    }

    /**
     * Checks whether mod/...:view capability restricts the user's access.
     * @see \cm_info::is_user_access_restricted_by_capability()
     * #core_moodle - you have to update this function to moodle core functionality
     *
     * @param \cm_info|numeric $cm_or_id
     * @param object|numeric   $user_or_id
     *
     * @return bool True if the user access is restricted.
     */
    static public function cm_is_user_access_restricted_by_capability($cm_or_id, $user_or_id=null){
        $cm = static::get_cm_by_cmorid($cm_or_id);
        $capability = 'mod/' . $cm->modname . ':view';
        $capabilityinfo = get_capability_info($capability);
        if (!$capabilityinfo) {
            // Capability does not exist, no one is prevented from seeing the activity.
            return false;
        }

        // You are blocked if you don't have the capability.
        return !has_capability($capability, $cm->context, $user_or_id);
    }

    /**
     * Get cm visibility by NED logic and using custom user
     * @see _calc_activity_visibility
     *
     * WARNING: Function is supposed to be used for check many users - for one activity,
     *      if you need checks only one user, it will be better to load cm(s) for this user and uses check_activity_visible_by_cm()
     *      or load already checked activities with get_important_activities()
     * @see get_important_activities()
     *
     * @param \cm_info|numeric $cm_or_id
     * @param object|numeric   $user_or_id               - if null, use current(global) $USER by default
     * @param bool             $unavailable_as_invisible - if true, then with false uservisible - return false, despite available info
     * @param bool             $check_global_visibility  - if true, return false, if loaded cm (or $USER) can't see this activity, despite $userid
     *
     * @return bool - visibility $cm for the $user_or_id by NED rules
     */
    static public function get_cm_visibility_by_user($cm_or_id, $user_or_id=null, $unavailable_as_invisible=false, $check_global_visibility=true){
        $cmid = static::get_id($cm_or_id);
        if (!$user_or_id){
            $check_global_visibility = false;
        }
        $userid = static::get_userid_or_global($user_or_id);
        $unavailable_as_invisible = (int)$unavailable_as_invisible;
        $check_global_visibility = (int)$check_global_visibility;

        $res = static::g_get(__FUNCTION__, [$cmid, $userid, $unavailable_as_invisible, $check_global_visibility]);
        if (is_null($res)){
            do {
                if ($check_global_visibility){
                    $g_userid = static::get_userid_or_global();
                    // we do not need check global user, if $userid - is global
                    if ($g_userid != $userid){
                        $res = static::get_cm_visibility_by_user($cm_or_id, $g_userid, $unavailable_as_invisible);
                        if (!$res){
                            break;
                        }
                    }
                }

                $v_data = static::_get_visibility_data_by_cm_user($cm_or_id, $user_or_id);
                $res = static::_calc_activity_visibility($unavailable_as_invisible, ...$v_data);
            } while(false);

            static::g_set(__FUNCTION__, [$cmid, $userid, $unavailable_as_invisible, $check_global_visibility], $res);
        }

        return $res;
    }

    /**
     * Get cm visibility by NED logic and using custom user
     * @see _calc_activity_visibility
     *
     * WARNING: Function is supposed to be used for check many users - for one activity,
     *      if you need checks only one user, it will be better to load cm(s) for this user and uses check_activity_visible_by_cm()
     *      or load already checked activities with get_important_activities()
     * @see get_important_activities()
     *
     * @param \cm_info|numeric     $cm_or_id
     * @param array|object|numeric $users_or_ids             - if empty, check only for current(global) $USER by default
     * @param bool                 $unavailable_as_invisible - if true, then with false uservisible - return false, despite available info
     * @param bool                 $check_global_visibility  - if true, return false, if loaded cm (or $USER) can't see this activity
     * @param bool                 $rule_any                 - if true, check that at least one user can see activity, otherwise - that all user can see activity
     *
     * @return bool - visibility $cm for the $user_or_id by NED rules
     */
    static public function get_cm_visibility_by_userlist($cm_or_id, $users_or_ids=[], $unavailable_as_invisible=false,
        $check_global_visibility=true, $rule_any=true){
        if (empty($users_or_ids)){
            // check only global user
            return static::get_cm_visibility_by_user($cm_or_id, null, $unavailable_as_invisible, false);
        } else {
            if ($check_global_visibility){
                if (!static::get_cm_visibility_by_user($cm_or_id, null, $unavailable_as_invisible, false)){
                    return false;
                }
            }

            $users_or_ids = static::val2arr($users_or_ids);
            foreach ($users_or_ids as $user_or_id){
                if (static::get_cm_visibility_by_user($cm_or_id, $user_or_id, $unavailable_as_invisible, false)){
                    if ($rule_any){
                        return true;
                    }
                } elseif (!$rule_any){
                    return false;
                }
            }

            return !$rule_any;
        }
    }

    /**
     * NED standard check, to show activity in some list or not
     * NOTE: visibility check for user, for whom cm was loaded (or for global $USER, if you provided cm id)
     * @see get_cm_visibility_by_user()
     *
     * @param \cm_info|numeric $cm_or_id
     * @param bool             $unavailable_as_invisible - if true, then with false uservisible - return false, despite available info
     *
     * @return bool - visibility $cm for the loaded user by NED rules
     */
    static public function check_activity_visible_by_cm($cm_or_id, $unavailable_as_invisible=false){
        $cm = static::get_cm_by_cmorid($cm_or_id);
        return static::get_cm_visibility_by_user($cm, $cm->get_modinfo()->userid, $unavailable_as_invisible, false);
    }
    //endregion

    //region Get Activities (cm) by visibility
    /**
     * Use instead of get_gradable_activities to get activities for Student/Class progress
     *
     * @param numeric|object $course_or_id
     * @param numeric|object $user_or_id
     * @param bool           $use_tt_sla_option        - if true, check TT option for "only sla activities", and if it true - return only SLA activities
     * @param bool           $unavailable_as_invisible - if true, then with false uservisible - return false, despite available info
     *
     * @return \cm_info[]|array
     */
    static public function get_important_activities($course_or_id, $user_or_id=null, $use_tt_sla_option=false, $unavailable_as_invisible=false){
        $courseid = static::get_courseid_or_global($course_or_id);
        $userid = static::get_userid_or_global($user_or_id);

        if ($use_tt_sla_option){
            $use_tt_sla_option = (int)static::get_site_and_course_block_config($course_or_id, C::TT_NAME, 'excludewithoutsla');
        } else {
            $use_tt_sla_option = 0;
        }
        $unavailable_as_invisible = (int)$unavailable_as_invisible;
        $g_key = [$courseid, $userid, $unavailable_as_invisible, $use_tt_sla_option];
        $res = static::g_get(__FUNCTION__, $g_key);

        if (is_null($res)){
            $cms = static::get_course_cms($courseid, $userid);
            $res = [];

            foreach ($cms as $key => $cm){
                if (!static::get_cm_visibility_by_user($cm, $userid, $unavailable_as_invisible, false)) continue;
                if ($use_tt_sla_option && !static::cm_is_sla($cm, $course_or_id)) continue;

                $res[$key] = $cm;
            }

            static::g_set(__FUNCTION__, $g_key, $res);
        }

        return $res;
    }

    /**
     * Use instead of get_gradable_activities to get activities for Student/Class progress
     * if you need it with some filter activities list
     * Alias for get_important_activities_by_users_and_compare_list()
     * @see get_important_activities_by_users_and_compare_list()
     *
     * @param object|numeric $courseid
     * @param object|numeric $user_or_id
     * @param array          $compare_list
     * @param bool           $use_tt_sla_option        - if true, check TT option for "only sla activities", and if it true - return only SLA activities
     * @param bool           $unavailable_as_invisible - if true, then with false uservisible - return false, despite available info
     *
     * @return \cm_info[]|array
     */
    static public function get_important_activities_by_compare_list($courseid, $user_or_id=null, $compare_list=[],
        $use_tt_sla_option=false, $unavailable_as_invisible=false){
        return static::get_important_activities_by_users_and_compare_list($courseid, $user_or_id, null, $compare_list,
            $use_tt_sla_option, $unavailable_as_invisible);
    }

    /**
     * Use instead of get_gradable_activities to get activities for Student/Class progress
     * if you need it with some filter user list
     * Alias for get_important_activities_by_users_and_compare_list()
     * @see get_important_activities_by_users_and_compare_list()
     *
     * @param object|numeric   $courseid
     * @param object|numeric   $main_user_or_id          - main user, for whom cm will be loaded (global $USER by default)
     * @param array|int|object $users_or_ids             - list of users or userids (or single user/id)
     * @param bool             $use_tt_sla_option        - if true, check TT option for "only sla activities", and if it true - return only SLA activities
     * @param bool             $unavailable_as_invisible - if true, then with false uservisible - return false, despite available info
     * @param bool             $rule_any                 - if true, check that at least one user can see activities, otherwise - that all user can see activity
     *
     * @return \cm_info[]|array
     */
    static public function get_important_activities_by_users($courseid, $main_user_or_id=null, $users_or_ids=[],
        $use_tt_sla_option=false, $unavailable_as_invisible=false, $rule_any=true){
        return static::get_important_activities_by_users_and_compare_list($courseid, $main_user_or_id, $users_or_ids, null,
            $use_tt_sla_option, $unavailable_as_invisible, $rule_any);
    }

    /**
     * Use instead of get_gradable_activities to get activities for Student/Class progress
     * if you need it with some filter user list by global $USER as main user (teacher)
     * Alias @see data_util::get_important_activities_by_users_and_compare_list()
     *
     * If you need to specify other main user than global, you can use get_important_activities_by_users()
     * @see get_important_activities_by_users()
     *
     * @param object|numeric   $courseid
     * @param array|int|object $users_or_ids             - list of users or userids (or single user/id)
     * @param bool             $use_tt_sla_option        - if true, check TT option for "only sla activities", and if it true - return only SLA activities
     * @param bool             $unavailable_as_invisible - if true, then with false uservisible - return false, despite available info
     * @param bool             $rule_any                 - if true, check that at least one user can see activities, otherwise - that all user can see activity
     *
     * @return \cm_info[]|array
     */
    static public function get_important_activities_for_users($courseid, $users_or_ids=[],
        $use_tt_sla_option=false, $unavailable_as_invisible=false, $rule_any=true){
        return static::get_important_activities_by_users_and_compare_list($courseid, null, $users_or_ids, null,
            $use_tt_sla_option, $unavailable_as_invisible, $rule_any);
    }

    /**
     * Use instead of get_gradable_activities to get activities for Student/Class progress
     *      if you need it with some filter user list and filter compare list
     *
     * @param object|numeric   $course_or_id
     * @param object|numeric   $main_user_or_id          - main user, for whom cm will be loaded (global $USER by default)
     * @param array|int|object $users_or_ids             - list of users or userids (or single user/id)
     * @param array|null       $compare_list             - you get empty result, if $compare_list is empty, use NULL to not check this variable
     * @param bool             $use_tt_sla_option        - if true, check TT option for "only sla activities", and if it true - return only SLA activities
     * @param bool             $unavailable_as_invisible - if true, then with false uservisible - return false, despite available info
     * @param bool             $rule_any                 - if true, check that at least one user can see activities, otherwise - that all user can see activity
     *
     * @return \cm_info[]|array
     */
    static public function get_important_activities_by_users_and_compare_list($course_or_id, $main_user_or_id=null, $users_or_ids=[],
        $compare_list=null, $use_tt_sla_option=false, $unavailable_as_invisible=false, $rule_any=true){
        $check_compare_list = !is_null($compare_list);
        $check_users = !empty($users_or_ids);
        if ($check_compare_list && empty($compare_list)){
            return [];
        }

        $activities = static::get_important_activities($course_or_id, $main_user_or_id, $use_tt_sla_option, $unavailable_as_invisible);
        if (!$check_compare_list && !$check_users){
            return $activities;
        }

        $users_or_ids = static::val2arr($users_or_ids);
        $user_list = $check_users ? static::val2arr($users_or_ids) : [null];
        $skip_activity = [];
        $res = [];

        /**
         * It's important to get users in the outer circle as
         * activity visibility checks much faster (around 10 times) when users in the outer circle
         * (and activities - in the inner circle)
         */
        foreach ($user_list as $user_or_id){
            foreach ($activities as $key => $cm){
                if ($check_compare_list && !isset($compare_list[$cm->id])) continue;
                if (!empty($res[$cm->id]) || !empty($skip_activity[$cm->id])) continue;

                $add = true;
                if ($check_users){
                    $add = !$rule_any;
                    if (static::get_cm_visibility_by_user($cm, $user_or_id, $unavailable_as_invisible, false)){
                        if ($rule_any){
                            $res[$key] = $cm;
                            continue;
                        }
                    } elseif (!$rule_any){
                        $skip_activity[$key] = true;
                        continue;
                    }
                }

                if ($add){
                    $res[$key] = $cm;
                }
            }
        }

        return $res;
    }
    //endregion

    //region CM & Grades
    /**
     * More grade methods you can find in the {@see \local_ned_controller\shared\grade_util}
     * But here there are some specific cm/mod grade methods
     */

    /**
     * Check user capabilities to grade cm.
     *
     * @param numeric|object|\cm_info $cm_or_id
     * @param numeric|object          $user_or_id (optional) A user, who will do the editing. By default, (null) checks for the current $USER.
     *
     * @return bool
     */
    static public function cm_can_grade_cm($cm_or_id, $user_or_id=null){
        $cm = static::get_cm_by_cmorid($cm_or_id);
        if (!$cm || !$cm->has_view() || !static::get_cm_visibility_by_user($cm, $user_or_id, true, false)){
            return false;
        }

        $cap = static::mod_get_grade_capability($cm->modname);
        if (empty($cap)) return false;

        if (is_array($cap)) return static::has_any_capability($cap, $cm->context, $user_or_id);
        else return static::has_capability($cap, $cm->context, $user_or_id);
    }

    /**
     * Get mod grade capability
     *
     * @param string $modname
     *
     * @return string|string[]|false - string for capability full name (or array of them), or false if activity doesn't support grades
     */
    static public function mod_get_grade_capability($modname){
        switch ($modname){
            case C::ASSIGN:
                return 'mod/assign:grade';
            case C::QUIZ:
                return 'mod/quiz:grade';
            case C::FORUM:
                return ['mod/forum:grade', 'mod/forum:rate'];
            case C::JOURNAL:
                return 'mod/journal:manageentries';
            case C::H5PACTIVITY:
                return 'mod/hvp:viewallresults';
            case C::FEEDBACK:
                return false;
        }

        // Try to find capabilities for all others activities
        $caps = [];
        if (static::mod_supports_grades($modname)){
            $cap = "mod/$modname:grade";
            if (get_capability_info($cap)){
                $caps[] = $cap;
            } else {
                // we can't find a special activity capability, so use the moodle general one
                $caps[] = 'moodle/grade:edit';
            }
        }
        if (static::mod_supports_rates($modname)){
            $cap = "mod/$modname:rate";
            if (get_capability_info($cap)){
                $caps[] = $cap;
            } else {
                // we can't find a special activity capability, so use the moodle general one
                $caps[] = 'moodle/rating:rate';
            }
        }

        if (empty($caps)) return false;
        elseif (count($caps) == 1) return reset($caps);
        else return $caps;
    }

    /**
     * Check, that activity type has grade feature
     *
     * @param string $modname
     *
     * @return bool
     */
    static public function mod_supports_grades($modname){
        return static::mod_supports_feature($modname, FEATURE_GRADE_HAS_GRADE);
    }

    /**
     * Check, that activity type has rate feature
     *
     * @param string $modname
     *
     * @return bool
     */
    static public function mod_supports_rates($modname){
        return static::mod_supports_feature($modname, FEATURE_RATE);
    }

    /**
     * Get link to the grader page, which redirect to real cm grader page
     *
     * @param \cm_info|object|numeric $cm_or_id
     * @param object|numeric          $user_or_id
     * @param object|numeric          $course_or_id - (optional) course of the CM
     * @param array                   $add_params   - (optional) add params to the URL
     *
     * @return \moodle_url
     */
    static public function cm_get_grader_url($cm_or_id, $user_or_id, $course_or_id=null, $add_params=[]){
        $cmid = static::get_id($cm_or_id);
        $userid = static::get_id($user_or_id);
        if (empty($course_or_id) && is_object($cm_or_id)){
            $courseid = $cm_or_id->course ?? null;
        } else {
            $courseid = static::get_id($course_or_id);
        }

        $params = [static::PAR_CM => $cmid, static::PAR_USER => $userid, static::PAR_COURSE => $courseid];
        if (!empty($add_params)){
            $params = array_merge($params, $add_params);
        }

        return static::$C::url('~grade_page.php', $params);
    }
    //endregion

    //region CM & Tags

    /**
     * @param int|\cm_info|object   $cm_or_id   - Id of course-module, or database object
     *
     * @return string[]|array tags[$id => $name]
     */
    static public function cm_get_tags($cm_or_id){
        $cmid = static::get_id($cm_or_id);
        $tags = static::g_get(__FUNCTION__, $cmid);
        if (is_null($tags)){
            $tags = \core_tag_tag::get_item_tags_array('core', 'course_modules', $cmid);
            static::g_set(__FUNCTION__, $cmid, $tags);
        }

        return $tags;
    }

    /**
     * Get course modules ids by tag names OR tag ids
     *
     * @param array   $tags_name - array of raw (human read) tag names
     * @param array   $tags_id - array of tag id
     * @param object|numeric $course_or_id
     *
     * @return array [cmid => cmid]
     */
    static public function cmids_get_by_tags($tags_name=[], $tags_id=[], $course_or_id=null){
        global $DB;
        if (empty($tags_name) && empty($tags_id)){
            return [];
        }

        $tags_name = static::val2arr($tags_name);
        $tags_id = static::val2arr($tags_id);
        $courseid = static::get_id($course_or_id);

        $select = 'DISTINCT cm.id';
        $from = "
            JOIN {tag_instance} AS ti
                ON ti.tagid = tag.id
                AND ti.component = 'core'
                AND ti.itemtype = 'course_modules'
            JOIN {course_modules} AS cm
                ON cm.id = ti.itemid
        ";
        $where = [];
        $params = [];
        if (!empty($tags_name)){
            [$tg_sql, $tg_params] = $DB->get_in_or_equal($tags_name, SQL_PARAMS_NAMED, 'tag_name');
            $where[] = 'tag.rawname '.$tg_sql;
        } else {
            [$tg_sql, $tg_params] = $DB->get_in_or_equal($tags_id, SQL_PARAMS_NAMED, 'tag_ids');
            $where[] = 'tag.id '.$tg_sql;
        }
        $params = array_merge($params, $tg_params);

        if ($courseid){
            $where[] = 'cm.course = :courseid';
            $params['courseid'] = $courseid;
        }

        $sql = static::sql_generate($select, $from, 'tag', 'tag', $where);
        $records = $DB->get_records_sql($sql, $params);
        $cmids = array_keys($records);

        return array_combine($cmids, $cmids);
    }

    /**
     * Return list of course module ids by Final Tag and course id
     *
     * @param numeric $courseid
     *
     * @return array [cmid => cmid]
     */
    static public function cmids_get_final($courseid){
        $res = static::g_get(__FUNCTION__, $courseid);
        if (is_null($res)){
            $block_config = static::get_site_and_course_block_config($courseid, C::TT_NAME);
            if (!empty($block_config->fe_coursecompletion)){
                $tags = $block_config->fe_tags ?? [];
            } else {
                $tags = [];
            }
            $res = static::cmids_get_by_tags($tags, [], $courseid);
            static::g_set(__FUNCTION__, $courseid, $res);
        }

        return $res ?: [];
    }

    /**
     * Check, that this course-module is final activity (has final tags)
     * @see \local_ned_controller\shared\tt_and_sm::cmids_get_final()
     *
     * @param \cm_info|object|numeric $cm_or_id
     * @param object|numeric          $course_or_id - optional course (or id) if it already loaded,
     *                                              otherwise try to get it from the cm_info object
     *
     * @return bool
     */
    static public function cm_is_final($cm_or_id, $course_or_id=null){
        $courseid = static::get_id($course_or_id);
        if (!$courseid){
            $cm = static::get_cm_by_cmorid($cm_or_id);
            if (!$cm) return false;

            $courseid = $cm->course;
        }

        $cmid = static::get_id($cm_or_id);
        if (!$cmid) return false;

        $final_cmids = static::cmids_get_final($courseid);
        return isset($final_cmids[$cmid]);
    }

    /**
     * Return list of course module ids by midterm tag and course id
     *
     * @param numeric $courseid
     *
     * @return array [cmid => cmid]
     */
    static public function cmids_get_midterm($courseid){
        $res = static::g_get(__FUNCTION__, $courseid);
        if (is_null($res)){
            $res = static::cmids_get_by_tags([C::TAG_MIDTERM], [], $courseid);
            static::g_set(__FUNCTION__, $courseid, $res);
        }

        return $res ?: [];
    }

    /**
     * Check, that this course-module is midterm activity (has midterm tags)
     * @see \local_ned_controller\shared\tt_and_sm::cmids_get_midterm()
     *
     * @param \cm_info|object|numeric $cm_or_id
     * @param object|numeric          $course_or_id - optional course (or id) if it already loaded,
     *                                              otherwise try to get it from the cm_info object
     *
     * @return bool
     */
    static public function cm_is_midterm($cm_or_id, $course_or_id=null){
        $courseid = static::get_id($course_or_id);
        if (!$courseid){
            $cm = static::get_cm_by_cmorid($cm_or_id);
            if (!$cm) return false;

            $courseid = $cm->course;
        }

        $cmid = static::get_id($cm_or_id);
        if (!$cmid) return false;

        $midterm_cmids = static::cmids_get_midterm($courseid);
        return isset($midterm_cmids[$cmid]);
    }

    /**
     * Return list of course module ids by SLA tags and course id
     *
     * @param numeric $courseid
     *
     * @return array [cmid => cmid]
     */
    static public function cmids_get_sla($courseid){
        $res = static::g_get(__FUNCTION__, $courseid);
        if (is_null($res)){
            $res = static::cmids_get_by_tags(C::SLA_TAGS, [], $courseid);
            static::g_set(__FUNCTION__, $courseid, $res);
        }

        return $res ?: [];
    }

    /**
     * Check, that this course-module is SLA activity (has SLA tags)
     * @see \local_ned_controller\shared\tt_and_sm::cmids_get_sla()
     *
     * @param \cm_info|object|numeric $cm_or_id
     * @param object|numeric          $course_or_id - optional course (or id) if it already loaded,
     *                                              otherwise try to get it from the cm_info object
     *
     * @return bool
     */
    static public function cm_is_sla($cm_or_id, $course_or_id=null){
        $courseid = static::get_id($course_or_id);
        if (!$courseid){
            $cm = static::get_cm_by_cmorid($cm_or_id);
            if (!$cm) return false;

            $courseid = $cm->course;
        }

        $cmid = static::get_id($cm_or_id);
        if (!$cmid) return false;

        $sla_cmids = static::cmids_get_sla($courseid);
        return isset($sla_cmids[$cmid]);
    }

    /**
     * Return list of course module ids by CTA tag and course id
     *
     * @param numeric $courseid
     *
     * @return array [cmid => cmid]
     */
    static public function cmids_get_cta($courseid){
        $res = static::g_get(__FUNCTION__, $courseid);
        if (is_null($res)){
            $res = static::cmids_get_by_tags([C::TAG_CTA], [], $courseid);
            static::g_set(__FUNCTION__, $courseid, $res);
        }

        return $res ?: [];
    }

    /**
     * Check, that this course-module is CTA activity (has CTA tag)
     * @see \local_ned_controller\shared\tt_and_sm::cmids_get_cta()
     *
     * @param \cm_info|object|numeric $cm_or_id
     * @param object|numeric          $course_or_id - optional course (or id) if it already loaded,
     *                                              otherwise try to get it from the cm_info object
     *
     * @return bool
     */
    static public function cm_is_cta($cm_or_id, $course_or_id=null){
        $courseid = static::get_id($course_or_id);
        if (!$courseid){
            $cm = static::get_cm_by_cmorid($cm_or_id);
            if (!$cm) return false;

            $courseid = $cm->course;
        }

        $cmid = static::get_id($cm_or_id);
        if (!$cmid) return false;

        $cta_cmids = static::cmids_get_cta($courseid);
        return isset($cta_cmids[$cmid]);
    }

    /**
     * Checked, that activity is "Test" or "Assignment"
     * Both of them are summative assign or quiz,
     *  and are differentiated from other activities by the presence or absence of specific tags
     *
     * @param \cm_info|object|numeric $cm_or_id
     *
     * @return bool
     */
    public static function cm_is_test_or_assignment($cm_or_id){
        $cm = static::get_cm_by_cmorid($cm_or_id);
        if (!$cm || ($cm->modname !== 'assign' && $cm->modname !== 'quiz')) return false;

        $tags = static::cm_get_tags($cm_or_id);
        if (empty($tags) || !in_array(C::TAG_SUMMATIVE, $tags)) return false;

        $has_test_or_exam_tag = in_array(C::TAG_UNIT_TEST, $tags) || in_array(C::TAG_FINAL_EXAM, $tags);
        if ($cm->modname == 'assign'){
            return !$has_test_or_exam_tag;
        } else {
            // $cm->modname == 'quiz'
            return $has_test_or_exam_tag;
        }
    }
    //endregion

    //region Activity output
    /**
     * Get url for cm
     * If theme wasn't set up, try to get icon without theme initialization
     * If you need to set up theme any way, you can call {@see \cm_info::get_icon_url()} directly
     *
     * @param \cm_info|object|numeric $cm_or_id
     *
     * @return \moodle_url|string
     */
    static public function cm_get_icon_url($cm_or_id){
        $cm = static::get_cm_by_cmorid($cm_or_id);
        if (static::is_theme_initialised()){
            $icon_url = $cm->get_icon_url();
        } else {
            $icon_url = new \moodle_url("/mod/$cm->modname/pix/icon.png");
        }

        return $icon_url;
    }

    /**
     * Return name/icon/link for course module
     * It's base method with a lot of parameters, more for other methods than for common use.
     *
     * Usually you can use more simple methods:
     * @see \local_ned_controller\shared\output::q_cm_link() - simple cm name & link
     * @see \local_ned_controller\shared\output::q_cm_student_link() - link to cm as for student view (checks student restriction)
     * @see \local_ned_controller\shared\output::q_cm_grade_link() - link to the MM page with cm text
     * @see \local_ned_controller\shared\output::q_cm_name() - get icon+name or only cm name (without link)
     * @see \local_ned_controller\shared\output::mod_link() - get icon+name with icon and html settings
     *
     * @param \cm_info|int|string $cm_or_id        - Id of course-module, database object or cm-info
     * @param numeric|object|null $course_or_id    - Optional course object (or its id) if already loaded
     * @param numeric|object|null $user_or_id      - Optional user object (or its id; default = current)
     * @param bool                $without_icon    - if true, don't load cm icon
     * @param bool                $without_text    - if true, don't use cm name
     * @param bool                $dimmed          - if true, return only name and/or icon, without link
     * @param bool                $without_wrapper - if true, and result will be without link (or $dimmed) - it will be without html wrappers
     * @param string              $add_text        - add some text to result
     * @param string|\moodle_url  $replace_url     - set other url to text; if result has icon, it would still have cm url
     * @param numeric|string      $icon_size       - icon size for html attributes
     * @param array|string        $add_class       - additional classes, if result will be a link or has html wrapper
     * @param array               $html_attributes - additional html attributes, if result will be a link or has html wrapper
     *
     * @return string - html string / cm icon / cm name / cm link
     */
    static public function cm_get_text_icon_link($cm_or_id, $course_or_id=null, $user_or_id=null, $without_icon=false, $without_text=false,
        $dimmed=false, $without_wrapper=false, $add_text='', $replace_url=null, $icon_size=null, $add_class=[], $html_attributes=[]){

        $cm = static::get_cm_by_cmorid($cm_or_id, $course_or_id, $user_or_id);
        if (!$cm) return '';

        if ($without_icon && $without_text && empty($add_text)){
            static::debugging('You can\'t use cm_get_text_icon_link with empty content');
            $without_text = false;
        }

        $cm_name = $cm_icon = '';
        if (!$without_text){
            $cm_name = $cm->get_formatted_name();
        }
        if (!$without_icon){
            $attr = [];
            if ($icon_size){
                $attr['height'] = $attr['width'] = $icon_size;
            }
            $cm_icon = static::img(static::cm_get_icon_url($cm), 'icon', '', $attr, $cm_name);
        }

        $use_link = !$dimmed && $cm->uservisible && static::course_can_view_course_info($cm->course, $user_or_id);
        if ($use_link || !$without_wrapper){
            $add_class = static::val2arr($add_class);
            $html_attributes = $html_attributes ?: [];
            if (!isset($html_attributes['title'])){
                $html_attributes['title'] = $cm_name ?: $cm->get_formatted_name();
            }
        }

        // Set base value
        $add_text = $add_text ?: '';
        if ($use_link && !empty($replace_url)){
            $value = '';
            if ($cm_icon){
                $value .= static::link($cm->url, $cm_icon, '', $cm_name ? [] : $html_attributes);
            }
            if ($cm_name){
                $value .= static::link(new \moodle_url($replace_url), $cm_name.$add_text, '', $html_attributes);
            }
            $use_link = false;
        } else {
            $value = $cm_icon.$cm_name.$add_text;
            $replace_url = null;
        }

        // Return result
        if ($use_link && !$replace_url){
            return static::link($cm->url, $value, $add_class, $html_attributes);
        } elseif ($without_wrapper){
            return $value;
        } else {
            if (!$replace_url){
                $add_class[] = 'dimmed dimmed_text';
            }
            return static::span($value, $add_class, $html_attributes);
        }
    }

    /**
     * Get formatted availability info
     *
     * @param \cm_info|numeric $cmorid
     * @param object|numeric   $user_or_id - by default check current $USER
     * @param bool             $plain_text - return plain text (true) or HTML (false)
     *
     * @return string
     */
    static public function cm_get_cm_unavailable_info($cmorid, $user_or_id=null, $plain_text=false){
        [, $availableinfo,] = static::_get_visibility_data_by_cm_user($cmorid, $user_or_id);
        if (empty($availableinfo)) return $availableinfo;

        $courseid = static::get_courseid_by_cmorid($cmorid);
        $availableinfo = \core_availability\info::format_info($availableinfo, $courseid);
        if ($plain_text){
            $availableinfo = trim(strip_tags(str_replace('<li>', PHP_EOL."• ", $availableinfo)));
        }

        return $availableinfo;
    }
    //endregion

    //region Utils
    /**
     * Return array [cmid => cmid] from the array of "something" with cm data
     *
     * @param numeric|object|\cm_info|array|object[]|\cm_info[] $cms_or_ids
     *
     * @return array|int[] - [cmid => cmid]
     */
    static public function cm_get_cmids_filter($cms_or_ids){
        $some_array = static::val2arr($cms_or_ids);
        $key = key($some_array);
        $val = reset($some_array);
        $cmids = [];
        if (is_numeric($val)){
            if ($key && $key == $val){
                // it looks like already cmids array
                $cmids = $some_array;
            } else {
                // list of cmids
                $cm_keys = array_values($some_array);
                $cmids = array_combine($cm_keys, $cm_keys);
            }
        } elseif (is_object($val) && !empty($val->id)){
            if ($key && $val->id == $key){
                // it looks like [cmid => cm]
                $cm_keys = array_keys($some_array);
                $cmids = array_combine($cm_keys, $cm_keys);
            } else {
                // oh, it's list of the cm, we will need to go through it
                foreach ($some_array as $cm){
                    $cmids[$cm->id] = $cm->id;
                }
            }
        }

        return $cmids;
    }

    /**
     * Search activity by its name in some course
     *
     * @param numeric|object $course_or_id
     * @param string $name - activity name to search
     *
     * @return \cm_info|null - cm object or null, if it finds nothing
     */
    static public function cm_find_activity_by_name($course_or_id=null, $name=''){
        if (empty($name) || empty($course_or_id)) return null;

        $name = s(trim(strip_tags($name)));
        if (empty($name)) return null;

        // it quicker than search name in the DB
        $cms = static::get_course_cms($course_or_id);
        if (empty($cms)) return null;

        foreach ($cms as $cm){
            if (s(trim(strip_tags($cm->name))) == $name) return $cm;
        }

        return null;
    }

    /**
     * Check, that activity type has some feature, with cache support
     * @see FEATURE_GRADE_HAS_GRADE for features list
     *
     * @param string $modname - Name of module e.g. 'forum'
     * @param string $feature - FEATURE_xx constant for requested feature
     *
     * @return bool
     */
    static public function mod_supports_feature($modname, $feature){
        if (empty($modname) || empty($feature)) return false;

        $res = static::g_get(__FUNCTION__, [$modname, $feature]);
        if (is_null($res)){
            $res = plugin_supports('mod', $modname, $feature, false) ?: false;
            static::g_set(__FUNCTION__, [$modname, $feature], $res);
        }
        return $res;
    }
    //endregion
}
