<?php
/**
 * @package    local_ned_controller
 * @subpackage shared
 * @category   NED
 * @copyright  2022 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller\shared;

defined('MOODLE_INTERNAL') || die();

/**
 * Trait controller_util
 *
 * @package local_ned_controller\shared
 */
trait controller_util {
    use plugin_dependencies, util;

    /**
     * Save admin alerts data to the config
     *
     * @param array $admin_alerts
     *
     * @return bool
     */
    static protected function _ctrl_admin_alert_save_config($admin_alerts){
        if (empty($admin_alerts)){
            return static::$C::unset_config(C::CONFIG_ADMIN_ALERTS);
        }

        return static::$C::set_config(C::CONFIG_ADMIN_ALERTS, json_encode($admin_alerts));
    }

    /**
     * Return all admin alerts
     *
     * @return array - [plugin => [key => [$text, $type]]]
     */
    static public function ctrl_admin_alert_get_all(){
        $admin_alerts = static::$C::get_config(C::CONFIG_ADMIN_ALERTS);
        if (empty($admin_alerts)){
            return [];
        }

        return json_decode($admin_alerts, true);
    }

    /**
     * Return all admin alerts by plugin or only one by key
     *
     * @param string|null $key - if key provided, return only notification by this key
     * @param string|null $plugin - optional plugin name (i.e. 'local_ned_controller'), $PLUGIN_NAME used by default
     *
     * @return array - return empty array, if nothing found
     */
    static public function ctrl_admin_alert_get($key=null, $plugin=null){
        $plugin = $plugin ?: static::$PLUGIN_NAME;
        $admin_alerts = static::ctrl_admin_alert_get_all();
        if (empty($admin_alerts[$plugin])){
            return [];
        }

        return $key ? ($admin_alerts[$plugin][$key] ?? []) : $admin_alerts[$plugin];
    }

    /**
     * Add notification block to the admin Notifications page
     *
     * @param string        $key - key to control notification data
     * @param string        $type - use one of the NOTIFY types, NOTIFY_INFO by default
     * @param string|null   $text - if empty, try to translate $key by added plugin
     * @param string|null   $plugin - optional plugin name (i.e. 'local_ned_controller'), $PLUGIN_NAME used by default
     *
     * @return bool
     */
    static public function ctrl_admin_alert_add($key, $type=C::NOTIFY_INFO, $text=null, $plugin=null){
        $admin_alerts = static::ctrl_admin_alert_get_all();
        if (empty($text)){
            $text = '';
        }

        $plugin = $plugin ?: static::$PLUGIN_NAME;
        if (!isset($admin_alerts[$plugin])){
            $admin_alerts[$plugin] = [];
        }
        // Admin page has some differences from the usual notifications
        if ($type == C::NOTIFY_ERROR){
            $type = 'danger';
        }

        $admin_alerts[$plugin][$key] = [$text, $type];
        return static::_ctrl_admin_alert_save_config($admin_alerts);
    }

    /**
     * Remove notification block from the admin Notifications page by key, or all for plugin
     *
     * @param string|null $key - key to control notification data
     * @param string|null $plugin - optional plugin name (i.e. 'local_ned_controller'), $PLUGIN_NAME used by default
     *
     * @return bool
     */
    static public function ctrl_admin_alert_rem($key=null, $plugin=null){
        $admin_alerts = static::ctrl_admin_alert_get_all();
        $plugin = $plugin ?: static::$PLUGIN_NAME;
        if ($key){
            if (!isset($admin_alerts[$plugin][$key])){
                return true;
            }

            unset($admin_alerts[$plugin][$key]);
            if (empty($admin_alerts[$plugin])){
                return static::ctrl_admin_alert_rem(null, $plugin);
            }
        } else {
            if (!isset($admin_alerts[$plugin])){
                return true;
            }

            unset($admin_alerts[$plugin]);
            if (empty($admin_alerts)){
                return static::ctrl_admin_alert_clear();
            }
        }

        return static::_ctrl_admin_alert_save_config($admin_alerts);
    }

    /**
     * Clear all Admin Alerts
     *
     * @return bool
     */
    static public function ctrl_admin_alert_clear(){
        return static::_ctrl_admin_alert_save_config([]);
    }

    /**
     * Check that current server is Maple
     *
     * @return bool
     */
    public static function ctrl_server_is_maple(){
        $home_url = new \moodle_url('/');
        return $home_url->get_host() == 'maple.rosedaleedu.com';
    }
}
