<?php
/**
 * @package    local_ned_controller
 * @subpackage shared
 * @category   NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller\shared;

defined('MOODLE_INTERNAL') || die();

/**
 * Trait db_util
 *
 * @package local_ned_controller\shared
 */
trait db_util {
    use util;

    /**
     * @return \moodle_database|\mysqli_native_moodle_database|\readonlydriver
     */
    static public function db(){
        global $DB;
        return $DB;
    }

    /**
     * Remove all empty values from param array
     *
     * @param array $params
     *
     * @return array
     */
    static public function sql_filter_params($params){
        foreach ($params as $key => $value){
            if (empty($value)){
                unset($params[$key]);
            }
        }

        return $params;
    }

    /**
     * Return sql with FROM, JOIN and WHERE for getting user_enrolments
     *  if sent $params, necessary data will be saved here
     *
     * Courseid and cmid work by next rules:
     *  if null - check none of them
     *  if false (0) - check all of them
     *  if true - check by this id
     *  Exception: if cmid === true (bool), then it try to use courseid, but will not check course enrollments
     *
     * Examples:
     *  Courseid = null, cmid = null - return empty result
     *  Courseid = false, cmid = false - return users, with such role anywhere on site
     *  Courseid = null, cmid = false - return users, with such role on any cm
     *  Courseid = false, cmid = null - return users, with such role on any course
     *  Courseid = false, cmid = true - return users, with such role on cm + course of this cm
     *  Courseid = true, cmid = false - return users, with such role on course + any from course cm
     *  Courseid = null, cmid = true - return users, with such role only on cm
     *  Courseid = true, cmid = null - return users, with such role only on course
     *  Courseid = true, cmid = true - if cmid is id - return users, with such role on course + current cm,
     *      otherwise users with role on any cm from this course, but not course itself
     *
     * @param string             $select    - by default return user_enrolments id,
     *                                      use "ra_cm.id" if you need to get cm->id data (instead of "cm.id"), when $cmid is not NULL
     * @param array|string       $where     - array of conditions for AND
     * @param array              $params    - array for query params, new params also will be saved here
     * @param string|array       $rolenames - shortname(s) from the role table
     * @param numeric|false|null $courseid  - course id, null to check nothing, false to check all, or id
     * @param numeric|array      $groupids  - filter by the group
     * @param numeric|bool|null  $cmid      - course module id, null to check nothing, false to check all, true - check all by course, or id
     * @param numeric|array      $schoolids - filter by the school
     * @param bool|int           $is_active - if false - get all, if true, return only active users, if === -1 - return only inactive
     * @param string             $other_sql - any other sql string to add after WHERE conditions
     * @param string|array       $add_join  - add other tables to join, if == 'group' - will add group table to join
     * @param string             $prefix    - prefix to sql params
     *
     * @return string
     */
    static public function sql_user_enrolments($select='ue.id', $where=[], &$params=[], $rolenames=C::ROLE_STUDENT,
        $courseid=null, $groupids=[], $cmid=null, $schoolids=[], $is_active=true, $other_sql='', $add_join='', $prefix='ue_'){

        $select = $select ?: 'ue.id';
        $where = static::val2arr($where);
        $from = [];
        if ($add_join == 'group'){
            $add_group_join = true;
            $add_join = [];
        } else {
            $add_join = static::val2arr($add_join);
            $add_group_join = !empty($groupids);
        }

        if (is_null($courseid) && is_null($cmid)){
            $where []= C::SQL_NONE_COND;
        }

        $p = function($key) use (&$prefix){ return $prefix.$key; };
        $isset_param = function($key) use (&$prefix, &$params){ return isset($params[$prefix.$key]); };
        $set_param = function($key, $value) use (&$prefix, &$params){ return $params[$prefix.$key] = $value; };

        if (!empty($rolenames) && !$isset_param('rolename')){
            [$rolename_where, $rolename_params] = static::db()->get_in_or_equal($rolenames, SQL_PARAMS_NAMED, $prefix);
            $params = array_merge($params, $rolename_params);
        } else {
            $rolename_where = '= '.$p('rolename');
        }

        $from []= "
            JOIN {user} u
                ON u.id = ue.userid
            JOIN {enrol} e
                ON e.id = ue.enrolid
            JOIN {role} r 
                ON r.shortname $rolename_where
            JOIN {course} course
                ON course.id = e.courseid
        ";

        if (!empty($schoolids) && static::is_school_manager_exists()){
            $from[] = static::sql_get_school_join('u.id', 'school');
            static::sql_add_get_in_or_equal_options('school.id', $schoolids, $where, $params, $prefix);
        }

        $e_join = 'JOIN';
        $join_course = !is_null($courseid) && $cmid !== true;
        $join_cm = !is_null($cmid);
        if ($join_course && $join_cm){
            $e_join = 'LEFT JOIN';
            $where[] = "COALESCE(ra_course.id, ra_cm.id, 0) > 0";
        }

        if ($join_course){
            $from []= "
                $e_join {context} ctx_course
                    ON ctx_course.contextlevel = :{$prefix}ctx_course
                    AND ctx_course.instanceid = course.id
                $e_join {role_assignments} ra_course
                    ON ra_course.contextid = ctx_course.id
                    AND ra_course.userid = ue.userid
                    AND ra_course.roleid = r.id
            ";
            $set_param('ctx_course', CONTEXT_COURSE);
        }

        if ($join_cm){
            $from []= "
                $e_join {course_modules} cm
                    ON cm.course = course.id
                $e_join {context} ctx_cm
                    ON ctx_cm.contextlevel = :{$prefix}ctx_module
                    AND ctx_cm.instanceid = cm.id
                $e_join {role_assignments} ra_cm
                    ON ra_cm.contextid = ctx_cm.id
                    AND ra_cm.userid = ue.userid
                    AND ra_cm.roleid = r.id
            ";
            $set_param('ctx_module', CONTEXT_MODULE);
        }

        if ($courseid){
            $set_param('courseid', $courseid);
            $where[] = "course.id = :".$p('courseid');
        }
        if ($cmid && $cmid !== true){
            $set_param('cmid', $cmid);
            $where[] = "cm.id = :".$p('cmid');
        }

        if ($is_active){
            $now = C::SQL_NOW;
            $active_sql = "(
                ue.status = :{$prefix}active AND e.status = :{$prefix}enabled AND 
                ue.timestart <= $now AND (ue.timeend = 0 OR ue.timeend > $now) AND
                u.suspended = 0 AND u.deleted = 0
            )";
            $set_param('enabled', ENROL_INSTANCE_ENABLED);
            $set_param('active', ENROL_USER_ACTIVE);

            if ($is_active === -1){
                $active_sql = "NOT".$active_sql;
            }
            $where[] = $active_sql;
        }

        if ($add_group_join){
            $from[] = "
                LEFT JOIN (
                    SELECT grp.id, g_m.userid, grp.courseid
                    FROM {groups} grp
                    JOIN {groups_members} g_m
                       ON g_m.groupid = grp.id
                ) gr 
                    ON gr.courseid = e.courseid
                    AND gr.userid = ue.userid
            ";

            if (!empty($groupids)){
                static::sql_add_get_in_or_equal_options('gr.id', $groupids, $where, $params, $prefix);
            }
        }

        $select = static::arr2str($select, '', ',');
        $groupby = '';
        if (empty($other_sql) && !static::str_has($select, ['DISTINCT', ' AS '], 0, false)){
            $groupby = $select;
        }

        $from = array_merge($from, $add_join);
        $sql = static::sql_generate($select, $from, 'user_enrolments', 'ue', $where, $groupby);
        $sql .= $other_sql;

        return $sql;
    }


    /**
     * @param string|array $select
     * @param string|array $joins
     * @param string       $table_name - table name [without {}] or custom table query [with ()], can't be empty
     * @param string       $table_alias - if empty, uses first letter of table name, or "t" for custom tables
     * @param string|array $where
     * @param string|array $groupby
     * @param string|array $orderby
     * @param string|array $limit
     *
     * @return string - $sql query
     */
    static public function sql_generate($select=[], $joins=[], $table_name='', $table_alias='', $where=[], $groupby=[], $orderby=[], $limit=[]){
        [$select, $joins, $where, $groupby, $orderby, $limit] =
            self::val2arr_multi(true, $select, $joins, $where, $groupby, $orderby, $limit);

        $table_name = trim($table_name);
        if (empty($table_name)) return '';

        $select = "SELECT ". (empty($select) ? "$table_alias.*" : join(', ', $select));
        if ($table_name[0] == '(' && $table_name[-1] == ')'){
            // it's custom table
            $table = $table_name;
            $table_alias = $table_alias ?: 't';
        } else {
            $table = '{'.$table_name.'}';
            $table_alias = $table_alias ?: $table_name[0];
        }
        $from = array_merge(["\nFROM $table $table_alias"], $joins);
        $from = join("\n", $from);
        $where = static::sql_where($where);
        $groupby = !empty($groupby) ? ("\nGROUP BY " . join(',', $groupby)) : '';
        $orderby = !empty($orderby) ? ("\nORDER BY " . join(',', $orderby)) : '';
        $limit = !empty($limit) ? ("\nLIMIT " . join(',', $limit)) : '';

        return $select.$from.$where.$groupby.$orderby.$limit;
    }

    /**
     * Transform array where condition to the string
     *
     * @param array  $where
     * @param string $condition -  AND or OR
     * @param bool   $without_word_where - if true, not including WHERE in result string
     *
     * @return string
     */
    static public function sql_where($where=[], $condition="AND", $without_word_where=false){
        $where = static::val2arr($where);
        if (empty($where)) return '';

        $start = $without_word_where ? "\n" : "\nWHERE ";
        return $start . static::sql_condition($where, $condition);
    }

    /**
     * Transform array condition to the sql string
     * @see \local_ned_controller\shared\db_util::sql_where()
     *
     * @param array  $conditions
     * @param string $condition -  "AND" or "OR"
     *
     * @return string
     */
    static public function sql_condition($conditions=[], $condition="AND"){
        return !empty($conditions) ? ('((' . join(") $condition (", $conditions) . '))') : '';
    }

    /**
     * Union array sql-tables into the one sql-string
     *
     * @param string[] $tables - sql strings to union as tables
     * @param bool $custom_table - if true, add round braces () around the table [compatibility with sql_generate()]
     * @param bool $union_all - if true, uses "UNION ALL"
     *
     * @return string - sql strings with union tables
     */
    static public function sql_union_tables($tables=[], $custom_table=false, $union_all=false){
        if (empty($tables)) return "";

        $tables = static::val2arr($tables);
        $union = $union_all ? "UNION ALL" : "UNION";
        $table = "\n".join("\n$union\n", $tables)."\n";
        if ($custom_table){
            $table = '('.$table.')';
        }

        return $table;
    }

    /**
     * Add construct with some condition SQL fragment for $where & $params
     *
     * Warning: use only simple value. For arrays - {@see sql_add_get_in_or_equal_options()}
     * Also, function doesn't check param array for having such key already
     *
     * @param string $sql_field  - name of where condition
     * @param mixed  $val        - value for where condition
     * @param array  $where      - where list, if you already have it
     * @param array  $params     - params array, if you already have it
     * @param string $condition - one of the {@see C::SQL_CONDITIONS} values
     * @param string $param_name - (optional) param name for $params, if empty - uses field name
     * @param string $param_postfix - (optional) - if $param_name is empty, add it to the $sql_field to create $param_name
     *
     * @return void - result saves in the $where and $params
     */
    static public function sql_add_condition($sql_field, $val=null, &$where=[], &$params=[], $condition=C::SQL_COND_EQ,
        $param_name='', $param_postfix=''){
        if (empty($sql_field)) return;

        if (empty($param_name)){
            $param_name = str_replace('.', '_', $sql_field);
            if (!empty($param_postfix)){
                $param_name .= $param_postfix;
            }
        }

        $where[] = "$sql_field $condition :$param_name";
        $params[$param_name] = $val;
    }

    /**
     * Add construct '=' sql fragment for $where & $params
     * Alias for {@see sql_add_condition()}
     *
     * Warning: use only simple value. For arrays - {@see sql_add_get_in_or_equal_options()}
     * Also, function doesn't check param array for having such key already
     *
     * @param string $sql_field  - name of where condition
     * @param mixed  $val        - value for where condition
     * @param array  $where      - where list, if you already have it
     * @param array  $params     - params array, if you already have it
     * @param string $param_name - (optional) param name for $params, if empty - uses field name
     * @param string $param_postfix - (optional) - if $param_name is empty, add it to the $sql_field to create $param_name
     *
     * @return void - result saves in the $where and $params
     */
    static public function sql_add_equal($sql_field, $val=null, &$where=[], &$params=[], $param_name='', $param_postfix=''){
        static::sql_add_condition($sql_field, $val, $where, $params, C::SQL_COND_EQ, $param_name, $param_postfix);
    }

    /**
     * Add construct FIND_IN_SET(value, list) sql fragment for $where & $params
     *
     * Warning: use only simple value. For arrays - {@see sql_add_get_in_or_equal_options()}
     * Also, function doesn't check param array for having such key already
     *
     * @param string $sql_list   - SQL field or string with values separated by ','
     * @param mixed  $val        - value to search in the $sql_list
     * @param array  $where      - where list, if you already has it
     * @param array  $params     - params array, if you already has it
     * @param string $param_name - (optional) param name for $params, if empty - uses field name
     *
     * @return void - result saves in the $where and $params
     */
    static public function sql_add_find_in_set($sql_list, $val=null, &$where=[], &$params=[], $param_name=''){
        if (empty($sql_list)) return;

        $param_name = $param_name ?: str_replace(['.', ',', ' '], '_', $sql_list);
        $where[] = "FIND_IN_SET(:$param_name, $sql_list)";
        $params[$param_name] = $val;
    }

    /**
     * Add constructs 'IN()' or '=' sql fragment for $where & $params
     *
     * @param string      $cond_name - name of where condition
     * @param array|mixed $items     - items adding in where condition as params
     * @param array       $where     - where list, if you already has it
     * @param array       $params    - params array, if you already has it
     * @param string|null $prefix    - prefix for SQL parameters, null means default "param" value
     * @param bool        $equal     - true means we want to equate to the constructed expression, false means we don't want to equate to it.
     *
     * @return void - result saves in the $where and $params
     */
    static public function sql_add_get_in_or_equal_options($cond_name, $items, &$where=[], &$params=[], $prefix='param', $equal=true){
        $prefix = $prefix ?: 'param';
        [$col_sql, $col_params] = static::db()->get_in_or_equal($items, SQL_PARAMS_NAMED, $prefix.'_', $equal);
        $where[] = "$cond_name $col_sql";
        $params = array_merge($params, $col_params);
    }

    /**
     * Constructs 'IN()' or '=' sql fragment for all options
     * Useful for DB->get_records_select()
     *
     * @param array  $options - options to transform in where and parameters
     * @param string $prefix  - prefix for SQL parameters
     * @param array|null  $columns - key filter for options, if not null
     * @param false  $return_where_as_array - return where as array, to continue work with it, otherwise it will be string
     * @param string $condition -  AND or OR for $where options
     * @param bool   $without_word_where - if true, not including WHERE in result $where string
     *
     * @return array($where, $params)
     */
    static public function sql_get_in_or_equal_options($options, $prefix='param', $columns=null,
        $return_where_as_array=false, $condition="AND", $without_word_where=true){
        if (empty($options)){
            return [$return_where_as_array ? [] : '', []];
        }
        $columns = $columns ?? array_keys($options);
        $all_params = [];
        $where = [];

        foreach ($columns as $column){
            if (!array_key_exists($column, $options)){
                continue;
            }

            $items = $options[$column];
            if (is_array($items) && empty($items)){
                // moodle_database::get_in_or_equal() does not accept empty arrays
                continue;
            }

            $prefix_column = str_replace('.', '_', $column);
            [$col_sql, $col_params] = static::db()->get_in_or_equal($items, SQL_PARAMS_NAMED, $prefix.'_'.$prefix_column.'_');
            $where[] = "$column $col_sql";
            $all_params[] = $col_params;
        }

        if (empty($where)){
            return [$return_where_as_array ? [] : '', []];
        }

        if (!$return_where_as_array){
            $where = static::sql_where($where, $condition, $without_word_where);
        }
        $params = array_merge(...$all_params);

        return [$where, $params];
    }

    /**
     * Constructs 'IN()' or '=' sql fragment for all options
     * Adding them to your where and params arrays, and return them
     * Alias @see db_util::sql_get_in_or_equal_options()
     *
     * @param array         $options - options to transform in where and parameters
     * @param array         $where - your where array
     * @param array         $params - your params array
     * @param array|null    $columns - key filter for options, if not null
     * @param string        $prefix  - prefix for SQL parameters
     *
     * @return array($where, $params)
     */
    static public function sql_get_in_or_equal_options_list($options, &$where=[], &$params=[], $columns=null, $prefix='param'){
        [$new_where, $new_params] = static::sql_get_in_or_equal_options($options, $prefix, $columns, true);
        $where = array_merge($where, $new_where);
        $params = array_merge($params, $new_params);
        return [$where, $params];
    }

    /**
     * Return sql for join school and school members
     *
     * @param string       $userid_cond     - SQL condition to join userid data
     * @param string       $school_alias    - alias for school table
     * @param string       $member_alias    - alias for school member table
     * @param bool         $left_join       - if true, uses LEFT JOIN, simple JOIN otherwise (default)
     * @param string|array $school_add_join - add custom join condition to school table
     * @param string|array $member_add_join - add custom join condition to member table
     *
     * @return string
     */
    static public function sql_get_school_join($userid_cond='u.id', $school_alias='school', $member_alias='cohort_m', $left_join=false,
    $school_add_join='', $member_add_join=''){
        if (!static::is_school_manager_exists()) return '';

        $school_alias = $school_alias ?: 'school';
        $member_alias = $member_alias ?: 'cohort_m';

        $school_table = \local_schoolmanager\school_manager::TABLE_SCHOOL;
        $member_table = \local_schoolmanager\school_manager::TABLE_MEMBERS;
        $join = $left_join ? 'LEFT JOIN' : 'JOIN';

        $school_add_join = static::sql_condition($school_add_join);
        $school_add_join = empty($school_add_join) ? '' : ('AND '.$school_add_join);
        $member_add_join = static::sql_condition($member_add_join);
        $member_add_join = empty($member_add_join) ? '' : ('AND '.$member_add_join);

        return "$join {{$member_table}} $member_alias ON $member_alias.userid = $userid_cond $member_add_join"."\n".
            "$join {{$school_table}} $school_alias ON $school_alias.id = $member_alias.cohortid $school_add_join";
    }

    /**
     * Return sql for join group and group members
     *
     * @param string       $courseid_cond   - SQL condition to join courseid data
     * @param string       $userid_cond     - SQL condition to join userid data
     * @param string       $group_alias     - alias for group table
     * @param string       $member_alias    - alias for group member table
     * @param bool         $left_join       - if true, uses LEFT JOIN, simple JOIN otherwise (default)
     * @param string|array $group_add_join  - add custom join condition to school table
     * @param string|array $member_add_join - add custom join condition to member table
     *
     * @return string
     */
    static public function sql_get_group_join($courseid_cond='c.id', $userid_cond='u.id', $group_alias='grp', $member_alias='g_m',
        $left_join=false, $group_add_join='', $member_add_join=''){

        $group_alias = $group_alias ?: 'grp';
        $member_alias = $member_alias ?: 'g_m';

        $join = $left_join ? 'LEFT JOIN' : 'JOIN';

        $group_add_join = static::sql_condition($group_add_join);
        $group_add_join = empty($group_add_join) ? '' : ('AND '.$group_add_join);
        $member_add_join = static::sql_condition($member_add_join);
        $member_add_join = empty($member_add_join) ? '' : ('AND '.$member_add_join);

        return "$join {groups_members} $member_alias ON $member_alias.userid = $userid_cond $member_add_join"."\n".
            "$join {groups} $group_alias ON $group_alias.id = $member_alias.groupid AND $group_alias.courseid = $courseid_cond $group_add_join";
    }

    /**
     * Get SQL code to join school tables as subquery
     * In $res_names_alias you will get all user school names; In $res_code_alias you will get all user school codes.
     *  You can set separator $name_sep, which is ', ' by default.
     * In $res_ids_alias you will get all user school id. Separator for it is always ',',
     *  so you can use FIND_IN_SET() for filtering records by id
     * If you set empty $res_*, their data will be not added to query select
     *
     * @param string       $table_alias         - alias for result school-member table
     * @param string       $userid_cond         - SQL condition to join member ids
     * @param string       $res_ids_alias       - alias to store result school ids (i.e. "$table_alias.$res_ids_alias" = 1,2, ...)
     * @param string       $res_names_alias     - alias to store result school names (i.e. "$table_alias.$res_names_alias" = school1, school1, ...)
     * @param string       $res_code_alias      - alias to store result school codes (short names)
     * @param string       $name_sep            - separator for name result values
     * @param bool         $left_join           - if true, uses LEFT JOIN (default), simple JOIN otherwise
     * @param string|array $add_join_conditions - add custom join condition to result table
     * @param string|array $add_select          - add custom select code to result table
     *
     * @return string - SQL join code
     */
    static public function sql_join_school_as_subquery_table($table_alias='grp', $userid_cond='u.id',
        $res_ids_alias='school_ids', $res_names_alias='school_names', $res_code_alias='school_codes', $name_sep=', ',
        $left_join=true, $add_join_conditions='', $add_select=[]){
        if (!static::is_school_manager_exists()) return '';

        $school_table = \local_schoolmanager\school_manager::TABLE_SCHOOL;
        $member_table = \local_schoolmanager\school_manager::TABLE_MEMBERS;
        $join_word = $left_join ? 'LEFT JOIN' : 'JOIN';
        $name_sep = $name_sep ?: ', ';

        $select = ['sm.userid'];
        if (!empty($res_ids_alias)){
            $select[] = "GROUP_CONCAT(DISTINCT sch.id) AS $res_ids_alias";
        }
        if (!empty($res_names_alias)){
            $select[] = "GROUP_CONCAT(DISTINCT sch.name ORDER BY sch.name SEPARATOR '$name_sep') AS $res_names_alias";
        }
        if (!empty($res_code_alias)){
            $select[] = "GROUP_CONCAT(DISTINCT sch.code ORDER BY sch.code SEPARATOR '$name_sep') AS $res_code_alias";
        }
        if (!empty($add_select)){
            $select[] = static::arr2str($add_select, '', ', ');
        }

        $join_cond = ["$table_alias.userid = $userid_cond"];
        if (!empty($add_join_conditions)){
            if (is_array($add_join_conditions)){
                $join_cond[] = static::sql_condition($add_join_conditions);
            } else {
                $join_cond[] = $add_join_conditions;
            }
        }

        $select = static::arr2str($select, '', ', ');
        $join_cond = static::sql_condition($join_cond);

        return "$join_word
               (SELECT $select
                FROM {{$school_table}} sch
                JOIN {{$member_table}} sm
                   ON sm.cohortid = sch.id
                GROUP BY sm.userid
            ) $table_alias ON $join_cond
        ";
    }

    /**
     * Get SQL code to join group tables as subquery
     * In $res_names_alias you will get all user group names. You can set separator $name_sep, which is ', ' by default
     * In $res_ids_alias you will get all user group id. Separator for it is always ',',
     *  so you can use FIND_IN_SET() for filtering records by id
     * If you set empty $res_*, their data will be not added to query select
     *
     * @param string       $table_alias         - alias for result group-member table
     * @param string       $userid_cond         - SQL condition to join member ids
     * @param string       $courseid_cond       - SQL condition to join course ids, can be empty
     * @param string       $res_ids_alias       - alias to store result group ids (i.e. "$table_alias.$res_ids_alias" = 1,2, ...)
     * @param string       $res_names_alias     - alias to store result group names (i.e. "$table_alias.$res_names_alias" = group1, group2, ...)
     * @param string       $name_sep            - separator for name result values
     * @param bool         $left_join           - if true, uses LEFT JOIN (default), simple JOIN otherwise
     * @param string|array $add_join_conditions - add custom join condition to result table
     * @param string|array $add_select          - add custom select code to result table
     *
     * @return string - SQL join code
     */
    static public function sql_join_groups_as_subquery_table($table_alias='grp', $userid_cond='u.id', $courseid_cond='c.id',
        $res_ids_alias='group_ids', $res_names_alias='group_names', $name_sep=', ', $left_join=true, $add_join_conditions='', $add_select=''){

        $join_word = $left_join ? 'LEFT JOIN' : 'JOIN';
        $name_sep = $name_sep ?: ', ';

        $select = ['g.courseid', 'gm.userid'];
        if (!empty($res_ids_alias)){
            $select[] = "GROUP_CONCAT(DISTINCT g.id) AS $res_ids_alias";
        }
        if (!empty($res_names_alias)){
            $select[] = "GROUP_CONCAT(DISTINCT g.name ORDER BY g.name SEPARATOR '$name_sep') AS $res_names_alias";
        }
        if (!empty($add_select)){
            $select[] = static::arr2str($add_select, '', ', ');
        }


        $join_cond = ["$table_alias.userid = $userid_cond"];
        if (!empty($courseid_cond)){
            $join_cond[] = "$table_alias.courseid = $courseid_cond";
        }
        if (!empty($add_join_conditions)){
            if (is_array($add_join_conditions)){
                $join_cond[] = static::sql_condition($add_join_conditions);
            } else {
                $join_cond[] = $add_join_conditions;
            }
        }

        $select = static::arr2str($select, '', ', ');
        $join_cond = static::sql_condition($join_cond);

        return "$join_word
               (SELECT $select
                FROM {groups} g
                JOIN {groups_members} gm
                   ON gm.groupid = g.id
                GROUP BY g.courseid, gm.userid
            ) $table_alias ON $join_cond
        ";
    }

    /**
     * Return sql join tables, to get first time of last period of grading
     *
     * $first_last_grade_time - Time, when first grade was granted in last period of grading,
     *      after all periods of ungraded time or user changes submission
     * If grade were ungraded, $first_last_grade_time will be after last ungraded time
     * If main grade (from the moodle gradebook) was changed after last user modified time,
     *      $first_last_grade_time will be after last user modified time
     *
     * To get $first_last_grade_time value use:
     *  LEAST(COALESCE(ggh_first_graded_time.timemodified, gg.timemodified), gg.timemodified) AS first_last_grade_time
     *
     * To make it work correctly, you should add to where clause:
     *  "ggh_last_ungraded_time_fake.id IS NULL AND ggh_first_graded_time_fake.id IS NULL"
     *
     * @param string $sql_user_timemodified         - sql alias to get modification time from user (it depends on activity type),
     *                                                  you can set it to empty for ignoring this condition
     * @param string $sql_grade_item_id             - sql alias to the grade item id, "gi.id" by default
     * @param string $sql_userid                    - sql alias to the user id, "u.id" by default
     * @param string $sql_grade_grade_timemodified  - sql alias to the grade_grade timemodified, "gg.timemodified" by default
     *
     * @return string - sql to join tables
     */
    static public function sql_get_first_last_grade_join($sql_user_timemodified=null,
        $sql_grade_item_id='gi.id', $sql_userid = 'u.id', $sql_grade_grade_timemodified='gg.timemodified'){

        if (empty($sql_user_timemodified)){
            $after_time_condition = '0';
        } else {
            $after_time_condition = "IF(COALESCE($sql_user_timemodified, $sql_grade_grade_timemodified) > $sql_grade_grade_timemodified, ".
                "0, $sql_user_timemodified)";
        }

        return "
            LEFT JOIN {grade_grades_history} ggh_last_ungraded_time
                ON ggh_last_ungraded_time.itemid = $sql_grade_item_id
                AND ggh_last_ungraded_time.userid = $sql_userid
                AND ggh_last_ungraded_time.source IS NOT NULL
                AND ggh_last_ungraded_time.finalgrade IS NULL
            LEFT JOIN {grade_grades_history} ggh_last_ungraded_time_fake
                ON ggh_last_ungraded_time_fake.itemid = $sql_grade_item_id
                AND ggh_last_ungraded_time_fake.userid = $sql_userid
                AND ggh_last_ungraded_time_fake.source IS NOT NULL
                AND ggh_last_ungraded_time_fake.finalgrade IS NULL
                -- filter fake
                AND ggh_last_ungraded_time_fake.id <> ggh_last_ungraded_time.id
                AND ((ggh_last_ungraded_time_fake.timemodified > ggh_last_ungraded_time.timemodified) OR
                    (ggh_last_ungraded_time_fake.timemodified = ggh_last_ungraded_time.timemodified AND 
                        ggh_last_ungraded_time_fake.id > ggh_last_ungraded_time.id))
                
            LEFT JOIN {grade_grades_history} ggh_first_graded_time
                ON ggh_first_graded_time.itemid = $sql_grade_item_id
                AND ggh_first_graded_time.userid = $sql_userid
                AND ggh_first_graded_time.source IS NOT NULL
                AND ggh_first_graded_time.finalgrade IS NOT NULL
                AND ggh_first_graded_time.timemodified > COALESCE(ggh_last_ungraded_time.timemodified, 0)
                AND ggh_first_graded_time.timemodified >= $after_time_condition
            LEFT JOIN {grade_grades_history} ggh_first_graded_time_fake
                ON ggh_first_graded_time_fake.itemid = $sql_grade_item_id
                AND ggh_first_graded_time_fake.userid = $sql_userid
                AND ggh_first_graded_time_fake.source IS NOT NULL
                AND ggh_first_graded_time_fake.finalgrade IS NOT NULL
                AND ggh_first_graded_time_fake.timemodified > COALESCE(ggh_last_ungraded_time.timemodified, 0)
                AND ggh_first_graded_time_fake.timemodified >= $after_time_condition
                -- filter fake
                AND ggh_first_graded_time_fake.id <> ggh_first_graded_time.id
                AND ((ggh_first_graded_time_fake.timemodified < ggh_first_graded_time.timemodified) OR
                    (ggh_first_graded_time_fake.timemodified = ggh_first_graded_time.timemodified AND 
                        ggh_first_graded_time_fake.id < ggh_first_graded_time.id))
        ";
    }
}
