<?php
/**
 * @package    local_ned_controller
 * @subpackage shared
 * @category   NED
 * @copyright  2022 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller\shared;

use local_ned_controller\support\ned_proxy_record;

defined('MOODLE_INTERNAL') || die();

/**
 * Trait dm_util
 *
 * @package local_ned_controller\shared
 * @mixin cm_util
 */
trait dm_util {
    use plugin_dependencies, data_util, db_util, util;

    /**
     * @return \block_ned_teacher_tools\deadline_manager|string|null
     * @noinspection PhpReturnDocTypeMismatchInspection
     */
    static public function get_DM(){
        if (!static::is_tt_exists()){
            return null;
        }

        return '\block_ned_teacher_tools\deadline_manager';
    }

    /**
     * Alias @see \local_ned_controller\shared\dm_util::get_DM()
     *
     * @return \block_ned_teacher_tools\deadline_manager|string|null
     */
    static public function dm_get_DM(){
        return static::get_DM();
    }

    /**
     * Return deadline manager mod by course module
     *
     * @param \cm_info|int|string $cm_or_id
     * @param object|numeric      $courseorid - Optional course object (or its id) if loaded, improves optimization if $cm_or_id is represented as ID
     *
     * @return \block_ned_teacher_tools\mod\deadline_manager_mod|false|null
     */
    static public function dm_get_dmm_by_cm($cm_or_id, $courseorid=null){
        $DM = static::get_DM();
        if (!$DM || !$cm_or_id){
            return null;
        }

        return $DM::get_dmm_by_cm($cm_or_id, $courseorid);
    }

    /**
     * Get deadline through Deadline Manager, if TT is installed, otherwise get deadline form cm object
     *
     * @param int|\cm_info|object   $cm_or_id   - Id of course-module, or database object
     * @param \stdClass|int         $userorid   - Optional user object (or its id; default = current)
     * @param \stdClass|int         $courseorid - Optional course object (or its id) if already loaded
     *
     * @return int
     */
    static public function get_deadline_by_cm($cm_or_id, $userorid=null, $courseorid=null){
        $dm_module = null;
        /** @var \cm_info|null $cm */
        $cm = static::get_cm_by_cmorid($cm_or_id, $courseorid);
        if (empty($cm)){
            return 0;
        }

        if (static::is_tt_exists()){
            $dm_module = \block_ned_teacher_tools\deadline_manager::get_dmm_by_cm($cm, $courseorid);
        }

        if ($dm_module){
            $userid = static::get_userid_or_global($userorid);
            $finaldeadline = $dm_module->get_user_final_deadline($userid);
        } elseif ($instance = static::get_module_instance_by_cm($cm)){
            if ($cm->modname == 'assign') {
                $finaldeadline = $instance->duedate ?? null;    // NOT cutoffdate;
            } elseif ($cm->modname == 'quiz') {
                $finaldeadline = $instance->timeclose ?? null;
            } else {
                $finaldeadline = $instance->duedate ?? null;
            }
        } else {
            $finaldeadline = 0;
        }

        return (int)($finaldeadline ?? 0);
    }

    /**
     * Get deadline through Deadline Manager, if TT is installed, otherwise get deadline form cm object
     * Alias @see \local_ned_controller\shared\dm_util::get_deadline_by_cm()
     *
     * @param int|\cm_info|object   $cm_or_id   - Id of course-module, or database object
     * @param \stdClass|int         $userorid   - Optional user object (or its id; default = current)
     * @param \stdClass|int         $courseorid - Optional course object (or its id) if already loaded
     *
     * @return int
     */
    static public function dm_get_deadline_by_cm($cm_or_id, $userorid=null, $courseorid=null){
        return static::get_deadline_by_cm($cm_or_id, $userorid, $courseorid);
    }

    /**
     * Return available cm options for the proxy plugin
     * It's not allowed to chose itself or activity, which already choose this activity
     *
     * @param \cm_info|numeric $cm_or_id    - course-module, the main work bases on it, but you can also provide $course when $cm is none.
     * @param numeric|object $course_or_id - (optional) you can provide it instead of $cm,
     *                                     to get data on the whole course without linking to the activity
     *
     * @return array|\cm_info[]
     */
    static public function dm_get_proxy_activities_choice_list($cm_or_id=null, $course_or_id=null){
        $cm = static::get_cm_by_cmorid($cm_or_id, $course_or_id);
        if (empty($cm) && empty($course_or_id)) return [];

        if ($cm){
            $cms = static::get_important_activities($cm->get_course());
            unset($cms[$cm->id]);

            $proxy_point_for_this = static::dm_get_proxy_configs(null, $cm->id, null, false, true);
            foreach ($proxy_point_for_this as $proxy_point_config){
                unset($cms[$proxy_point_config->cmid]);
            }
        } else {
            $cms = static::get_important_activities($course_or_id);
        }

        return $cms;
    }

    /**
     * Return is course-module enabled on the course by DM
     *
     * @param \cm_info|int|string $cm_or_id - it's better to use cm_info, if you already loaded it
     * @param int|string          $courseid
     *
     * @return bool|null
     */
    static public function dm_is_cm_enabled($cm_or_id, $courseid=null){
        $DM = static::get_DM();
        if ($DM){
            return $DM::is_cm_enabled($cm_or_id, $courseid);
        }

        return null;
    }

    /**
     * Return active deadline-proxy custom configs
     * Activities can have more than one proxy plugin,
     *  but the first correct one from the C::PROXY_DEADLINE_PLUGINS will be the top
     *
     * Note: If both $proxy_cmids and $proxy_point_cmids are provided,
     *          the result will be the intersection of applying both of them, not the union
     *
     * @param numeric|array|null    $proxy_cmids - get configs only for these course-modules
     * @param numeric|array|null    $proxy_point_cmids - get configs, where these course-modules are point activities
     * @param string|string[]|null  $filter_plugins - (optional) filter result by some proxy plugin names
     *                                              (removes results, where plugin from the filter is not the top)
     * @param bool                  $only_one - (optional) return only the first suitable object, otherwise return array
     * @param bool                  $get_raw_records - (optional) return raw records from the DB query,
     *                                               partially ignores $proxy_point_cmids and $filter_plugins
     *
     * @return ned_proxy_record|ned_proxy_record[]|object|object[]|null - if $only_one is true, return single object or null,
     *                                                                  otherwise return array [proxy activity cmid => config object]
     */
    static protected function _get_proxy_configs($proxy_cmids=null, $proxy_point_cmids=null,
        $filter_plugins=null, $only_one=false, $get_raw_records=false){
        $def = $only_one ? null : [];
        if (empty(C::PROXY_DEADLINE_PLUGINS)){
            return $def;
        }

        $proxy_cmids = static::val2arr($proxy_cmids);
        $filter_plugins = static::val2arr($filter_plugins);
        $proxy_point_cmids = static::val2arr($proxy_point_cmids);

        $i = 1;
        $order_cond = [];
        $plugins = [];
        $params = [];
        $where = [];
        $filter_result_plugin = [];
        foreach (C::PROXY_DEADLINE_PLUGINS as $proxy_plugin => $proxy_name){
            if (!static::is_plugin_exists($proxy_plugin)) continue;

            $p = $proxy_name.'_i';
            $params[$p] = $proxy_name;
            $order_cond[] = "WHEN :$p THEN $i";
            $plugins[] = $proxy_name;
            if ($filter_plugins){
                if (in_array($proxy_plugin, $filter_plugins) || in_array($proxy_name, $filter_plugins)){
                    $filter_result_plugin[] = $proxy_name;
                }
            }
            $i++;
        }

        if ( empty($plugins) || ($filter_plugins && empty($filter_result_plugin)) ){
            return $def;
        }

        $select = [
            "DISTINCT CONCAT(cm.id, '_', apc_e.id) AS id",
            "cm.id AS cmid",
            "apc_e.plugin AS plugin",
            "apc_s.value AS submissionwindow",
            "apc_a.value AS point_activity",
        ];
        $select[] = "(CASE apc_e.plugin ".join(' ', $order_cond)." END) AS plugin_order";

        $joins = ["
            JOIN {modules} m
                ON m.id = cm.module
                AND m.name = 'assign'
            JOIN {assign} a
                ON a.id = cm.instance
            JOIN {assign_plugin_config} AS apc_e
                ON apc_e.assignment = cm.instance
                AND apc_e.name = 'enabled'
                AND apc_e.value = 1
            JOIN {assign_plugin_config} AS apc_s
                ON apc_s.assignment = cm.instance
                AND apc_s.plugin = apc_e.plugin
                AND apc_s.name = 'submissionwindow'
                AND apc_s.value > 0
            JOIN {assign_plugin_config} AS apc_a
                ON apc_a.assignment = cm.instance
                AND apc_a.plugin = apc_e.plugin
                AND apc_a.name = CONCAT(apc_a.plugin, 'activity')
                AND apc_a.value > 0
        "];

        static::sql_get_in_or_equal_options_list(['apc_e.plugin' => $plugins], $where, $params);

        if ($proxy_cmids){
            static::sql_get_in_or_equal_options_list(['cm.id' => $proxy_cmids], $where, $params);
        }
        if ($proxy_point_cmids){
            $joins[] = "
                JOIN {assign_plugin_config} AS apc_p
                    ON apc_p.assignment = cm.instance
                    AND apc_p.name = CONCAT(apc_p.plugin, 'activity')
                    AND apc_p.value > 0
            ";

            static::sql_get_in_or_equal_options_list(['apc_p.plugin' => $plugins], $where, $params);
            static::sql_get_in_or_equal_options_list(['apc_p.value' => $proxy_point_cmids], $where, $params);
        }

        $sql = static::sql_generate($select, $joins, 'course_modules', 'cm', $where, [], 'plugin_order');
        $records = static::db()->get_records_sql($sql, $params, 0, (int)($get_raw_records && $only_one));

        if ($get_raw_records){
            return $only_one ? (reset($records) ?: null) : $records;
        }

        $real_data = [];
        $return_data = [];
        /** @var ned_proxy_record|object $record */
        foreach ($records as $record){
            if ($real_data[$record->cmid] ?? false) continue;

            $real_data[$record->cmid] = true;

            if ($filter_result_plugin){
                if (!in_array($record->plugin, $filter_result_plugin)) continue;
            }
            if ($proxy_point_cmids){
                if (!in_array($record->point_activity, $proxy_point_cmids)) continue;
            }

            unset($record->id);
            unset($record->plugin_order);

            if ($only_one){
                return $record;
            }

            $return_data[$record->cmid] = $record;
        }

        return $only_one ? null : $return_data;
    }

    /**
     * Return active deadline-proxy custom configs
     * Activities can have more than one proxy plugin,
     *  but the first correct one from the C::PROXY_DEADLINE_PLUGINS will be the top
     *
     * Require to provide $proxy_cmids or $proxy_point_cmids
     * If you need to return all configs without limits, @see \local_ned_controller\shared\dm_util::dm_get_all_proxy_configs
     *
     * Note: If both $proxy_cmids and $proxy_point_cmids are provided,
     *          the result will be the intersection of applying both of them, not the union
     *
     * @param numeric|array|null    $proxy_cmids - get configs only for these course-modules
     * @param numeric|array|null    $proxy_point_cmids - get configs, where these course-modules are point activities
     * @param string|string[]|null  $filter_plugins - (optional) filter result by some proxy plugin names
     *                                              (removes results, where plugin from the filter is not the top)
     * @param bool                  $only_one - (optional) return only the first suitable object, otherwise return array
     * @param bool                  $get_raw_records - (optional) return raw records from the DB query,
     *                                               partially ignores $proxy_point_cmids and $filter_plugins
     *
     * @return ned_proxy_record|ned_proxy_record[]|object|object[]|null - if $only_one is true, return single object or null,
     *                                                                  otherwise return array [proxy activity cmid => config object]
     */
    static public function dm_get_proxy_configs($proxy_cmids=null, $proxy_point_cmids=null,
        $filter_plugins=null, $only_one=false, $get_raw_records=false){
        if (empty($proxy_cmids) && empty($proxy_point_cmids)){
            static::debugging('You need to provide $proxy_cmids or $proxy_point_cmids!');
            return $only_one ? null : [];
        }

        return static::_get_proxy_configs($proxy_cmids, $proxy_point_cmids, $filter_plugins, $only_one, $get_raw_records);
    }

    /**
     * Return All active deadline-proxy custom configs
     * Activities can have more than one proxy plugin,
     *  but the first correct one from the C::PROXY_DEADLINE_PLUGINS will be the top
     *
     * @param string|string[]|null  $filter_plugins - (optional) filter result by some proxy plugin names
     *                                              (removes results, where plugin from the filter is not the top)
     * @param bool                  $only_one - (optional) return only the first suitable object, otherwise return array
     * @param bool                  $get_raw_records - (optional) return raw records from the DB query,
     *                                               partially ignores $proxy_point_cmids and $filter_plugins
     *
     * @return ned_proxy_record|ned_proxy_record[]|object|object[]|null - if $only_one is true, return single object or null,
     *                                                                  otherwise return array [proxy activity cmid => config object]
     */
    static public function dm_get_all_proxy_configs($filter_plugins=null, $only_one=false, $get_raw_records=false){
        return static::_get_proxy_configs(null, null, $filter_plugins, $only_one, $get_raw_records);
    }

    /**
     * Return true, if this course-module is active proxy activity
     *
     * @param \cm_info|numeric  $cm_or_id
     *
     * @return bool
     */
    static public function dm_is_proxy_activity($cm_or_id){
        if (!static::has_proxy_deadline_plugins()) return false;

        $cm = static::get_cm_by_cmorid($cm_or_id, null, null, C::ASSIGN);
        if (!$cm) return false;

        return (bool)static::_get_proxy_configs($cm->id, null, null, true, false);
    }

    /**
     * Return true, if this course-module has any 'enabled' proxy activity (it doesn't check any other settings)
     *
     * @param \cm_info|numeric  $cm_or_id
     *
     * @return bool
     */
    static public function dm_is_proxy_activity_enabled($cm_or_id){
        if (!static::has_proxy_deadline_plugins()) return false;

        $cm = static::get_cm_by_cmorid($cm_or_id, null, null, C::ASSIGN);
        if (!$cm) return false;

        $where = [
            "assignment = :instance",
            "name = 'enabled'",
            "subtype = 'assignsubmission'",
            "value = '1'",
        ];
        $params = ['instance' => $cm->instance];
        static::sql_add_get_in_or_equal_options('plugin', array_values(C::PROXY_DEADLINE_PLUGINS), $where, $params);
        $where = static::sql_condition($where);

        return static::db()->record_exists_select('assign_plugin_config', $where, $params);
    }

    /**
     * Return true, if this course-module is proxy point activity for some other active proxy activity
     *
     * @param \cm_info|numeric  $cm_or_id
     * @param bool              $include_not_top - return true, even if this activity is not active because of other top proxy
     *
     * @return bool
     */
    static public function dm_is_proxy_activity_pointer($cm_or_id, $include_not_top=false){
        $cmid = static::get_id($cm_or_id);
        return (bool)static::_get_proxy_configs(null, $cmid, null, true, $include_not_top);
    }


    /**
     * Update proxy deadline for one proxy
     * If plugin name provided, and its name will not be the same as active proxy plugin for the proxy,
     *  updates only this proxy default (bases on any plugin config)
     *
     * @param \cm_info|object|numeric $cmorid_proxy - course module (or id) of updated proxy
     * @param string|null             $proxy_plugin_check - string short name of plugin proxy, ie 'proxy'
     * @param bool                    $has_changes - required it in true, and correct plugin check, to update all overrides
     *
     * @return string - short name of the current active proxy plugin
     */
    static public function dm_proxy_updated($cmorid_proxy, $proxy_plugin_check=null, $has_changes=false){
        $cm = static::get_cm_by_cmorid($cmorid_proxy, null, null, C::ASSIGN);
        if (!$cm) return '';

        $ned_proxy = static::dm_get_proxy_configs($cm->id, null, null, true);
        if (!$ned_proxy){
            return '';
        }

        if (!$proxy_plugin_check || ($ned_proxy->plugin == $proxy_plugin_check && $has_changes)){
            static::dm_update_proxy_overrides($ned_proxy, true);
        } else {
            static::dm_update_proxy_default_deadlines($ned_proxy);
        }

        return $ned_proxy->plugin;
    }

    /**
     * Update one type of deadline overrides in the activities based on point proxy activity
     *
     * If $groupid passed - update group overriding, if $userid passed - update user overriding,
     *  otherwise update default overriding
     * You can pass deadline, if it already loaded,
     *  otherwise it will be getting from the proxy point activity based on ned_proxy records
     *
     * @param \cm_info|object|numeric $cmorid_point - course module (or id) of the proxy point activity
     * @param int|null  $groupid               - set override for this group, ignores $userid
     * @param int|null  $userid                - set override for this user, in no group id
     * @param int|null  $deadline              - if null, load deadline from the proxy point activity
     * @param bool      $delete_empty_deadline - if true, and deadline is empty, remove deadline,
     *                                              set unlimited deadline for empty deadline otherwise
     */
    static public function dm_point_update_proxy($cmorid_point, $groupid=null, $userid=null, $deadline=null, $delete_empty_deadline=true){
        $cmid = static::get_id($cmorid_point);
        $ned_proxies = static::dm_get_proxy_configs(null, $cmid);
        if (!empty($ned_proxies)){
            static::dm_update_proxy_deadlines($ned_proxies, $groupid, $userid, $deadline, $delete_empty_deadline);
        }
    }

    /**
     * Remove all deadline overrides in the activities based on point proxy activity
     *
     * @param \cm_info|object|numeric $cmorid_point - course module (or id) of the proxy point activity
     * @param bool $including_default - if true, remove default deadline too
     */
    static public function dm_point_delete_proxy_deadlines($cmorid_point, $including_default=false){
        $cmid = static::get_id($cmorid_point);
        $ned_proxies = static::dm_get_proxy_configs(null, $cmid);
        if (!empty($ned_proxies)){
            static::dm_delete_proxy_deadlines($ned_proxies, $including_default);
        }
    }

    /**
     * Update all proxy overrides based on the ned_proxy records
     * @see \local_ned_controller\shared\dm_util::dm_get_proxy_configs()
     *
     * If can't get DM module for the point proxy activity, will tries to update defaults
     *
     * @param ned_proxy_record|ned_proxy_record[]|object|object[] $ned_proxies
     * @param bool $remove_old - if true, remove all old overrides
     */
    static public function dm_update_proxy_overrides($ned_proxies, $remove_old=false){
        if (empty($ned_proxies)) return;

        $load_only_defaults = [];
        $ned_proxies = static::val2arr($ned_proxies);

        /** @var ned_proxy_record|object $ned_proxy */
        foreach ($ned_proxies as $ned_proxy){
            $dmm_proxy = static::dm_get_dmm_by_cm($ned_proxy->cmid);
            // if there is no DM module for proxy activity, we can do nothing
            if (!$dmm_proxy) continue;

            $dmm_point = static::dm_get_dmm_by_cm($ned_proxy->point_activity);
            if (!$dmm_point){
                // without DM module of point activity, we can only try to set proxy default
                $load_only_defaults[] = $ned_proxy;
                continue;
            }

            $overrides = $dmm_point->get_all_mod_deadlines(true);
            $dmm_proxy->set_deadlines_by_records($overrides, $ned_proxy->submissionwindow, $remove_old);
        }

        if (!empty($load_only_defaults)){
            static::dm_update_proxy_default_deadlines($load_only_defaults);
        }
    }

    /**
     * Update one type of deadline overrides in the activities based on the ned_proxy records
     * @see \local_ned_controller\shared\dm_util::dm_get_proxy_configs()
     *
     * If $groupid passed - update group overriding, if $userid passed - update user overriding,
     *  otherwise update only default overriding
     * You can pass deadline, if it already loaded,
     *  otherwise it will be getting from the proxy point activity based on ned_proxy records
     *
     * @param ned_proxy_record|ned_proxy_record[]|object|object[] $ned_proxies - configs from the dm_get_proxy_configs() method
     * @param int|null  $groupid               - set override for this group, ignores $userid
     * @param int|null  $userid                - set override for this user, in no group id
     * @param int|null  $deadline              - if null, load deadline from the proxy point activity
     * @param bool      $delete_empty_deadline - if true, and deadline is empty, remove deadline,
     *                                              set unlimited deadline for empty deadline otherwise
     */
    static public function dm_update_proxy_deadlines($ned_proxies, $groupid=null, $userid=null, $deadline=null, $delete_empty_deadline=true){
        if (empty($ned_proxies)) return;

        $ned_proxies = static::val2arr($ned_proxies);
        $get_deadline = is_null($deadline);
        $update_default = empty($groupid) && empty($userid);
        /**
         * @param int $time
         * @param ned_proxy_record|object $config
         *
         * @return int
         */
        $d = function($time, $config){
            return $time ? ($time + $config->submissionwindow) : 0;
        };

        /** @var ned_proxy_record|object $ned_proxy */
        foreach ($ned_proxies as $ned_proxy){
            $dmm_proxy = static::dm_get_dmm_by_cm($ned_proxy->cmid);
            // if there is no DM module for proxy activity, we can do nothing
            if (!$dmm_proxy) continue;

            $dmm_point = null;
            if ($get_deadline){
                $dmm_point = static::dm_get_dmm_by_cm($ned_proxy->point_activity);
                if (!$dmm_point){
                    // there is no DM module for point activity, try to get usual deadline
                    $deadline = static::get_deadline_by_cm($ned_proxy->point_activity);
                }
            }
            $load_deadline = $get_deadline && $dmm_point;

            if ($update_default){
                if ($load_deadline){
                    $deadline = $dmm_point->get_default_deadline();
                }
                $dmm_proxy->set_default_deadline($d($deadline, $ned_proxy));
            } else {
                if ($load_deadline){
                    $deadline = $dmm_point->get_override($groupid, $userid, true);
                }

                if ($delete_empty_deadline && !$deadline){
                    $dmm_proxy->delete_override($groupid, $userid);
                } else {
                    $dmm_proxy->set_override($groupid, $userid, $d($deadline, $ned_proxy), null, !$deadline);
                }
                $dmm_proxy->delete_extension($userid);
            }
        }
    }

    /**
     * Remove all deadline overrides in the activities based on the ned_proxy records
     * @see \local_ned_controller\shared\dm_util::dm_get_proxy_configs()
     *
     * @param ned_proxy_record|ned_proxy_record[]|object|object[] $ned_proxies - configs from the dm_get_proxy_configs() method
     * @param bool $including_default - if true, remove default deadline too
     */
    static public function dm_delete_proxy_deadlines($ned_proxies, $including_default=false){
        if (empty($ned_proxies)){
            return;
        }

        $ned_proxies = static::val2arr($ned_proxies);
        /** @var ned_proxy_record|object $ned_proxy */
        foreach ($ned_proxies as $ned_proxy){
            $dmm_proxy = static::dm_get_dmm_by_cm($ned_proxy->cmid);
            // if there is no DM module for proxy activity, we can do nothing
            if (!$dmm_proxy) continue;

            $dmm_proxy->delete_all_overrides($including_default);
        }
    }

    /**
     * Update deadline defaults in the activities based on ned_proxy records
     * @see \local_ned_controller\shared\dm_util::dm_get_proxy_configs()
     *
     * You can pass deadline, if it already loaded,
     *  otherwise it will be getting from the proxy point activity based on ned_proxy records
     *
     * @param ned_proxy_record|ned_proxy_record[]|object|object[] $ned_proxies - configs from the dm_get_proxy_configs() method
     * @param int|null  $deadline - if null, load deadline from the proxy point activity
     */
    static public function dm_update_proxy_default_deadlines($ned_proxies, $deadline=null){
        static::dm_update_proxy_deadlines($ned_proxies, null, null, $deadline);
    }

    /**
     * Return format start and end of submission window, and its numeric values, based on proxy assign plugin
     *
     * @param \cm_info|object|numeric $cm_or_id      - assign course module
     * @param string                  $proxy_name    - one of the C::PROXY_DEADLINE_PLUGINS
     * @param object|numeric          $user_or_id    - get deadline for this user
     * @param object|numeric          $group_or_id   - get deadline for this group
     * @param bool                    $day_start_end - if true, update values through get_day_start_end()
     *
     * @return array - list($window_start_str, $window_end_str, $window_start, $window_end)
     */
    static public function dm_get_submission_window($cm_or_id, $proxy_name, $user_or_id=null, $group_or_id=null, $day_start_end=false){
        $def = ['-', '-', 0, 0];
        if (!$group_or_id && !$user_or_id) return $def;

        $cm = static::get_cm_by_cmorid($cm_or_id, null, null, C::ASSIGN);
        if (!$cm) return $def;

        // get submission window
        $schools = static::get_user_schools($user_or_id);
        if (!empty($schools)) {
            $school = reset($schools);
            if (!empty($school->forceproxysubmissionwindow)) {
                $submissionwindow = $school->forceproxysubmissionwindow;
            }
        }

        if (empty($submissionwindow)) {
            $ned_assign = static::ned_assign_by_cm($cm, $cm->course);
            $assign_proxy = $ned_assign->get_submission_plugin_by_type(C::PROXY_DEADLINE_PLUGINS[$proxy_name] ?? $proxy_name);
            $cfg = null;
            if ($assign_proxy) {
                $cfg = $assign_proxy->get_config();
            }

            $submissionwindow = $cfg->submissionwindow ?? 0;
        }

        // get deadline
        if ($user_or_id){
            $deadline = static::get_deadline_by_cm($cm, $user_or_id);
        } else {
            $dmm = static::dm_get_dmm_by_cm($cm);
            $deadline = $dmm ? $dmm->get_group_final_deadline($group_or_id) : 0;
        }

        // calculate
        if ($deadline){
            $window_start = $deadline - $submissionwindow;
            $window_end = $deadline;
            if ($day_start_end){
                [$window_start,] = static::get_day_start_end($window_start);
                [, $window_end] = static::get_day_start_end($window_end);
            }
        } else {
            $window_start = $window_end = 0;
        }

        // format
        $dm_timezone = static::NED_TIMEZONE;
        if (static::is_tt_exists()){
            $dm_timezone = \block_ned_teacher_tools\shared_lib::get_user_dm_timezone_ctx($cm->context);
        }
        $window_start_str = static::ned_date_tz($window_start, $dm_timezone);
        $window_end_str = static::ned_date_tz($window_end, $dm_timezone);

        return [$window_start_str, $window_end_str, $window_start, $window_end];
    }

    /**
     * Check, should DM use other extension capability then default
     *
     * @param numeric|object    $user_or_id - for who should add extension
     *
     * @return bool|null - true if "yes", false if "no", null in case of indeterminacy, for example, if there is a contradiction in the data
     */
    static public function dm_should_use_other_extension_capability($user_or_id=null){
        if (!static::is_schm_exists()) return false;

        $schools = static::get_user_schools($user_or_id);
        if (empty($schools)) return false;

        $ext_manager_ct = $ext_manager_sa = false;
        foreach ($schools as $school){
            if (isset($school->extmanager) && $school->extmanager == \local_schoolmanager\school::EXT_MANAGE_SA){
                $ext_manager_sa = true;
            } else {
                $ext_manager_ct = true;
            }

            if ($ext_manager_ct && $ext_manager_sa) break;
        }

        if ($ext_manager_ct && $ext_manager_sa) return null;
        if ($ext_manager_sa) return true;
        return false;
    }
}
