<?php

/**
 * @package    local_ned_controller
 * @subpackage task
 * @category   NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller\task;
use core\task\adhoc_task;
use local_ned_controller\shared_lib as NED;

defined('MOODLE_INTERNAL') || die;

/**
 * Class base_adhoc_task
 * IMPORTANT: When use it as parent class, don't forget add "use \local_ned_controller\task\base_task;"
 *
 * @package local_ned_controller\task
 */
class base_adhoc_task extends adhoc_task {
    use base_task;

    const TABLE = 'task_adhoc';

    /**
     * Returns default concurrency limit for this task.
     *
     * @return int default concurrency limit
     */
    protected function get_default_concurrency_limit(): int{
        return $this->_concurrency_limit ?? parent::get_default_concurrency_limit();
    }

    /**
     * Queue new task
     *
     * @param null|base_adhoc_task $task
     * @param int  $time_delay
     * @param bool $remove_same_tasks
     *
     * @return base_adhoc_task
     */
    static protected function _add_new_job($task=null, $time_delay=0, $remove_same_tasks=false){
        $task = $task ?? new static();
        if ($time_delay){
            $task->set_next_run_time(time() + $time_delay);
        }
        if ($remove_same_tasks){
            static::remove_same_tasks($task);
        }

        $taskid = \core\task\manager::queue_adhoc_task($task);
        $task->set_id($taskid);

        return $task;
    }

    /**
     * Simple add new task
     * You can use this method, if there none other else
     *
     * @param int  $time_delay
     * @param null $task
     *
     * @return base_adhoc_task
     */
    static public function just_add_new_job($time_delay=0, $task=null, $show_debug_messages=true){
        $task = static::_add_new_job($task, $time_delay);
        $name = $task->get_name();

        if (!$show_debug_messages) return $task;

        if ($time_delay){
            static::print("Next run $name through $time_delay seconds.");
        } else {
            static::print("Queue $name task with zero delay.");
        }

        return $task;
    }

    /**
     * Simple add new task, if there none copies in the adhoc queue
     * * You can use this method, if there none other else
     * Return new task or null, if there is the same task in the queue
     *
     * @param int  $time_delay
     * @param null $task
     *
     * @return base_adhoc_task|null
     */
    static public function just_add_new_job_without_doubles($time_delay=0, $task=null){
        if (static::has_same_tasks_warning($task)){
            return null;
        }

        return static::just_add_new_job($time_delay, $task);
    }

    /**
     * Remove all current tasks form DB
     * Checks only by classname & component
     *
     * @param base_adhoc_task|null $task
     *
     * @return bool true.
     */
    static public function remove_same_tasks($task=null){
        global $DB;
        if ($task){
            $component = $task->get_component();
            $classname = $task->get_classname();
        } else {
            $component = static::get_plugin();
            $classname = static::get_canonical_class_name();
        }
        $params = ['classname' => $classname, 'component' => $component];
        return $DB->delete_records(static::TABLE, $params);
    }

    /**
     * Does queue has this task
     * Checks only by classname & component
     *
     * @param base_adhoc_task|null $task
     *
     * @return bool
     */
    static public function has_same_tasks($task=null){
        global $DB;
        if ($task){
            $component = $task->get_component();
            $classname = $task->get_classname();
        } else {
            $component = static::get_plugin();
            $classname = static::get_canonical_class_name();
        }
        $params = ['classname' => $classname, 'component' => $component];
        return $DB->record_exists(static::TABLE, $params);
    }

    /**
     * Does queue has this task
     * Checks only by classname & component
     * If yes, also print $text or default warning
     * @see has_same_tasks()
     *
     * @param base_adhoc_task|null $task
     * @param string|null $text
     *
     * @return bool
     */
    static public function has_same_tasks_warning($task=null, $text=null){
        if (static::has_same_tasks($task)){
            if (is_null($text)){
                $text = NED::str('othertasks_warning');
            } else {
                $text = NED::str_check($text, null, null, static::get_plugin());

            }
            static::print($text);
            return true;
        }

        return false;
    }

}
