<?php

/**
 * @package    local_ned_controller
 * @subpackage task
 * @category   NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_ned_controller\task;
use local_ned_controller\shared_lib as NED;


defined('MOODLE_INTERNAL') || die;

/**
 * Trait base_task
 *
 * @package local_ned_controller\task
 */
trait base_task {
    protected $_data = null;

    /**
     * If return false, task will not execute
     *
     * @return bool
     */
    public function can_execute(){
        return true;
    }

    /**
     * Do the job.
     * Throw exceptions on errors (the job will be retried).
     *
     * @return bool (true if all was fine, false if there was error)
     */
    public function execute() {
        $res = false;

        try {
            static::print('Start '.NED::ned_date(time()));
            if ($this->can_execute()){
                $this::do_job($this);
                static::print("Done ".NED::ned_date(time()));
                $res = true;
            } else {
                static::print("Can't execute, return");
            }
        } catch (\Throwable $ex){
            static::print('Catch Exception');
            NED::cron_print_error($ex);
        } finally {
            return $res;
        }
    }

    /**
     * Return the name of the component.
     * Need for task class
     *
     * @return string The name of the component.
     */
    public function get_component() {
        return static::get_plugin();
    }

    /**
     * Return the name of the component.
     *
     * @return string The name of the component.
     */
    static public function get_plugin(){
        static $_name = null;
        if (is_null($_name)){
            $path_name = explode('\\', __CLASS__);
            $_name = reset($path_name);
        }
        return $_name;
    }

    /**
     * Get a descriptive name for this task (shown to admins).
     * Need for task class
     *
     * @return string
     */
    public function get_name() {
        return static::get_task_name();
    }

    /**
     * Get a descriptive name for this task (shown to admins).
     *
     * @return string
     */
    static public function get_task_name(){
        static $_name = null;
        if (is_null($_name)){
            $_name = NED::str_check(static::get_technical_name(), null, null, static::get_plugin());
        }
        return $_name;
    }

    /**
     * @return mixed|string|null
     */
    static public function get_technical_name(){
        static $_name = null;
        if (is_null($_name)){
            $path_name = explode('\\', __CLASS__);
            $_name = end($path_name);
        }
        return $_name;
    }

    /**
     * Gets class name for use in database table. Always begins with a \.
     *
     * @return string classname for task table
     */
    public function get_classname() {
        return static::get_canonical_class_name();
    }

    /**
     * Gets class name for use in database table. Always begins with a \.
     *
     * @return string classname for task table
     */
    static public function get_canonical_class_name() {
        $classname = __CLASS__;
        NED::str_add_prefix($classname, '\\');

        return $classname;
    }

    /**
     * @param string $text
     * @param bool   $use_time
     */
    static function print($text, $use_time=true){
        $prefix = '';
        if ($use_time){
            $prefix .= '['.date('H:i:s').']';
        }
        $prefix .= '['.static::get_technical_name().']';
        $res = $prefix.' '.$text;
        if (CLI_SCRIPT){
            mtrace($res);
        } else {
            debugging($res);
        }
    }

    /**
     * Do task job without any checks
     *
     * @param base_task|array|null $task_or_data
     *
     * @return void
     */
    static public function do_job($task_or_data=[]){
        static::get_data_from_task_or_data($task_or_data);
        static::print('base_task::do_job()');
    }

    /**
     * @param array|object|base_task|null $task_or_data
     *
     * @return array
     */
    static public function get_data_from_task_or_data($task_or_data){
        if (is_array($task_or_data)){
            return $task_or_data;
        } elseif (is_object($task_or_data)){
            if (method_exists($task_or_data, 'get_data')){
                return $task_or_data->get_data();
            } elseif (method_exists($task_or_data, 'get_custom_data')){
                return (array)$task_or_data->get_custom_data();
            }
        }

        return [];
    }

    /**
     * Check, that current server is Maple, and log warning about it
     *
     * @param string $add_custom_warning   - if not empty, log this message, if current sever is Maple
     * @param bool   $show_default_warning - if true, show default Maple warning for the Maple server
     *
     * @return bool - return true, if task are running on the Maple server now
     */
    static public function is_maple_warning($add_custom_warning='', $show_default_warning=true){
        if (NED::ctrl_server_is_maple()){
            if ($show_default_warning){
                static::print(NED::$C::str('base_task_maple_warning', static::get_task_name()));
            }

            if (!empty($add_custom_warning)){
                static::print($add_custom_warning);
            }

            return true;
        }

        return false;
    }

    /**
     * Getter for $customdata.
     *
     * @return array|null
     */
    public function get_data(){
        if (is_null($this->_data)){
            if (method_exists($this, 'get_custom_data')){
                $this->_data = (array)$this->get_custom_data();
            } else {
                $this->_data = [];
            }
        }

        return $this->_data;
    }

    /**
     * Getter for $blocking.
     *
     * @return bool
     */
    public function is_blocking() {
        return $this->_blocking ?? parent::is_blocking();
    }

}
