<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * @package    local_ned_controller
 * @subpackage NED
 * @copyright  2018 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @developer  G J Barnard - {@link http://about.me/gjbarnard} and
 *                           {@link http://moodle.org/user/profile.php?id=442195}
 */

namespace local_ned_controller;

defined('MOODLE_INTERNAL') || die();

use moodle_url;

class toolbox {

    protected static $instance;
    protected $categoryicons = null;
    // Constants.
    public static $fontawesomekey = 'fa';

    private function __construct() {
    }

    public static function get_instance() {
        if (!is_object(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function get_categoryicons() {
        if (is_null($this->categoryicons)) {
            $categoryicons = \get_config('local_ned_controller', 'categoryicons');
            if (!empty($categoryicons)) {
                $this->categoryicons = json_decode($categoryicons, true);
            } else {
                // Not set or empty so no categories to customise.
                $this->categoryicons = array();
            }
        }

        return $this->categoryicons;
    }

    /**
     * If allowed then will return the moodle URL object otherwise false.
     */
    public function get_courseparticipants_quicklink($context, $courseid, $position = 'courseheader') {
        if ($this->quicklink_allowed('quicklinks'.$position.'courseparticpants', $context, 'moodle/course:viewparticipants')) {
            global $CFG;
            if ($CFG->branch < 34) {
                $url = new moodle_url('/enrol/users.php', array('id' => $courseid));
            } else {
                $url = new moodle_url('/user/index.php', array('id' => $courseid));
            }
            return $url;
        }
        return false;
    }

    /**
     * If allowed then will return the moodle URL object otherwise false.
     */
    public function get_editsettings_quicklink($context, $courseid, $position = 'coursename') {
        if ($this->quicklink_allowed('quicklinks'.$position.'editsettings', $context, 'moodle/course:update')) {
            return new moodle_url('/course/edit.php', array('id' => $courseid));
        }
        return false;
    }

    /**
     * If allowed then will return the moodle URL object otherwise false.
     */
    public function get_gradebook_quicklink($context, $courseid, $position = 'coursename') {
        if ($this->quicklink_allowed('quicklinks'.$position.'gradebook', $context, 'moodle/grade:viewall')) {
            return new moodle_url('/grade/report/grader/index.php', array('id' => $courseid));
        }
        return false;
    }

    /**
     * If allowed then will return the moodle URL object otherwise false.
     */
    public function get_manualenrollment_quicklink($context, $courseid, $position = 'courseheader') {
        if ($this->quicklink_allowed('quicklinks'.$position.'manualenrollment', $context, 'enrol/manual:manage')) {
            /* Get the enrolment instances for the course if they are enabled.  If manual is in the list
               then we can use it. */
            $enrolments = \enrol_get_instances($courseid, true);
            foreach ($enrolments as $enrolment) {
                if ($enrolment->enrol == 'manual') {
                    return new moodle_url('/enrol/manual/manage.php', array('enrolid' => $enrolment->id));
                }
            }
        }
        return false;
    }

    /**
     * If allowed then will return the moodle URL object otherwise false.
     */
    public function get_nedprogressreport_quicklink($context, $courseid, $position = 'coursename') {
        global $CFG;
        if (file_exists("$CFG->dirroot/blocks/ned_teacher_tools/progress_report.php")) {
            if ($this->quicklink_allowed('quicklinks'.$position.'nedprogressreport', $context, 'block/ned_teacher_tools:viewblock')) {
                return new moodle_url('/blocks/ned_teacher_tools/progress_report.php', array('courseid' => $courseid));
            }
        }
        return false;
    }

    /**
     * If allowed then will return the moodle URL object otherwise false.
     */
    public function get_nedmarkingmanager_quicklink($context, $courseid, $position = 'coursename') {
        global $CFG;
        if (file_exists("$CFG->dirroot/blocks/ned_teacher_tools/marking_manager.php")) {
            if ($this->quicklink_allowed('quicklinks'.$position.'nedmarkingmanager', $context, 'block/ned_teacher_tools:viewblock')) {
                return new moodle_url('/blocks/ned_teacher_tools/marking_manager.php', array('courseid' => $courseid, 'show' => 'unmarked'));
            }
        }
        return false;
    }

    /**
     * If allowed then will return the moodle URL object otherwise false.
     */
    public function get_nedstudentmenu_quicklink($context, $courseid, $position = 'coursename') {
        global $CFG;
        if (is_siteadmin()){
            return false;
        }
        if (file_exists("$CFG->dirroot/blocks/ned_student_menu/student_progress.php")) {
            if ($this->quicklink_allowed('quicklinks'.$position.'nedstudentmenu', $context, 'block/ned_student_menu:viewblock')) {
                return new moodle_url('/blocks/ned_student_menu/student_progress.php', array('courseid' => $courseid));
            }
        }
        return false;
    }

    private function quicklink_allowed($name, $context, $capability) {
        /* Note: Context can be fetched by the theme by $this->page->context->get_course_context(false) in the core_renderer for
                 the course context, but initially checking the page context until more is understood about contexts and their
                 use with this functionality. */
        $allowed = false;

        if (has_capability($capability, $context)) {
            $quicklink = \get_config('local_ned_controller', $name);
            if (!empty($quicklink)) {
                // Then exists and is ticked.
                $allowed = true;
            }
        }

        return $allowed;
    }
}
