<?php
/**
 * @package    local_ned_controller
 * @subpackage NED
 * @copyright  2021 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

use \local_ned_controller\shared_lib as NED;

/**
 * All upgrades checks from Moodle 3
 *
 * @param int $oldversion
 */
function local_ned_controller_moodle3_upgrades($oldversion): void{
    global $DB;

    $dbman = $DB->get_manager();

    if ($oldversion < 2018080800) {

        // Define table local_ned_controller to be created.
        $table = new xmldb_table('local_ned_controller');

        // Adding fields to table local_ned_controller.
        $table->add_field('id', XMLDB_TYPE_INTEGER, '11', null, XMLDB_NOTNULL, XMLDB_SEQUENCE, null);
        $table->add_field('name', XMLDB_TYPE_CHAR, '255', null, null, null, null);
        $table->add_field('repository', XMLDB_TYPE_CHAR, '255', null, null, null, null);
        $table->add_field('plugintype', XMLDB_TYPE_CHAR, '255', null, null, null, null);
        $table->add_field('version', XMLDB_TYPE_CHAR, '255', null, null, null, null);
        $table->add_field('installed', XMLDB_TYPE_INTEGER, '2', null, null, null, '0');
        $table->add_field('timecreated', XMLDB_TYPE_INTEGER, '11', null, null, null, '0');
        $table->add_field('timemodified', XMLDB_TYPE_INTEGER, '11', null, null, null, '0');

        // Adding keys to table local_ned_controller.
        $table->add_key('id', XMLDB_KEY_PRIMARY, array('id'));
        $table->add_key('ix_mdl_repo', XMLDB_KEY_UNIQUE, array('repository'));

        // Conditionally launch create table for local_ned_controller.
        if (!$dbman->table_exists($table)) {
            $dbman->create_table($table);
        }

        // Initial plugins.
        $plugins = [
            'moodle-block_ned_teacher_tools' => ['NED Teacher Tools', 'block'],
            'moodle-block_ned_student_menu' => ['NED Student Menu', 'block'],
            'moodle-block_mentor' => ['NED Mentor Manager', 'block'],
            'moodle-block_custom_menu' => ['NED Custom Menu', 'block'],
            'moodle-mod_nedactivitycluster' => ['NED Activity Cluster', 'mod'],
            'moodle-format_ned' => ['NED Format', 'format'],
            'moodle-theme_ned_boost' => ['NED Boost', 'theme'],
            'moodle-theme_ned_clean' => ['NED Clean', 'theme'],
            'moodle-filter_ned' => ['NED Filter', 'filter'],
            'moodle-local_ned_controller' => ['NED Controller', 'local']
        ];

        foreach ($plugins as $repository => $plugin) {
            $rec = new stdClass();
            $rec->name = $plugin[0];
            $rec->repository = $repository;
            $rec->plugintype = $plugin[1];
            if ($update = $DB->get_record('local_ned_controller', array('repository' => $repository))) {
                $rec->id = $update->id;
                $rec->timemodified = time();
                $DB->update_record('local_ned_controller', $rec);
            } else {
                $rec->timecreated = time();
                $DB->insert_record('local_ned_controller', $rec);
            }
        }
        // Ned_controller savepoint reached.
        upgrade_plugin_savepoint(true, 2018080800, 'local', 'ned_controller');
    }
    if ($oldversion < 2018122100) {

        // Define table local_rosedale_rules to be created.
        $table = new xmldb_table('local_ned_controller_rules');

        // Adding fields to table local_rosedale_rules.
        $table->add_field('id', XMLDB_TYPE_INTEGER, '11', null, XMLDB_NOTNULL, XMLDB_SEQUENCE, null);
        $table->add_field('name', XMLDB_TYPE_CHAR, '255', null, null, null, null);
        $table->add_field('availabilityconditionsjson', XMLDB_TYPE_TEXT, null, null, null, null, null);
        $table->add_field('disabled', XMLDB_TYPE_INTEGER, '1', null, null, null, 0);
        $table->add_field('weight', XMLDB_TYPE_INTEGER, '3', null, null, null, 0);
        $table->add_field('customcss', XMLDB_TYPE_TEXT, null, null, false, false, null);
        $table->add_field('additionalhtmlhead', XMLDB_TYPE_TEXT, null, null, null, null, null);
        $table->add_field('additionalhtmltopofbody', XMLDB_TYPE_TEXT, null, null, null, null, null);
        $table->add_field('additionalhtmlfooter', XMLDB_TYPE_TEXT, null, null, null, null, null);
        $table->add_field('timecreated', XMLDB_TYPE_INTEGER, '11', null, null, null, '0');
        $table->add_field('timemodified', XMLDB_TYPE_INTEGER, '11', null, null, null, '0');

        // Adding keys to table local_rosedale_rules.
        $table->add_key('id', XMLDB_KEY_PRIMARY, array('id'));

        // Conditionally launch create table for local_ned_controller.
        if (!$dbman->table_exists($table)) {
            $dbman->create_table($table);
        }

        // Rosedale_rules savepoint reached.
        upgrade_plugin_savepoint(true, 2018122100, 'local', 'ned_controller');
    }

    if ($oldversion < 2019121600) {

        $table = new xmldb_table('local_ned_controller_notify');

        $table->add_field('id', XMLDB_TYPE_INTEGER, '11', null, XMLDB_NOTNULL, XMLDB_SEQUENCE, null);
        $table->add_field('name', XMLDB_TYPE_CHAR, '255', null, null, null, null);
        $table->add_field('weight', XMLDB_TYPE_INTEGER, '3', null, null, null, 0);
        $table->add_field('disabled', XMLDB_TYPE_INTEGER, '1', null, null, null, 0);
        $table->add_field('notification_type', XMLDB_TYPE_INTEGER, '2', null, null, null, 0);
        $table->add_field('notification_rule', XMLDB_TYPE_INTEGER, '2', null, null, null, 0);
        $table->add_field('description', XMLDB_TYPE_TEXT, null, null, null, null, null);
        $table->add_field('conditions', XMLDB_TYPE_TEXT, null, null, null, null, null);
        $table->add_field('courselevel', XMLDB_TYPE_INTEGER, '1', null, null, null, 0);
        $table->add_field('foreachcourse', XMLDB_TYPE_INTEGER, '1', null, null, null, 0);
        $table->add_field('message', XMLDB_TYPE_TEXT, null, null, null, null, null);
        $table->add_field('iteration', XMLDB_TYPE_INTEGER, '3', null, null, null, 0);
        $table->add_field('dontshow', XMLDB_TYPE_INTEGER, '1', null, null, null, 1);
        $table->add_field('userids_remove', XMLDB_TYPE_TEXT, null, null, null, null, null);
        $table->add_field('userids_add', XMLDB_TYPE_TEXT, null, null, null, null, null);
        $table->add_field('timeout', XMLDB_TYPE_INTEGER, '11', null, null, null, 0);
        $table->add_field('timemodified', XMLDB_TYPE_INTEGER, '11', null, null, null, 0);

        $table->add_key('id', XMLDB_KEY_PRIMARY, array('id'));

        if (!$dbman->table_exists($table)) {
            $dbman->create_table($table);
        }


        $table = new xmldb_table('local_ned_controller_unotes');

        $table->add_field('id', XMLDB_TYPE_INTEGER, '11', null, XMLDB_NOTNULL, XMLDB_SEQUENCE, null);
        $table->add_field('notifyid', XMLDB_TYPE_INTEGER, '11', null, null, null, null);
        $table->add_field('userid', XMLDB_TYPE_INTEGER, '11', null, null, null, null);
        $table->add_field('courseid', XMLDB_TYPE_INTEGER, '11', null, null, null, 0);
        $table->add_field('iteration', XMLDB_TYPE_INTEGER, '3', null, null, null, 0);
        $table->add_field('dontshow', XMLDB_TYPE_INTEGER, '1', null, null, null, 1);
        $table->add_field('timemodified', XMLDB_TYPE_INTEGER, '11', null, null, null, 0);

        $table->add_key('id', XMLDB_KEY_PRIMARY, array('id'));

        if (!$dbman->table_exists($table)) {
            $dbman->create_table($table);
        }

        upgrade_plugin_savepoint(true, 2019121600, 'local', 'ned_controller');
    }

    if ($oldversion < 2019122100) {
        $table = new xmldb_table('local_ned_controller_notify');
        $field = new xmldb_field('courselevel', XMLDB_TYPE_INTEGER, '1', null, null, null, 0);
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        $field = new xmldb_field('foreachcourse', XMLDB_TYPE_INTEGER, '1', null, null, null, 0);
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        $field = new xmldb_field('timeout', XMLDB_TYPE_INTEGER, '11', null, null, null, 0);
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        $table = new xmldb_table('local_ned_controller_unotes');
        $field = new xmldb_field('courseid', XMLDB_TYPE_INTEGER, '11', null, null, null, 0);
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        $field = new xmldb_field('timemodified', XMLDB_TYPE_INTEGER, '11', null, null, null, 0);
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        upgrade_plugin_savepoint(true, 2019122100, 'local', 'ned_controller');
    }

    if ($oldversion < 2019122400) {
        $table = new xmldb_table('local_ned_controller_notify');
        $field = new xmldb_field('description', XMLDB_TYPE_TEXT, null, null, null, null, null);

        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        upgrade_plugin_savepoint(true, 2019122400, 'local', 'ned_controller');
    }

    if ($oldversion < 2020031000) {
        // Define table local_rosedale_rules to be created.
        $table = new xmldb_table('local_ned_controller_rules');

        // Adding fields to table local_rosedale_rules.
        $field = new xmldb_field('customcss', XMLDB_TYPE_TEXT, null, null, false, false, null);

        if ($dbman->field_exists($table, $field)) {
            $dbman->change_field_default($table, $field);
        }

        upgrade_plugin_savepoint(true, 2020031000, 'local', 'ned_controller');
    }

    if ($oldversion < 2020113000) {
        // Define table local_rosedale_rules to be created.
        $table = new xmldb_table('local_ned_controller_rules');
        $field = new xmldb_field('loggedinredirect', XMLDB_TYPE_TEXT,
            null, null, false, false, null, 'additionalhtmlfooter');
        if (!$dbman->field_exists($table, $field)) {
            $dbman->add_field($table, $field);
        }

        upgrade_plugin_savepoint(true, 2020113000, 'local', 'ned_controller');
    }

    if ($oldversion < 2021020400){
        set_config('redirecttodraft', true, 'local_ned_controller');
        upgrade_plugin_savepoint(true, 2021020400, 'local', 'ned_controller');
    }

    if ($oldversion < 2021033100) {
        $table = new xmldb_table('local_ned_controller_dlnotif');

        // Adding fields to table block_ned_teacher_tools_cued.
        $table->add_field('id', XMLDB_TYPE_INTEGER, '20', null, XMLDB_NOTNULL, XMLDB_SEQUENCE);
        $table->add_field('courseid', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null);
        $table->add_field('groupid', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, 0);
        $table->add_field('cmid', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null);
        $table->add_field('userid', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null);
        $table->add_field('deadline', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, 0);
        $table->add_field('ct_id', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, 0);
        $table->add_field('ot_id', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, 0);

        // Adding keys to table
        $table->add_key('id', XMLDB_KEY_PRIMARY, ['id']);
        $table->add_key('courseid', XMLDB_KEY_FOREIGN, ['courseid'], 'course', ['id']);
        $table->add_key('groupid', XMLDB_KEY_FOREIGN, ['groupid'], 'groups', ['id']);
        $table->add_key('cmid', XMLDB_KEY_FOREIGN, ['cmid'], 'course_modules', ['id']);
        $table->add_key('userid', XMLDB_KEY_FOREIGN, ['userid'], 'user', ['id']);
        $table->add_key('ct_id', XMLDB_KEY_FOREIGN, ['ct_id'], 'user', ['id']);
        $table->add_key('ot_id', XMLDB_KEY_FOREIGN, ['ot_id'], 'user', ['id']);

        // Adding index to table
        $table->add_index('uniqueattempt', XMLDB_INDEX_UNIQUE, ['cmid', 'userid']);

        // Conditionally launch create table
        if (!$dbman->table_exists($table)) {
            $dbman->create_table($table);
        }

        local_ned_controller\task\disposable_adhoc_task::add_new_job('',
            'local_ned_controller\task\deadline_notification_update', 'do_job');

        upgrade_plugin_savepoint(true, 2021033100, 'local', 'ned_controller');
    }

    if ($oldversion < 2021072200) {
        // Define table local_ned_controller_grades to be created.
        $table = new xmldb_table('local_ned_controller_grades');

        // Adding fields to table local_ned_controller_grades.
        $table->add_field('id', XMLDB_TYPE_INTEGER, '20', null, XMLDB_NOTNULL, XMLDB_SEQUENCE, null);
        $table->add_field('courseid', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, null);
        $table->add_field('cmid', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, null);
        $table->add_field('userid', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, null);
        $table->add_field('graderid', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, '0');
        $table->add_field('cm_type', XMLDB_TYPE_INTEGER, '3', null, XMLDB_NOTNULL, null, '0');
        $table->add_field('grade_type', XMLDB_TYPE_INTEGER, '3', null, XMLDB_NOTNULL, null, '0');
        $table->add_field('grade_change', XMLDB_TYPE_NUMBER, '10', null, null, null, null);
        $table->add_field('reason', XMLDB_TYPE_INTEGER, '3', null, XMLDB_NOTNULL, null, '0');
        $table->add_field('relatedid', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, '0');
        $table->add_field('deadline', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, '0');
        $table->add_field('note', XMLDB_TYPE_TEXT, null, null, null, null, null);
        $table->add_field('status', XMLDB_TYPE_INTEGER, '3', null, XMLDB_NOTNULL, null, '0');
        $table->add_field('authorid', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, null);
        $table->add_field('timecreated', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, null);
        $table->add_field('timemodified', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, null);

        // Adding keys to table local_ned_controller_grades.
        $table->add_key('id', XMLDB_KEY_PRIMARY, ['id']);
        $table->add_key('courseid', XMLDB_KEY_FOREIGN, ['courseid'], 'course', ['id']);
        $table->add_key('cmid', XMLDB_KEY_FOREIGN, ['cmid'], 'course_modules', ['id']);
        $table->add_key('userid', XMLDB_KEY_FOREIGN, ['userid'], 'user', ['id']);
        $table->add_key('graderid', XMLDB_KEY_FOREIGN, ['graderid'], 'user', ['id']);
        $table->add_key('authorid', XMLDB_KEY_FOREIGN, ['authorid'], 'user', ['id']);

        // Adding indexes to table local_ned_controller_grades.
        $table->add_index('uniquesubmission', XMLDB_INDEX_UNIQUE, ['cmid', 'userid']);

        // Conditionally launch create table for local_ned_controller_grades.
        if (!$dbman->table_exists($table)) {
            $dbman->create_table($table);
        }

        // Ned_controller savepoint reached.
        upgrade_plugin_savepoint(true, 2021072200, 'local', 'ned_controller');
    }

    if ($oldversion < 2021090700){
        // Set new config defaults
        $config = get_config('local_ned_controller');
        $new_config = [
            'cron_ngc_enable' => 1,
            'cron_ngc_time' => 15,
            'cron_ngc_delay' => 2,
            'cron_ngc_update_enable' => 1,
            'cron_ngc_missed_deadlines_enable' => 1,
            'cron_ngc_late_submissions_enable' => 1,
        ];
        foreach ($new_config as $name => $value){
            if (!property_exists($config, $name)){
                set_config($name, $value, 'local_ned_controller');
            }
        }

        // update NGC statuses
        $NGC = NED::$ned_grade_controller;
        $records = $NGC::get_records(['status' => $NGC::ST_WAIT]);
        $changed_id = [];
        foreach ($records as $record){
            if ($NGC::apply_record($record)){
                $changed_id[] = $record->id;
            }
        }
        if (!empty($changed_id)){
            $NGC::update_records_value('status', $NGC::ST_DONE, ['id' => $changed_id]);
        }

        // Ned_controller savepoint reached.
        upgrade_plugin_savepoint(true, 2021090700, 'local', 'ned_controller');
    }

    if ($oldversion < 2021091500){
        local_ned_controller\task\disposable_adhoc_task::add_new_job('local_ned_controller_remove_unavailable_ngc_records',
            '', '', [], 0, null,
            __DIR__ . '/upgradelib.php'
        );
        // Ned_controller savepoint reached.
        upgrade_plugin_savepoint(true, 2021091500, 'local', 'ned_controller');
    }

    if ($oldversion < 2021101200){
        // Add new config defaults
        $config = get_config('local_ned_controller');
        $new_config = [
            'cron_ngc_fixed_submissions_enable' => 1,
        ];
        foreach ($new_config as $name => $value){
            if (!property_exists($config, $name)){
                set_config($name, $value, 'local_ned_controller');
            }
        }

        // Ned_controller savepoint reached.
        upgrade_plugin_savepoint(true, 2021101200, 'local', 'ned_controller');
    }

    if ($oldversion < 2021101800){
        local_ned_controller\task\disposable_adhoc_task::add_new_job('local_ned_controller_update_previous_ngc_wrong_files_records',
            '', '', [], 0, null,
            __DIR__ . '/upgradelib.php'
        );
        // Ned_controller savepoint reached.
        upgrade_plugin_savepoint(true, 2021101800, 'local', 'ned_controller');
    }

    if ($oldversion < 2021121100){
        local_ned_controller\task\disposable_adhoc_task::add_new_job('local_ned_controller_set_ngc_resubmissions_obsoleted',
            '', '', [], 0, null,
            __DIR__ . '/upgradelib.php'
        );
        // Ned_controller savepoint reached.
        upgrade_plugin_savepoint(true, 2021121100, 'local', 'ned_controller');
    }

    if ($oldversion < 2022120300){
        $table = new xmldb_table('local_ned_controller_notify');
        if ($dbman->table_exists($table)){
            $field = new xmldb_field('redirect_url', XMLDB_TYPE_TEXT,
                null, null, null, null, null, 'message');
            if (!$dbman->field_exists($table, $field)){
                $dbman->add_field($table, $field);
            }

            $index = new xmldb_index('disabled', XMLDB_INDEX_NOTUNIQUE, ['disabled']);
            if (!$dbman->index_exists($table, $index)){
                $dbman->add_index($table, $index);

            }
        }

        $table = new xmldb_table('local_ned_controller_unotes');
        if ($dbman->table_exists($table)){
            $foreign_keys = [
                'notifyid' => 'local_ned_controller_notify',
                'userid' => 'user',
                'courseid' => 'courseid',
            ];
            foreach ($foreign_keys as $key => $ref_table){
                $xmldb_key = new xmldb_key($key, XMLDB_KEY_FOREIGN, [$key], $ref_table, ['id']);
                $dbman->add_key($table, $xmldb_key);
            }
        }

        // Ned_controller savepoint reached.
        upgrade_plugin_savepoint(true, 2022120300, 'local', 'ned_controller');
    }

    if ($oldversion < 2023012700){
        // Set new config defaults
        $config = get_config('local_ned_controller');
        $new_config = [
            'ned_school_year_start' => make_timestamp(2022, 9, 1),
        ];
        foreach ($new_config as $name => $value){
            if (!property_exists($config, $name)){
                set_config($name, $value, 'local_ned_controller');
            }
        }

        // Ned_controller savepoint reached.
        upgrade_plugin_savepoint(true, 2023012700, 'local', 'ned_controller');
    }

    if ($oldversion < 2023031000){
        $table = new xmldb_table('local_ned_controller_notify');
        if ($dbman->table_exists($table)){
            $old_field = new xmldb_field('redirect_url', XMLDB_TYPE_TEXT,
                null, null, null, null, null, 'message');
            if ($dbman->field_exists($table, $old_field)){
                $dbman->rename_field($table, $old_field, 'redirect_data');
            }

            $field = new xmldb_field('redirect_option', XMLDB_TYPE_INTEGER, '2', null, true, null, 0, 'message');
            if (!$dbman->field_exists($table, $field)){
                $dbman->add_field($table, $field);

                $DB->set_field_select('local_ned_controller_notify', 'redirect_option',
                    \local_ned_controller\ned_notifications::REDIRECT_PAGE, 'redirect_data IS NOT NULL');
            }
        }

        // Ned_controller savepoint reached.
        upgrade_plugin_savepoint(true, 2023031000, 'local', 'ned_controller');
    }

    if ($oldversion < 2023052600){
        $ngc_table = NED::$ned_grade_controller::TABLE;
        $table = new xmldb_table($ngc_table);
        if ($dbman->table_exists($table)){
            $field = new xmldb_field('authorid', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, 0, 'status');
            if ($dbman->field_exists($table, $field)){
                // drop index to change field default
                $index = new xmldb_index('authorid', XMLDB_INDEX_NOTUNIQUE, ['authorid']);
                if ($dbman->index_exists($table, $index)){
                    $dbman->drop_index($table, $index);
                }

                $dbman->change_field_default($table, $field);
            } else {
                $dbman->add_field($table, $field);
            }
            // update index
            $key = new xmldb_key('authorid', XMLDB_KEY_FOREIGN, ['authorid'], 'user', ['id']);
            $dbman->add_key($table, $key);

            $field = new xmldb_field('editorid', XMLDB_TYPE_INTEGER, '10', null, XMLDB_NOTNULL, null, 0, 'authorid');
            if (!$dbman->field_exists($table, $field)){
                $dbman->add_field($table, $field);

                $key = new xmldb_key('editorid', XMLDB_KEY_FOREIGN, ['editorid'], 'user', ['id']);
                $dbman->add_key($table, $key);

                $DB->execute("UPDATE {{$ngc_table}} SET `editorid` = `authorid`");
            }
        }

        // Ned_controller savepoint reached.
        upgrade_plugin_savepoint(true, 2023052600, 'local', 'ned_controller');
    }
}

function local_ned_controller_remove_unavailable_ngc_records(){
    $NGC = NED::$ned_grade_controller;
    $records = $NGC::get_records(['reason' => $NGC::REASON_SUBMISSION]);
    $to_del = [];
    $deleted = 0;
    foreach ($records as $record){
        if (!NED::get_cm_visibility_by_user($record->cmid, $record->userid, false, false)){
            $to_del[] =  $record;
        }
    }

    if (!empty($to_del)){
        $deleted = $NGC::check_and_delete($to_del, false);
        if ($deleted){
            mtrace("Removed $deleted record(s) from the Grade Penalty Controller.");
        }
    }
    if (!$deleted){
        mtrace("There are no removed records from the Grade Penalty Controller.");
    }
}

function local_ned_controller_update_previous_ngc_wrong_files_records(){
    mtrace('local_ned_controller_update_previous_ngc_wrong_files_records start');
    $NGC = NED::$ned_grade_controller;
    $T_CHANGED = 1634067240;
    $records = $NGC::get_records(['reason' => $NGC::REASON_FILE, 'grade_type' => $NGC::GT_AWARD_ZERO, 'status' => $NGC::ST_DONE]);
    $c = count($records);
    mtrace("There are $c records to check");
    $i = 0;
    foreach ($records as $record){
        if ($record->timemodified > $T_CHANGED){
            continue;
        }

        $i++;
        if (!($i % 10)){
            mtrace(" * $i updated...");
        }
        $record->status = $NGC::ST_WAIT;
        $NGC::apply_record($record);
    }
    mtrace("There are $i from $c records have been updated");
    mtrace('local_ned_controller_update_previous_ngc_wrong_files_records end');
}

function local_ned_controller_set_ngc_resubmissions_obsoleted(){
    mtrace('local_ned_controller_set_ngc_resubmissions_obsoleted start');
    $NGC = NED::$ned_grade_controller;
    $t = $NGC::TABLE;
    $assign_key = NED::MOD_ASSIGN_KEY;
    $st_done = $NGC::ST_DONE;
    $sql = "
        SELECT ngc.*
        FROM {{$t}} ngc
        JOIN {course_modules} cm
          ON cm.id = ngc.cmid
          AND ngc.cm_type = $assign_key
        JOIN {assign} a
          ON a.id = cm.instance
        JOIN {assign_submission} a_s
          ON a_s.assignment = a.id 
          AND a_s.userid = ngc.userid
          AND a_s.attemptnumber > 0
          AND a_s.timemodified > ngc.timecreated
        WHERE ngc.status = $st_done 
    ";
    $ngc_records = NED::db()->get_records_sql($sql);
    $count = $NGC::check_and_change_obsolete_state($ngc_records, true, false);

    mtrace("There are $count records have been updated");
    mtrace('local_ned_controller_set_ngc_resubmissions_obsoleted end');
}
