<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Plugins overview
 *
 * @package     local_package
 * @category    string
 * @copyright   2018 Michael Gardener <mgardener@cissq.com>
 * @license     http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

use local_ned_controller\shared_lib as NED;

require_once(dirname(dirname(dirname(__FILE__))) . '/config.php');
require_once(__DIR__.'/lib.php');

$check = optional_param('check', 0, PARAM_INT);

// Security.
$context = context_system::instance();
require_login();
require_capability('moodle/site:config', $context);

$html_status_img = function($type, $text) use ($OUTPUT){
    return html_writer::div(
        html_writer::img(
            $OUTPUT->image_url($type, NED::$PLUGIN_NAME), NED::str($text),
            ['width' => '24']
        ),
        'license-status'
    );
};

// Page boilerplate stuff.
$url = NED::url('~/index.php');
$PAGE->set_context($context);
$PAGE->set_url($url);
$PAGE->set_pagelayout('admin');
$title = NED::str('pluginsoverview');
$PAGE->set_title($title);
$PAGE->set_heading($title);

if ($check || $DB->record_exists_select(NED::$PLUGIN_NAME, "version =''")) {
    local_ned_controller_update_plugin_info();
}

// LICENSE TABLE.
$licensetable = new html_table();

$line = new html_table_row();
$line->cells[] = $html_status_img('transparent', 'valid') . NED::str('teacher');
$line->cells[] = 4;
$line->cells[] = NED::str('forum');
$line->cells[] = '';
$data[] = $line;

$line = new html_table_row();
$line->cells[] = $html_status_img('transparent', 'invalid') . NED::str('department');
$line->cells[] = 10;
$line->cells[] = NED::str('forum') . ' + ' . NED::str('email');
$line->cells[] = '';
$data[] = $line;

$line = new html_table_row();
$line->attributes = array('class' => 'active');
$line->cells[] = $html_status_img('valid', 'invalid') . NED::str('school');
$line->cells[] = NED::str('unlimited');
$line->cells[] = NED::str('forum') .
    html_writer::span(' + ' . NED::str('email'), 'dimmed_text').
    html_writer::span(' + ' . NED::str('qrticket'), 'dimmed_text');
$line->cells[] = '10 January 2020';
$data[] = $line;

$licensetable->head  = array(
    NED::str('license'),
    NED::str('courseinstances'),
    NED::str('support'),
    NED::str('expires'),
);
$licensetable->align = array('left', 'left');
$licensetable->attributes['class'] = 'licensetable generaltable';
$licensetable->data  = $data;

$ned_settings_block = '';
// Plugin settings
$ned_settings_block .= NED::link(local_ned_controller_get_setting_local_urlpath(NED::$PLUGIN),
    NED::fa('fa-cog').html_writer::span(NED::str('controllersettings')),
    'settings-link'
);
// Theme customization
$ned_settings_block .= NED::link('~/rules_edit.php', 'managerules', 'btn btn-primary');
// Pop-up notifications
$ned_settings_block .= NED::link('~/notifications_edit.php', 'popupnotifications', 'btn btn-primary');
// Grade Controller
$ned_settings_block .= NED::link(NED::PAGE_GRADE_CONTROLLER, 'gradecontroller', 'btn btn-primary');
// DB Benchmark
$ned_settings_block .= NED::link('~/db_benchmark.php', 'db_benchmark', 'btn btn-primary');

$top_table_links = '';
// Check for updates link.
$checkforupdates = clone $url;
$checkforupdates->param('check', 1);
$top_table_links .= NED::link($checkforupdates, 'checkforupdates');

// OVERVIEW TABLE.
$table = new html_table();

$strnotinstalled = NED::str('notinstalled');
$strupdateavailable = NED::str('updateavailable');
$strlatestversion = NED::str('latestversion');

if ($plugins = $DB->get_records(NED::$PLUGIN_NAME, null, 'installed DESC, name ASC')) {
    $data = array();
    foreach ($plugins as $plugin) {
        $component = explode('-', $plugin->repository);
        if ($component[1] == 'block_custom_menu') {
            $component[1] = 'block_ned_custom_menu';
        }
        if ($plugininfo = core_plugin_manager::instance()->get_plugin_info($component[1])) {
            $clsrow = 'installed';
            $version = $plugininfo->versiondb;
        } else {
            $clsrow = 'notinstalled';
            $version = $strnotinstalled;
        }

        $line = new html_table_row();
        $line->attributes = array('class' => $clsrow);
        $line->cells[] = $plugin->name;
        $line->cells[] = NED::str($plugin->plugintype);
        $line->cells[] = local_ned_controller_count_instances($plugin);
        $line->cells[] = local_ned_controller_count_course_instances($plugin);

        // Documentation page link.
        if ($documentationurl = local_ned_controller_get_documentation_url($plugin)) {
            $documentation = NED::link($documentationurl, 'info', '', ['target' => '_blank']);
        } else {
            $documentation = '-';
        }
        $line->cells[] = $documentation;

        // Setting page link.
        if ((!$plugininfo || $plugininfo->name != NED::$PLUGIN) && $settingurl = local_ned_controller_get_setting_url($plugin)) {
            $setting = NED::link($settingurl, 'open', '', ['target' => '_blank']);
        } else {
            $setting = '-';
        }
        $line->cells[] = $setting;

        $line->cells[] =  $version;

        if ($clsrow == 'notinstalled') {
            $line->cells[] = '';
        } else if ($plugin->version > $version) {
            $line->cells[] =  html_writer::div($strupdateavailable, 'label label-danger label-important');
        } else {
            $line->cells[] =  html_writer::div($strlatestversion, 'label label-success');
        }

        $data[] = $line;

    }
    $table->head  = NED::str_arr([
        'plugin_name',
        'type',
        'instances',
        'courseinstances',
        'info',
        'settings',
        'version',
        '',
    ]);
    $table->align = ['left', 'left', 'center','center', 'center', 'left'];
    $table->attributes['class'] = 'generaltable';
    $table->data  = $data;
}

$o = $OUTPUT->header();
$o .= $OUTPUT->heading(NED::str('license'), 3, 'main');
$o .= html_writer::start_div('d-flex justify-content-between table-buttons-wrapper');
$o .= html_writer::table($licensetable);
$o .= html_writer::div($ned_settings_block, 'ned-controller-settings');
$o .= html_writer::end_div();
$o .= $OUTPUT->heading($title, 3, 'main float-left');
$o .= html_writer::div($top_table_links, 'top_table_links');
$o .= html_writer::table($table);
$o .= $OUTPUT->footer();

echo $o;
